/*  $Id: CircularPointBuffer.h,v 1.1 2012/07/08 09:35:52 sarrazip Exp $
    CircularPointBuffer.h - Fixed-size circular buffer containing 2D positions.

    quadrupleback - A video game where intruders must be circled.
    Copyright (C) 2012 Pierre Sarrazin <http://sarrazip.com/>

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
    02110-1301, USA.
*/

#ifndef _H_CircularPointBuffer
#define _H_CircularPointBuffer

#include "Intersection.h"

#include "util.h"


class CircularPointBuffer
{
public:
    CircularPointBuffer(size_t _capacity)
    :   points(),
        capacity(0),
        indexOfFirstElement(0),
        numElements(0),
        intersections()
    {
        reset(_capacity);
    }

    ~CircularPointBuffer()
    {
        deleteVectorElements(intersections);
    }

    void reset(size_t _capacity)
    {
        points.resize(_capacity);
        capacity = _capacity;
        indexOfFirstElement = 0;
        numElements = 0;
        deleteVectorElements(intersections);
    }

    size_t size() const
    {
        return numElements;
    }

    // If buffer is full, oldest element is removed.
    //
    void addPoint(const flatzebra::RCouple &c);

    // inter: relative to game area.
    // indexOfFirstSegment, indexOfSecondSegment: logical indexes.
    // The indexes stored in the added Intersection object will be physical.
    // Returns a reference to the added Intersection.
    //
    Intersection &addIntersection(const flatzebra::RCouple &inter,
                                  size_t indexOfFirstSegment,
                                  size_t indexOfSecondSegment);

    // Finds the loop created by the last call to addIntersection().
    //
    bool findNewestLoop(std::vector<flatzebra::RCouple> &polygon);

    size_t logToPhys(size_t logicalIndex) const
    {
        assert(logicalIndex <= numElements);

        return (indexOfFirstElement + logicalIndex) % capacity;
    }

    size_t physToLog(size_t physicalIndex) const
    {
        assert(physicalIndex <= numElements);

        // "+ capacity" is there to avoid sending a negative number to the modulo operator.
        return (physicalIndex + capacity - indexOfFirstElement) % capacity;
    }

    const flatzebra::RCouple &getPoint(size_t index) const
    {
        assert(index < numElements);
        return points[logToPhys(index)];
    }

    const flatzebra::RCouple &back() const
    {
        return getPoint(numElements - 1);
    }

    const std::vector<Intersection *> &getIntersections() const
    {
        return intersections;
    }

private:

    const Intersection *findIntersectionAtSegment(size_t logicalIndex) const;

    void removeIntersection(size_t indexOfRemovedSegment);

    void eraseVectorElement(std::vector<Intersection *>::iterator it);

private:

    // Forbidden operations:
    CircularPointBuffer(const CircularPointBuffer &);
    CircularPointBuffer &operator = (const CircularPointBuffer &);

private:

    // There are 2 kinds of indexes into 'points':
    // a physical index, which is an index usable with operator[] on 'points'.
    // a logical index, which is dependent on 'indexOfFirstElement', modulo 'capacity'.
    //
    std::vector<flatzebra::RCouple> points;

    size_t capacity;  // points[] is always this size
    size_t indexOfFirstElement;  // physical index of the first logical element
    size_t numElements;

    // Indexes into 'points' stored in an Intersection must be *physical* indexes,
    // because such indexes do not change when a point is added to the circular buffer.
    // See 'points'.
    // The Intersection objects stored in this vector must come from 'new'.
    //
    std::vector<Intersection *> intersections;
};


#endif  /* _H_CircularPointBuffer */
