#include <sys/types.h>
#include <sys/stat.h>
#include <stdio.h>
#include <unistd.h>
#include <dirent.h>
#include <time.h>
#include "qconfirm.h"
#include "qconfirm_conf_get.h"
#include "sgetopt.h"
#include "error.h"
#include "open.h"
#include "lock.h"
#include "strerr.h"
#include "stralloc.h"
#include "direntry.h"
#include "cdb.h"
#include "cdb_make.h"
#include "scan.h"
#include "str.h"
#include "fmt.h"
#include "env.h"
#include "buffer.h"
#include "tai.h"
#include "uint32.h"

#define USAGE " [-vrf] [-c cdb] [-t sec] [-m min] [-l 1..1000] [-s state]"
#define VERSION "$Id: qconfirm-cdb-update.c,v 1.7 2003/11/21 13:45:53 pape Exp $"
#define FATAL "qconfirm-cdb-update: fatal: "
#define WARNING "qconfirm-cdb-update: warning: "
#define INFO "qconfirm-cdb-update: info: "

const char *progname;

void usage() { strerr_die4x(111, "usage: ", progname, USAGE, "\n"); }
void die_nomem() { strerr_die2x(111, FATAL, "out of memory."); }
void info(char *m0, char *m1) {
  buffer_puts(buffer_1, INFO);
  buffer_puts(buffer_1, m0);
  buffer_puts(buffer_1, m1);
  buffer_putsflush(buffer_1, "\n");
}
void warn(char *m0, char *m1) {
  strerr_warn4(WARNING, m0, m1, ": ", &strerr_sys);
}
void fatal(char *m0, char *m1) { strerr_die4sys(100, FATAL, m0, m1, ": "); }

char *qconfirm_dir;
char *cdbfn =0;
char *state ="ok";
unsigned long min =100;
unsigned long limit =300;
unsigned long timeout =15724786;
unsigned int rem =0;
unsigned int skip =1;
unsigned int verbose =0;
stralloc tmp ={0};
stralloc todo[1000];
int todolen;
int rmlen =0;
stralloc sa ={0};
struct cdb_make c;
struct cdb cdb;
int fdcdb;
int fdtmp;

void get(buffer *b, char *buf, unsigned int l) {
  int r;

  while (l > 0) {
    r =buffer_get(b, buf, l);
    if (r == -1) fatal("unable to read: ", "data.cdb");
    if (r == 0)
      strerr_die2x(111, FATAL, "unable to read: data.cdb: truncated file");
    buf +=r;
    l -=r;
  }
}

int main(int argc, const char **argv) {
  int opt;
  int i;
  int mydir;
  DIR *dir;
  direntry *d;
  struct stat st;
  time_t now;
  mode_t mode;
  buffer bcdb;
  char bcdbspace[4096];
  struct tai ts;
  char tspack[8];
  char uintbuf[4];
  uint32 pos =0;
  uint32 eod;
  uint32 klen;
  uint32 dlen;
  char numbuf[FMT_ULONG];

  progname =*argv;

  qconfirm_dir =env_get("QCONFIRM_DIR");
  cdbfn =env_get("QCONFIRM_OKCDB");

  while ((opt =getopt(argc, argv, "Vc:s:m:l:t:rfv")) != opteof) {
    switch(opt) {
    case 'c': cdbfn =(char*) optarg; break;
    case 's': state =(char*) optarg; break;
    case 'm': scan_ulong(optarg, &min); break;
    case 'l': scan_ulong(optarg, &limit); break;
    case 't': scan_ulong(optarg, &timeout); break;
    case 'r': rem =1; break;
    case 'f': skip =0; break;
    case 'v': verbose =1; break;
    case 'V': strerr_warn1(VERSION, 0);
    case '?': usage();
    }
  }
  argv +=optind;
  umask(077);

  if (limit > 1000) usage();

  if (! qconfirm_dir) qconfirm_dir =QCONFIRMDIR;
  if (! cdbfn)
    if (! (cdbfn =conf_get_dflt(qconfirm_dir, "QCONFIRM_OKCDB", "ok.cdb")))
      fatal("unable to read config: ", "QCONFIRM_OKCDB");

  for (i =0; i < 1000; ++i) todo[i].s =0;

  if (chdir(qconfirm_dir) == -1)
    fatal("unable to change directory: ", qconfirm_dir);
  if ((mydir =open_read(".")) == -1)
    fatal("unable to open directory: ", qconfirm_dir);

  if (stat(cdbfn, &st) == -1) {
    if (errno != error_noent) fatal("unable to stat: ", cdbfn);
    if (verbose) warn("unable to stat: ", cdbfn);
    mode =S_IRUSR | S_IWUSR;
  }
  else
    mode =st.st_mode;

  /* open cdb */
  fdcdb =open_read(cdbfn);
  if (fdcdb != -1) cdb_init(&cdb, fdcdb);
  /* scan dir */
  if (chdir(state) == -1) fatal("unable to change directory: ", state);
  if (! (dir =opendir("."))) fatal("unable to open directory: ", state);
  now =time(NULL);
  errno =0;
  todolen =0;
  while ((d =readdir(dir))) {
    if (d->d_name[0] == '.') continue;
    if (stat(d->d_name, &st) == -1) {
      warn("unable to stat: ", d->d_name);
      errno =0;
      continue;
    }
    if (st.st_size != 0) continue;
    if ((st.st_mode & S_IWUSR) == 0) continue;
    if (skip && ((now -st.st_mtime) < 86400)) {
      if (verbose) info("skip: ", d->d_name);
      continue;
    }
    if (fdcdb != -1)
      if (cdb_find(&cdb, d->d_name, str_len(d->d_name))) {
	if (rem) if (unlink(d->d_name) == -1)
	  warn("unable to delete: ", d->d_name);
	if (verbose) info("known: ", d->d_name);
	continue;
      }
    if (! stralloc_copys(&todo[todolen], d->d_name)) die_nomem();
    if (++todolen >= limit) break;
  }
  if (errno) fatal("unable to read directory: ", state);
  if (todolen < min) {
    closedir(dir);
    info(state, ": nothing to do.");
    _exit(0);
  }
  closedir(dir);

  /* update cdb */
  if (fchdir(mydir) == -1) fatal("unable to change dir: ", qconfirm_dir);
  tai_now(&ts);
  tai_pack(tspack, &ts);
  /* open data.tmp */
  if (! stralloc_copys(&tmp, cdbfn)) die_nomem();
  if (! stralloc_cats(&tmp, ".tmp")) die_nomem();
  if (! stralloc_0(&tmp)) die_nomem();
  if ((fdtmp =open_trunc(tmp.s)) == -1) fatal("unable to create: ", tmp.s);
  if (cdb_make_start(&c, fdtmp) == -1) fatal("unable to create: ", tmp.s);
  if (fdcdb != -1) {
    /* lock data.cdb */
    if (lock_exnb(fdcdb) == -1)
      if (verbose) warn("lock is busy, waiting...: ", cdbfn);
    if (lock_ex(fdcdb) == -1) fatal("unable to lock: ", cdbfn);
    buffer_init(&bcdb, buffer_unixread, fdcdb, bcdbspace, sizeof bcdbspace);
    /* size of database */
    get(&bcdb, uintbuf, 4);
    pos +=4;
    uint32_unpack(uintbuf, &eod);
    while (pos < 2048) {
      get(&bcdb, uintbuf, 4);
      pos +=4;
    }
    while (pos < eod) {
      get(&bcdb, uintbuf, 4);
      pos +=4;
      uint32_unpack(uintbuf, &klen);
      get(&bcdb, uintbuf, 4);
      pos +=4;
      uint32_unpack(uintbuf, &dlen);
      if (! stralloc_ready(&sa, klen +dlen)) die_nomem();
      get(&bcdb, sa.s, klen +dlen);
      pos +=klen +dlen;
      if (timeout > 0) {
	struct tai t;
	
	tai_unpack(sa.s +klen, &t);
	t.x +=timeout;
	if (tai_less(&t, &ts)) {
	  sa.s[klen] =0;
	  ++rmlen;
	  if (verbose) info("remove: ", sa.s);
	  continue;
	}
      }
      if (cdb_make_add(&c, sa.s, klen, sa.s +klen, dlen) == -1)
	fatal("unable to write: ", "data.tmp");
    }
  }
  for (i =0; i < todolen; ++i) {
    /* add */
    if (cdb_make_add(&c, todo[i].s, todo[i].len, tspack, 8) == -1)
      fatal("unable to add to: ", tmp.s);
    if (! stralloc_0(&todo[i])) die_nomem();
    if (verbose) info("add: ", todo[i].s);
  }

  if (cdb_make_finish(&c) == -1) fatal("unable to write cdb: ", tmp.s);
  if (fsync(fdtmp) == -1) fatal("unable to write cdb: ", tmp.s);
  close(fdtmp);

  if (fchdir(mydir) == -1)
    fatal("unable to change directory: ", qconfirm_dir);

  if (rename(tmp.s, cdbfn) == -1) fatal("unable to replace: ", cdbfn);
  if (chmod(cdbfn, mode) == -1) fatal("unable to chmod: ", cdbfn);

  buffer_puts(buffer_1, INFO);
  buffer_puts(buffer_1, "update: ");
  buffer_puts(buffer_1, cdbfn);
  buffer_puts(buffer_1, ": add ");
  buffer_put(buffer_1, numbuf, fmt_ulong(numbuf, todolen));
  buffer_puts(buffer_1, " remove ");
  buffer_put(buffer_1, numbuf, fmt_ulong(numbuf, rmlen));
  buffer_putsflush(buffer_1, "\n");

  /* clean up */
  if (chdir(state) == -1) {
    warn("unable to change dir: ", state);
    _exit(111);
  }
  for (i =0; i < todolen; ++i)
    if (unlink(todo[i].s) == -1) warn("unable to delete: ", todo[i].s);
  if (fchdir(mydir) == -1) {
    warn("unable to change dir: ", qconfirm_dir);
    _exit(111);
  }
  return(0);
}
