//
//  PLP - An implementation of the PSION link protocol
//
//  Copyright (C) 1999  Philip Proudman
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
//  e-mail philip.proudman@btinternet.com

#include <stream.h>
#include <stdlib.h>
#include <fstream.h>
#include <iomanip.h>
#include <time.h>
#include <string.h>

#include "bool.h"
#include "bool.h"
#include "rfsv32.h"
#include "bufferstore.h"
#include "ppsocket.h"
#include "../defaults.h"
#include "bufferarray.h"

rfsv32::rfsv32(ppsocket *_skt) :
  serNum(0)
{
  skt = _skt;
  bufferStore a;
  a.addStringT( getConnectName() );
  if (skt->sendBufferStore(a)) {
    if (skt->getBufferStore(a) == 1) {
      if (strcmp(a.getString(0), "Ok")) {
        cerr << "Not got ok over socket\n";
	abort();
      }
    }
  }
}

rfsv32::~rfsv32() {
  bufferStore a;
  a.addStringT("Close");
  skt->sendBufferStore(a);
  skt->closeSocket();
}

const char *rfsv32::getConnectName() {
  return "SYS$RFSV.*";
}

long rfsv32::fopen(long attr, const char* file, long &handle) { 
  bufferStore a;
  a.addDWord(attr);
  a.addWord(strlen(file));
  a.addString(file);
  sendCommand(OPEN_FILE, a);
  
  long res = getResponse(a);
  if (!res && a.getLen() == 4) {
    handle = a.getDWord(0);
    return 0;
  }
  return res;
}

long rfsv32::fcreatefile(long attr, const char* file, long &handle) { 
  bufferStore a;
  a.addDWord(attr);
  a.addWord(strlen(file));
  a.addString(file);
  sendCommand(CREATE_FILE, a);
  
  long res = getResponse(a);
  if (!res && a.getLen() == 4) {
    handle = a.getDWord(0);
    return 0;
  }
  return res;
}

long rfsv32::freplacefile(long attr, const char* file, long &handle) { 
  bufferStore a;
  a.addDWord(attr);
  a.addWord(strlen(file));
  a.addString(file);
  sendCommand(REPLACE_FILE, a);
  
  long res = getResponse(a);
  if (!res && a.getLen() == 4) {
    handle = a.getDWord(0);
    return 0;
  }
  return res;
}

long rfsv32::fopendir(long attr, const char* file, long &handle) { 
  bufferStore a;
  a.addDWord(attr);
  a.addWord(strlen(file));
  a.addString(file);
  sendCommand(OPEN_DIR, a);
  
  long res = getResponse(a);
  if (!res && a.getLen() == 4) {
    handle = a.getDWord(0);
    return 0;
  }
  return res;
}

long rfsv32::fclose(long fileHandle) {
  bufferStore a;
  a.addDWord(fileHandle);
  sendCommand(CLOSE_HANDLE, a);
  long res = getResponse(a);
  return res;
}

int rfsv32::dir(const char* dirName, bufferArray *files) {
  long fileHandle;
  char realName[200];
  int rv = convertName(dirName, realName);
  if (rv) return rv;

  long status = fopendir(HIDDEN|SYSTEM|DIRECTORY, realName, fileHandle);
  if (status != 0) {
    opErr(status);
    cerr << endl;
    return status;
  }

  bufferStore a;
  a.addDWord(fileHandle);
  sendCommand(READ_DIR, a);
  long res = getResponse(a);
  if (res == EoF) {
    // Empty directory
    fclose(fileHandle);
    return 0;
  }

  if (!res) {
    while (a.getLen() > 16) {
      long shortLen = a.getDWord(0);
      long attributes = a.getDWord(4);
      long size = a.getDWord(8);
      unsigned long modLow = a.getDWord(12);
      unsigned long modHi = a.getDWord(16);
      long uid1 = a.getDWord(20);
      long uid2 = a.getDWord(24);
      long uid3 = a.getDWord(28);
      long longLen = a.getDWord(32);

      if (!files) {
	cout << ((attributes & READ_ONLY) ? "-" : "w");
	cout << ((attributes & HIDDEN) ? "h" : "-");
	cout << ((attributes & SYSTEM) ? "s" : "-");
	cout << ((attributes & DIRECTORY) ? "d" : "-");
	cout << ((attributes & ARCHIVE) ? "a" : "-");
	cout << ((attributes & VOLUME) ? "v" : "-");
	cout << ((attributes & NORMAL) ? "n" : "-");
	cout << ((attributes & TEMPORARY) ? "t" : "-");
	cout << ((attributes & COMPRESSED) ? "c" : "-");

	cout << " " << dec << setw(10) << setfill(' ') << size;
	
	{
	  unsigned long long us = modHi;
	  us <<= 32;
	  us += modLow;
	  us /= 1000000;
	  us -= (1970*365 + 490)*60*60*24;
	  long date = us;
	  char dateBuff[100];
	  struct tm *t;
	  t = gmtime(&date);
	  strftime(dateBuff, 100, "%d/%m/%y %H:%M:%S", t);
	  cout << " " << dateBuff;
	}
      }
      
      int d = 36;
      bufferStore s;
      for (int i=0; i<longLen; i++, d++) s.addByte(a.getByte(d));
      s.addByte(0);
      while (d%4) d++;
      if (!files) cout << " " << s.getString() << endl;
      else if ((attributes & DIRECTORY) == 0) {
	files->pushBuffer(s);
      }
      // Read the short version of the filename if present
      if (shortLen) {
	// Read the short filename
	bufferStore sn;
	for (int i=0; i<shortLen; i++, d++) sn.addByte(a.getByte(d));
      }
      while (d%4) d++;
      a.discardFirstBytes(d);
    }
  }
  else {
    cerr << "Unknown response to dir - ";
    opErr(res);
    cerr << " " << a<<endl;
    fclose(fileHandle);
    return 1;
  }
  
  fclose(fileHandle);
  
  return 0;
}

bool rfsv32::sendCommand(enum commands cc, bufferStore &data) {
  bufferStore a;
  a.addWord(cc);
  a.addWord(serNum);
  if (serNum < 0xffff)
    serNum++;
  else
    serNum=0;
  a.addBuff(data);
  return skt->sendBufferStore(a);
}

long rfsv32::getResponse(bufferStore &data) {
  if (skt->getBufferStore(data) == 1 &&
      data.getWord(0) == 0x11) {
    long status = data.getDWord(4);
    data.discardFirstBytes(8);
    return status;
  }
  return (enum errs)GENERAL;
}

void rfsv32::opErr(long status) {
  enum errs e = (enum errs)status;
  switch (e) {
  case NONE:
    return;
  case NOT_FOUND:
    cerr << "NOT_FOUND";
    break;
  case GENERAL:
    cerr << "GENERAL";
    break;
  case CANCEL:
    cerr << "CANCEL";
    break;
  case NO_MEMORY:
    cerr << "NO_MEMORY";
    break;
  case NOT_SUPPORTED:
    cerr << "NOT_SUPPORTED";
    break;
  case ARGUMENT:
    cerr << "ARGUMENT";
    break;
  case TOTAL_LOSS_OF_PRECISION:
    cerr << "TOTAL_LOSS_OF_PRECISION";
    break;
  case BAD_HANDLE:
    cerr << "BAD_HANDLE";
    break;
  case OVERFLOW:
    cerr << "OVERFLOW";
    break;
  case UNDERFLOW:
    cerr << "UNDERFLOW";
    break;
  case ALREADY_EXISTS:
    cerr << "ALREADY_EXISTS";
    break;
  case PATH_NOT_FOUND:
    cerr << "PATH_NOT_FOUND";
    break;
  case DIED:
    cerr << "DIED";
    break;
  case IN_USE:
    cerr << "IN_USE";
    break;
  case SERVER_TERMINATED:
    cerr << "SERVER_TERMINATED";
    break;
  case SERVER_BUSY:
    cerr << "SERVER_BUSY";
    break;
  case COMPLETION:
    cerr << "COMPLETION";
    break;
  case NOT_READY:
    cerr << "NOT_READY";
    break;
  case UNKNOWN:
    cerr << "UNKNOWN";
    break;
  case CORRUPT:
    cerr << "CORRUPT";
    break;
  case ACCESS_DENIED:
    cerr << "ACCESS_DENIED";
    break;
  case LOCKED:
    cerr << "LOCKED";
    break;
  case WRITE:
    cerr << "WRITE";
    break;
  case DISMOUNTED:
    cerr << "DISMOUNTED";
    break;
  case EoF:
    cerr << "EOF";
    break;
  case DISK_FULL:
    cerr << "DISK_FULL";
    break;
  case BAD_DRIVER:
    cerr << "BAD_DRIVER";
    break;
  case BAD_NAME:
    cerr << "BAD_NAME";
    break;
  case COMMS_LINE_FAIL:
    cerr << "COMMS_LINE_FAIL";
    break;
  case COMMS_FRAME:
    cerr << "COMMS_FRAME";
    break;
  case COMMS_OVERRUN:
    cerr << "COMMS_OVERRUN";
    break;
  case COMMS_PARITY:
    cerr << "COMMS_PARITY";
    break;
  case TIMEOUT:
    cerr << "TIMEOUT";
    break;
  case COULD_NOT_CONNECT:
    cerr << "COULD_NOT_CONNECT";
    break;
  case COULD_NOT_DISCONNECT:
    cerr << "COULD_NOT_DISCONNECT";
    break;
  case DISCONNECTED:
    cerr << "DISCONNECTED";
    break;
  case BAD_LIBRARY_ENTRY_POINT:
    cerr << "BAD_LIBRARY_ENTRY_POINT";
    break;
  case BAD_DESCRIPTOR:
    cerr << "BAD_DESCRIPTOR";
    break;
  case ABORT:
    cerr << "ABORT";
    break;
  case TOO_BIG:
    cerr << "TOO_BIG";
    break;
  case DIVIDE_BY_ZERO:
    cerr << "DIVIDE_BY_ZERO";
    break;
  case BAD_POWER:
    cerr << "BAD_POWER";
    break;
  case DIR_FULL:
    cerr << "DIR_FULL";
    break;
  default:
    cerr << "Unknown error";
    break;
  }
}

int rfsv32::read(const char* psionName, const char* localName) {
  long fileHandle;
  char realName[200];
  int rv = convertName(psionName, realName);
  if (rv) return rv;

  long status = fopen(SHARE_READERS|BINARY, realName, fileHandle);
  if (status != 0) {
    opErr(status);
    cerr << endl;
    return status;
  }

  ofstream op(localName);
  if (!op) {
    cerr << "Could not open UNIX file " << localName <<endl;
    fclose(fileHandle);
    return 1;
  }
    
  do {
    bufferStore a;
    a.addDWord(fileHandle);
    a.addDWord(2000);
    sendCommand(READ_FILE, a);
      
    long res = getResponse(a);
    if (!res) {
      if (a.getLen() == 0) break;
      op.write( a.getString(), a.getLen() );
    }
    else {
      cerr << "Unknown response to fread - ";
      opErr(res);
      cerr << " " <<a<<endl;
      fclose(fileHandle);
      return 1;
    }
  } while (true);
  
  fclose(fileHandle);
  op.close();
  return 0;
}

int rfsv32::write(const char* localName, const char* psionName) {
  long fileHandle;
  char realName[200];
  int rv = convertName(psionName, realName);
  if (rv) return rv;

  long status = fcreatefile(BINARY|SHARE_EXCLUSIVE, realName, fileHandle);
  if (status != 0) {
    status = freplacefile(BINARY|SHARE_EXCLUSIVE, realName, fileHandle);
    if (status != 0) {
      opErr(status);
      cerr << endl;
      return status;
    }
  }
  
  ifstream ip(localName);
  if (!ip) {
    cerr << "Could not open UNIX file " << localName << endl;
    fclose(fileHandle);
    return 1;
  }
  unsigned char * buff = new unsigned char [RFSV_SENDLEN];
  while (ip &&!ip.eof()) {
    ip.read(buff, RFSV_SENDLEN);
    bufferStore tmp(buff, ip.gcount());
    if (tmp.getLen() == 0) break;
    bufferStore a;
    a.addDWord(fileHandle);
    a.addBuff(tmp);
    sendCommand(WRITE_FILE, a);    
    
    long res = getResponse(a);
    if (res) {
      cerr << "Unknown response to fwrite - ";
      opErr(res);
      cerr << " "<<a<<endl;
      fclose(fileHandle);
      return 1;
    }
  }
  delete [] buff;
  
  fclose(fileHandle);
  ip.close();
  return 0;
}

int rfsv32::mkdir(const char* dirName) {
  char realName[200];
  int rv = convertName(dirName, realName);
  if (rv) return rv;
  if (realName[strlen(realName)-1] != '\\') strcat(realName, "\\");
  bufferStore a;
  a.addWord(strlen(realName));
  a.addString(realName);
  sendCommand(MK_DIR_ALL, a);
  long res = getResponse(a);
  if (!res) {
    // Correct response
    return 0;
  }
  cerr << "Unknown response from mkdir - ";
  opErr(res);
  cerr << " "<< a <<endl;
  return 1;
}

int rfsv32::del(const char* psionName) {
  char realName[200];
  int rv = convertName(psionName, realName);
  if (rv) return rv;
  bufferStore a;
  a.addWord(strlen(realName));
  a.addString(realName);
  sendCommand(DELETE, a);
  long res = getResponse(a);
  if (!res) {
    // Correct response
    return 0;
  }
  cerr << "Unknown response from delete - ";
  opErr(res);
  cerr << " " << a <<endl;
  
  return 1;
}

int rfsv32::convertName(const char* orig, char *retVal) {
  int len = strlen(orig);
  char *temp = new char [len+1];

  for (int i=0; i <= len; i++) {
    if (orig[i] == '/')
      temp[i] = '\\';
    else
      temp[i] = orig[i];
  }

  if (len == 0 || temp[1] != ':') {
    // We can automatically supply a drive letter
    strcpy(retVal, DEFAULT_DRIVE);

    if (len == 0 || temp[0] != '\\') {
      strcat(retVal, DEFAULT_BASE_DIRECTORY);
    }
    else {
      retVal[strlen(retVal)-1] = 0;
    }

    strcat(retVal, temp);
  }
  else {
    strcpy(retVal, temp);
  }

  delete [] temp;
  cout << retVal << endl;
  return 0;
}


