
/*
 *   The conversation daemon --- does all the main work 
 *   for one conversation.  It is invoked with stdin(fd 0)
 *   on the service port; it listens for requests there and
 *   does the right thing.
 *
 *   See the comments in ../client/readstream.c for an
 *   explanation of the command encoding scheme used here.
 *
 *   NOTE: this code relies heavily upon the writev() system call
 *   which provides for scatter/gather arrays of data, thus allowing
 *   us to to write out multiple arrays of characters in a single
 *   system call, thus avoiding having to copy data from one buffer
 *   to another.
 *
 *   Also ... note that we don't use slot #0 - the client program
 *   wants to remap the user's window into slot zero, so we help
 *   out by never assigning *anyone* that slot.
 */


/*
 * Log:    convd.c,v 
 * Revision 1.1	 85/10/29  14:20:06  broome
 * Initial revision
 */


#include <sys/types.h>
#include <sys/param.h>
#include <sys/uio.h>
#include <sys/socket.h>
#include <sys/ioctl.h>
#include <netinet/in.h>
#include <stdio.h>
#include <unistd.h>
#include <stdlib.h>
#include <err.h>
#include <poll.h>
#include <stdarg.h>
#include <string.h>
#include <fcntl.h>
#include <netdb.h>
#include <signal.h>
#include <syslog.h>
#include <errno.h>

#include "../common.h"

#ifndef lint
#if 0
static char __unused char RCSid[] = "Header: convd.c,v 1.1 85/10/29 14:20:06 broome Exp ";
#else
static char __unused RCSid[] = "$Phone: convd.c,v 1.2 2013/01/02 23:00:43 christos Exp $";
#endif
#endif



#define MAXSLOTS  32		    /* max users/conversation	      */

struct slot {
    int	   inuse;		    /* this slot in use?	      */
    char   *info;		    /* user's login, host, tty, etc.  */
    char   buffer[BUFSIZ];	    /* text buffer		      */
    int	   new;			    /* strchr of most recent character */
    int	   old;			    /* strchr of oldest character      */
    int	   fd;			    /* open stream file descriptor    */
} slots[MAXSLOTS];		    /* all users in this conversation */
struct pollfd pfd[MAXSLOTS + 1];

int	users;			    /* number of users on	      */
int	stayaround;		    /* still waiting for first users  */
int	currslot;		    /* current slot		      */
int	highslot;		    /* highest slot number in use     */


#ifdef	KERNEL_BUG
int	warned = 0;
#endif

void initslots(void);
void service(void);
void deluser(int);
char *strsave(const char *);
void intro(int);
void sigalrm(int s);
void fatal(int);
void sendit(const char *fmt, ...) __attribute__((__format__(__printf__, 1, 2)));

int
main(int argc, char *argv[])
{
    struct slot *cslot;
    char *c;
    int	 i;
    int	 sl;
    int	 fds;
    int	 r;
    int	 changed;
    static	int  new, old;
    struct	iovec iov[2];
    char	recvbuf[BUFSIZ];
    char	sel[1];	    /* for selecting current window */
    int		ind;

#ifdef NICE
   (void) nice(-3);	  /* get a little bit of priority */
#endif

    users = 0;		   /*  noone here yet */
    stayaround = 1;
    currslot = -1;
    highslot = -1;
    changed = 0;

    signal(SIGPIPE, SIG_IGN);  /* we'll find out soon enough */
    signal(SIGALRM, sigalrm);  /* to handle timeout */
    alarm(60 * 60);	       /* go away if noone home */
    initslots();	       /* clean everything out first */

    iov[0].iov_base = sel;
    iov[0].iov_len = 1;
    iov[1].iov_base = recvbuf;
    pfd[0].fd = 0;
    pfd[0].events = POLLIN;

    do {
	if ((fds = poll(pfd, highslot + 2, INFTIM)) <= 0)
	    continue;

	if (pfd[0].revents & POLLIN)
	    service();

	for (sl = 0; sl <= highslot && fds; sl++) {	/* client port */
	    cslot = &slots[sl];
	    if (cslot->inuse == 0)
		continue;

	    if (pfd[sl + 1].revents & (POLLHUP|POLLERR|POLLNVAL)) {
		    deluser(sl);
		    if (sl == currslot)		/* have to switch windows */
			currslot = -1;		/* just in case ... */
		    continue;
	    }
	    if (pfd[sl + 1].revents & POLLIN) {
		fds--;			    /* decrement slots to check */
		if ((r = read(cslot->fd, recvbuf, BUFSIZ)) <= 0) {   /* EOF */
		    deluser(sl);
		    if (sl == currslot)		/* have to switch windows */
			currslot = -1;		/* just in case ... */
		} else {
		    iov[1].iov_len = r;
		    new = cslot->new;		/* strchr of where to add */
		    old = cslot->old;		/* strchr of oldest char  */
		    c = &cslot->buffer[new];	/* so point to newest	 */
		    for (i = 0; i < r; i++) {
			*c++ = recvbuf[i];
			new++;
			if (new == BUFSIZ) {	/* at end of buffer    */
			    new = 0;		/* so loop back around */
			    c = cslot->buffer;
			} else if (new == old) {    /* full buffer	   */
			    old++;		    /* so advance the end  */
			    if (old == BUFSIZ)	    /* wrapped around here */
				old = 0;
			}
		    }
		    cslot->new = new;  cslot->old = old;    /* save pointers */

		    if (sl != currslot) {     /* switch to this slot */
			sel[0] = (META | sl);
			currslot = sl;
			ind = 0;
		    } else {
			ind = 1;
		    }
		    for (i = 0; i <= highslot; i++)	/* ship out to others */
			if (slots[i].inuse) {
			    if (writev(slots[i].fd, &iov[ind], 2-ind) == -1) {
				deluser(i);
				if (i == currslot) /* have to switch windows */
				    currslot = -1; /* just in case ... */
			    }
			}
		}
	    }
	}
    } while ((users > 1) ||(stayaround == 1));

    shutdown(0, 2);
    exit(17);
}



/*
 *  Set all the slots to an unused state before starting...
 */

void
initslots(void)
{
    int i;
    for (i = 0; i < MAXSLOTS; i++) {
	slots[i].inuse = 0;
	slots[i].fd = -1;
    }
}


/*
 *  Handle a request on the service port.
 */

void
service(void)
{
    int	 new;
    char buf[BUFSIZ];
    int	 j;
    struct	sockaddr_in addr;
    socklen_t	len;
    int		r;


    len = sizeof(addr);
    if ((new = accept(0, (void *)&addr, &len)) < 0) {
	if (errno != EINTR) 
#ifdef	KERNEL_BUG
	    if (warned++ == 0)
#endif /*  KERNEL_BUG */
		fatal(errno);
	return;
    }


    for (j = 0; j < MAXSLOTS; j++)
	if (slots[j].inuse == 0)
	    break;

    if (j == MAXSLOTS) {
	write(new, "Too many users!\n", 16);
	shutdown(new, 2);
	close(new);
	return;
    }
    if ((r = read(new, buf, BUFSIZ)) == 0) {  /* EOF ?? */
	close(new);
	return;
    }
    buf[r] = '\0';

    /* save name, host, tty, realname */
    slots[j].info = strsave(buf);

    if (j > highslot)
	highslot = j;

    slots[j].inuse = 1;
    slots[j].fd = new;
    slots[j].new = 0;
    slots[j].old = 0;
    pfd[j + 1].fd = new;
    pfd[j + 1].events = POLLIN;
    users++;
    r = 1;
    ioctl(new, FIONBIO, &r);		   /* mark socket as non-blocking */

    sendit("%c%s%c", META | ADDUSER | j, slots[j].info, META);
    intro(new);			   /* and fill me in on things */

    return;
}


/*
 *  Retransmit a message to all the users.
 */

void
sendit(const char *fmt, ...)
{
    int	  i, len;
    char buf[1024];
    va_list ap;

    va_start(ap, fmt);
    len = vsnprintf(buf, sizeof(buf), fmt, ap);
    va_end(ap);

    for (i = 0; i <= highslot; i++)
	if (slots[i].inuse) {
	    fprintf(stderr, "%s <- %s %d\n", slots[i].info, buf, buf[0] & 037);
	    if (write(slots[i].fd, buf, len) != len)
		warn("sendit: write");
	}
}


/*
 *  Delete a user from this conversation.
 */

void
deluser(int u)
{
    if (--users < 2) {
	shutdown(0, 2);
	close(0);
	exit(0);
    }
    stayaround = 0;
    pfd[u + 1].fd = -1;
    slots[u].inuse = 0;

    close(slots[u].fd);
    free(slots[u].info);

    sendit("%c",  META | DELUSER | u);
}



/*
 *  Save a string.
 */

char *
strsave(const char *s)
{
    char *new;

    if ((new = strdup(s)) == NULL)
	abort();
    return new;
}



/*
 *   Send the new filedes all the users and all the buffers.
 *   We first send UPDATE | 0 to tell the user to delay updating
 *   the screen until we've sent all the text buffers, at which 
 *   point we send UPDATE | 01 to signal that we're done and
 *   the screen should be updated.
 */

void
intro(int fd)
{
    struct   iovec iov[3];	/* used for multi-buffer writes */
    int old, new;
    int	 s;	       /* slot number */
    int num;	       /* number of buffers to write out */
    char     sc;		/* used for slot number selection */

    /* tell user not to update screen until done with intro */
    sc = META | UPDATE | 00;	
    write(fd, &sc, 1);

    /* first go through and add all the windows */
    for (s = 0; s <= highslot; s++) {
	int len;
	char buf[1024];
	if (slots[s].inuse == 0 || slots[s].fd == fd) 
	    continue;
	len = snprintf(buf, sizeof(buf), 
	    "%c%s%c", META | ADDUSER | s, slots[s].info, META);
	fprintf(stderr, "%d adduser %s %d\n", __LINE__, buf, s);
	if (write(fd, buf, len) < 0)
		warn("write");
    }

    /* now go through and give him all the buffers */
    for (s = 0; s <= highslot; s++) {
	if (slots[s].inuse == 0 || slots[s].fd == fd)
	    continue;

	sc = META | s;		    /* switch to this slot */
	iov[0].iov_base = &sc;
	iov[0].iov_len	= 1;

#ifdef notdef
	/raboof ======== foo bar baz zot blip/
	      ^ new	 ^ old		      ^ BUFSIZ
	|-----|		 |-------------------|
	slots[s].new	 BUFSIZ - slots[s].old
#endif /* notdef */

	new = slots[s].new;
	old = slots[s].old;
	iov[1].iov_base = &slots[s].buffer[old];
	iov[1].iov_len	= (old > new ? BUFSIZ - old : new - old);
	if (old > new) {
	    iov[2].iov_base = slots[s].buffer;
	    iov[2].iov_len  = new;
	    num = 3;
	} else 
	    num = 2;

	writev(fd, iov, num);		   /* write all at once!!! */
    }

    /* now he can update the screen */
    sc = META | UPDATE | 01;
    write(fd, &sc, 1);
}


/*
 *  Come here on alarm signal. If less than 2 users, exit.
 */

void
sigalrm(int s __unused)
{
    if (users < 2) {
	shutdown(0, 2);
	close(0);
	exit(0);
    }
    return;
}


/*
 *  We have encountered some kind of nasty error.
 *  Tell all the users about it and go away.
 */

void
fatal(int e)
{
    char   mesg[1024];
    char   host[MAXHOSTNAMELEN];
    int	   s, len;

    gethostname(host, sizeof(host));

    len = snprintf(mesg, sizeof(mesg),
	"\nMessage from phone conversation daemon @ %s:\n"
	"Fatal error: %s\n"
#ifdef KERNEL_BUG	/* try to keep going */
	"Warning: no more users can join this conversation! Sorry.\n"
#endif /* KERNEL_BUG */
	, host, strerror(e));

    for (s = 0; s <= highslot; s++)
	if (slots[s].inuse)
	    write(slots[s].fd, mesg, len);
    
#ifndef KERNEL_BUG
    chdir("/");
    abort();
    exit(1);
#endif /*  KERNEL_BUG */
}
