#include "defs.h"
#ifndef lint
static char __unused RCSid[] = "$Phone: message.c,v 1.2 2013/01/02 23:00:42 christos Exp $";
#endif

/*
 *  Routines for dealing with the messages printed 
 *  down on the bottom line of the screen.
 */

struct mesg {
    char   *text;	   /* text of this message */
    struct mesg *next;	   /* link to next message */
};

static struct mesg *mesghead =NULL;  /* top of queue	 */
static struct mesg *last     =NULL;  /* bottom of queue	 */
static struct mesg *freelist =NULL;  /* stuff to free up */
int    msgpending  = 0;			/* number of pending messages  */


/*
 *  Add a message to the pending message queue.
 */

void
message(const char *fmt, ...)
{
    struct mesg *new;
    sigset_t nss, oss;
    va_list ap;
    char *str;

    sigemptyset(&nss);
    sigaddset(&nss, SIGALRM);
    sigprocmask(SIG_BLOCK, &nss, &oss);

    if (!freelist) {	    /* need to malloc some up */
	new = malloc(sizeof(*new));
	if (!new)
	    error(1, "cannot malloc message buffer!");
    } else {		    /* take off the top of the list */
	new = freelist;
	freelist = freelist->next;
	free(new->text);
    }
    va_start(ap, fmt);
    vasprintf(&str, fmt, ap);
    va_end(ap);
    new->text = str,
    new->next = NULL;

    if (last)			    /* add to end of currently pending list */
	last->next = new;
    if (!mesghead)		    /* empty queue - add to top */
	mesghead = new;
    last = new;			    /* update bottom pointer */
    if (msgpending <= 0) {	    /* need to restart alarms */
	alarm(1);
	msgpending = 1;
    } else
	msgpending++;
    sigprocmask(SIG_SETMASK, &oss, NULL);
}


/*
 *  Called by the alarm interrupt routine - print the 
 *  next pending message and remove it from the queue.
 */

void
showmessage(void)
{
    static int ticks = 0;
    struct mesg *m;

    if ((ticks =(ticks + 1) % Interval) != 0)
	return;

    if (msgpending == 0) {     /* just clean up after last message */
	putmessage("");
	msgpending = -1;
	return;
    }

    m = mesghead;	    /* save pointer to this message */

    putmessage("%s", m->text);
    mesghead = m->next;	    /* move down top of queue */
    if (last == m)	    /* was the last item, too */
	last =NULL;

    m->next = freelist;	    /* add this item to the free list */
    freelist = m;	    /* this way we don't free in an interrupt !!! */
    msgpending--;	    /* and decrement counter  */
}



/*
 *  Flush the pending message buffer.  Not actually used yet.
 */

void
flush(void)
{
    last->next = freelist;		/* add curr freelist to end of list */
    last =NULL;
    freelist = mesghead;		/* move pending list to free list */
    mesghead =NULL;	  /* zap the head */
    msgpending = 0;			/* and reset the counter */
}
