#include "defs.h"
#include "../common.h"
#ifndef lint
static char __unused RCSid[] = "$Phone: calls.c,v 1.2 2013/01/02 23:00:42 christos Exp $";
#endif

/*  client pending call structure  */
struct	call {
    char    *user;		    /* user's name	       */
    char    *host;		    /* his host		       */
    char    *tty;		    /* his tty		       */
    struct  sockaddr_in addr;	    /* socket address	       */
    char    id[11];		    /* used to talk to daemon  */
    int	    rings;		    /* ticks since last ring   */
    int	    status;		    /* has daemon acted on it? */
    struct  call *next;		    /* next one in the list    */
};

static	struct call *calls = NULL;  /* list of pending calls   */
int	pending = 0;		    /* number of pending calls */

/*
**  Place a call.
**
**  Parameters:
**	user -	the login name of the person to call
**	host -	his hostname, in either normal form('ucbarpa') or 
**		internet dotted quad form('128.32.149.5')
**	tty  -	a specified tty, if any, other the null string("")
**
**  What we do:
**	1.  Send a request packet to the phone daemon at his address,
**	    packet looks like "<esc> P callno : user : tty : myname : convaddr".
**
**	    Callno is a unique request identifier that the daemon uses
**	    to guard against that duplicate messages occurring due to
**	    the time delay between placing the call and getting the response
**	    causing the client to retransmit the request. [Got that?]
**
**	    If we are using the master/slave protocol, the convaddr is
**	    preceded by either 'y' or 'n' to indicate whether or not we 
**	    will be willing to give up our convd and go to his.(This 
**	    is used for joining an existing conversation.)
**	    [Not used yet ...]
**
**	2.  We expect to get a response back from his phoned of the form
**	    "<ESC> P y id" where `id' is a small 5-character request id which
**	    is to be used when 'renewing' a page request. 
**
**	3.  Then save all of the info(user, host, tty, message id) in 
**	    a 'call' structure which is added to the linked list of pending
**	    calls.
**
**	4.  As a side effect, the global variable 'pending' is incremented.
*/

void
placecall(const char *user, const char *host, const char *tty)
{
    struct  call    *new;	    /* new call structure */
    struct  hostent *hp;	    /* host info */
    struct  sockaddr_in sin;	    /* network address */
    static  int	    callno = 0;
    ssize_t rval;
    char buf[1024];

    if (isdigit((unsigned char)*host))		  /* numeric address ??? */
	sin.sin_addr.s_addr = inet_addr(host);
    else {
	if ((hp = gethostbyname(host)) == NULL) {
	    putmessage("Unknown host: %s", host);
	    return;
	}
	memcpy(&sin.sin_addr, hp->h_addr, sizeof(sin.sin_addr));
    }
    sin.sin_family = AF_INET;
    sin.sin_port   = port;

    message("Calling %s@%s", user, host);

#ifdef	SLAVE
    rval = sendit(buf, sizeof(buf), &sin,
	"%c%c%03d:%s:%s:%s:%c%s", ESC, PAGE, callno++,
	user, tty ? tty : "", login, users > 1 ? NAK : ACK,  convaddr); 
#else /*   !SLAVE */
    rval = sendit(buf, sizeof(buf), &sin,
	"%c%c%03d:%s:%s:%s:%s", ESC, PAGE, callno++, 
	user, tty ? tty : "", mylogin, convaddr);
#endif /*  SLAVE */

    if (rval < 0) { 
	message("No response to call request.");
	return;
    }

    /*	got a response - text is in buf.  */
    buf[rval] = '\0';
    new = malloc(sizeof(*new));
    new->user  = strsave(user);
    new->host  = strsave(host);
    new->tty   = strsave(tty);
    new->rings = 1;
    strncpy(new->id, buf+3, 10);   /* save the mesg id # */
    memcpy(&new->addr, &sin, sizeof(new->addr));
    new->next = calls;		    /* put list together */
    calls = new;

    pending++;	    /* increment the number of pending calls */
    return;
}


/*
**  Go through the list of pending calls and reinvite 
**  all the calls that have expired.  Returns 1 if the timer
**  should be reset, 0 if the timer should expire.
**
**  If SLAVE is defined, we check every call to see they have 
**  tried to call us. If they indicate that they are *not* willing
**  to submit(they are in an existing conversation), then we close
**  our existing convd connection and connect to their convd; else
**  we go over only if our convd address is lower than theirs.
** (Yes, this looks like a hack, but it seems to work best.)
*/

void
reinvite(void)
{
    struct call *ptr;

    if (pending == 0)	    /* nothing to do */
	return;
    
    for (ptr = calls; ptr; ptr = ptr->next) {
	/* 
	 *  See if we've connected to anyone,
	 *  if not then first call our local host and see if the
	 *  intended recipient is trying to call us...
	 */
#ifdef SLAVE
	if (users < 2 && pending < 2) {
	    ssize_t r;
	    char buf[1024];
	    if (Debug)
		putmessage("Checking for reverse calls...");
	    r = sendit(buf, sizeof(buf), localaddr, "%c%c%s:%s", ESC, INQUIRE,
		login, ptr->user);
	    if (r > 0) {
		buf[r] = '\0';
		if (Debug)
		    printf("Received \"%s\"\r\n", buf+1);
		if (buf[2] == ACK && 
		  (buf[3] == NAK || strcmp(convaddr, buf + 4) < 0) &&
		   connect_daemon(buf + 4) == 0) {
		    putmessage("Switching to conversation already in progress.");
		    delete(ptr->id);
		    /* need to reset status so windows work */
		    users = 0;
		    return;
		}
	    }
	    putmessage("");
	}
#endif /* SLAVE */

	if (ptr->rings == 0) {	    /* time to reinvite */
	    char buf[1024];

	    if (sendit(buf, sizeof(buf), &ptr->addr, "%c%c%s", ESC, REINVITE,
		ptr->id) < 0)
		error(0, "reinvite: sendto");
	    /* should check response value!!! */
	}
	ptr->rings = (ptr->rings + 1) % 20;	/* reinvite every 20 secs. */
    }
    return;
}


/*
**  Delete the call `id' from the pending list.
*/

void
delete(const char *id)
{
    struct call *curr;
    struct call *prev = NULL;
    struct call *next;

    for (curr = calls; curr; curr = next) {
	next = curr->next;
	if (strncmp(id, curr->id, 5) == 0) {   /* this is the one */
	    if (prev)
		prev->next = next;
	    else
		calls = next;
	    free(curr->user);
	    free(curr->host);
	    free(curr->tty);
	    free(curr);
	    pending--;
	    return;
	}
    }
}



/*
**  Front end routine --- Page a user to join in the conversation.
*/
void
page(int argc, char *argv[])
{
    char    *hishost;
    char    *tty;
    char    *user;

    if ((user = expalias(argv[0])) == NULL)
	user = argv[0];

    if ((hishost = strchr(user, '@')) != NULL)
	*hishost++ = '\0';
    else
	hishost = myhost;
    tty = (argc == 2) ? argv[1] : NULL;
    
    placecall(user, hishost, tty);
}

/*
**  Cancel a call sent out by simply not bothering to "renew" it.
*/
void
cancel(int argc, char *argv[])
{
    char    *user;
    char    *hishost;
    char    *tty;
    struct  call *curr, *prev, *next;

    if ((user = expalias(argv[0])) == NULL)
	user = argv[0];

    tty = (argc == 2) ? argv[1] : NULL;
    if ((hishost = strchr(user, '@')) != NULL)
	*hishost++ = '\0';
    else
	hishost = myhost;
    
    prev = NULL;
    for (curr = calls; curr; curr = next) {
	next = curr->next;
	if (equal(user, curr->user) && equal(hishost, curr->host)) {
	    if (prev)
		prev->next = next;
	    else
		calls = next;
	    free(curr->user);
	    free(curr->host);
	    free(curr->tty);
	    free(curr);
	    pending--;
	    if (tty)
		message("Cancelling call to %s@%s on %s", user, hishost, tty);
	    else
		message("Cancelling call to %s@%s", user, hishost);
	    return;
	}
    }
    message("No pending calls to %s@%s", user, hishost);
}
