#  Keyring.pm
#    - manage a GnuPG-based keyring
#
#  Copyright (C) 2000 Frank J. Tobin <ftobin@uiuc.edu>
#
#  This file is part of pgpenvelope
#
#  This program is free software; you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation; either version 2 of the License, or
#  (at your option) any later version.
#
#  This program is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program; if not, visit the following URL:
#  http://www.gnu.org/copyleft/gpl.html
#
#  $Id: Keyring.pm,v 1.5 2001/04/15 05:28:24 ftobin Exp $
#

package PGPEnvelope::Keyring;

use strict;
use Symbol;
use Carp;
use IO::File;

use GnuPG::Handles;

use Class::MethodMaker
  boolean       => [ qw( keys_loaded indexed_key_info_loaded ) ],
  list          => [ qw( keys indexed_key_info ) ],
  object        => [ qw( GnuPG::Interface  gnupg ) ],
  new_hash_init => 'new';


sub print
{
    my ( $self ) = @_;
    print $self->get_indexed_key_info();
}


sub import_keys
{
    my ( $self, @key_ids ) = @_;
    return unless @key_ids;
    
    my $fileno = fileno select;
    my $newfh = IO::File->new( '&' . $fileno );
    
    my $handles = GnuPG::Handles->new( stdin  => gensym,
				       stdout => $newfh,
				     ); 
    
    $self->gnupg->recv_keys( handles            => $handles,
			     gnupg_command_args => [ @key_ids ],
			   );
    wait;
}


sub load_indexed_key_info
{
    my ( $self, %args ) = @_;
    
    my @keys = $self->keys();
    
    my $max_index = scalar @keys;
    my $blank_prefix  = ' ' x ( length( "$max_index) " ) );
    
    $self->clear_indexed_key_info();

    my $i = 0;
    
    foreach my $key ( @keys )
    {
	$i++;
	
	my $index_blank_prefix = ' ' x ( length( "$max_index" ) - length( "$i" ) );
	my @info = $key->info( %args );
	
	$self->push_indexed_key_info( $index_blank_prefix . $i . ') ' .( shift @info ) );
	
	foreach my $data ( @info )
	{
	    $self->push_indexed_key_info( $blank_prefix . $data );
	}
    }
    
    $self->set_indexed_key_info_loaded();
}


sub get_indexed_key_info
{
    my ( $self ) = @_;
    
    $self->load_indexed_key_info()
      unless $self->indexed_key_info_loaded();
    
    return $self->indexed_key_info();
}


sub get_short_indexed_key_info
{
    my ( $self ) = @_;
    
    $self->load_indexed_key_info( short => 1 )
      unless $self->indexed_key_info_loaded();
      
    return $self->indexed_key_info();
}



sub print_keys
{
    my ( $self, @key_ids ) = @_;
    
    $self->print_keys_slave( key_ids    => [ @key_ids ],
			     with_sigs  => 0,
			   );
}


sub print_keys_with_sigs
{
    my ( $self, @key_ids ) = @_;
    
    $self->print_keys_slave( key_ids    => [ @key_ids ],
			     with_sigs  => 1,
			   );
}


sub print_keys_slave
{
    my ( $self, %args ) = @_;
    
    my $key_ids   = $args{key_ids};
    my $with_sigs = $args{with_sigs} || 0;
    
    my $fileno = fileno select;
    my $newfh = '>&' . $fileno;
    
    my $handles = GnuPG::Handles->new( stdin  => gensym,
				       stdout => $newfh,
				     );
    
    my $listing_call = $with_sigs ? 'list_sigs' : 'list_public_keys';
    
    $self->gnupg->$listing_call( handles            => $handles,
				 gnupg_command_args => $key_ids,
			       );
    
    wait;
}



1;

