package models_test

import (
	"encoding/json"
	"testing"
	"time"

	"github.com/google/osv-scanner/pkg/models"
	"gopkg.in/yaml.v3"
)

func TestAffected_MarshalJSONWithPackage(t *testing.T) {
	t.Parallel()
	v := models.Vulnerability{
		ID: "TEST-0000",
		Affected: []models.Affected{
			{
				Package:  models.Package{Ecosystem: models.EcosystemPyPI, Name: "requests"},
				Versions: []string{"1.0.0"},
			},
		},
	}
	got, err := json.Marshal(v)
	if err != nil {
		t.Fatalf("Marshal() = %v; want no error", err)
	}
	want := `{"modified":"0001-01-01T00:00:00Z","id":"TEST-0000","affected":[{"package":{"ecosystem":"PyPI","name":"requests"},"versions":["1.0.0"]}]}`
	if string(got) != want {
		t.Errorf("Marshal() = %v; want %v", string(got), want)
	}
}

func TestAffected_MarshalJSONWithoutPackage(t *testing.T) {
	t.Parallel()
	v := models.Vulnerability{
		ID: "TEST-0000",
		Affected: []models.Affected{
			{
				Versions: []string{"1.0.0"},
			},
		},
	}
	got, err := json.Marshal(v)
	if err != nil {
		t.Fatalf("Marshal() = %v; want no error", err)
	}
	want := `{"modified":"0001-01-01T00:00:00Z","id":"TEST-0000","affected":[{"versions":["1.0.0"]}]}`
	if string(got) != want {
		t.Errorf("Marshal() = %v; want %v", string(got), want)
	}
}

func TestAffected_MarshalYAMLWithPackage(t *testing.T) {
	t.Parallel()
	v := models.Vulnerability{
		ID: "TEST-0000",
		Affected: []models.Affected{
			{
				Package:  models.Package{Ecosystem: models.EcosystemPyPI, Name: "requests"},
				Versions: []string{"1.0.0"},
			},
		},
	}
	got, err := yaml.Marshal(v)
	if err != nil {
		t.Fatalf("Marshal() = %v; want no error", err)
	}
	want := `id: TEST-0000
modified: 0001-01-01T00:00:00Z
affected:
    - package:
        ecosystem: PyPI
        name: requests
      versions:
        - 1.0.0
`
	if string(got) != want {
		t.Errorf("Marshal() = %v; want %v", string(got), want)
	}
}

func TestAffected_MarshalYAMLWithoutPackage(t *testing.T) {
	t.Parallel()
	v := models.Vulnerability{
		ID: "TEST-0000",
		Affected: []models.Affected{
			{
				Versions: []string{"1.0.0"},
			},
		},
	}
	got, err := yaml.Marshal(v)
	if err != nil {
		t.Fatalf("Marshal() = %v; want no error", err)
	}
	want := `id: TEST-0000
modified: 0001-01-01T00:00:00Z
affected:
    - versions:
        - 1.0.0
`
	if string(got) != want {
		t.Errorf("Marshal() = %v; want %v", string(got), want)
	}
}

func TestVulnerability_MarshalJSONTimes(t *testing.T) {
	t.Parallel()
	losAngeles, err := time.LoadLocation("America/Los_Angeles")
	if err != nil {
		panic(err)
	}

	tests := []struct {
		name string
		vuln models.Vulnerability
		want string
	}{
		{
			name: "empty",
			vuln: models.Vulnerability{
				ID: "TEST-0000",
			},
			want: `{"modified":"0001-01-01T00:00:00Z","id":"TEST-0000"}`,
		},
		{
			name: "no withdraw",
			vuln: models.Vulnerability{
				ID:        "TEST-0000",
				Modified:  time.Date(2023, 12, 1, 12, 30, 30, 0, time.UTC),
				Published: time.Date(2021, 6, 30, 1, 0, 0, 0, time.UTC),
			},
			want: `{"modified":"2023-12-01T12:30:30Z","published":"2021-06-30T01:00:00Z","id":"TEST-0000"}`,
		},
		{
			name: "all UTC",
			vuln: models.Vulnerability{
				ID:        "TEST-0000",
				Modified:  time.Date(2023, 12, 1, 12, 30, 30, 0, time.UTC),
				Published: time.Date(2021, 6, 30, 1, 0, 0, 0, time.UTC),
				Withdrawn: time.Date(2022, 1, 15, 23, 59, 59, 0, time.UTC),
			},
			want: `{"modified":"2023-12-01T12:30:30Z","published":"2021-06-30T01:00:00Z","withdrawn":"2022-01-15T23:59:59Z","id":"TEST-0000"}`,
		},
		{
			name: "all Los Angeles",
			vuln: models.Vulnerability{
				ID:        "TEST-0000",
				Modified:  time.Date(2023, 12, 1, 12, 30, 30, 0, losAngeles),
				Published: time.Date(2021, 6, 30, 1, 0, 0, 0, losAngeles),
				Withdrawn: time.Date(2022, 1, 15, 23, 59, 59, 0, losAngeles),
			},
			want: `{"modified":"2023-12-01T20:30:30Z","published":"2021-06-30T08:00:00Z","withdrawn":"2022-01-16T07:59:59Z","id":"TEST-0000"}`,
		},
	}
	for _, test := range tests {
		innerTest := test
		t.Run(innerTest.name, func(t *testing.T) {
			t.Parallel()
			got, err := json.Marshal(innerTest.vuln)
			if err != nil {
				t.Fatalf("Marshal() = %v; want no error", err)
			}
			if string(got) != innerTest.want {
				t.Errorf("Marshal() = %v; want %v", string(got), innerTest.want)
			}
		})
	}
}

func TestVulnerability_MarshalYAMLTimes(t *testing.T) {
	t.Parallel()
	losAngeles, err := time.LoadLocation("America/Los_Angeles")
	if err != nil {
		panic(err)
	}

	tests := []struct {
		name string
		vuln models.Vulnerability
		want string
	}{
		{
			name: "empty",
			vuln: models.Vulnerability{
				ID: "TEST-0000",
			},
			want: "id: TEST-0000\nmodified: 0001-01-01T00:00:00Z\n",
		},
		{
			name: "no withdraw",
			vuln: models.Vulnerability{
				ID:        "TEST-0000",
				Modified:  time.Date(2023, 12, 1, 12, 30, 30, 0, time.UTC),
				Published: time.Date(2021, 6, 30, 1, 0, 0, 0, time.UTC),
			},
			want: "id: TEST-0000\nmodified: 2023-12-01T12:30:30Z\npublished: 2021-06-30T01:00:00Z\n",
		},
		{
			name: "all UTC",
			vuln: models.Vulnerability{
				ID:        "TEST-0000",
				Modified:  time.Date(2023, 12, 1, 12, 30, 30, 0, time.UTC),
				Published: time.Date(2021, 6, 30, 1, 0, 0, 0, time.UTC),
				Withdrawn: time.Date(2022, 1, 15, 23, 59, 59, 0, time.UTC),
			},
			want: "id: TEST-0000\nmodified: 2023-12-01T12:30:30Z\npublished: 2021-06-30T01:00:00Z\nwithdrawn: 2022-01-15T23:59:59Z\n",
		},
		{
			name: "all Los Angeles",
			vuln: models.Vulnerability{
				ID:        "TEST-0000",
				Modified:  time.Date(2023, 12, 1, 12, 30, 30, 0, losAngeles),
				Published: time.Date(2021, 6, 30, 1, 0, 0, 0, losAngeles),
				Withdrawn: time.Date(2022, 1, 15, 23, 59, 59, 0, losAngeles),
			},
			want: "id: TEST-0000\nmodified: 2023-12-01T20:30:30Z\npublished: 2021-06-30T08:00:00Z\nwithdrawn: 2022-01-16T07:59:59Z\n",
		},
	}
	for _, test := range tests {
		innerTest := test
		t.Run(innerTest.name, func(t *testing.T) {
			t.Parallel()
			got, err := yaml.Marshal(innerTest.vuln)
			if err != nil {
				t.Fatalf("Marshal() = %v; want no error", err)
			}
			if string(got) != innerTest.want {
				t.Errorf("Marshal() = %v; want %v", string(got), innerTest.want)
			}
		})
	}
}
