package ci_test

import (
	"testing"

	"github.com/google/osv-scanner/internal/ci"
	"github.com/google/osv-scanner/internal/testutility"
	"github.com/google/osv-scanner/pkg/models"
)

func TestDiffVulnerabilityResults(t *testing.T) {
	t.Parallel()
	type args struct {
		oldRes models.VulnerabilityResults
		newRes models.VulnerabilityResults
	}
	tests := []struct {
		name     string
		args     args
		wantPath string
	}{
		{
			args: args{
				oldRes: testutility.LoadJSONFixture[models.VulnerabilityResults](t, "fixtures/vulns/test-vuln-results-a.json"),
				newRes: testutility.LoadJSONFixture[models.VulnerabilityResults](t, "fixtures/vulns/test-vuln-results-a.json"),
			},
			// `newRes` is the same as `oldRes`, so the result should be empty
			wantPath: "fixtures/vulns/test-vuln-diff-a-a.json",
		},
		{
			args: args{
				oldRes: testutility.LoadJSONFixture[models.VulnerabilityResults](t, "fixtures/vulns/test-vuln-results-a.json"),
				newRes: testutility.LoadJSONFixture[models.VulnerabilityResults](t, "fixtures/vulns/test-vuln-results-a-1.json"),
			},
			// `newRes` is the same as `oldRes`, but with the source path moved, so this should return all of the moved source
			wantPath: "fixtures/vulns/test-vuln-diff-a-a-1.json",
		},
		{
			args: args{
				oldRes: testutility.LoadJSONFixture[models.VulnerabilityResults](t, "fixtures/vulns/test-vuln-results-a.json"),
				newRes: testutility.LoadJSONFixture[models.VulnerabilityResults](t, "fixtures/vulns/test-vuln-results-b.json"),
			},
			// `newRes` has one new GO vuln compared to `oldRes`, so the result should contain just the extra vuln
			wantPath: "fixtures/vulns/test-vuln-diff-a-b.json",
		},
		{
			args: args{
				oldRes: testutility.LoadJSONFixture[models.VulnerabilityResults](t, "fixtures/vulns/test-vuln-results-b.json"),
				newRes: testutility.LoadJSONFixture[models.VulnerabilityResults](t, "fixtures/vulns/test-vuln-results-a.json"),
			},
			// `oldRes` has one new GO vuln compared to `newRes`, so the result should be empty
			wantPath: "fixtures/vulns/test-vuln-diff-b-a.json",
		},
		{
			args: args{
				oldRes: testutility.LoadJSONFixture[models.VulnerabilityResults](t, "fixtures/vulns/test-vuln-results-c.json"),
				newRes: testutility.LoadJSONFixture[models.VulnerabilityResults](t, "fixtures/vulns/test-vuln-results-b.json"),
			},
			// `newRes` has one new GO vuln, and a new `cargo.toml` source compared to `oldRes`, so the result should contain all the new vulns
			wantPath: "fixtures/vulns/test-vuln-diff-c-b.json",
		},
		{
			args: args{
				oldRes: testutility.LoadJSONFixture[models.VulnerabilityResults](t, "fixtures/vulns/test-vuln-results-b.json"),
				newRes: testutility.LoadJSONFixture[models.VulnerabilityResults](t, "fixtures/vulns/test-vuln-results-c.json"),
			},
			// opposite of above test case, result should be empty
			wantPath: "fixtures/vulns/test-vuln-diff-b-c.json",
		},
	}
	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			t.Parallel()
			got := ci.DiffVulnerabilityResults(tt.args.oldRes, tt.args.newRes)
			testutility.NewSnapshot().MatchJSON(t, got)
		})
	}
}

func TestDiffVulnerabilityByUniqueVulnCountResults(t *testing.T) {
	t.Parallel()
	type args struct {
		oldRes models.VulnerabilityResults
		newRes models.VulnerabilityResults
	}
	tests := []struct {
		name     string
		args     args
		wantPath string
	}{
		{
			args: args{
				oldRes: testutility.LoadJSONFixture[models.VulnerabilityResults](t, "fixtures/vulns/test-vuln-results-a.json"),
				newRes: testutility.LoadJSONFixture[models.VulnerabilityResults](t, "fixtures/vulns/test-vuln-results-a-1.json"),
			},
			// `newRes` is the same as `oldRes`, but with the source path moved, so this should be empty
			wantPath: "fixtures/vulns/test-vuln-unique-diff-a-a-1.json",
		},
		{
			args: args{
				oldRes: testutility.LoadJSONFixture[models.VulnerabilityResults](t, "fixtures/vulns/test-vuln-results-a.json"),
				newRes: testutility.LoadJSONFixture[models.VulnerabilityResults](t, "fixtures/vulns/test-vuln-results-a.json"),
			},
			// `newRes` is the same as `oldRes`
			wantPath: "fixtures/vulns/test-vuln-unique-diff-a-a.json",
		},
		{
			args: args{
				oldRes: testutility.LoadJSONFixture[models.VulnerabilityResults](t, "fixtures/vulns/test-vuln-results-a.json"),
				newRes: testutility.LoadJSONFixture[models.VulnerabilityResults](t, "fixtures/vulns/test-vuln-results-b.json"),
			},
			// `newRes` has one new GO vuln compared to `oldRes`, so the result should contain just the extra vuln
			wantPath: "fixtures/vulns/test-vuln-unique-diff-a-b.json",
		},
		{
			args: args{
				oldRes: testutility.LoadJSONFixture[models.VulnerabilityResults](t, "fixtures/vulns/test-vuln-results-b.json"),
				newRes: testutility.LoadJSONFixture[models.VulnerabilityResults](t, "fixtures/vulns/test-vuln-results-a.json"),
			},
			// `oldRes` has one new GO vuln compared to `newRes`, so the result should be empty
			wantPath: "fixtures/vulns/test-vuln-unique-diff-b-a.json",
		},
	}
	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			t.Parallel()
			got := ci.DiffVulnerabilityResultsByOccurrences(tt.args.oldRes, tt.args.newRes)
			testutility.NewSnapshot().MatchJSON(t, got)
		})
	}
}
