/*
 * stream.c
 *
 * Copyright (c) 2002 Maksim Yevmenkin <m_evmenkin@yahoo.com>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * $Id: stream.c,v 1.11 2010/10/29 19:34:58 max Exp $
 * $FreeBSD$
 */

#include <bluetooth.h>
#include <errno.h>
#include <fcntl.h>
#include <limits.h>
#include <obex.h>
#include <stdio.h>
#include <string.h>
#include <time.h>
#include <unistd.h>

#include "compat.h"
#include "obexapp.h"
#include "log.h"
#include "stream.h"
#include "util.h"

/*
 * Process OBEX_EV_STREAMEMPTY event
 */

void
obexapp_stream_write(obex_t *handle, obex_object_t *object, 
			__unused int obex_cmd, __unused int obex_rsp)
{
	context_p		context = (context_p) OBEX_GetUserData(handle);
	int			length;
	obex_headerdata_t	hv;

	log_debug("%s()", __func__);

	length = obexapp_util_read(context->sfd, context->sbuffer, OBEXAPP_BUFFER_SIZE);
	if (length < 0)
		log_err("%s(): Could not read(%d). %s (%d)",
			__func__, context->sfd, strerror(errno), errno);
	else
		log_debug("%s(): Got %d bytes of stream data",
			__func__, length);

	hv.bs = (length >= 0)? context->sbuffer : NULL;

	/* XXX what to do in case of error? */
	if (OBEX_ObjectAddHeader(handle, object, OBEX_HDR_BODY, hv, 
			(length < 0)? 0 : length,
			(length <= 0)? OBEX_FL_STREAM_DATAEND : 
				OBEX_FL_STREAM_DATA) < 0)
		log_err("%s(): Could not add HDR_BODY", __func__);

	if (length <= 0) {
		log_debug("%s(): Closing sfd=%d", __func__, context->sfd);

		close(context->sfd);
		context->sfd = -1;
	} else {
		context->stotal += length;
	}
} /* obexapp_stream_write */

/*
 * Process OBEX_EV_STREAMAVAIL event
 */

void
obexapp_stream_read(obex_t *handle, obex_object_t *object, 
			__unused int obex_cmd, __unused int obex_rsp)
{
	context_p		 context = (context_p) OBEX_GetUserData(handle);
	obex_headerdata_t	 hv;
	uint8_t			 hi;
	uint32_t		 hlen;
	int			 length;
	const uint8_t		*buffer = NULL;

	log_debug("%s()", __func__);

	if (context->sfd < 0) {
		if (!context->server) {
			log_err("%s(): Invalid streaming descriptor " \
				"in client mode", __func__);
			return;
		}

		if (context->pstream != 0) {
			log_err("%s(): Invalid streaming descriptor " \
				"in server mode, pstream=%d",
				__func__, context->pstream);

			OBEX_ObjectSetRsp(object,
				OBEX_RSP_INTERNAL_SERVER_ERROR,
				OBEX_RSP_INTERNAL_SERVER_ERROR);

			return;
		}

		context->temp[0] = '\0';
		context->file[0] = '\0';

		while(OBEX_ObjectGetNextHeader(handle,object,&hi,&hv,&hlen)) {
			switch(hi) {
			case OBEX_HDR_NAME:
				if (hlen == 0 || hlen / 2 > PATH_MAX) {
					log_err("%s(): Invalid Name, hlen=%d",
						__func__, hlen);

					OBEX_ObjectSetRsp(object,
						OBEX_RSP_BAD_REQUEST,
						OBEX_RSP_BAD_REQUEST);

					return;
				}

				if (obexapp_util_locale_from_utf16be(
						hv.bs, hlen,
						context->file, PATH_MAX) < 0) {
					log_err("%s(): Could not convert from UTF-16BE",
						__func__);

					OBEX_ObjectSetRsp(object,
						OBEX_RSP_BAD_REQUEST,
						OBEX_RSP_BAD_REQUEST);

					return;
				}

				log_debug("%s(): Name - %s, hlen=%d",
					__func__, context->file, hlen);
				break;

			default:
				log_debug("%s(): Skipping header, hi=%#x, " \
					"hlen=%d", __func__, hi, hlen);
				break;
			}
		}

		/*
		 * Re-parse OBEX headers so out PUT callback will get exactly 
		 * the same set of headers as we received here.
		 * XXX do we need to check for errors here?
		 */

		OBEX_ObjectReParseHeaders(handle, object);

		/* XXX must have NAME in PUT */
		if (context->file[0] == '\0') {
			log_warning("%s(): Rejecting PUT without NAME",
				__func__);

			OBEX_ObjectSetRsp(object,
				OBEX_RSP_BAD_REQUEST, OBEX_RSP_BAD_REQUEST);

			return;
		}

		context->sfd = obexapp_util_mkstemp(context->file, context->temp, PATH_MAX);
		if (context->sfd < 0) {
			log_err("%s(): Could not obexapp_util_mkstemp. %s (%d)",
				__func__, strerror(errno), errno);

			OBEX_ObjectSetRsp(object,
				OBEX_RSP_INTERNAL_SERVER_ERROR,
				OBEX_RSP_INTERNAL_SERVER_ERROR);

			return;
		}

		context->pstream = OBEXAPP_PSTREAM_START;
	} else
		context->pstream = OBEXAPP_PSTREAM_CONTINUE;
         
	length = OBEX_ObjectReadStream(handle, object, &buffer);

	/* 
	 * XXX FIXME can length be < 0 ?
	 * XXX FIXME should we OBEX_ObjectSetRsp in case of error?
	 */

	if (length <= 0) {
		log_debug("%s(): Closing sfd=%d, length=%d",
			__func__, context->sfd, length);

                close(context->sfd); 
		context->sfd = -1;
		context->pstream = OBEXAPP_PSTREAM_END;

		return;
	}

	/* XXX FIXME should we OBEX_ObjectSetRsp in case of error? */
	if (obexapp_util_write(context->sfd, buffer, length) < 0) {
		log_err("%s(): Could not write(%d). %s (%d)",
			__func__, context->sfd, strerror(errno), errno);

                close(context->sfd); 
		context->sfd = -1;
		context->pstream = OBEXAPP_PSTREAM_END;

		return;
	}

	context->stotal += length;
	log_debug("%s(): Wrote %d bytes of stream data", __func__, length);

	if (context->connection_id != OBEXAPP_INVALID_CONNECTION_ID) {
		hv.bq4 = context->connection_id;
		OBEX_ObjectAddHeader(handle, object,
			OBEX_HDR_CONNECTION, hv, sizeof(hv.bq4), 0);
	}
} /* obexapp_stream_read */

void
obexapp_stream_stats_reset(context_t *context)
{
	context->stotal = 0;
	context->stime = time(NULL);
}

void
obexapp_stream_stats_print(context_t *context)
{
	int tm;

	tm = (int)(time(NULL) - context->stime);
	printf("%zu bytes streamed in %d second%s",
	    context->stotal, tm, (tm == 1 ? "" : "s"));
	if (tm > 1)
		printf(" (%zu bytes/sec)", context->stotal / tm);
	printf("\n");
}
