/*
 * main.c
 *
 * Copyright (c) 2002-2009 Maksim Yevmenkin <m_evmenkin@yahoo.com>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * $Id: main.c,v 1.18 2010/11/03 18:28:50 max Exp $
 * $FreeBSD$
 */

#include <bluetooth.h>
#include <ctype.h>
#include <err.h>
#include <errno.h>
#include <libgen.h>
#include <limits.h>
#include <obex.h>
#include <sdp.h>
#include <signal.h>
#include <stdarg.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#include "compat.h"
#include "obexapp.h"
#include "client.h"
#include "event.h"
#include "log.h"
#include "server.h"
#include "sdpc.h"
#include "transport.h"
#include "util.h"

static void usage(char const *prog);

/*
 * Main
 */

int
main(int argc, char *argv[])
{
	struct sigaction	sa;
	char			*ep = NULL, *pri_name = NULL;
	int			n, service, detach;
	context_t		context;
	obex_ctrans_t		custfunc;

	/* Initialize locale */
	if (obexapp_util_locale_init() < 0)
		errx(1, "Could not initialize locale");

	/* Ignore SIGCHLD */
	memset(&sa, 0, sizeof(sa));
	sa.sa_handler = SIG_IGN;
	sa.sa_flags = SA_NOCLDWAIT;
	if (sigaction(SIGCHLD, &sa, NULL) < 0)
		err(1, "Could not sigaction(SIGCHLD)");

	/* Prepare context */
	memset(&context, 0, sizeof(context));
	context.tfd = context.sfd = -1;
	detach = 1;

	context.ls_size = OBEXAPP_BUFFER_SIZE;
	if ((context.ls = (char *) malloc(context.ls_size)) == NULL)
		errx(1, "Could not allocate context.ls");

	if ((context.root = (char *) malloc(PATH_MAX)) == NULL)
		errx(1, "Could not allocate context.root");
	context.root[0] = '\0';

	if ((context.file = (char *) malloc(PATH_MAX)) == NULL)
		errx(1, "Could not allocate context.file");

	if ((context.temp = (char *) malloc(PATH_MAX)) == NULL)
		errx(1, "Could not allocate context.temp");

	context.tbuffer = (uint8_t *) malloc(OBEXAPP_BUFFER_SIZE);
	if (context.tbuffer == NULL)
		errx(1, "Could not allocate context.tbuffer");

	context.sbuffer = (uint8_t *) malloc(OBEXAPP_BUFFER_SIZE);
	if (context.sbuffer == NULL)
		errx(1, "Could not allocate context.sbuffer");

	context.mtu = OBEX_MAXIMUM_MTU;
	context.connection_id = OBEXAPP_INVALID_CONNECTION_ID;

	memset(&custfunc, 0, sizeof(custfunc));
	custfunc.connect = obexapp_transport_connect;
	custfunc.disconnect = obexapp_transport_disconnect;
	custfunc.listen = obexapp_transport_listen;
	custfunc.write = obexapp_transport_write;
	custfunc.handleinput = obexapp_transport_handle_input;
	custfunc.customdata = &context;

	/* Process command line options */
	service = 0;
	while ((n = getopt(argc, argv, "a:A:cC:dDfhl:m:nr:RsSu:")) != -1) {
		switch (n) {
		case 'a':
			if (!bt_aton(optarg, &context.raddr)) {
				struct hostent	*he = bt_gethostbyname(optarg);

				if (he == NULL)
					errx(1, "%s: %s", optarg,
						hstrerror(h_errno));

				memcpy(&context.raddr, he->h_addr,
					sizeof(context.raddr));
			}
			break;

		case 'A':
#ifdef HAVE_BT_DEVADDR
			if (!bt_devaddr(optarg, &context.laddr))
				err(1, "%s", optarg);
#else
			if (!bt_aton(optarg, &context.laddr)) {
				struct hostent	*he = bt_gethostbyname(optarg);

				if (he == NULL)
					errx(1, "%s: %s", optarg,
						hstrerror(h_errno));

				memcpy(&context.laddr, he->h_addr,
						sizeof(context.laddr));
			}
#endif /* HAVE_BT_DEVADDR */
			break;

		case 'c': /* client */
			context.server = 0;
			break;

		case 'C': /* channel to listen on or connect to */
			context.channel = strtoul(optarg, &ep, 10);
			if (*ep != '\0') {
				context.channel = 0;

				switch (tolower((int)optarg[0])) {
				case 'i': /* IrMC */
					service = SDP_SERVICE_CLASS_IR_MC_SYNC;
					break;

				case 'f': /* OBEX File Transfer */
					service = SDP_SERVICE_CLASS_OBEX_FILE_TRANSFER;
					/* force fbs, because we can :) */
					context.fbs = 1;
					break;

				case 'o': /* OBEX Push */
					service = SDP_SERVICE_CLASS_OBEX_OBJECT_PUSH;
					break;
				}
			}
			break;

		case 'd': /* do not detach server */
			detach = 0;
			break;

		case 'D': /* use stdin/stdout */
			context.direct = 1;
			break;

		case 'f': /* connect to Folder Browsing Service */
			context.fbs = 1;
			break;

		case 'l': /* log priority */
			pri_name = optarg;
			break;

		case 'm': /* OBEX MTU */
			context.mtu = strtoul(optarg, &ep, 10);

			if (*ep != '\0' ||
			    context.mtu < OBEX_MINIMUM_MTU ||
			    context.mtu > OBEX_MAXIMUM_MTU)
				errx(1, "Invalid OBEX MTU: %s", optarg);
			break;

		case 'n': /* non-interactive client mode */
			if (context.server)
				usage(basename(argv[0]));
				/* NOT REACHED */

			context.ni = 1;
			break;

		case 'r': /* root */
			if (realpath(optarg, context.root) == NULL)
				err(1, "Could not realpath(%s)", optarg);
			break;

		case 'R': /* virtualize root for each device */
			context.vroot = 1;
			context.secure = 1;
			break;

		case 's': /* server */
			if (context.ni)
				usage(basename(argv[0]));
				/* NOT REACHED */

			context.server = 1;
			break;

		case 'S': /* secure */
			context.secure = 1;
			break;

		case 'u': /* user */
			context.user = optarg;
			break;

		case 'h':
		default:
			usage(basename(argv[0]));
			/* NOT REACHED */
		}
	}

	argc -= optind;
	argv += optind;

	if (!context.server) {
		if (bdaddr_any(&context.raddr))
			errx(1, "Must specify server BD_ADDR"); 

		/* Check channel, if was not set then obtain it via SDP */
		if (context.channel == 0 && service != 0) {
			int	channel;

			if (rfcomm_channel_lookup(&context.laddr,
						&context.raddr,
						service, &channel, &n) != 0)
				errx(1, "Could not obtain RFCOMM channel: %s",
					strerror(n));

			context.channel = (uint8_t) channel;
		}
	}

	if ((context.channel == 0 && !context.server) || context.channel > 30)
		errx(1, "Invalid RFCOMM channel %d",
			context.channel);

	log_open("obexapp", pri_name, 0);

	/* Detach server (if required) */
	if (context.server && detach && daemon(0, 0) < 0) {
		log_err("%s(): Could not daemon. %s (%d)",
			__func__, strerror(errno), errno);
		exit(1);
	}

	/* Initialize OBEX */
	context.handle = OBEX_Init(OBEX_TRANS_CUSTOM, obexapp_event_handler, 0);
	if (context.handle == NULL) {
		log_err("%s(): OBEX_Init failed", __func__);
		exit(1);
	}

	OBEX_SetUserData(context.handle, &context);

	/* Register custom transport */
	if (OBEX_RegisterCTransport(context.handle, &custfunc) < 0) {
		log_err("%s(): OBEX_RegisterCTransport failed", __func__);
		exit(1);
        }

	if (context.server)
		n = obexapp_server(context.handle);
	else if (context.ni)
		n = obexapp_non_interactive_client(context.handle, argc, argv);
	else
		n = obexapp_client(context.handle);

	OBEX_Cleanup(context.handle);

	free(context.ls);
	free(context.root);
	free(context.file);
	free(context.temp);
	free(context.tbuffer);
	free(context.sbuffer);
	memset(&context, 0, sizeof(context));

	obexapp_util_locale_fini();

	log_close();

	exit(n);
} /* main */

/*
 * Display usage and exit
 */

static void
usage(char const *prog)
{
	fprintf(stderr,
"Usage: %s options\n" \
"Options:\n" \
"-a bdaddr      remote device address (required for client)\n" \
"-A bdaddr      local device address\n" \
"-c             act as client (default)\n" \
"-C channel     RFCOMM channel (required)\n" \
"-d             run server in foreground\n" \
"-D             use stdin/stdout instead of socket\n" \
"-f             use Folder Browsing Service\n" \
"-h             display this message\n" \
"-l priority    set least severe log priority\n" \
"-m MTU         set OBEX MTU\n" \
"-n             non-interactive mode (client mode)\n" \
"-r path        set path to root directory\n" \
"-s             act as server\n" \
"-S             secure mode (recommended)\n" \
"-u user        set user ID to 'user'\n" \
"",		prog);
	exit(255);
} /* usage */

