/*
 * ntesla - a program for designing Tesla Coils
 * 
 * Copyright (C) 1997  Steven A. Falco
 * 
 * Email contact: sfalco@worldnet.att.net
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#define EXTERN 
#define GLOBALS
#include "protos.h"

int
main(int argc, char *argv[])
{
	char buf[BS];

	/* have we been handed the save-file name? */
	if(argc > 1) {
		/* yes - set the filename */
		Filename = argv[1];
	} else {
		/* no - use the default */
		Filename = PARM_FILE;
	}

#ifdef unix
	sprintf(Tmpname, "%s.%d", TMP_FILE, getpid());
#endif

	/* Is there an existing design file we can use? */
	if(access(Filename, R_OK | W_OK) == 0) {

		/* This should always succeed */
		if((Fd_current_parms = open(Filename, O_RDWR | O_BINARY)) == -1) {
			P "Cannot open %s\n", Filename);
			exit(1);
		}

		/* opened OK - now load up the structure */
		if(read(Fd_current_parms, &Tcparms, sizeof(Tcparms))
		  != sizeof(Tcparms)) {
			P "Save file %s not compatible or corrupted - winging it!\n", Filename);
			P "Hit <ENTER> to continue\n");
			fgets(buf, BS, stdin);
		}

		/* Then get back to the beginning */
		if(lseek(Fd_current_parms, 0, SEEK_SET) == -1) {
			P "Cannot seek %s\n", Filename);
			exit(1);
		}

		/* Make sure it is the right version */
		if(Tcparms.version != VERSION) {
			P "WARNING: save file %s is version %.1f, but this program is version %.1f\n",
			  Filename, Tcparms.version, VERSION);
			P "Results cannot be guaranteed!\n");
			P "Hit <ENTER> to continue\n");
			fgets(buf, BS, stdin);
			Tcparms.version = VERSION;
		}
	} else {
	
		/* No existing design - init values to something reasonable */
		init();
		
		/* Create a new parameter file.  May not succeed if directory
		 * is locked.
		 */
		if((Fd_current_parms = open(Filename,
		   (O_TRUNC | O_RDWR | O_CREAT | O_BINARY), 0666)) == -1) {
			P "Cannot create %s\n", Filename);
			exit(1);
		}

		/* And write it */
		if(write(Fd_current_parms, &Tcparms, sizeof(Tcparms))
		  != sizeof(Tcparms)) {
			P "Cannot write %s\n", Filename);
			exit(1);
		}

		/* Then get back to the beginning */
		if(lseek(Fd_current_parms, 0, SEEK_SET) == -1) {
			P "Cannot seek %s\n", Filename);
			exit(1);
		}
	}
    
    Dialectric_count = sizeof(Cap_table) / sizeof(CAP_TABLE);
    
	/* We have opened or created the data file, and have a usable in-core
	 * copy of the data.  Put up the main menu.
	 */
	main_menu();

	EXIT(0);
}

void
main_menu()
{
	char buf[BS];
	int iii;

	while(1) {

		/* Set up the menu in memory */
		Row = 0;
S "Welcome to ntesla version %.1f - yet another Tesla coil design tool.\n",
	VERSION);
S "Copyright (C) 1997 Steven A. Falco\n");
S "                   sfalco@worldnet.att.net\n");
S "\n");
S "Enter the item letter you wish to select followed by <ENTER>.  (You can use\n");
S "upper-case or lower-case.)  The items are in logical order - it is best to\n");
S "start at the top.\n");
S "\n");
S "I - Introduction to using the ntesla program\n");
S "H - Help getting additional information\n");
S "E - Environment (inch or metric, printer parameters)\n");
S "X - Specify transformer parameters\n");
S "T - Design Toroid (top terminal)\n");
S "S - Design Secondary\n");
S "P - Design Primary\n");
S "C - Design Capacitor\n");
S "M - Miscellaneous design\n");
S "D - Dump all parameters to the printer\n");
S "Q - Quit program\n");
S "\n");
S "Make sure to exit ntesla via \"q\" from any menu, so ntesla can save your\n");
S "parameters.  Help is available on each design screen via \"h\".\n");
S "\n");

		/* Clear the screen, then display the menu and errors */
		CLEAR_SCREEN; for(iii = 0; iii < Row; iii++) P Display[iii]);
		SHOW_ERROR;
		P "[ Main Menu ] > ");

		/* Get the users choice */
		if(fgets(buf, BS, stdin) == NULL) {
			/* We are done */
			break;
		}

		switch(tolower(buf[0])) {

		  case 'i':
			intro();
		  	break;

		  case 'e':
			set_environment();
		  	break;

		  case 'x':
		  	transformer();
		  	break;

		  case 's':
		  	secondary();
		  	break;

		  case 'p':
		  	primary();
		  	break;

		  case 'c':
		  	capacitor();
		  	break;

		  case 't':
		  	toroid();
		  	break;

		  case 'm':
		  	misc();
		  	break;

		  case 'd':
		    dump_all();
		    break;
		    
		  case 'h':
		  	main_help();
		  	break;

		  case 'q':
		  	save_parms();
		  	return;

		  case '\n':
		  case '\r':
		  	break;

		  default:
		  	E "\007Sorry - %c is illegal - try again ", buf[0]);
			break;
		}
	}

	return;
}

void
set_environment()
{
	char buf[BS], *bp;
	int iii;

	while(1) {

		/* Build the screen image */
		Row = 0;
S "  *Units: %s\n", Tcparms.units ? "inch (English)" : "metric");
S "Set units to 'i' for inch (English) or 'm' for metric.\n");
S "\n");
S "  *Output: %s\n", Tcparms.printer);
#ifdef unix
S "Note - you can set \"output\" to 'cat >> foo' to keep appending your output\n");
S "to file 'foo'.  You can use 'pr | lpr' to paginate and send your output to\n");
S "the printer.\n");
#endif
#ifdef _DOS
S "Note - you can set \"output\" to 'PRN' for the standard printer, or any\n");
S "other device that will work on your system.  You can also select a file\n");
S "name if you want your output sent to that file.\n");
#endif
S "\n");
S "  *Separator: %s\n", show_sep());
S "Separator is a string which is output each time you use the 'p' (print)\n");
S "option.  Separator can be empty, or can be a form-feed, or a line like\n");
S "'-------------------------------' or '@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@'\n");
S "or whatever you like.  To enter a form-feed, type \\f.  Similarly, a\n");
S "new-line is entered as \\n, and a carriage-return as \\r.   Only the\n");
S "first 60 characters of the separator will be shown on the screen, but\n");
S "the entire separator will be printed.\n");
S "\n");

		/* Clear the screen, then display the menu */
		CLEAR_SCREEN; for(iii = 0; iii < Row; iii++) P Display[iii]);

		/* Put up general instructions */
		general_instructions();

		/* Get the users choice */
		if(fgets(buf, BS, stdin) == NULL) {
			/* We are done */
			break;
		}

		switch(tolower(buf[0])) {

		  case 'u':
		  	bp = buf + 1;
			while(*bp != 0 && *bp == ' ') bp++;
			if(tolower(*bp) == 'i') {
				Tcparms.units = INCH;
			} else if(tolower(*bp) == 'm') {
				Tcparms.units = METRIC;
			} else {
				E "\007Sorry - units cannot be set to %c - try again ", *bp);
			}
		  	break;
		  
		  case 'o':
		    /* Never want line terminator */
		  	strcpy(Tcparms.printer, get_sep(buf + 1, 0));
			break;

		  case 's':
		    /* Want line terminator if line ends with printable char */
		  	strcpy(Tcparms.separate, get_sep(buf + 1, 1));
			break;

		  case 'p':
		  	print_buf();
			break;

		  case 'h':
		  	env_help();
			break;

		  case 'q':
		  	save_parms();
			EXIT(0);

		  case 'r':
		  	return;

		  case '\n':
		  case '\r':
		  	break;

		  default:
			E "\007Sorry - %c is illegal - try again ", buf[0]);
			break;
		}
	}
}

/* Save parameters before we go */
void
save_parms()
{
	if(write(Fd_current_parms, &Tcparms, sizeof(Tcparms)) != sizeof(Tcparms)) {
		P "Cannot write %s\n", Filename);
	}
}

void
print_buf()
{
	/* Print from Display buffer */

	FILE *fp;
	int iii;
#ifdef unix
	char buf[BS];
#endif

#ifdef unix
	/* Dump to a temp. file, then toss it to the printer, and clean up */
	if((fp = fopen(Tmpname, "w")) == NULL) {
		E "\007Cannot open/create %s ", Tmpname);
#endif
#ifdef _DOS
	/* Dump it to selected device or file directly */
	if((fp = fopen(Tcparms.printer, "at")) == NULL) {
		E "\007Cannot open/create %s ", Tcparms.printer);
#endif
		
		return;
	}
	for(iii = 0; iii < Row; iii++) {
		fprintf(fp, Display[iii]);
	}
	fprintf(fp, Tcparms.separate);
	fclose(fp);
#ifdef unix
	/* Unix supports pipes for any program that reads/writes stdio. */
	sprintf(buf, "cat %s | %s", Tmpname, Tcparms.printer);
	system(buf);
	unlink(Tmpname);
#endif
}

void
general_instructions()
{
P "To change a parameter, enter the first letter of its name, the new value\n");
P "and <ENTER>.  To print this screen, use 'p'.  To return to the main menu,\n");
P "use 'r'.  To quit, use 'q'.  For help, use 'h'.\n");
P "\n");
SHOW_ERROR;
P "[ Design Menu ] > ");
}

void
intro()
{
	char buf[BS];
	int iii;

	while(1) {
		
		/* Build the screen image */
		Row = 0;

S "Ntesla helps you design the components of a Tesla Coil.  You select the\n");
S "desired topics from the main menu; ntesla then displays a design screen\n");
S "for that component.  On each screen, you will find parameters marked with\n");
S "an asterisk (*).  You can change these parameters by typing the first letter\n");
S "of the parameter name (upper or lower case) followed by the new value.\n");
S "Then hit the <ENTER> key to make the change happen.  For example, on the\n");
S "\"secondary design screen\" you will see *Gauge.  If you wanted to change to\n");
S "12 gauge wire, you would type g12 then hit <ENTER>.\n");
S "\n");
S "Ntesla remembers all the parameters you set in a file called \"%s\"\n",
   Filename);
S "so you only have to enter them once.  (The file name is the first argument on\n");
S "the command line; it defaults to \"%s\" if no file name is specified.)\n",
   PARM_FILE);
S "\n");
S "Please note that some parameters are dependent on other parameters.  You will\n");
S "be warned of this where appropriate.  You can temporarily override these\n");
S "parameters to see what effect they have, but they will revert to their\n");
S "previous values when you go back to the screen where they are calculated.  To\n");
S "make a permanent change in these parameters, you have to change the items from\n");
S "which they are calculated.\n");
S "\n");

		/* Clear the screen, then display the menu */
		CLEAR_SCREEN; for(iii = 0; iii < Row; iii++) P Display[iii]);

		/* Put up help instructions */
		help_instructions();

		/* Get the users choice */
		if(fgets(buf, BS, stdin) == NULL) {
			/* We are done */
			break;
		}

		switch(tolower(buf[0])) {

		  case 'p':
		  	print_buf();
			break;

		  case 'q':
		  	save_parms();
			EXIT(0);

		  case 'r':
		  	return;

		  case '\n':
		  case '\r':
		  	break;

		  default:
			E "\007Sorry - %c is illegal - try again ", buf[0]);
			break;
		}
	}

	return;

}

/* No existing design - init it */
void
init()
{
	Tcparms.version		= VERSION;
	Tcparms.units		= INCH;
#ifdef unix
	strcpy(Tcparms.printer, "pr | lpr");
#endif
#ifdef _DOS
	strcpy(Tcparms.printer, "PRN");
#endif
	strcpy(Tcparms.separate, "");
	Tcparms.primary_volts	= 12000.0;
	Tcparms.primary_amps	= 0.030;
	Tcparms.primary_va	= Tcparms.primary_volts * Tcparms.primary_amps;
	Tcparms.primary_type	= SHUNTED;
	Tcparms.primary_gap	= FIXED;
	Tcparms.primary_cap	= 0.00469;
	Tcparms.cap_type	= ROLLED;
	Tcparms.cap_dielectric	= 0;		/* LDPE */
	Tcparms.cap_length	= 39.0;
	Tcparms.cap_width	= 14.0;
	Tcparms.cap_gap		= 0.093;
	Tcparms.cap_diameter	= 5.0;
	Tcparms.cap_plates	= 2;
	Tcparms.primary_thick	= 0.25;
	Tcparms.primary_spacing	= 0.25;
	Tcparms.primary_inner	= 8.0;
	Tcparms.primary_turns	= 1;
	Tcparms.secondary_wire.gauge = 22;
	Tcparms.secondary_wire.thick =
		gauge_to_inch(Tcparms.secondary_wire.gauge);
	Tcparms.secondary_diam	= 6.0;
	Tcparms.secondary_leng	= 20.0;
	Tcparms.secondary_lfreq	= 0.0;
	Tcparms.terminal_minor	= 4.0;
	Tcparms.terminal_major	= 12.0;
	Tcparms.terminal_cap	= toroid_cap();
	Tcparms.misc_amp	= 1.0;
	Tcparms.misc_volt	= 1.0;
	Tcparms.misc_res	= 1.0;
	Tcparms.misc_freq	= 1.0;
	Tcparms.misc_cap	= 1.0;
	Tcparms.misc_ind	= 1.0;
	Tcparms.xf_freq		= 60.0;
	Tcparms.spool_wd	= 6.0;
	Tcparms.spool_cd	= 2.0;
	Tcparms.spool_len	= 4.0;
	Tcparms.spool_wire.gauge = 22;
	Tcparms.spool_wire.thick = 
		gauge_to_inch(Tcparms.spool_wire.gauge);
}

char *
UnitOut(void *val, int type, char *bp)
{
	if(type == TOBS) {
		/* use the saved copy rather than estimating gauge */
		sprintf(bp, "%d %s",
		  ((WIRE *)val)->gauge, Cvt[type].unit);
	} else if(type == TOWMM) {
		sprintf(bp, "%g %s",
		  ((WIRE *)val)->thick * Cvt[type].factor, Cvt[type].unit);
	} else {
		sprintf(bp, "%g %s",
		  *(double *)val * Cvt[type].factor, Cvt[type].unit);
	}

	return bp;
}

void
UnitIn(double inval, void *val, int type)
{
	if(type == TOBS) {
		/* save a copy of the gauge for later printout */
		((WIRE *)val)->thick = gauge_to_inch((int)inval);
		((WIRE *)val)->gauge = (int)inval;
	} else if(type == TOWMM) {
		((WIRE *)val)->thick = inval / Cvt[type].factor;
		((WIRE *)val)->gauge = inch_to_gauge(inval / Cvt[type].factor);
	} else {
		*(double *)val = inval / Cvt[type].factor;
	}
}

void
main_help()
{
	char buf[BS];
	int iii;

	while(1) {
		
		/* Build the screen image */
		Row = 0;
S "This program can help you design the components of a Tesla Coil, but it\n");
S "assumes that you already understand the general concepts.  It is not practical\n");
S "to provide schematics or detailed construction information here, but you will\n");
S "need this information to proceed.\n");
S "\n");
S "If you have access to the Internet, you should visit:\n");
S "        http://www.funet.fi/pub/sci/electrical/tesla\n");
S "where you will find a large quantity of information on Tesla Coils.\n");
S "\n");
S "If you cannot access the 'net, you should try a good library for the\n");
S "information you will need.\n");
S "\n");
S "CAUTION: Tesla Coils use and produce high voltage electricity.  You can easily\n");
S "electrocute yourself with one of these machines.  You assume all risk and\n");
S "liability for your creation.  Make sure you know how to proceed safely before\n");
S "you start!\n");
S "\n");

		/* Clear the screen, then display the menu */
		CLEAR_SCREEN; for(iii = 0; iii < Row; iii++) P Display[iii]);

		/* Put up help instructions */
		help_instructions();

		/* Get the users choice */
		if(fgets(buf, BS, stdin) == NULL) {
			/* We are done */
			break;
		}

		switch(tolower(buf[0])) {

		  case 'p':
		  	print_buf();
			break;

		  case 'q':
		  	save_parms();
			EXIT(0);

		  case 'r':
		  	return;

		  case '\n':
		  case '\r':
		  	break;

		  default:
			E "\007Sorry - %c is illegal - try again ", buf[0]);
			break;
		}
	}

	return;
}


void
env_help()
{
	char buf[BS];
	int iii;

	while(1) {
		
		/* Build the screen image */
		Row = 0;
S "Ntesla can use either the inch (English) system of measurement, or the Metric\n");
S "system.  All values are stored internally in the English system, and are\n");
S "converted to Metric on input and output as requested.  So, you can switch your\n");
S "design from one measurement system to the other without re-entering your data.\n");
S "\n");
S "Note that the English system uses the B&S 'gauge' to specify wire diameter.\n");
S "Ntesla uses a closed-form approximation to convert the gauge number to a\n");
S "diameter.  This is quite accurate (better than 0.1%%%% error in all cases).\n");
S "So, you can even use ntesla to convert wire sizes from B&S gauge to mm and\n");
S "vice versa.\n");
S "\n");
S "The output feature varies between Unix and DOS.  Unix is multi-tasking, so a\n");
S "complex pipeline of filters can be set up to post-process your output.  The\n");
S "most useful are 'pr | lpr' to paginate and print the output, and 'cat >> foo'\n");
S "to append all output to a file for subsequent printing.  DOS is much more\n");
S "limited; output is sent directly to the printer device (default is PRN)\n");
S "or to a file.\n");
S "\n");

		/* Clear the screen, then display the menu */
		CLEAR_SCREEN; for(iii = 0; iii < Row; iii++) P Display[iii]);

		/* Put up help instructions */
		help_instructions();

		/* Get the users choice */
		if(fgets(buf, BS, stdin) == NULL) {
			/* We are done */
			break;
		}

		switch(tolower(buf[0])) {

		  case 'p':
		  	print_buf();
			break;

		  case 'q':
		  	save_parms();
			EXIT(0);

		  case 'r':
		  	return;

		  case '\n':
		  case '\r':
		  	break;

		  default:
			E "\007Sorry - %c is illegal - try again ", buf[0]);
			break;
		}
	}

	return;
}

void
help_instructions()
{
P "To print this screen, use 'p'.  To return to the previous menu, use 'r'.\n");
P "To quit, use 'q'.\n");
P "\n");
SHOW_ERROR;
P "[ Help Menu ] > ");
}

/* This just affects the on-screen display.  Separator can be longer. */
#define WIDEST_SEP 60
char *
show_sep()
{
	char *bp, *sp;
	static char buf[BS];

	for(bp = buf, sp = Tcparms.separate; *sp; bp++, sp++) {
		switch(*sp) {
		  case '\f':
		  	*bp++ = '\\';
			*bp = 'f';
			break;

		  case '\r':
		  	*bp++ = '\\';
			*bp = 'r';
			break;

		  case '\n':
		  	*bp++ = '\\';
			*bp = 'n';
			break;

		  default:
		  	*bp = *sp;
			break;
		}
	}

	/* make sure we don't exceed the buffer length */
	if((bp - buf) > WIDEST_SEP) {
		buf[WIDEST_SEP] = (char)NULL;
	} else {
		*bp = (char)NULL;
	}
	return buf;
}

char *
get_sep(char *sp, int flag)
{
	char *bp;
	static char buf[BS];

	for(bp = buf; *sp; bp++, sp++) {
		if((bp - buf) >= BS) {
			break;
		}
		
		if(*sp == '\n' || *sp == '\r') {
			/* end of line - don't copy this character */
			break;
		} else if(*sp == '\\') {
			/* escape - do the substitution */
			switch(*++sp) {
			  case 'f':
			  	*bp = '\f';
				break;

			  case 'r':
			  	*bp = '\r';
				break;

			  case 'n':
			  	*bp = '\n';
				break;

			  default:
				/* unrecognized - treat as plain char */
				*bp = *sp;
				break;
			}
		} else {
			/* plain char */
		  	*bp = *sp;
		}
	}

	if(flag && isprint(*(bp - 1))) {
		/* want terminator */
		*bp++ = '\n';
	}
	*bp = (char)NULL;
	return buf;
}

void
dump_all()
{
	int iii;
	FILE *fp;
#ifdef unix
	char buf[BS];
#endif

#ifdef unix
	/* Dump to a temp. file, then toss it to the printer, and clean up */
	if((fp = fopen(Tmpname, "w")) == NULL) {
		E "\007Cannot open/create %s ", Tmpname);
#endif
#ifdef _DOS
	/* Dump it to selected device or file directly */
	if((fp = fopen(Tcparms.printer, "at")) == NULL) {
		E "\007Cannot open/create %s ", Tcparms.printer);
#endif
		
		return;
	}

	fprintf(fp, "Ntesla version %.1f\n", Tcparms.version);

	fprintf(fp, Tcparms.separate); fprintf(fp, "\nTransformer design:\n");
	transgen(0); for(iii = 0; iii < Row; iii++) fprintf(fp, Display[iii]);

	fprintf(fp, Tcparms.separate); fprintf(fp, "\nToroid design:\n");
	toroidgen(0); for(iii = 0; iii < Row; iii++) fprintf(fp, Display[iii]);

	fprintf(fp, Tcparms.separate); fprintf(fp, "\nSecondary design:\n");
	secgen(0); for(iii = 0; iii < Row; iii++) fprintf(fp, Display[iii]);

	fprintf(fp, Tcparms.separate); fprintf(fp, "\nPrimary design:\n");
	prigen(0); for(iii = 0; iii < Row; iii++) fprintf(fp, Display[iii]);

	fprintf(fp, Tcparms.separate); fprintf(fp, "\nCapacitor design:\n");
	capgen(0); for(iii = 0; iii < Row; iii++) fprintf(fp, Display[iii]);

	fprintf(fp, Tcparms.separate);
	fclose(fp);
#ifdef unix
	/* Unix supports pipes for any program that reads/writes stdio. */
	sprintf(buf, "cat %s | %s", Tmpname, Tcparms.printer);
	system(buf);
	unlink(Tmpname);
#endif	
}
