/* $Header: /home/vikas/src/nocol/trapmon/RCS/utils.c,v 1.1 2000/01/27 05:15:58 vikas Exp $ */

/*
 * Nocol specific trapmon code.
 *
 * AUTHOR:
 *	Vikas Aggarwal, vikas@navya.com, Nov 1999
 *
 */

/*
 * $Log: utils.c,v $
 * Revision 1.1  2000/01/27 05:15:58  vikas
 * Initial revision
 *
 */

#include	"trapmon.h"
#include	"snmp/snmp.h"		/* part of CMU snmp code */

char		*prognm = "trapmon";		/* Program name */
int		debug = 0;
static char	*datafile, *configfile;		/* nocol specific */

/*
 * Call nocol startup, and open various files.
 */
init_nocol()
{
  int fd;

  nocol_startup(&configfile, &datafile);
  if (debug) 
  {
    fprintf(stderr, 
	    "prognm set to %s, datafile to %s.\n", prognm, datafile);
    fflush(stderr);
  }

  unlink(datafile);
  fd = open(datafile, O_WRONLY|O_CREAT|O_TRUNC, DATAFILE_MODE);
  if (fd < 0) {
    closeeventlog();

    fprintf(stderr, "open() failed on %s\n", datafile);
    perror(prognm);
    exit(1);
  }
  close(fd);
  openeventlog();			/* Logging daemon */

  return 1;
}	/* init_nocol() */

/*
 * Appends a new event to the end of the datafile. For an enterprise specific
 * trap, it adds the 'specific_type' of the trap for identification.
 */
add_nocol_event(addr, name, pdu)
    char *addr;					/* Address of trap sender */
    char *name;					/* Name of trap sender */
    struct snmp_pdu *pdu;			/* trap PDU */
{
    int fd;
    struct tm *tm;
    time_t clock;
    EVENT v;

    bzero((char *)&v, sizeof(v));

    /*
     * In the following 'strncpy's, the null character at the end is
     * auto appended since to the bzero of the event struct and also
     * because we are doing a copy for 1 less than the size of the storage
     */
    strncpy(v.sender, prognm, sizeof(v.sender) - 1);	/* Set sender */
    strncpy(v.site.addr, addr, sizeof(v.site.addr) -1);	/* From address */
    strncpy(v.site.name, name, sizeof(v.site.name) -1);

    v.var.value = 0;				/* No value field */
    v.var.threshold = 0;			/* No threshold field */
    strncpy (v.var.units, VARUNITS, sizeof (v.var.units) - 1);

    clock = time(NULL);				/* Get the time of day... */
    tm = localtime(&clock);
    v.mon = tm->tm_mon + 1;			/* Don't forget to add 1! */
    v.day = tm->tm_mday;			/* Straightforward... */
    v.hour = tm->tm_hour;
    v.min = tm->tm_min;

    if (pdu->command == TRP_REQ_MSG)	/* V1 trap */
    {
      int trap = pdu->trap_type;			/* Trap number */
      strncpy(v.var.name, trap_desc[trap].tname, sizeof(v.var.name) - 1);
      if (trap == SNMP_TRAP_ENTERPRISESPECIFIC)	/* specific type */
      { 
	static char spdutype[16] ;		/* convert int to string */
	sprintf (spdutype, "/%d\0", pdu->specific_type) ;
	if ( (sizeof(v.var.name) - strlen(v.var.name)) > 5)
	  strncpy (v.var.name + strlen(v.var.name), spdutype, 4) ;
	else
	  strncpy (v.var.name + sizeof(v.var.name) - 5, spdutype, 4) ;
      }

      v.severity = trap_desc[trap].tseverity;	/* Set the severity level */
      v.nocop = SETF_UPDOUN (v.nocop, trap_desc[trap].nocop);
      v.loglevel = trap_desc[trap].loglevel ;	/* logging level */
    }
    else if (pdu->command == SNMP_PDU_V2TRAP)		/* V2 trap */
    {
      struct variable_list *vars;
      char varbuf[4096];

      varbuf[0] = '\0';
      for (vars = pdu->variables; vars; vars = vars->next_variable)
	sprint_variable(varbuf, vars->name, vars->name_length, vars);

      strncpy(v.var.name, varbuf, sizeof(v.var.name) - 1);
      v.severity = E_ERROR;	/* Set a default severity level for all v2 */
      v.nocop = SETF_UPDOUN (v.nocop, n_DOWN);
      v.loglevel = E_ERROR ;	/* default logging level for all v2*/
      
    }	/* end if v1/v2 trap */
    else 
    {
      fprintf(stderr, "Unknown trap PDU type %d\n", pdu->command);
      return 0;
    }

    eventlog(&v);				/* Log the event */

    fd = open(datafile, O_WRONLY | O_APPEND);	/* Add the new EVENT */
    if (write(fd, &v, sizeof(v)) != sizeof(v)) {
	fprintf(stderr, "write() failed on %s\n", datafile);
	perror("addevent()");

	closeeventlog();
	exit(1);
    }
    close(fd);

    /* Store the time after which the trap event is to be deleted */
    expire_at[numtraps++] = clock + TIME_TO_LIVE;	/* When to be expired... */

    if (debug)
	printf("Added: sendr=%s sitenm=%s siteaddr=%s varnm=%s severity=%d mon=%d day=%d hr=%d min=%d expire_at=%d\n",
	    v.sender, name, addr, v.var.name, v.severity, v.mon, v.day,
	    v.hour, v.min, expire_at[numtraps - 1]);
}



/*+
 * Open the datafile twice - readonly and writeonly, which gives us two
 * independent file pointers for the datafile. Then:
 *
 *	(1) Read an EVENT and advance the read-pointer.
 *	(2) If this EVENT must be expired, go to step 4.
 *	(3) Otherwise, write the EVENT and advance the write-pointer.
 *	(4) If end-of-file not reached yet, go back to 1.
 *	(5) Truncate the file.
 *
 * If, when reading an EVENT, we discover it needs to be expired, then 
 * dont write it out i.e. only the read-pointer advances (and the
 * write-pointer remains at the same location).
 * After the whole file has been read and all the timed-out EVENTs have
 * been expired, the file must be truncated; it must be shortened by an
 * amount proportional to the number of EVENTs expired.
 *
 */
int expire_nocol_events()
{
  int rd;				/* Read-only pointer */
  int wr;				/* Write-only pointer */
  int i;				/* Read index into expire_at[] */
  int j;				/* Write index into expire_at[] */
  EVENT v;				/* EVENT buffer */
  time_t expire_date;			/* expire_at[] buffer */
  time_t clock;				/* Current time */

  i = j = 0;
  clock = time(NULL);				/* Get the current time */
  if ((rd = open(datafile, O_RDONLY)) < 0)	/* Init read pointer */
  {
    fprintf(stderr, "%s (expire_nocol_events): %s ", prognm, datafile) ;
    perror ("read open()");
    closeeventlog();
    return (-1);
  }
  /* Init write pointer */
  if ((wr = open(datafile, O_WRONLY, DATAFILE_MODE)) < 0)
  {
    fprintf(stderr, "%s (expire_nocol_events): %s ", prognm, datafile) ;
    perror ("write open()");
    closeeventlog();
    return (-1);
  }	

  if (debug)
    printf("Entered expire_nocol_events() numtraps=%d clock=%d... ",
	   numtraps, clock);

  while (read(rd, &v, sizeof(v)) == sizeof(v))/* Read EVENTs one by one */
    if ((expire_date = expire_at[i++]) > clock) {  /* expire this EVENT? */
      write(wr, &v, sizeof(v));		/* No - keep this EVENT */
      expire_at[j++] = expire_date;
      if (debug)
	printf("kept, ");
    }
    else if (debug)			/* Yes - expire this EVENT */
      printf("expired, ");

  numtraps = j;
  ftruncate(wr, (off_t) j * sizeof(EVENT));	/* Shorten the file */

  if (debug)
    printf("truncated, ");

  close(rd);					/* Done with read pointer */
  close(wr);					/* Done with write pointer */

  if (debug) {
    printf("Done.\n");
    fflush(stdout);
  }
}	/* expire_nocol_events() */

