/*
 * $Revision: 1.1 $
 */ 

/*+ ntpmon
 * 	Network Time Protocol monitoring package.
 * This is a part of the nocol package that monitors NTP strata of
 * remote hosts.
 *
 * Method :
 * 	Create a standard ntp control package with only the header(no data)
 *	Send it over a UDP socket.Then wait for a reply.
 *	The stratum value is contained in the data field of the reply.
 *	Stratum ranges from 0 to 16. Based on this calculate the error level.
 *
 * Copyright 1997 Netplex Technologies Inc., info@netplex-tech.com
 *
 */	

 /*
  * $Log: main.c,v $
  * Revision 1.1  1998/06/30 20:26:03  vikas
  * Initial revision
  *
  *
  */ 

#include <stdio.h>
#include <fcntl.h>
#include "nocol.h"
#include "ntpmon.h"

char *prognm;
static char *configfile, *datafile, *sender;

/* We keep a linked list of all the sites that we poll and store the
 * various thresholds in this linked list.
 */
struct site_info *site_info_list = NULL;	

void free_site_list();
static char *Strdup();			/* replacement for strdup() */

static time_t	pollinterval ;		/* Time between polls */
int debug = 0;

int main(argc, argv)
  int argc;
  char **argv;
{
  int fdout = 0;
  time_t starttm, polltime ;
  extern char *optarg;
  extern int optind;
  register int c ;

  prognm = argv[0];
#ifdef SENDER
    sender = SENDER ;
#else						/* delete the directory name */
    if ((sender = (char *)strrchr (prognm , '/')) == NULL)
      sender = prognm ;				/* no path in program name */
    else
      sender++ ;				/* skip leading '/' */
#endif

    while ((c = getopt(argc, argv, "do:")) != EOF)
      switch (c) 
      {
      case 'd':
	debug++ ;
	break ;
	
      case 'o':				/* output datafile */
	datafile = (char *)Strdup(optarg);
	break ;
      case '?':
      default:

	help() ;
	goto Cleanup ;
      }

    switch (argc - optind) 
    {
    case 0:					/* default input file */
      break ;
    case 1:
      configfile = (char *)Strdup(argv[optind]);
      break ;
    default:
      fprintf (stderr, "%s Error: Too many 'hosts' files\n\n", prognm);
      help() ;
      goto Cleanup;
    }


  nocol_startup(&configfile, &datafile);
  
  if (fdout != 0)			/* Say, recovering from longjmp	*/
    close(fdout);

  if ( (fdout = open(datafile, O_RDWR|O_CREAT|O_TRUNC, DATAFILE_MODE)) < 0) {
    fprintf(stderr, "(%s) ERROR in open datafile ", prognm);
    perror (datafile);
    goto Cleanup ;
  }

    openeventlog() ;			/* Event logging */
    if (init_sites(fdout, configfile) == -1 )
      goto Cleanup ;

    if (!pollinterval)
      pollinterval = POLLINTERVAL ;		/* default value */

  
    /* poll_sites makes one complete pass over the list of nodes */
    while (1)			      		/* forever */
    {
	starttm = time((time_t *)NULL) ;	/* time started this cycle */
	if (poll_sites(fdout) == -1)		/* Polling error */
	  break ;

	if ( (polltime = time((time_t *)NULL) - starttm) < pollinterval)
	{
	  if(debug)
	    fprintf(stderr, "(debug)%s: Sleeping for %u seconds.ZZZ...\n",
		    prognm, (unsigned)(pollinterval - polltime));
	 
	  sleep ((unsigned)(pollinterval - polltime));
	}
    }

    /* HERE ONLY IF ERROR */
  Cleanup:
    nocol_done();
 
}	/* end main() */

/*
 * Read the config file.Returns 1 on success, 0 on error.
 * Do other initializtions.
 * Format :
 * 	POLLINTERVAL  <time in seconds>
 * 	<hostname> <address> <Warn level> <Error Level> <Critical level>
 */
int init_sites(fdout, configfile)
  int fdout;
  char *configfile;
{
  char w1[MAXLINE], w2[MAXLINE], w3[MAXLINE], w4[MAXLINE], w5[MAXLINE];	
  /* Confg words	*/
  struct tm *loctime ;
  time_t locclock ;			/* Careful, don't use 'long'	*/
  FILE *p_nodes ;
  char record[MAXLINE];
  EVENT v;				/* Defined in NOCOL.H		*/
  struct site_info *lastnode = NULL, *newnode;

  if(site_info_list)
    free_site_list(&site_info_list);	/* In case rereading confg file */

  if ((p_nodes = fopen(configfile, "r")) == NULL)
  {
    fprintf(stderr, "%s error (init_sites) ", prognm) ;
    perror (configfile);
    return (-1);
  }
  
  /*
   * Fill in the static data stuff
   */
    bzero (&v, sizeof(v)) ;
    locclock = time((time_t *)0);
    loctime = localtime((long *)&locclock);

    v.mon = loctime->tm_mon + 1;	v.day = loctime->tm_mday;
    v.hour = loctime->tm_hour;		v.min = loctime->tm_min;

    /*
     * in the following strncpy's, the NULL is already appended because
     * of the bzero, and copying one less than size of the arrays.
     */
    strncpy (v.sender, sender, sizeof(v.sender) - 1);

    strncpy (v.var.name, VARNM, sizeof (v.var.name) - 1);
    strncpy (v.var.units, VARUNITS, sizeof (v.var.units) - 1);
    v.var.value = 0 ; v.var.threshold = 0 ;	/* threshold not used */
    v.nocop = SETF_UPDOUN (v.nocop, n_UNKNOWN);	/* Set all to UNKNOWN	*/
    v.severity = E_INFO ;

  while(fgetLine(p_nodes,record,MAXLINE) > 0 ) 
  {

    int rc;						/* return code	*/

    v.nocop = 0 ;				/* Init options to zero	*/
    *w1 = *w2 = *w3 = *w4 = *w5 = '\0' ;
    rc = sscanf(record,"%s %s %s %s %s", w1, w2, w3, w4, w5);
    if (rc == 0 || *w1 == '\0' || *w1 == '#')  /* Comment or blank 	*/
      continue;

    /* Looking for "POLLINTERVAL xxx" 	*/
    if (strncasecmp(w1, "POLLINTERVAL", 8) == 0)
    {
      char *p; 					/* for strtol */
      pollinterval = (u_long)strtol(w2, &p, 0) ;
      if (p == w2)
      {
	fprintf(stderr,"(%s): Error in format for POLLINTERVAL '%s'\n",
		prognm, w2) ;
	pollinterval = POLLINTERVAL ;		/* reset to default above */
      }
      
      continue;
    }
    
    /* Only <name> <addr> <Wlevel> <Elevel> <Critlevel>  remain */
    strncpy(v.site.name, w1, sizeof(v.site.name) - 1);
    strncpy(v.site.addr, w2, sizeof(v.site.addr) - 1);	/* no checks */
    
    if (inet_addr(w2) == -1)	/* bad address */
    {
      fprintf(stderr,
	      "	(%s): Error in address '%s' for site '%s', ignoring\n",
	      prognm, w2, w1);
      continue ;
    }

    if (*w3 == '\0' || *w4 == '\0' || *w5 == '\0' )
    {
      fprintf(stderr, "%s: Warning levels not specified correctly\n", prognm);
      fprintf(stderr, "\t '%s'\n", record);
      continue;
    }

    /*
     * Create a site info node and add to the tail of linked list
     */
    newnode = (struct site_info *)calloc(1, sizeof(struct site_info));
    if(!newnode)
    {
      fprintf(stderr,"%s: Out of Memory ",prognm);
      return -1;
    }

    newnode->wlevel = atoi(w3);    
    newnode->elevel = atoi(w4);    
    newnode->critlevel = atoi(w5);
    newnode->next = NULL;
	
    if(!site_info_list)
      site_info_list = newnode;		/* This is the first node (head) */
    else
      lastnode->next = newnode;
	  
    lastnode = newnode;

    write (fdout, (char *)&v, sizeof(v)) ;
  }	/* end: while			*/
}

/*
 * This func. calls ntpmon, gets the stratum  for
 * every site and writes the output
 */

poll_sites(fdout)
     int fdout;				/* Descriptors to open files	*/
{
  EVENT v;			    	/* described in nocol.h		*/
  struct tm *ltime ;    
  time_t locclock ;			/* careful: don't use 'long'	*/
  int status;		       		/* site status			*/
  int bufsize;				/* for reading in file */
  int sigpid;				/* PID of program to get signal	*/
  struct site_info *curnode;		/* current site being processed */

  if ( lseek(fdout, (off_t)0, SEEK_SET) == -1) 
  {			 /* rewind the file	*/
    perror (prognm);
    return (-1);
  }

  /* 
   * until end of the file or erroneous data... one entire pass
   */

  curnode = site_info_list;	/* Point to the head of Linked List */

  while ( (bufsize = read (fdout, (char *)&v, sizeof(v))) == sizeof(v) ) 
  {
   int maxseverity, thres, stratum;

   stratum = ntpmon(v.site.addr);
   status = calc_status(stratum, curnode->wlevel, curnode->elevel, 
			curnode->critlevel, &thres, &maxseverity) ;

   if (debug)
   {
     fprintf(stderr, "(debug) %s: NTP status is %d/%s.Val=%lu\n",
	     prognm, status, status ? "UP" : "DOWN",  (u_long)stratum);
     fflush(stderr);
   }

   update_event(&v, status, /* VALUE */ stratum, maxseverity);
      
   /* Now rewind to start of present record and write to the file  */
   lseek(fdout,-(off_t)sizeof(v), SEEK_CUR);
   write (fdout, (char *)&v, sizeof(v));

   curnode = curnode->next;
     
  }	/* end of:    while (read..)	*/
    
  /**** Now determine why we broke out of the above loop *****/
    
  if (bufsize == 0)			/* reached end of the file	*/
    return (1);
  else {				/* error in output data file 	*/
    fprintf (stderr, "%s: Insufficient read data in output file", prognm);
    return (-1);
  }

}	/* end of:  poll_sites		*/

/*
 *      calc_status()
 *  Useful to extract the status and maximum severity in monitors which
 * have 3 thresholds. Given the three thresholds and the value, it returns
 * the 'status, thres, maxseverity'.
 * Handle increasing or decreasing threshold values.
 * Adapted from the perl library version.
 * Should be in the library..
 */

int calc_status(val, warnt, errt, critt, thres, maxseverity)
  int  val, warnt, errt, critt;		/* the value and 3 thresholds */
  int *thres, *maxseverity;		/* Return */

{
    int status = 1;
    int gt = 1 ;   	/* test if value *greater* than thresholds */

    if (critt < warnt)  
      gt = 0 ; 		/*  test if value *less* than thres */

    if ( (gt && val >= warnt) || (gt == 0 && val <= warnt) ) 
      status = 0;          /* value exceeds warning threshold */

    if ( (gt && val >= critt) || (gt == 0 && val <= critt) ) 
    { 
      *thres = critt;
      *maxseverity = E_CRITICAL;
    }
    else if ( (gt && val >= errt) || (gt == 0 && val <= errt) )
    { 
      *thres = errt;
      *maxseverity = E_ERROR;
    }
    else if ( (gt && val >= warnt) || (gt == 0 && val <= warnt) )
    { 
      *thres = critt;
      *maxseverity = E_WARNING;
    }
    else    
    { 
      *thres = warnt;
      *maxseverity = E_INFO;
    } /* status should be UP */

    return status; 

}       /* end calc_status */


help()
{
  fprintf(stderr,"Usage:%s [-d] [-o outputfile] configfile \n", prognm);
}

void free_site_list(si_list)
  struct site_info **si_list;
{
  struct site_info *curptr, *nextptr;

  for(curptr = *si_list; curptr ; curptr = nextptr)
  {
    nextptr = curptr->next;		/* store next cell before freeing */
    free(curptr);
  }

  *si_list = NULL;
}	/* end free_site_list() */

static char *Strdup(s)
  char *s ;
{
  char *t ;

  if (s == NULL)
  {
    t = (char *)malloc(1);
    *t = '\0';
  }
  else
  {
    t = (char *)malloc(strlen(s) + 1);
    if (t != NULL)
      (char *)strcpy(t, s);
  }

  return (t);
}
