#ifndef lint
static char *RCSid = "$Header: /home/vikas/src/nocol/etherload/RCS/nocol_specific.c,v 1.5 1997/08/22 06:08:57 vikas Exp $" ;
#endif

#include "os.h"

#ifdef NOCOL

/*
 * NOCOL specific routines for the 'etherload' program.
 *
 *	Vikas Aggarwal, vikas@navya.com  Feb 1994
 *
 *
 * $Log: nocol_specific.c,v $
 * Revision 1.5  1997/08/22 06:08:57  vikas
 * Now typecasting all the strtok() to char * instead of a global
 * declaration (was messing up on some system somewhere)
 *
 * Revision 1.4  1997/02/11 10:38:06  vikas
 * Including file os.h to allow undefining NOCOL for standalone
 *
 * Revision 1.3  1994/04/13 18:22:45  vikas
 * Now can specify 'sleepsecs' also instead of 'pollinterval'
 *
 * Revision 1.2  1994/04/13  15:50:00  vikas
 * Now just copying pointers over in the interface info.
 * Also some other cleanups.
 *
 * Revision 1.1  1994/03/28  14:38:14  vikas
 * Initial revision
 *
 *
 */


#include <string.h>

#include "nocol.h"
#include "etherload.h"
#include "externs.h"		/* defines  extern bw, pps, debug */


static int	fdout = 0;			/* output data file */
static char	*datafile, *configfile, *sender;

/*
 * Note that E_CRITICAL = 1 & E_INFO = 4. We only need 3 spaces in the
 * thres[x][], but easier to reference using these ENUM types.
 */
enum { BW, PPS, EOVARS } ;		/* list of variables checked */
static char *varnames[] = {		/* names of the variables */
  "bw", "pps", "" };

struct _nocol_if			/* store thresholds for each var */
{
  char	*name ;				/* device name in the config file */
  int	thres[E_INFO][EOVARS];		/* the three thresholds */
} cfnocol_if[MAXINTERFACES], *pnocol_if[MAXINTERFACES] ;


/*
 * Called upon startup to do device specific stuff
 */
nocol_begin(cf_file)
  char *cf_file;
{

#ifdef SENDER
  sender = SENDER ;
#else					/* delete the directory name */
  if ((sender = (char *)strrchr (prognm , '/')) == NULL)
    sender = prognm ;                         /* no path in program name */
  else
    sender++ ;                                /* skip leading '/' */
#endif

  configfile = cf_file;		/* if specified from calling prog */
  nocol_startup(&configfile, &datafile);

  if ( (fdout = open(datafile, O_RDWR|O_CREAT|O_TRUNC, DATAFILE_MODE)) < 0)
  {
    fprintf(stderr, "(%s) ERROR in open datafile ", prognm);
    perror (datafile);
    finish(-1);
  }

  if (readconfig(configfile) == -1)
    finish(1);

}	/* end nocol_begin() */


/*
 * Called just before beginning the tests.... Write out the initial
 * EVENT structures to the output datafile.
 */

nocol_prep()
{
  register int i, j;
  int nvar, nif;
  extern int	ninterfaces;		/* number of valid interfaces found */
  char myhostname[256];
  EVENT v;                            	/* Defined in NOCOL.H */
  struct tm *loctime ;
  time_t locclock ;                   	/* Careful, don't use 'long'  */

  /*
   * Get rid of the unavailable interfaces from the 'nocol_if' struct
   * so that the 'if_stats' interfaces match the 'nocol_if' struct.
   * DONT 'free' the char strings- they are pointed to by a number
   * of other vars like devlist, etc.
   */

  for (i=0; i < ninterfaces; ++i)
    for (j=0; cfnocol_if[j].name != NULL; ++j)
      if (strcmp(cfnocol_if[j].name, if_stats[i].name) == 0)
      {
	pnocol_if[i] = &(cfnocol_if[j]);

	if (debug)
	{
	  int s, t;

	  fprintf (stderr, "(debug)%s thresholds:\n",
		   pnocol_if[i]->name);
	  for (s = 0; s < EOVARS; ++s)
	  {
	    fprintf (stderr, "%s: ", varnames[s]);
	    for (t = E_WARNING ; t >= E_CRITICAL; --t)
	      fprintf (stderr, " %d", pnocol_if[i]->thres[t][s]);
	    fprintf (stderr, "\n");
	  }
	}
	      
	break;
      }

  /*
   * Fill in the static data stuff
   */
  bzero (&v, sizeof(v)) ;

  locclock = time((time_t *)0);
  loctime = localtime((long *)&locclock);

  v.mon = loctime->tm_mon + 1;        v.day = loctime->tm_mday;
  v.hour = loctime->tm_hour;          v.min = loctime->tm_min;

  strncpy (v.sender, sender, sizeof(v.sender) - 1);

  v.nocop = SETF_UPDOUN (0, n_UNKNOWN); /* Set all to UNKNOWN   */
  v.severity = E_INFO ;

  gethostname(v.site.name, sizeof(v.site.name) - 1) ;	/* null from bzero */

  for (nif = 0; nif < ninterfaces; ++nif)
    for (nvar = 0; nvar < EOVARS; ++nvar)
    {
      strncpy(v.site.addr, if_stats[nif].name, sizeof(v.site.addr) - 1);
      strncpy(v.site.addr + strlen(v.site.addr), "/",
	      sizeof(v.site.addr) - strlen(v.site.addr) - 1);
      strncpy(v.site.addr + strlen(v.site.addr),
	      if_stats[nif].type,
	      sizeof(v.site.addr) - strlen(v.site.addr) - 1);
      switch (nvar)
      {
      case BW:
	strncpy (v.var.units, "%age", sizeof (v.var.units) - 1);
	strncpy(v.var.name, "Bandwidth", sizeof(v.var.name) - 1);
	v.var.threshold = 0 ;	/* any value for now */
	break;

      case PPS:
	strncpy (v.var.units, "pps", sizeof (v.var.units) - 1);
	strncpy(v.var.name, "PktsPerSec", sizeof(v.var.name) - 1);
	v.var.threshold = 0 ;	/* any value for now */
	break;

      default:
	fprintf(stderr, "%s: fatal, went past EOVARS in nocol_prep\n",
		prognm);
	finish(-1);
      }

      if (write (fdout, (char *)&v, sizeof(v)) != sizeof(v))
      {
	fprintf(stderr, "%s (write): %s\n", prognm,sys_errlist[errno]);
	finish(-1) ;
      }
    }

  return(1);                          /* All OK  */

}	/* end:  nocol_prep */


/*
 * Called just before the program exits.
 */
nocol_wrapup()
{
  nocol_done();	/* delete the pid and data files and exit */
}


/*
 * Print statistics- NOCOL specific.
 */
void
nocol_printstats()
{
  int nif, nvar;
  extern int debug;

  (void) gettimeofday(&endtime, (struct timezone *) 0);
  totaltime = (u_long)endtime.tv_sec - (u_long)starttime.tv_sec ;

  if (totaltime <= 0)
    return;

  lseek(fdout, (off_t)0, SEEK_SET);	/* rewind data file */

  for (nif = 0; nif < ninterfaces; nif++)
  {
    dostats (&if_stats[nif]);
    if (debug)
      fprintf(stderr, "BW= %d PPS=%d\n", bw, pps);
    for (nvar = 0; nvar < EOVARS; nvar++)
    {
      int status ;
      int maxseverity;
      u_long value;	/* u_long because of VAR in nocol struct */
      EVENT v;		/* Do PPS AND BandWidth */

      switch (nvar)
      {
      case BW:
	value = bw; break;
      case PPS:
	value = pps; break;
      default:
	fprintf(stderr, "%s: fatal error, went past EOVARS\n", prognm);
	finish (-1);
      }

      status = calc_status(value, nif, nvar);
      maxseverity = calc_maxseverity(value, nif, nvar);

      read(fdout, &v, sizeof v);
      update_event(&v, /* up or down */status, value, maxseverity) ;
      /* update with current threshold */
      if (maxseverity == E_INFO)
	v.var.threshold = pnocol_if[nif]->thres[E_WARNING][nvar];
      else
	v.var.threshold = pnocol_if[nif]->thres[maxseverity][nvar];
      lseek(fdout, -(off_t)sizeof(v), SEEK_CUR);
      write(fdout, (char *)&v, sizeof(v));
    }	/* end: for(nvar) */

  }	/* end:  for(nif) */

}	/* end:  nocol_printstats()  */


/*
 * Given the var-index and interface-index into the nocol_if[] array, it
 * compares the value with the WARNING threshold to see if the event should
 * be flagged as up or down.
 */
calc_status(value, nif, nvar)
  long value;
int nif, nvar;
{
  if (value >= pnocol_if[nif]->thres[E_WARNING][nvar]) /* exceed thres */
    return 0;
  else
    return 1;		/* okay */
}

/*
 * Given the var-index and interface-index into the nocol_if[] array, it
 * compares the value against the 3 thresholds to see which threshold is
 * exceeded.
 */
calc_maxseverity(value, nif, nvar)
  long value;
int nif, nvar;
{
  int i;

  for (i = E_CRITICAL ; i < E_INFO ; ++i)
  {
    if (debug > 1)
      fprintf(stderr, "calc_maxsev: checking %ld > %ld\n",
	      value, pnocol_if[nif]->thres[i][nvar]);
    if (value >= pnocol_if[nif]->thres[i][nvar])
      return (i);
  }
  return E_INFO;
}


/*+ 
 * Read in the configuration file.
 *
 * SLEEPSECS ppp (or POLLINTERVAL secs)
 * SCANSECS sss
 * DEVICE device
 * BW	thres1 thres2 thres3
 * PPS	thres1 thres2 thres3
 *
 */

readconfig(cfgfile)
  char *cfgfile;
{
  register int i;
  int ndevs = -1;
  int bwdone =1, ppsdone =1;
  char line[MAXLINE];
  FILE *cfile = NULL;
  extern char *skip_spaces(), *Strdup();
  extern char **devlist;		/* in etherload.h */
    

  if (debug)
    fprintf (stderr, "Opening config file '%s'\n", configfile);

  if ((cfile = fopen(configfile, "r")) == NULL)
  {
    fprintf(stderr, "%s error (init_sites) ", prognm) ;
    perror (configfile);
    return (-1);
  }

  while (fgets(line, sizeof (line), cfile) != NULL)
  {
    char *s;
    static int linenum = 0;

    s = (char *)strchr(line, '\n');
    *s = '\0';			/* rid of newline */
    s = line;
    ++linenum;
    if (debug > 3)
      fprintf (stderr, "Config %.2d: %s\n", linenum, line);

    s = skip_spaces(s);

    if (*s == '#' || *s == '\0')
      continue;

    if (strncasecmp(s, "pollinterval", strlen("pollinterval")) == 0  ||
	strncasecmp(s, "sleep", strlen("sleep")) == 0)
    {
      s = (char *)strtok(s, " \t");
      sleepsecs = atoi (skip_spaces((char *)strtok(NULL, " \t\n")));
      if (debug > 1)
	fprintf(stderr, "readconfig(): sleepsecs/pollinterval= %d\n",
		sleepsecs);
      continue;
    }

    if (strncasecmp(s, "scan", strlen("scan")) == 0)
    {
      s = (char *)strtok(s, " \t");
      scansecs = atoi (skip_spaces((char *)strtok(NULL, " \t\n")));
      if (debug > 1)
	fprintf(stderr, "readconfig(): scansecs= %d\n", scansecs);
      continue;
    }

    if (strncasecmp(s, "device", strlen("device")) == 0)
    {
      if (bwdone != 1 && ppsdone != 1)
	fprintf (stderr,
		 "%s [%d]: error- incomplete DEVICE config, ignoring\n",
		 prognm, linenum);
      else
	++ndevs ;			/* increment only if prev was okay */
      s = (char *)strtok(s, " \t");
      s = skip_spaces((char *)strtok(NULL, " \t\n"));
      cfnocol_if[ndevs].name = Strdup(s);

      bwdone =0; ppsdone =0;
      if (debug > 1)
	fprintf(stderr, "readconfig(): device = %s\n", s);
      continue;
    }

    if (strncasecmp(s, "pps", strlen("pps")) == 0)
    {
      if (ppsdone)
      {
	fprintf(stderr,"%s [%d]: error- got PPS line without DEVICE\n",
		prognm, linenum);
	continue ;
      }
      if (debug > 1)
	fprintf(stderr, "readconfig(): PPS thres= ");

      s = (char *)strtok(s, " \t");
      for (i=E_WARNING ; i >= E_CRITICAL; --i)
      {
	if ((s = (char *)strtok(NULL, " \t\n")) != NULL)
	  cfnocol_if[ndevs].thres[i][PPS] = atoi(skip_spaces(s));
	else
	  cfnocol_if[ndevs].thres[i][PPS] = 0;

	if (debug > 1)
	  fprintf (stderr, 
		   "%d(%s) ", cfnocol_if[ndevs].thres[i][PPS],
		   severity_txt[i]);
      }
      ppsdone =1;
      if (debug > 1)
	fprintf (stderr, "\n");
      continue;
    }	

    if (strncasecmp(s, "bw", strlen("bw")) == 0)
    {
      if (bwdone)
      {
	fprintf(stderr, "%s [%d]: error- got BW line without DEVICE\n",
		prognm, linenum);
	continue;
      }
      if (debug > 1)
	fprintf(stderr, "readconfig(): BW thres= ");

      s = (char *)strtok(s, " \t");
      for (i=E_WARNING ; i >= E_CRITICAL; --i)
      {
	if ((s = (char *)strtok(NULL, " \t\n")) != NULL)
	  cfnocol_if[ndevs].thres[i][BW] = atoi(skip_spaces(s));
	else
	  cfnocol_if[ndevs].thres[i][BW] = 0;

	if (debug > 1)
	  fprintf (stderr, "%d(%s) ", cfnocol_if[ndevs].thres[i][BW],
		   severity_txt[i]);
      }
      bwdone =1;
      if (debug > 1)
	fprintf (stderr, "\n");
      continue;
    }	

    fprintf (stderr, "%s [%d]: Ignoring config '%s'\n", prognm, linenum,
	     line);

  }	/* end: while()  */

  fclose(cfile);
  if (++ndevs < MAXINTERFACES)
    cfnocol_if[ndevs].name = NULL;	/* last one in list */

  /*
   * Create list of devices read in from the config file. Used by the
   * main module to see which interfaces it should look for.
   */
  devlist = (char **)malloc((ndevs + 1) * sizeof (char *));
  for (i = 0 ; i <= ndevs ; ++i)
    *(devlist + i) = cfnocol_if[i].name ;


  return (0);
}


#endif NOCOL

