#! /usr/bin/perl
#
# Copyright 2005-2021 The Mumble Developers. All rights reserved.
# Use of this source code is governed by a BSD-style license
# that can be found in the LICENSE file at the root of the
# Mumble source tree or at <https://www.mumble.info/LICENSE>.

use warnings;
use strict;
use Carp;

sub licenseFileToVar($$) {
  my ($var,$file)=@_;

  my $ret;


  open(IN, $file) or croak;
  my $l = join("", <IN>);
  $l =~ s/\r//g;
  $l =~ s/\f//g;
  $l =~ s/\"/\\\"/g;

  $l = join("\\n\"\n\t\"",split(/\n/, $l));

  return qq!static const char *${var} = \n\t\"! . $l . "\";\n";
}

sub printGuarded($$$) {
  my ($F, $S, $Guard)=@_;
  
  if ($Guard) {
    print $F "#ifdef " . $Guard . "\n";
  }
  print $F $S;
  
  if ($Guard) {
    print $F "#endif\n";
  }
}


open(my $F, "> ../src/licenses.h");
binmode $F; # Ensure consistent file endings across platforms

print $F "/*\n";
print $F " * This file was auto-generated by scripts/mklic.pl\n";
print $F " * DO NOT EDIT IT MANUALLY\n";
print $F " */\n";
print $F "#ifndef MUMBLE_LICENSES_H_\n";
print $F "#define MUMBLE_LICENSES_H_\n";
print $F "\n";
print $F "#include <QtGlobal>\n";
print $F "\n";
print $F "struct ThirdPartyLicense {\n";
print $F "	const char* name;\n";
print $F "	const char* url;\n";
print $F "	const char* license;\n";
print $F "\n";
print $F "	ThirdPartyLicense() : name(0), url(0), license(0) {}\n";
print $F "	ThirdPartyLicense(const char* name_, const char* url_, const char* license_)\n";
print $F "	    : name(name_), url(url_), license(license_) {}\n";
print $F "	bool isEmpty() const { return (name == 0 && url == 0 && license == 0); }\n";
print $F "};\n";
print $F "\n";


print $F licenseFileToVar("licenseMumble", "../LICENSE");
print $F "\n\n";

print $F licenseFileToVar("authorsMumble", "../AUTHORS");
print $F "\n\n";

# List of 3rd party licenses  [<variableName>, <pathToLicenseFile>, <DisplayName>, <URL>]
my @thirdPartyLicenses = (
    ["licenseCELT", "../3rdparty/celt-0.7.0-src/COPYING", "CELT", "http://www.celt-codec.org/"],
    ["licenseOpus", "../3rdparty/opus-src/COPYING", "Opus", "http://www.opus-codec.org/"],
    ["licenseSPEEX", "../3rdparty/speex-src/COPYING", "Speex", "http://www.speex.org/"],
    ["licenseOpenSSL", "../3rdPartyLicenses/openssl_license.txt", "OpenSSL", "http://www.openssl.org/"],
    ["licenseLibsndfile", "../3rdPartyLicenses/libsndfile_license.txt", "libsndfile", "http://www.mega-nerd.com/libsndfile/"],
    ["licenseOgg", "../3rdPartyLicenses/libogg_license.txt", "libogg", "http://www.xiph.org/"],
    ["licenseVorbis", "../3rdPartyLicenses/libvorbis_license.txt", "libvorbis", "http://www.xiph.org/"],
    ["licenseFLAC", "../3rdPartyLicenses/libflac_license.txt", "libFLAC", "http://flac.sourceforge.net/"],
    ["licenseMachOverride", "../3rdPartyLicenses/mach_override_license.txt", "mach_override", "https://github.com/rentzsch/mach_star", "Q_OS_MAC"],
    ["licenseMinHook", "../3rdparty/minhook-src/LICENSE.txt", "MinHook", "https://github.com/TsudaKageyu/minhook", "Q_OS_WIN64"],
    ["licenseQtTranslations", "../src/mumble/qttranslations/LICENSE",
        "Additional Qt translations", "https://www.virtualbox.org/ticket/2018", "USING_BUNDLED_QT_TRANSLATIONS"],
    ["licenseEmojiOne", "../3rdPartyLicenses/cc_by_sa_40_legalcode.txt", "Emoji One artwork", "http://emojione.com/"],
    ["licenseXInputCheck", "../3rdparty/xinputcheck-src/COPYING.txt", "XInputCheck (SDL_IsXInput function)", "https://www.libsdl.org/"],
    ["licenseQQBonjour", "../3rdparty/qqbonjour-src/LICENSE", "QQBonjour", "https://doc.qt.io/archives/qq/qq23-bonjour.html"],
    ["licenseSmallFT", "../3rdparty/smallft-src/LICENSE", "smallft", "https://www.xiph.org"],
    ["licenseOldStyleLicenseHeaders", "../3rdPartyLicenses/mumble-old-license-headers/LICENSE.txt", "Old-style Mumble license headers", "https://www.mumble.info"],
    ["licenseArc4RandomUniform", "../3rdparty/arc4random-src/LICENSE", "arc4random_uniform", "https://www.openbsd.org"],
    ["licenseMariaDBCConnector", "../3rdPartyLicenses/mariadb_c_connector_license.txt", "MariaDB Connector/C", "https://www.mariadb.org", "USE_BUILDENV"],
    ["licenseZlib", "../3rdPartyLicenses/zlib_license.txt", "ZLIB", "https://www.zlib.net", "USE_BUILDENV"],
    ["licenseProtobuf", "../3rdPartyLicenses/protobuf_license.txt", "Protobuf", "https://developers.google.com/protocol-buffers", "USE_BUILDENV"],
    ["licenseExpat", "../3rdPartyLicenses/expat_license.txt", "Expat", "http://libexpat.org", "USE_BUILDENV"],
    ["licenseBoost", "../3rdPartyLicenses/boost_license.txt", "Boost C++ Libraries", "https://www.boost.org/", "USE_BUILDENV"],
    ["licenseBzip2", "../3rdPartyLicenses/bzip2_license.txt", "bzip2", "http://www.bzip.org", "USE_BUILDENV"],
    ["licenseBerkeleyDB", "../3rdPartyLicenses/berkeleydb_license.txt", "BerkeleyDB", "https://www.oracle.com/database/berkeley-db", "USE_BUILDENV"],
    ["licenseFreetype", "../3rdPartyLicenses/freetype_license.txt", "Freetype", "https://www.freetype.org/", "USE_BUILDENV"],
    ["licenseHarfBuzzNG", "../3rdPartyLicenses/harfbuzzng_license.txt", "HarfBuzz", "https://www.freedesktop.org/wiki/Software/HarfBuzz", "USE_BUILDENV"],
    ["licenseLibPNG", "../3rdPartyLicenses/libpng_license.txt", "LibPNG", "http://www.libpng.org/pub/png/libpng.html", "USE_BUILDENV"],
    ["licenseLibjpegTurbo", "../3rdPartyLicenses/libjpeg_turbo_license.txt", "libjpeg-turbo", "http://www.libjpeg-turbo.org", "USE_BUILDENV"],
    ["licenseZeroCIce", "../3rdPartyLicenses/zeroc_ice_license.txt", "ZeroC Ice", "https://zeroc.com/", "USE_BUILDENV"],
    ["licenseMcpp", "../3rdPartyLicenses/mcpp_license.txt", "mcpp", "http://mcpp.sourceforge.net/", "USE_BUILDENV"],
    ["licenseMDNSResponder", "../3rdPartyLicenses/mdnsresponder_license.txt", "mDNSResponder", "https://opensource.apple.com/tarballs/mDNSResponder/", "USE_BUILDENV"],
    ["licensePCRE", "../3rdPartyLicenses/pcre_license.txt", "PCRE", "http://www.pcre.org/", "USE_BUILDENV"],
    ["licenseQt", "../3rdPartyLicenses/qt_license.txt", "Qt", "https://www.qt.io/", "USE_BUILDENV"],
    ["licenseSQLite3", "../3rdPartyLicenses/sqlite3_license.txt", "SQLite3", "http://sqlite.org", "USE_BUILDENV"],
    ["licenseXar", "../3rdPartyLicenses/xar_license.txt", "XAR", "https://opensource.apple.com/source/xar/", "USE_BUILDENV"], # macOS only
    ["licenseAvahi", "../3rdPartyLicenses/avahi_license.txt", "Avahi", "https://www.avahi.org/", "USE_BUILDENV"], # Linux only
    ["licenseDBus", "../3rdPartyLicenses/dbus_license.txt", "D-Bus", "https://www.freedesktop.org/wiki/Software/dbus", "USE_BUILDENV"], # Linux only
    ["licenseAppImageRuntime", "../3rdPartyLicenses/appimage_runtime_license.txt", "AppImage Runtime", "http://www.appimage.org", "USE_BUILDENV"], # Linux only
    ["licenseSquashfuse", "../3rdPartyLicenses/squashfuse_license.txt", "squashfuse", "https://github.com/vasi/squashfuse", "USE_BUILDENV"], # Linux only
    ["licenseLibfuse", "../3rdPartyLicenses/libfuse_license.txt", "libfuse", "https://github.com/libfuse/libfuse/", "USE_BUILDENV"], # Linux only
    ["licenseGlib", "../3rdPartyLicenses/glib_license.txt", "glib", "https://developer.gnome.org/glib/", "USE_BUILDENV"], # Linux only
    ["licenseLibcap2", "../3rdPartyLicenses/libcap2_license.txt", "libcap2", "https://sites.google.com/site/fullycapable/", "USE_BUILDENV"], # Linux only
    ["licenseLibdaemon", "../3rdPartyLicenses/libdaemon_license.txt", "libdaemon", "http://0pointer.de/lennart/projects/libdaemon/", "USE_BUILDENV"], # Linux only
    ["licenseLibffi", "../3rdPartyLicenses/libffi_license.txt", "libffi", "https://sourceware.org/libffi/", "USE_BUILDENV"] # Linux only
);

# Print 3rd party licenses
foreach (@thirdPartyLicenses) {
    printGuarded($F, licenseFileToVar(@$_[0], @$_[1]), @$_[4]);
    print $F "\n\n";
}


# Print list of 3rd party licenses
print $F "static const ThirdPartyLicense licenses3rdParties[] = {\n";
foreach (@thirdPartyLicenses) {
    printGuarded($F, "\tThirdPartyLicense(\"" . @$_[2] . "\", \"" . @$_[3] . "\", " . @$_[0] . "),\n", @$_[4]);
}
# Empty entry that marks the end of the array
printGuarded($F, "\tThirdPartyLicense(),\n", @$_[4]);
print $F "};\n\n\n";

print $F "#endif\n";

close($F);
