/*
 * MGTERM -- Terminal Emulator for MobileGear -
 * Copyright (C) 1998, 1999
 *      Koji Suzuki (suz@at.sakura.ne.jp)
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY KOJI SUZUKI ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE TERRENCE R. LAMBERT BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 */

#include <stdio.h>
#include "mgl2.h"

static int vk_attached=0;
static struct virtual_key *vk_canvas;


static char *vkeys="(789/)456*,123-C0.=+";

static char *keys[] = {
"","","","","",
"","","","","",
"+-","","","","",
"","","","","",
};
static int keyvals[] = {
'(','7','8','9','/',
')','4','5','6','*',
',','1','2','3','-',
'c','0','.','=','+',
};


#if 0
main() {
	open_graph();
	mcalc(-2);
	for (;;) {
		mcalc(get_key(10));
	}
}
#endif

#define MAX_VSCALE 9
#define MAX_VAL	1000000000
#define IS_NAN(a)	(((a)->scale >= 10000) || ((a)->scale <= -1000))
#define SET_NAN(a)	((a)->scale = 10000)

struct cnum {
	long long val;
	int scale;
};

cnum_fix(struct cnum *a) {
	int last=0;
	while (a->val > MAX_VAL || a->val < -MAX_VAL) {
		last = a->val % 10;
		a->val /= 10;
		a->scale += 1;
	}
	if (last > 4) {
		a->val ++;
	}
	if (a->val != 0) {
		while ((a->val % 10) == 0) {
			a->val /= 10;
			a->scale += 1;
		}
	}
	if (IS_NAN(a)) SET_NAN(a);
}

cnum_mult(struct cnum *a,const struct cnum *b,const struct cnum *c) {
	a->val = b->val * c->val;
	a->scale = b->scale + c->scale;
	cnum_fix(a);
}

cnum_div(struct cnum *a,const struct cnum *b,const struct cnum *c) {
	long long val;
	val = b->val * MAX_VAL;
	a->val = val / c->val;
	a->scale = b->scale - c->scale - MAX_VSCALE;
	cnum_fix(a);
}

cnum_add(struct cnum *a,const struct cnum *b,struct cnum *c) {
	int bscale,cscale;
	long long bval,cval;

	bscale = b->scale;
	cscale = c->scale;
	bval = b->val;
	cval = c->val;

	while (bscale > cscale) {
		bval *= 10;
		bscale--;
	}
	while (cscale > bscale) {
		cval *= 10;
		cscale--;
	}
	a->val = bval + cval;
	a->scale = bscale;
	cnum_fix(a);
}

cnum_negative(struct cnum *a,const struct cnum *b) {
	a->val = -  b->val;
	a->scale = b->scale;
}

cnum_sub(struct cnum *a,const struct cnum *b,const struct cnum *c) {
	struct cnum cc;
	cc.val = - c->val;
	cc.scale = c->scale;
	cnum_add(a,b,&cc);
}

int cnum_width(const struct cnum *a,int max_width,int *dot_pos) {
	long long val;
	int width;

	*dot_pos = 0;
	val = a->val;
	if (val == 0) return 1;
	if (a->scale > 0) {
	    width = a->scale;
	    if (val < 0) {
		width += 1;
		val = -val;
	   }
	   while (val > 0) {
		val /= 10;
		width += 1;
	   }
	   if (width > max_width) {
		return -1;
	   }
	   return width;
	} else {
	    *dot_pos = -a->scale;
	    if (*dot_pos >= max_width) return -1;
	    width = 0;
	    val = a->val;
	    if (val < 0) {
		width += 1;
		val = -val;
	    }
	    while (val > 0) {
		val /= 10;
		width += 1;
	    }
//printf("cnum_width %d\n",width);
	    return width;
	}
}

cnum_to_str(char *buf,int max_width,int *dot_pos,const struct cnum *a) {
	int width;
	long long val;
	int i,n;
	int scale;
	char *p;
	int sign = 0;

	memset(buf,' ',max_width);
	*dot_pos = 0;
	if (IS_NAN(a)) {
//printf("IS_NAN 1\n");
		buf[max_width-1] = 'E';
		return;
	}
	width = cnum_width(a,max_width,dot_pos);
	if (width < 0) {
//printf("IS_NAN 2\n");
		buf[max_width-1] = 'E';
		buf[max_width-1] = 'E';
		return;
	}
	val = a->val;
	if (val < 0) {
		sign = 1;
		val = -val;
		width --;
	}
	scale = a->scale;
	if (scale > 0) {
		n = scale;
		for (i=0; i<n; i++) {
			val *= 10;
		}
	}
	p = buf + max_width - 1;
	for (i=0; i<width; i++) {
		int d;
		d = val % 10;
		*p = '0' + d;
		p--;
		val /= 10;
	}
	for (; i<= *dot_pos; i++) {
		*p = '0';
		p--;
	}
	if (sign) {
		*p = '-';
	}
}

static struct cnum value[10];
static int dot;
static int op[10];
static int minus;
static int stk,value_is_valid,den,fixed;

#define XOFF	480
#define YOFF	8
#define XSIZE	160
static int ysize=224;
static int from_main = 0;
#define YSIZE	(ysize)
#define BYS	(ysize>=224?28:26)
#define BXS	28
#define BYOFF	(ysize>=224?92:64-4)

void mcalc(int c);
void clear_input(void);
void fix_input(void);
//void fix_dot(int d, int s);
void do_calc(void);

#ifdef MINIAPLI
#undef XOFF
#undef YOFF
#define XOFF	0
#define YOFF	0

void mcalc();

static char *icon_calc="\
#MGR000200160016
++++++++++++++++
++@@@@@@@@@@@@++
++@..........@++
++@.********.@++
++@.********.@++
++@.********.@++
++@..........@++
++@.@@.@@.@@.@++
++@.@@.@@.@@.@++
++@..........@++
++@.@@.@@.@@.@++
++@.@@.@@.@@.@++
++@..........@++
++@@@@@@@@@@@@++
++++++++++++++++
++++++++++++++++
";

main(int argc, char *argv[]) {
	int c;

	if (argc > 1 && !strcmp(argv[1],"-m")) {
		SCREEN_WIDTH = 160;
		SCREEN_HEIGHT = 240-16;
		open_graph();
	} else {
		mgl_apli_type = AT_MINIAPLI;
		open_graph();
		if (!mgl_client) {
			printf("server not running \n");
			exit(2);
		}
	}
	set_icon(icon_calc,"calc");
	from_main = 1;
	mcalc(-2);
	for (;;) {
		c = get_key(10);
		mcalc(c);
	}
}
#endif


void mcalc(c) {
	static struct textscreen *ts;
	char buf[512];
	int x,y;
	int i,j;
	int cc = 0;
	static int ri=0,rj=0;

	if (c == -1) return;
	if (ts == NULL) {
		if (SCREEN_HEIGHT < 224) {
			YSIZE = SCREEN_HEIGHT;
			if (!from_main) YSIZE -= 16;
		}
		ts = create_textscreen(NULL,XOFF,YOFF,XSIZE,YSIZE,TS_BORDER|TS_BLINE);
		ts_set_bgcolor(ts,COLOR_LIGHTGRAY);
	}

	if (c == -3) {
		if (!vk_attached) {
			vk_attach(NULL,vk_canvas);
			vk_attached = 1;
		}
		set_color(COLOR_BLACK);
		draw_rect(XOFF,YOFF,XSIZE-1,YSIZE-1);
		return;
	} else if (c == -4) {
		if (vk_attached) {
			vk_detach(vk_canvas,0);
			vk_attached = 0;
		}
		set_color(ts->bgcolor);
		draw_rect(XOFF,YOFF,XSIZE-1,YSIZE-1);
		return;
	}
	if (c == -2) {
		ts_clear(ts);
		stk = 0;
		clear_input();
		if (!vk_canvas) {
			struct virtual_key *v;
			vk_canvas = create_virtual_key(
				XOFF+4+8+1,YOFF+4+BYOFF+1
				,BXS*5,BYS*4,MK_V0);
			for (i=0; i<20; i++) {
				x = (i%5) * BXS;
				y = i/5 * BYS;
				v = create_virtual_key(x,y,BXS-4,BYS-4,vkeys[i]);
				vk_attach(vk_canvas,v);
			}
			vk_attach(NULL,vk_canvas);
			vk_attached = 1;
		}
		set_font(16,FA_BOLD);
		for (i=0; i<4; i++) {
			y = YOFF+4+BYOFF+i*BYS;
			for (j=0; j<5; j++) {
				x = XOFF+4+8+j*BXS;
				set_color(COLOR_DARKGRAY);
				fill_rect(x,y,BXS-2,BYS-2);
				set_color(COLOR_LIGHTGRAY);
				draw_pixel(x,y);
				draw_pixel(x,y+BYS-3);
				draw_pixel(x+BXS-3,y);
				draw_pixel(x+BXS-3,y+BYS-3);
				set_color(COLOR_WHITE);
				ts_goto(ts,8+j*BXS+1+4,BYOFF+i*BYS+1+4);
				ts_put_string(ts,keys[i*5+j],1);
				set_color(COLOR_BLACK);
			}
		}
	} 
	i = ri;
	j = rj;
	if (c == MK_LEFT && rj>0) { cc = 1; rj--;} 
	else if (c == MK_RIGHT && rj<4) { cc = 1; rj++;} 
	else if (c == MK_UP && ri>0) { cc = 1; ri--; }
	else if (c == MK_DOWN && ri<3) { cc = 1; ri++;} 
	if (cc && c != -2) {
		x = XOFF+4+8+j*BXS+1;
		y = YOFF+4+BYOFF+i*BYS+1;
		set_color(COLOR_REVERSE);
		draw_rect(x,y,BXS-4,BYS-4);
	}
	if (cc || c == -2) {
		x = XOFF+4+8+rj*BXS+1;
		y = YOFF+4+BYOFF+ri*BYS+1;
		set_color(COLOR_REVERSE);
		draw_rect(x,y,BXS-4,BYS-4);
	}
	if (cc) {
		refresh();
		return;
	}
	if (c == ' ') c = keyvals[ri*5+rj];
	switch(c) {
#ifdef MINIAPLI
	case '\033':
		exit(0);
#endif
	case 'c':
	case 'C':
		clear_input();
		break;
	case '.':
		if (fixed) clear_input();
		den = 1;
		value_is_valid = 1;
		break;
	case '0':
	case '1':
	case '2':
	case '3':
	case '4':
	case '5':
	case '6':
	case '7':
	case '8':
	case '9':
		value_is_valid = 1;
		if (fixed) clear_input();
		if (den) value[stk].scale--;
		value[stk].val *= 10;
		value[stk].val += c - '0';
		break;
	case '=':
		if (stk != 0 && op[stk-1]) do_calc();
		break;
	case '-':
		if (!value_is_valid) {
			minus = !minus;
		} else {
			if (stk != 0 && op[stk-1]) do_calc();
			fix_input();
			op[stk] = c;
			stk++;
			clear_input();
		}
		break;
	case ',':
		value[stk].val = -value[stk].val;
		break;
	case '(':
		if (stk < 10-1) {
			stk++;
			clear_input();
		}
		break;
	case ')':
		if (stk > 0) {
			value[stk-1] = value[stk];
			op[stk-1] = 0;
			stk--;
		}
		break;
	case '+':
	case '*':
	case '/':
		if (stk != 0 && op[stk-1]) do_calc();
		fix_input();
		op[stk] = c;
		stk++;
		clear_input();
	}
	set_font(24,0);
	set_color(COLOR_BLACK);
	set_color(COLOR_WHITE);
	fill_rect(XOFF+4+8,YOFF+4+4+4+4,12*10+12,24+16);
	set_color(COLOR_BLACK);
	draw_rect(XOFF+4+8,YOFF+4+4+4+4,12*10+12,24+16);
	set_font(12,0);
	ts_goto(ts,8+6,4);
	ts_set_bgcolor(ts,COLOR_WHITE);

	set_font(24,FA_BOLD);
	ts_goto(ts,8+6,4+4+12);

	if (fixed) {
		cnum_to_str(buf,10,&dot,value+stk);
	} else {
	 	struct cnum c;
		c = value[stk];
		if (minus) c.val = - c.val;
		cnum_to_str(buf,10,&dot,&c);
	}
	ts_put_string(ts,buf,0);

	ts_set_bgcolor(ts,COLOR_LIGHTGRAY);
	draw_pixel(XOFF+4+8+6+(10-dot)*12-1,YOFF+4+4+4+12+23);
	draw_pixel(XOFF+4+8+6+(10-dot)*12,YOFF+4+4+4+12+23);
	draw_pixel(XOFF+4+8+6+(10-dot)*12-1,YOFF+4+4+4+12+24);
	draw_pixel(XOFF+4+8+6+(10-dot)*12,YOFF+4+4+4+12+24);
refresh();
}

void clear_input() {
	value[stk].val = 0;
	value[stk].scale = 0;
	op[stk] = 0;
	den = 0;
	value_is_valid = 0;
	minus = 0;
	fixed = 0;
}

void fix_input() {
	if (minus) value[stk].val *= -1;
	minus = 0;
	cnum_fix(value+stk);
	fixed = 1;
}

void do_calc() {
	if (stk == 0) return;
	fix_input();

	if (op[stk-1] == '*') {
		cnum_mult(value+stk-1,value+stk-1,value+stk);
		op[stk] = 0;
		stk--;
	} else if (op[stk-1] == '/') {
		cnum_div(value+stk-1,value+stk-1,value+stk);
		op[stk-1] = 0;
		stk--;
	} else if (op[stk-1] == '+') {
		cnum_add(value+stk-1,value+stk-1,value+stk);
		op[stk-1] = 0;
		stk--;
	} else if (op[stk-1] == '-') {
		cnum_sub(value+stk-1,value+stk-1,value+stk);
		op[stk-1] = 0;
		stk--;
	}
}
