/*
 * MGL -- MobileGear Graphic Library -
 * Copyright (C) 1998, 1999, 2000
 *      Koji Suzuki (suz@at.sakura.ne.jp)
 *      Hiroyuki Yanai (fk200329@fsinet.or.jp)
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY KOJI SUZUKI AND YUKIHIKO SANO ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE TERRENCE R. LAMBERT BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 */
/*
 * Graphics driver for "desktop".
 *
 * March 15th, 1994 -- Lawrence Kesteloot
 *   Original -- written in the grf driver of the kernel.
 *
 * June 18th, 1994 -- Lawrence Kesteloot
 *   Added support for non-8-pixel-wide fonts.
 *
 * June 26th, 1994 -- Lawrence Kesteloot
 *   Pulled out of the kernel into user-land.
 *
 * October 31st, 1994 -- Monroe Williams
 *   Added fixes for non-640x480 video cards. [Is this only in kernel's grf.c? --DJH]
 *   Changed struct grf_softc to grf_Softc.
 *   Added improvement for scrolling and screen-clearing in grf_scrollup
 *      and grf_blankscreen.
 */
/*
 *      Fearful commenting is leaden
 *              servitor to dull delay
 *       (Richard III, Act IV, sc iii)
 */


#include <machine/grfioctl.h>

#include "fb2.h"
#include "fb4.h"
#include "fb16.h"

#define NO_VIRTUAL_CONSOLE

#ifdef ADB_MOUSE
const static int show_mouse=1;
#else
const static int show_mouse=0;
#endif
const static int same_format=0;
#define MD_PUT_SCANSEGMENT(dst_y,dst_x,src,nbytes)

#ifdef ADB_KEYBOARD
#include "md_mac_adbkbd.h"
#endif

#define MAXGRF          16
/* per display info */
struct grf_softc {
	struct grfmode g_display;       /* hardware description (for ioctl) */
};
static struct grf_softc grf_Softc[MAXGRF];
static int grf_fds[MAXGRF];
static int numgrf;

static caddr_t g_fbbase;
static u_int32_t g_fbsize;
static u_int16_t g_rowbytes;
static u_int16_t g_psize;
static u_int16_t g_width;
static u_int16_t g_height;


/*
	fake 2bpp mode (for real 8bpp mode)
*/
#define FAKE2BPP_BLACK		255
#define FAKE2BPP_LTGRAY		100
#define FAKE2BPP_DKGRAY		180
#define FAKE2BPP_WHITE		0

static int fake2bpp_ctable[] = {
	FAKE2BPP_BLACK,
	FAKE2BPP_DKGRAY,
	FAKE2BPP_LTGRAY,
	FAKE2BPP_WHITE
};


static int  create_mac68k_physical_screen();
static void dispose_mac68k_physical_screen();
static void setup_physical_screen_size();
static void grf_blankscreen(int grf, int inverse);



static void
MD_GRAPH_MODE()
{
	return;
}

static void
MD_TEXT_MODE()
{
	return;
}

static void
MD_SET_MOUSEXY(int x,int y)
{
	return;
}

static int
MD_MOUSE_EVENT()
{
	return 0;
}

static int
MD_MOUSE_X()
{
	return 0;
}

static int
MD_MOUSE_Y()
{
	return 0;
}

static int
MD_MOUSE_BUTTON()
{
	return 0;
}



int
MD_INIT()
{
	printf("md_mac init\n");
#ifdef ADB_KEYBOARD
	if(mac_adb_open() < 0)
		return -1;
#endif
	if (create_mac68k_physical_screen()) 
		return -1;

	setup_physical_screen_size();

	depth = g_psize;

	switch(depth) {
	case 2:
		/* reverse order and reverse color */
		fb2_setup(g_fbbase, g_rowbytes, ORD2_REVERSE, COL2_REVERSE);
		break;

	case 4:
		/* reverse order and normal color */
		fb4_setup(g_fbbase, g_rowbytes, ORD4_REVERSE, COL4_REVERSE);
		break;

	case 8:
		/* fb8_setup(g_fbbase, g_rowbytes, 0); */
		/* return -1; */

		/* fake 2bpp mode. reverse order and normal color */
		fb2_setup(g_fbbase, g_rowbytes, ORD2_REVERSE, COL2_NORMAL);
		break;

	case 16:
		fb16_setup(g_fbbase, g_rowbytes, ORD_RGB);
		break;

	default:
		/* reverse order and reverse color */
		fb2_setup(g_fbbase, g_rowbytes, ORD2_REVERSE, COL2_REVERSE);
		break;
	}
	return 0;
}


static int
create_mac68k_physical_screen()
{
	int     f;
	char    devstr[128];
	caddr_t physscreen;
	struct grf_softc *gp;
	struct grfmode *gi;
	int    i;
	int    psize;

	numgrf = 0;
	for (i = 0; i < MAXGRF; i++) {
		grf_fds[i] = -1;
		physscreen = NULL;

		sprintf(devstr, "/dev/grf%x", i);
		f = open(devstr, O_RDWR);
		if (f == -1) {
			if (i == 0)
				perror("error opening grf0 (primary screen)");
			continue;
		}
		gp = &grf_Softc[numgrf];

#ifndef USE_OLD_GRF
		if (ioctl(f, GRFIOCGMODE, &gp->g_display) == -1) {
#endif /* USE_OLD_GRF */

#ifdef GRF_COMPAT
			physscreen = grfcompat(&gp->g_display, f);
			if (!physscreen) {
				close(f);
				continue;
			}
#else
			perror("GRFIOCGMODE");
			close(f);
			continue;
#endif /* GRF_COMPAT */

#ifndef USE_OLD_GRF
		}
#endif /* USE_OLD_GRF */

		psize = gp->g_display.psize;
		if (psize > 16) {
			fprintf(stderr, "grf%d: not in 2, 4, 8, 16 bit mode (%d planes)\n", i, psize);
			close(f);
			continue;
		}

#ifdef GRF_COMPAT
		if (!physscreen)
#endif /* GRF_COMPAT */

		if ((physscreen = mmap(0, gp->g_display.fbsize, PROT_READ | PROT_WRITE, MAP_SHARED,
			 f, gp->g_display.fboff)) == (caddr_t) - 1) {
				perror("mmap");
				close(f);
				continue;
		}

		grf_fds[i] = f;
		gi = &gp->g_display;
		gi->fbbase = (caddr_t) physscreen;

		grf_blankscreen(numgrf, 1);
		numgrf++;
	}
	if (numgrf == 0) {
		/* dispose_mac68k_physical_screen(); */
		fprintf(stderr, "no usable grf-devices found\n");
		return -1;
	}
    
	g_fbbase = (grf_Softc[0]).g_display.fbbase;
	g_rowbytes = (grf_Softc[0]).g_display.rowbytes;
	g_width = (grf_Softc[0]).g_display.width;
	g_height = (grf_Softc[0]).g_display.height;
	g_psize = (grf_Softc[0]).g_display.psize;
	
	return 0;
}


/*
 * grf_blankscreen
 *
 *   Clear the grf to zeros or ones if INVERSE. (white)
 */

static void
grf_blankscreen(int grf, int inverse)
{
	long    longcnt;
/* MBW -- added the next two lines for the fix below */
	register long rowcnt, longstore, rowbytes;
	register unsigned long *rowptr;

	register unsigned long *screenptr;
	struct grf_softc *gp;

	if (grf == -1) {
		return;
	}
	gp = &grf_Softc[grf];

/*
* QuickDraw does not currently define a system for screens
* that do not have a row that is long aligned.
*/

/* MBW -- fix the really inefficient screen-clearing */
	longstore = (gp->g_display.width) / (32 / (gp->g_display.psize));
	rowcnt = gp->g_display.height;
	rowptr = (unsigned long *) gp->g_display.fbbase;

/* divide by 4 because rowbytes will be added to a (long *) */
	rowbytes = gp->g_display.rowbytes / sizeof(long);

	while (rowcnt--) {
		screenptr = rowptr;
		longcnt = longstore;

		while (longcnt--) {
			*screenptr++ = (inverse) ? (~0L) : (0L);
		}
		rowptr += rowbytes;
	}
	return;
}


static void
setup_physical_screen_size()
{
	if ((SCREEN_WIDTH > 320) && (SCREEN_HEIGHT > 240)) {

		/* ---- 13inch, 15inch portlait(normal VGA) */
		if (g_width >= 640 && g_height >= 480) {
			SCREEN_WIDTH = 640;
			SCREEN_HEIGHT = 480;
			return;
		}
printf("640x480 not supported\n");

		/* ---- portable, powerbook(640x400) */
		if(g_width >= 640 && g_height >= 240) {
			SCREEN_WIDTH = 640;
			SCREEN_HEIGHT = 240;
			return;
		}
printf("640x240 not supported\n");

		/* ---- SE, SE/30, Classic, ColorClassic, 12inch(512x384, 512x342) */
		if (g_width >= 320 && g_height >= 240) {
			SCREEN_WIDTH = 320;
			SCREEN_HEIGHT = 240;
		}
	}
	return;
}


static void
dispose_mac68k_physical_screen()
{
	int     grfnum;

	for (grfnum = 0; grfnum < numgrf; grfnum++) {
		if (grf_fds[grfnum] != -1) {
			grf_blankscreen(grfnum, 0);
			close(grf_fds[grfnum]);
		}
	}
	return;
}

static void
MD_TERM(void)
{
	dispose_mac68k_physical_screen();

#ifdef ADB_KEYBOARD
	mac_adb_close();
#endif
	return;
}

#ifdef ADB_KEYBOARD
#define MD_KEY_SELECT  md_key_select
static int
md_key_select(int nfds, fd_set *readfds, int timeout)
{
	return mac_key_select(nfds,readfds,timeout);
}
#endif

#define MD_PUT_PIXSTREAM2	fb2_put_pixstream
#define MD_PUT_PIXSTREAM4	fb4_put_pixstream
#define MD_PUT_PIXSTREAM16	fb16_put_pixstream

static void
MD_PUT_PIXSTREAM8(int x,int y,int *buf,int xs,int dir, int op)
{
	unsigned char *p;

	/* fake 2bpp mode */
	p = g_fbbase + y * g_rowbytes + x;
	while(0 < xs--) {
		*p++ = fake2bpp_ctable[*buf++ & 0x03];
	}
	return;
}

/*
static void
MD_PUT_SCANSEGMENT(int dst_y, int dst_x, unsigned char *src, int nbytes)
{
	memmove(g_fbbase + dst_y * g_rowbytes + dst_x, src, nbytes);
	return;
}
*/

