/*
 * MGL -- MobileGear Graphic Library -
 * Copyright (C) 1998, 1999
 *      Koji Suzuki (suz@at.sakura.ne.jp)
 *      Yukihiko Sano (yukihiko@yk.rim.or.jp)
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY KOJI SUZUKI AND YUKIHIKO SANO ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE TERRENCE R. LAMBERT BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 */
#define MGL_PREFIX
#include "config.h"
#include "mgl2.h"
#include "draw_engine.h"
#include <stdlib.h>

#ifdef USE_LOCAL_MEMSET
#define memset	mgl_memset
#endif
#ifdef USE_LOCAL_MEMMOVE
#define memmove	mgl_memmove
#endif

#ifndef NULL
#define NULL ((void *)0)
#endif

static struct draw_engine dec3k;

static struct screen *dec3k_create_memscreen(int xs,int ys,char *bitmap,int op) {
	struct screen *ret;
	int wb;
	int type = STK_GENERIC_FULLCOLOR | ST_ALLOCED;

	wb = xs * 2;
	ret = de_create_memscreen(xs,ys,bitmap,op);
	if (!ret) return NULL;
	if (!bitmap) {
		bitmap = (char *)malloc(ys * wb);
        	if(!bitmap){
			perror("malloc");
			return (struct screen *)NULL;
        	}
		type |= ST_ALLOCED_BITMAP;
		memset(bitmap,0,ys * wb);
	}

	ret->type |= type;
	ret->de = &dec3k;
	ret->wbytes = wb;
	ret->bitmap = bitmap;
	return ret;
}

static struct screen *dec3k_create_subscreen(struct screen *org, int x, int y,
                                int xs, int ys,int opt) {
	struct screen *ret;

	ret = de_create_subscreen(org,x,y,xs,ys,opt);
	if (!ret) return NULL;
	if (opt & CSS_AS_MEMSCREEN) {
		ret->type &= ~ST_SUBSCREEN;
		ret->bitmap = org->bitmap + (ret->off_y * ret->wbytes)
			+ (ret->off_x *2);
		ret->off_y = 0;
		ret->off_x = 0;
	}
	return ret;
}

static void dec3k_free_screen(struct screen *s) {
	de_free_screen(s);
}


static int dec3k_get_pixel(struct screen *s,int x, int y,int op) {
	unsigned char *p;
	int col,ret;

	if (x < 0 || x >= s->width
	   || y < 0 || y >= s->height) return 0;

	x += s->off_x;
	y += s->off_y;
	if (s->type & ST_SUBSCREEN) {
		p = ((struct screen *)(s->bitmap))->bitmap;
		s->wbytes = 
			((struct screen *)(s->bitmap))->wbytes;
	} else {
		p = (char *)s->bitmap;
	}
	p += y * s->wbytes + x * 2;
	col = *(unsigned short *)p;

	ret = col |= COLOR_DITHER;
	if ((op & BLT_MASKING) && ((ret & 0xfff) == (op & BLT_MASKCOL))) {
		ret = COLOR_TRANSPARENT;
//printf("dec3k_get_pixel (%d %d) = %08x\n",x,y,ret);
	}
	return ret;
}

static void dec3k_put_pixel(struct screen *s,int x, int y, int col) {
	char *p;
	if (col & COLOR_TRANSPARENT) return;
	if (col < 0) return;

	CLIP_POINT(s,x,y);

	x += s->off_x;
	y += s->off_y;
	if (s->type & ST_SUBSCREEN) {
		p = ((struct screen *)(s->bitmap))->bitmap;
		s->wbytes = 
			((struct screen *)(s->bitmap))->wbytes;
	} else {
		p = (char *)s->bitmap;
	}
	p += y * s->wbytes + x * 2;

	*(unsigned short *)p = col & 0xfff;
	return;
}

static void dec3k_get_pixstream(struct screen *s,int x, int y,int *buf,int length,int dir,int op
		,struct draw_engine *self) {
	char *p;
	int i;
	int d;
	int r;
	int len4;
	unsigned short *pp;

	x += s->off_x;
	y += s->off_y;
	if (s->type & ST_SUBSCREEN) {
		p = ((struct screen *)(s->bitmap))->bitmap;
		s->wbytes = 
			((struct screen *)(s->bitmap))->wbytes;
	} else {
		p = (char *)s->bitmap;
	}
	p += y * s->wbytes + x*2;
	pp = (unsigned short *)p;

	switch(dir) {
	case DIR_NORTH:
#if (OPTIMIZE_FOR == OPTTYPE_SLOW)
		if (op & BLT_MASKING) for (i=0; i<length; i++) {
			r = *(unsigned short *)p | COLOR_DITHER;
			if ((r & 0xfff) == (op & BLT_MASKCOL)) {
				r = COLOR_TRANSPARENT;
			}
			*buf++ = r;
			p += 2;
		} else for (i=0; i<length; i++) {
			*buf++ = *(unsigned short *)p | COLOR_DITHER;
			p += 2;
		}
#else
		if (op & BLT_MASKING) {
		    len4 = length / 4;
		    length -= len4 * 4;
		    for (i=0; i<len4; i++) {
			r = pp[0] | COLOR_DITHER;
			if ((r & 0xfff) == (op & BLT_MASKCOL)) {
				r = COLOR_TRANSPARENT;
			}
			buf[0] = r;
			r = pp[1] | COLOR_DITHER;
			if ((r & 0xfff) == (op & BLT_MASKCOL)) {
				r = COLOR_TRANSPARENT;
			}
			buf[1] = r;
			r = pp[2] | COLOR_DITHER;
			if ((r & 0xfff) == (op & BLT_MASKCOL)) {
				r = COLOR_TRANSPARENT;
			}
			buf[2] = r;
			r = pp[3] | COLOR_DITHER;
			if ((r & 0xfff) == (op & BLT_MASKCOL)) {
				r = COLOR_TRANSPARENT;
			}
			buf[3] = r;
			pp += 4;
			buf += 4;
		    }
		    for (i=0; i<length; i++) {
			r = pp[0] | COLOR_DITHER;
			if ((r & 0xfff) == (op & BLT_MASKCOL)) {
				r = COLOR_TRANSPARENT;
			}
			*buf++ = r;
			pp++;
		    }
		} else {
		    for (i=0; i<length; i++) {
			*buf++ = pp[0] | COLOR_DITHER;
			pp++;
		    }
		}
#endif
		break;
	case DIR_WEST:
		if (op & BLT_MASKING) for (i=0; i<length; i++) {
			r = *(unsigned short *)p | COLOR_DITHER;
			if ((r & 0xfff) == (op & BLT_MASKCOL)) {
				r = COLOR_TRANSPARENT;
			}
			*buf++ = r;
			p -= s->wbytes;
		} else for (i=0; i<length; i++) {
			*buf++ = *(unsigned short *)p | COLOR_DITHER;
			p -= s->wbytes;
		}
		break;
	case DIR_SOUTH:
		if (op & BLT_MASKING) for (i=0; i<length; i++) {
			r = *(unsigned short *)p | COLOR_DITHER;
			if ((r & 0xfff) == (op & BLT_MASKCOL)) {
				r = COLOR_TRANSPARENT;
			}
			*buf++ = r;
			p -= 2;
		} else for (i=0; i<length; i++) {
			*buf++ = *(unsigned short *)p | COLOR_DITHER;
			p -= 2;
		}
		break;
	case DIR_EAST:
		if (op & BLT_MASKING) for (i=0; i<length; i++) {
			r = *(unsigned short *)p | COLOR_DITHER;
			if ((op & BLT_MASKING) && ((r & 0xfff)== (op & BLT_MASKCOL))) {
				r = COLOR_TRANSPARENT;
			}
			*buf++ = r;
			p += s->wbytes;
		} else for (i=0; i<length; i++) {
			*buf++ = *(unsigned short *)p | COLOR_DITHER;
			p += s->wbytes;
		}
		break;
	}
}

static void dec3k_put_pixstream(struct screen *s,int x, int y,int *buf,int length,int dir
		,struct draw_engine *self) {
	char *p;
	int col,mask;
	int d,i;
	int dx,dy;

	x += s->off_x;
	y += s->off_y;
	if (s->type & ST_SUBSCREEN) {
		p = ((struct screen *)(s->bitmap))->bitmap;
		s->wbytes = 
			((struct screen *)(s->bitmap))->wbytes;
	} else {
		p = (char *)s->bitmap;
	}
	p += y * s->wbytes + x*2;

	switch(dir) {
	case DIR_NORTH:
	    for (i=0; i<length;i++,x++) {
		col = *buf++;
		if (!(col & COLOR_TRANSPARENT)) {
		        *(unsigned short *)p = col;
		}
		p += 2;
	    }
	    break;
	case DIR_WEST:
	    for (i=0; i<length;i++) {
		col = *buf++;
		if (!(col & COLOR_TRANSPARENT)) {
		        *(unsigned short *)p = col;
		}
		p -= s->wbytes;
	    }
	    break;
	case DIR_SOUTH:
	    for (i=0; i<length;i++,x--) {
		col = *buf++;
		if (!(col & COLOR_TRANSPARENT)) {
		        *(unsigned short *)p = col;
		}
	        p -= 2;
	    }
	    break;
	case DIR_EAST:
	    for (i=0; i<length;i++) {
		col = *buf++;
		if (!(col & COLOR_TRANSPARENT)) {
		        *(unsigned short *)p = col;
		}
		p += s->wbytes;
	    }
	    break;
	}
	return;
}

static void dec3k_set_color(struct screen *s,int col) {
	return;
}

static void dec3k_draw_pixel(struct screen *s,int x, int y) {
	char *p;
	int mask;

	CLIP_POINT(s,x,y);

	x += s->off_x;
	y += s->off_y;
	if (s->type & ST_SUBSCREEN) {
		p = ((struct screen *)(s->bitmap))->bitmap;
		s->wbytes = 
			((struct screen *)(s->bitmap))->wbytes;
	} else {
		p = (char *)s->bitmap;
	}
	p += y * s->wbytes + x * 2;

	if (pen_color == COLOR_REVERSE)
#if 0
		*(unsigned short *)p = ~(*(unsigned short *)p);
#else
		*(unsigned short *)p ^= 0x0f;
#endif
	else
		*(unsigned short *)p = pen_color;
      return;
}

static void dec3k_draw_line_horizontal(struct screen *s,int x1, int y1, int x2, int y2,struct draw_engine *self) {
	char *p;
	int wp;
	int i;
	int off_x,off_y;

	CLIP_HLINE(s,x1,y1,x2,y2);
	s->need_clipping--;

	off_x = s->off_x;
	off_y = s->off_y;

	x1 += off_x;
	x2 += off_x;
	y1 += off_y;
	y2 += off_y;

	wp = (x2 + 1 - x1);
	if (wp) {
	     if (s->type & ST_SUBSCREEN) {
		p = ((struct screen *)(s->bitmap))->bitmap;
		s->wbytes = 
			((struct screen *)(s->bitmap))->wbytes;
	     } else {
		p = (char *)s->bitmap;
	    }
	    p += y1 * s->wbytes + x1 * 2;

	    if (pen_color == COLOR_REVERSE) {
#if (OPTIMIZE_FOR == OPTTYPE_SLOW)
		for (i=0; i< wp; i++,p+=2) {
		    *(unsigned short *)p ^= 0x0f;
		}
#else
		mgl_memxor2(p, 0x0f, wp);
#endif
	    } else {
#if (OPTIMIZE_FOR == OPTTYPE_SLOW)
		for (i=0; i< wp; i++,p+=2) {
	           *(unsigned short *)p = pen_color & 0xfff;
		}
#else
		mgl_memset2(p, pen_color, wp);
#endif
	    }
	}
	s->need_clipping++;
    return;
}

static void dec3k_clear_screen(struct screen *s,struct draw_engine *self) {
	int x1,y1,x2,y2;
	int off_x,off_y;
	char *p,*pp;
	int wp;
	int i;
	int y;

	x1 = 0;
	x2 = s->width-1;
	y1 = 0;
	y2 = s->height-1;
	s->need_clipping--;

	off_x = s->off_x;
	off_y = s->off_y;
	x1 += off_x;
	x2 += off_x;
	y1 += off_y;
	y2 += off_y;

	wp = (x2 + 1 - x1);
	if (wp) {
	    if (s->type & ST_SUBSCREEN) {
		p = ((struct screen *)(s->bitmap))->bitmap;
		s->wbytes = 
			((struct screen *)(s->bitmap))->wbytes;
	    } else {
		p = (char *)s->bitmap;
	    }
	    p += y1 * s->wbytes + x1*2;
	    for (y=y1; y<=y2; y++) {
		if (pen_color == COLOR_REVERSE) {
#if (OPTIMIZE_FOR == OPTTYPE_SLOW)
		    pp = p;
		    for (i=0; i< wp; i++,pp+=2) {
			*(unsigned short *)pp ^= 0x0f;
		    }
#else
		    mgl_memxor2(p, 0x0f, wp);
#endif
		} else {
#if (OPTIMIZE_FOR == OPTTYPE_SLOW)
		    pp = p;
		    for (i=0; i< wp; i++,pp+=2) {
			*(unsigned short *)pp = pen_color & 0xfff;
		    }
#else
		    mgl_memset2(p, pen_color & 0xfff, wp);
#endif
		} 
		p += s->wbytes;
	    }
	}
	s->need_clipping++;
}

static void dec3k_bitblt_copy(struct screen *dst, int dx, int dy
	, struct screen *src, int sx, int sy, int xsize, int ysize, int op
	, struct draw_engine *self) {
	char *dp,*sp;
	int i,len;

	dx += dst->off_x;
	dy += dst->off_y;
	sx += src->off_x;
	sy += src->off_y;
	if (dst->type & ST_SUBSCREEN) {
		dp = ((struct screen *)(dst->bitmap))->bitmap;
		dst->wbytes = 
			((struct screen *)(dst->bitmap))->wbytes;
	} else {
		dp = (char *)dst->bitmap;
	}
	if (src->type & ST_SUBSCREEN) {
		sp = ((struct screen *)(src->bitmap))->bitmap;
		src->wbytes = 
			((struct screen *)(src->bitmap))->wbytes;
	} else {
		sp = (char *)src->bitmap;
	}
	dp += dy * dst->wbytes + dx*2;
	sp += sy * src->wbytes + sx*2;
	len = xsize*2;

	for (i=0; i< ysize; i++) {
		memmove(dp,sp,len);
		dp += dst->wbytes;
		sp += src->wbytes;
	}
	return;
}

static void dec3k_bitblt_scroll_forward(struct screen *dst, int dx, int dy
	, struct screen *src, int sx, int sy, int xsize, int ysize, int op
	, struct draw_engine *self) {
	char *p,*dp,*sp;
	int i,len;

	dx += dst->off_x;
	dy += dst->off_y;
	sx += src->off_x;
	sy += src->off_y;
	if (dst->type & ST_SUBSCREEN) {
		p = ((struct screen *)(dst->bitmap))->bitmap;
		dst->wbytes = 
			((struct screen *)(dst->bitmap))->wbytes;
	} else {
		p = (char *)dst->bitmap;
	}
	dp = p + dy * dst->wbytes + dx*2;
	sp = p + sy * dst->wbytes + sx*2;
	len = xsize*2;
	for (i=0; i< ysize; i++) {
		memmove(dp,sp,len);
		dp += dst->wbytes;
		sp += dst->wbytes;
	}
	return;
}

static void dec3k_bitblt_scroll_backward(struct screen *dst, int dx, int dy
	, struct screen *src, int sx, int sy, int xsize, int ysize, int op
	, struct draw_engine *self) {
	char *p,*dp,*sp;
	int i,len;

	dx += dst->off_x;
	dy += dst->off_y;
	sx += src->off_x;
	sy += src->off_y;
	if (dst->type & ST_SUBSCREEN) {
		p = ((struct screen *)(dst->bitmap))->bitmap;
		dst->wbytes = 
			((struct screen *)(dst->bitmap))->wbytes;
	} else {
		p = (char *)dst->bitmap;
	}
	dy += ysize - 1;
	sy += ysize - 1;
	dp = p + dy * dst->wbytes + dx*2;
	sp = p + sy * dst->wbytes + sx*2;
	len = xsize*2;
	for (i=0; i< ysize; i++) {
		memmove(dp,sp,len);
		dp -= dst->wbytes;
		sp -= dst->wbytes;
	}
	return;
}
void dec3k_init(int type) {
	dec3k._create_subscreen = dec3k_create_subscreen;
	dec3k._free_screen = dec3k_free_screen;
	dec3k._put_pixel = dec3k_put_pixel;
	dec3k._get_pixel = dec3k_get_pixel;

	dec3k._set_color = dec3k_set_color;
	dec3k._draw_pixel = dec3k_draw_pixel;

	//dec3k._draw_line = dec3k_draw_line;

	//dec3k._draw_line_vertical = dec3k_draw_line_vertical;
	dec3k._draw_line_horizontal = dec3k_draw_line_horizontal;

	//dec3k._clear_screen = dec3k_clear_screen;

	dec3k._get_pixstream = dec3k_get_pixstream;
	dec3k._put_pixstream = dec3k_put_pixstream;
	//dec3k._put_pixstream_rect = dec3k_put_pixstream__rect;

	//dec3k._bitblt = dec3k_bitblt;
	dec3k._bitblt_scroll_forward = dec3k_bitblt_scroll_forward;
	dec3k._bitblt_scroll_backward = dec3k_bitblt_scroll_backward;
	dec3k._bitblt_copy = dec3k_bitblt_copy;
	//dec3k._bitblt_reserved_mask = dec3k_bitblt_reserved_mask;
	//dec3k._bitblt_reserved_masktile = dec3k_bitblt_reserver_masktile;
	//dec3k._bitblt_gen = dec3k_bitblt_gen;
	setup_draw_engine(&dec3k,0);
	_create_memscreen[type] = dec3k_create_memscreen;
}
