/*
 * MGL -- MobileGear Graphic Library -
 * Copyright (C) 1998, 1999
 *      Koji Suzuki (suz@at.sakura.ne.jp)
 *      Yukihiko Sano (yukihiko@yk.rim.or.jp)
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY KOJI SUZUKI AND YUKIHIKO SANO ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE TERRENCE R. LAMBERT BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 */
#define MGL_PREFIX
#include "config.h"
#include "mgl2.h"
#include "draw_engine.h"
#include <stdlib.h>

#ifndef NULL
#define NULL ((void *)0)
#endif

#ifdef USE_LOCAL_MEMSET
#define memset	mgl_memset
#endif
#ifdef USE_LOCAL_MEMMOVE
#define memmove	mgl_memmove
#endif

static struct draw_engine dec16;

#define lp_mask		(s->_pen_color.local_color)

static struct screen *dec16_create_memscreen(int xs,int ys,char *bitmap,int op) {
	struct screen *ret;
	int wb;
	int type = STK_GENERIC_16COLOR | ST_ALLOCED;

	wb = (xs+1)/2;

	ret = (struct screen *)de_create_memscreen(xs,ys,bitmap,op);
	if (!ret) return NULL;

	if (!bitmap) {
		bitmap = (char *)malloc(ys * wb);
        	if(!bitmap){
			perror("malloc");
			return (struct screen *)NULL;
        	}
		type |= ST_ALLOCED_BITMAP;
		memset(bitmap,0,ys * wb);
	}

	ret->type |= type;
	ret->de = &dec16;
	ret->wbytes = wb;
	ret->bitmap = bitmap;
	return ret;
}

static struct screen *dec16_create_subscreen(struct screen *org, int x, int y,
                                int xs, int ys,int opt) {
	struct screen *ret;

	ret = de_create_subscreen(org,x,y,xs,ys,opt);
	if (!ret) return NULL;

	if (opt & CSS_AS_MEMSCREEN) {
		ret->type &= ~ST_SUBSCREEN;
		ret->bitmap = org->bitmap + (ret->off_y * ret->wbytes)
			+ (ret->off_x /2);
		ret->off_y = 0;
		ret->off_x = 0;
	}
	return ret;
}

static void dec16_free_screen(struct screen *s) {
	de_free_screen(s);
}


static int dec16_get_pixel(struct screen *s,int x, int y,int op) {
	unsigned char *p;
	int col,ret;

	if (x < 0 || x >= s->width
	   || y < 0 || y >= s->height) return 0;

	x += s->off_x;
	y += s->off_y;
	if (s->type & ST_SUBSCREEN) {
		p = ((struct screen *)(s->bitmap))->bitmap;
		s->wbytes = 
			((struct screen *)(s->bitmap))->wbytes;
	} else {
		p = (char *)s->bitmap;
	}
	p += y * s->wbytes + (x>>1);

	col = (*p >> (x & 1)*4 ) & 0xf;
	ret = CONV_FROM_COL16(col);
	if ((op & BLT_MASKING) && (ret == (op & BLT_MASKCOL))) {
		ret = COLOR_TRANSPARENT;
	}
	return ret;
}

static void dec16_put_pixel(struct screen *s,int x, int y, int col) {
	char *p;
	int mask;

	if (col & COLOR_TRANSPARENT) return;
	if (col < 0) return;

	CLIP_POINT(s,x,y);

	x += s->off_x;
	y += s->off_y;
	if (s->type & ST_SUBSCREEN) {
		p = ((struct screen *)(s->bitmap))->bitmap;
		s->wbytes = 
			((struct screen *)(s->bitmap))->wbytes;
	} else {
		p = (char *)s->bitmap;
	}
	p += y * s->wbytes + (x>>1);
	mask = 0xf << ((x & 1)*4);
	/* *p = (*p & ~mask)|(CONV_TO_COL16(col) & mask); */
	*p = (*p & ~mask)|((CONV_TO_COL16(col)<<((x & 1)*4)) & mask);
	
	return;
}

static void dec16_get_pixstream(struct screen *s,int x, int y,int *buf,int length,int dir,int op
		,struct draw_engine *self) {
	unsigned char *p;
	int i;
	int d;
	int col,col2,len4,r;

	x += s->off_x;
	y += s->off_y;
	if (s->type & ST_SUBSCREEN) {
		p = ((struct screen *)(s->bitmap))->bitmap;
		s->wbytes = 
			((struct screen *)(s->bitmap))->wbytes;
	} else {
		p = (char *)s->bitmap;
	}
	p += y * s->wbytes + (x>>1);

	switch(dir) {
	case DIR_NORTH:
		if (op & BLT_MASKING) for (i=0; i<length; i++,x++) {
			d = (x & 1)*4;
			col = (*p >> d ) & 0xf;
			r = CONV_FROM_COL16(col);
			if (r == (op & BLT_MASKCOL)) {
				r = COLOR_TRANSPARENT;
			}
			*buf++ = r;
			if (d == 4) p++;
		    }
		else {
#if (OPTIMIZE_FOR == OPTTYPE_SLOW)
		    for (i=0; i<length; i++,x++) {
			d = (x & 1)*4;
			col = (*p >> d ) & 0xf;
			r = CONV_FROM_COL16(col);
			*buf++ = r;
			if (d == 4) p++;
		    }
#else
		    if (x & 1) {
			col = (*p >> 4 ) & 0xf;
			*buf++ = CONV_FROM_COL16(col);
			x++;
			p++;
			length--;
		    }
		    if (length & 1) {
			col = p[length/2] & 0xf;
			buf[length-1] = CONV_FROM_COL16(col);
		    }
		    length >>= 1;
		    len4 = length/2;
		    length -= len4 * 2;
		    for (i=0; i<len4; i++) {
			col = p[0]  & 0xf;
			col2 = ( p[0] >> 4)  & 0xf;
			buf[0] = CONV_FROM_COL16(col);
			buf[1] = CONV_FROM_COL16(col2);
			col = p[1]  & 0xf;
			col2 = ( p[1] >> 4)  & 0xf;
			buf[2] = CONV_FROM_COL16(col);
			buf[3] = CONV_FROM_COL16(col2);
			p += 2;
			buf += 4;
		    }
		    for (i=0; i<length; i++,p++) {
			col = *p  & 0xf;
			col2 = ( *p >> 4)  & 0xf;
			buf[0] = CONV_FROM_COL16(col);
			buf[1] = CONV_FROM_COL16(col2);
		    }
#endif
		}
		break;
	case DIR_WEST:
		d = (x & 1)*4;
		if (op & BLT_MASKING) for (i=0; i<length; i++) {
			col = (*p >> d ) & 0xf;
			r = CONV_FROM_COL16(col);
			if (r == (op & BLT_MASKCOL)) {
				r = COLOR_TRANSPARENT;
			}
			*buf++ = r;
			p -= s->wbytes;
		} else  for (i=0; i<length; i++) {
			col = (*p >> d ) & 0xf;
			r = CONV_FROM_COL16(col);
			*buf++ = r;
			p -= s->wbytes;
		}
		break;
	case DIR_SOUTH:
		if (op & BLT_MASKING) for (i=0; i<length; i++,x--) {
			d = (x & 1)*4;
			col = (*p >> d ) & 0xf;
			r = CONV_FROM_COL16(col);
			if (r == (op & BLT_MASKCOL)) {
				r = COLOR_TRANSPARENT;
			}
			*buf++ = r;
			if (d == 0) p--;
		} else for (i=0; i<length; i++,x--) {
			d = (x & 1)*4;
			col = (*p >> d ) & 0xf;
			r = CONV_FROM_COL16(col);
			*buf++ = r;
			if (d == 0) p--;
		}
		break;
	case DIR_EAST:
		d = (x & 1)*4;
		if (op & BLT_MASKING) for (i=0; i<length; i++) {
			col = (*p >> d ) & 0xf;
			r = CONV_FROM_COL16(col);
			if (r == (op & BLT_MASKCOL)) {
				r = COLOR_TRANSPARENT;
			}
			*buf++ = r;
			p += s->wbytes;
		} else for (i=0; i<length; i++) {
			col = (*p >> d ) & 0xf;
			r = CONV_FROM_COL16(col);
			*buf++ = r;
			p += s->wbytes;
		}
		break;
	}
}

static void dec16_put_pixstream(struct screen *s,int x, int y,int *buf,int length,int dir
		,struct draw_engine *self) {
	char *p;
	int col,len4,mask;
	int d,i;

	x += s->off_x;
	y += s->off_y;
	if (s->type & ST_SUBSCREEN) {
		p = ((struct screen *)(s->bitmap))->bitmap;
		s->wbytes = 
			((struct screen *)(s->bitmap))->wbytes;
	} else {
		p = (char *)s->bitmap;
	}
	p += y * s->wbytes + (x>>1);

	switch(dir) {
	case DIR_NORTH:
#if (OPTIMIZE_FOR == OPTTYPE_SLOW)
	    for (i=0; i<length;i++,x++) {
		col = *buf++;
		d = (x & 1)*4;
		if (!(col & COLOR_TRANSPARENT)) {
		    mask = 0xf << d;
		    *p = (*p & ~mask) |((CONV_TO_COL16(col)<<d) & mask);
		}
		if (d == 4) p++;
	    }
#else
	    if (x & 1) {
		col = *buf++;
		if (!(col & COLOR_TRANSPARENT)) {
		    mask = 0xf0;
		    *p = (*p & ~mask) |((CONV_TO_COL16(col)<<4) & mask);
		}
		x++;
		p++;
		length--;
	    }
	    if (length & 1) {
		col = buf[length-1];
		if (!(col & COLOR_TRANSPARENT)) {
		    mask = 0xf;
		    p[length/2] 
			= (p[length/2] & ~mask) |((CONV_TO_COL16(col)) & mask);
		}
	    }
	    length >>= 1;
	    len4 = length / 2;
	    length -= len4 * 2;
	    for (i=0; i<len4;i++) {
		col = buf[0];
		d = p[0];
		if (!(col & COLOR_TRANSPARENT)) {
		    d = (d & ~0xf) |((CONV_TO_COL16(col)) & 0xf);
		}
		col = buf[1];
		if (!(col & COLOR_TRANSPARENT)) {
		    d = (d & ~0xf0) |((CONV_TO_COL16(col)<<4) & 0xf0);
		}
		p[0] = d;

		col = buf[2];
		d = p[1];
		if (!(col & COLOR_TRANSPARENT)) {
		    d = (d & ~0xf) |((CONV_TO_COL16(col)) & 0xf);
		}
		col = buf[3];
		if (!(col & COLOR_TRANSPARENT)) {
		    d = (d & ~0xf0) |((CONV_TO_COL16(col)<<4) & 0xf0);
		}
		p[1] = d;
		buf += 4;
		p += 2;
	    }
	    for (i=0; i<length;i++) {
		col = buf[0];
		d = p[0];
		if (!(col & COLOR_TRANSPARENT)) {
		    d = (d & ~0xf) |((CONV_TO_COL16(col)) & 0xf);
		}
		col = buf[1];
		if (!(col & COLOR_TRANSPARENT)) {
		    d = (d & ~0xf0) |((CONV_TO_COL16(col)<<4) & 0xf0);
		}
		p[0] = d;
		buf += 2;
		p++;
	    }
#endif
	    break;
	case DIR_WEST:
	    d = (x & 1)*4;
	    mask = 0xf << d;
	    for (i=0; i<length;i++) {
		col = *buf++;
		if (!(col & COLOR_TRANSPARENT)) {
	 	    *p = (*p & ~mask) | ((CONV_TO_COL16(col)<<d)& mask);
		}
		p -= s->wbytes;
	    }
	    break;
	case DIR_SOUTH:
	    for (i=0; i<length;i++,x--) {
		col = *buf++;
		d = (x & 1)*4;
		if (!(col & COLOR_TRANSPARENT)) {
		    mask = 0xf << d;
		    *p = (*p & ~mask) | ((CONV_TO_COL16(col)<<d) & mask);
		}
	    if (d == 0) p--;
	    }
	    break;
	case DIR_EAST:
	    d = (x & 1)*4;
	    mask = 0xf << d;
	    for (i=0; i<length;i++) {
		col = *buf++;
		if (!(col & COLOR_TRANSPARENT)) {
		    *p = (*p & ~mask) | ((CONV_TO_COL16(col)<<d) & mask);
		}
		p += s->wbytes;
	    }
	    break;
	}
	return;
}

static void dec16_set_color(struct screen *s,int col) {
	lp_mask = CONV_TO_COL16(col) | ( CONV_TO_COL16(col) << 4) ;
	return;
}

static void dec16_draw_pixel(struct screen *s,int x, int y) {
	char *p;
	int mask;

//printf("dec16_draw_pixel (%d,%d) %d\n",x,y,lp_color);
	CLIP_POINT(s,x,y);

	x += s->off_x;
	y += s->off_y;
	if (s->type & ST_SUBSCREEN) {
		p = ((struct screen *)(s->bitmap))->bitmap;
		s->wbytes = 
			((struct screen *)(s->bitmap))->wbytes;
	} else {
		p = (char *)s->bitmap;
	}
	p += y * s->wbytes + (x>>1);

	mask = 0xf << ((x & 1)*4);
	if (pen_color == COLOR_REVERSE) {
		*p = (*p ^ mask);
	} else {
		*p = (*p & ~mask) | (lp_mask & mask);
	}
      return;
}

#define ABS(a) (((a)<0) ? -(a) : (a))

static void dec16_draw_line_vertical(struct screen *s,int x1, int y1, int x2, int y2,struct draw_engine *self) {
	char *p;
	int mask;
	int i;
	int n = y2 - y1 +1;

	CLIP_VLINE(s,x1,y1,x2,y2);

	x1 += s->off_x;
	y1 += s->off_y;
	x2 += s->off_x;
	y2 += s->off_y;
	if (s->type & ST_SUBSCREEN) {
		p = ((struct screen *)(s->bitmap))->bitmap;
		s->wbytes = 
			((struct screen *)(s->bitmap))->wbytes;
	} else {
		p = (char *)s->bitmap;
	}
	p += y1 * s->wbytes + (x1>>1);

	mask = 0xf << ((x1 & 1)*4);
	if (pen_color == COLOR_REVERSE) {
		for (i=0; i<n; i++,p+=s->wbytes)
			*p = (*p ^ mask);
	} else {
		for (i=0; i<n; i++,p+=s->wbytes)
			*p = (*p & ~mask) | (lp_mask & mask);
	}
}

static void dec16_draw_line_horizontal(struct screen *s,int x1, int y1, int x2, int y2,struct draw_engine *self) {
	char *p;
	int wb;
	int i;
	int off_x,off_y;

//printf("dec16_draw_line_horizontal \n",x1,y1,x2,y2);
	CLIP_HLINE(s,x1,y1,x2,y2);
//printf("clip pass \n");
	s->need_clipping--;

	off_x = s->off_x;
	off_y = s->off_y;

	x1 += off_x;
	x2 += off_x;
	y1 += off_y;
	y2 += off_y;

	while ((x1 % 2) != 0) {
	    draw_pixel(s,x1-off_x,y1-off_y);
	    x1++;
	}
	wb = (x2 + 1 - x1)/2;
	if (wb) {
	     if (s->type & ST_SUBSCREEN) {
		p = ((struct screen *)(s->bitmap))->bitmap;
		s->wbytes = 
			((struct screen *)(s->bitmap))->wbytes;
	     } else {
		p = (char *)s->bitmap;
	    }
	    p += y1 * s->wbytes + (x1>>1);

	    if (pen_color == COLOR_REVERSE) {
#if (OPTIMIZE_FOR == OPTTYPE_SLOW)
		for (i=0; i< wb; i++) {
		    *p++ ^= 0xff;
		}
#else
		mgl_memxor(p, 0xff, wb);
#endif
	    } else {
		memset(p, lp_mask & 0xff, wb);
	    }
	}
	x1 += wb * 2;
	while (x1 <= x2) {
	    draw_pixel(s,x1-off_x, y1-off_y);
	    x1++;
	}
	s->need_clipping++;
    return;
}


static void dec16_clear_screen(struct screen *s,struct draw_engine *self) {
	int x1,y1,x2,y2;
	int off_x,off_y;
	char *p,*pp;
	int wb;
	int i;
	int y;

	x1 = 0;
	x2 = s->width-1;
	y1 = 0;
	y2 = s->height-1;
	s->need_clipping--;

	off_x = s->off_x;
	off_y = s->off_y;
	x1 += off_x;
	x2 += off_x;
	y1 += off_y;
	y2 += off_y;

	while (x1 % 2 != 0) {
	    draw_line_vertical(s,x1-off_x,y1-off_y,x1-off_x,y2-off_y,self);
	    x1++;
	}
	wb = (x2 + 1 - x1)/2;
	if (wb) {
	    if (s->type & ST_SUBSCREEN) {
		p = ((struct screen *)(s->bitmap))->bitmap;
		s->wbytes = 
			((struct screen *)(s->bitmap))->wbytes;
	    } else {
		p = (char *)s->bitmap;
	    }
	    p += y1 * s->wbytes + (x1>>1);
	    for (y=y1; y<=y2; y++) {
		if (pen_color == COLOR_REVERSE) {
#if (OPTIMIZE_FOR == OPTTYPE_SLOW)
		    pp = p;
		    for (i=0; i< wb; i++) {
			*pp++ ^= 0xff;
		    }
#else
		    mgl_memxor(p, 0xff, wb);
#endif
		} else {
		    memset(p, lp_mask & 0xff, wb);
	        }
		p += s->wbytes;
	    }
	}
	x1 += wb * 2;
	while (x1 <= x2) {
	    draw_line_vertical(s,x1-off_x,y1-off_y,x1-off_x,y2-off_y,self);
	    x1++;
	}
	s->need_clipping++;
}

static void dec16_bitblt_copy(struct screen *dst, int dx, int dy
	, struct screen *src, int sx, int sy, int xsize, int ysize, int op
	, struct draw_engine *self) {
	char *dp,*sp;
	int i,len,d,e;
	int mask_d,mask_e;
	if ((dx - sx) & 0x1) goto gen;

	dx += dst->off_x;
	dy += dst->off_y;
	sx += src->off_x;
	sy += src->off_y;
	if (dst->type & ST_SUBSCREEN) {
		dp = ((struct screen *)(dst->bitmap))->bitmap;
		dst->wbytes = 
			((struct screen *)(dst->bitmap))->wbytes;
	} else {
		dp = (char *)dst->bitmap;
	}
	if (src->type & ST_SUBSCREEN) {
		sp = ((struct screen *)(src->bitmap))->bitmap;
		src->wbytes = 
			((struct screen *)(src->bitmap))->wbytes;
	} else {
		sp = (char *)src->bitmap;
	}
	d = 0;
	if (dx & 1) {
		d = 2 - (dx & 1);
	}
	dx += d;
	sx += d;
	xsize -= d;
	e = (xsize & 1);
	xsize -= e;
	dp += dy * dst->wbytes + (dx>>1);
	sp += sy * src->wbytes + (sx>>1);
	len = xsize>>1;
	mask_d = mask_e = 0;
	for (i=0; i<d; i++) {
		mask_d |= (0xf<<4)>>(i*4);
	}
	for (i=0; i<e; i++) {
		mask_e |= 0xf<<(i*4);
	}
	for (i=0; i< ysize; i++) {
		if (d) {
		    *(dp-1) = (*(dp-1) & ~mask_d) | (*(sp-1) & mask_d);
		}
		memmove(dp,sp,len);
		if (e) {
		    *(dp+len) = (*(dp+len) & ~mask_e) | (*(sp+len) & mask_e);
		}
		dp += dst->wbytes;
		sp += src->wbytes;
	}
	return;
gen:
	bitblt_generic(dst,dx,dy,src,sx,sy,xsize,ysize,op);
}

static void dec16_bitblt_scroll_forward(struct screen *dst, int dx, int dy
	, struct screen *src, int sx, int sy, int xsize, int ysize, int op
	, struct draw_engine *self) {
	char *p,*dp,*sp;
	int i,len,d,e;
	int mask_d,mask_e;
	if ((dx - sx) & 0x1) goto gen;

	dx += dst->off_x;
	dy += dst->off_y;
	sx += src->off_x;
	sy += src->off_y;
	if (dst->type & ST_SUBSCREEN) {
		p = ((struct screen *)(dst->bitmap))->bitmap;
		dst->wbytes = 
			((struct screen *)(dst->bitmap))->wbytes;
	} else {
		p = (char *)dst->bitmap;
	}
	d = 0;
	if (dx & 1) {
		d = 2 - (dx & 1);
	}
	dx += d;
	sx += d;
	xsize -= d;
	e = (xsize & 1);
	xsize -= e;
//printf("bitblt_forward (%d,%d)-(%d,%d) [%d,%d] d = %d e = %d"
//		,dx,dy,sx,sy,xsize,ysize,d,e);
	dp = p + dy * dst->wbytes + (dx>>1);
	sp = p + sy * dst->wbytes + (sx>>1);
	len = xsize>>1;
	mask_d = mask_e = 0;
	for (i=0; i<d; i++) {
		mask_d |= (0xf<<4)>>(i*4);
	}
	for (i=0; i<e; i++) {
		mask_e |= 0xf<<(i*4);
	}
	for (i=0; i< ysize; i++) {
		if (d) {
		    *(dp-1) = (*(dp-1) & ~mask_d) | (*(sp-1) & mask_d);
		}
		memmove(dp,sp,len);
		if (e) {
		    *(dp+len) = (*(dp+len) & ~mask_e) | (*(sp+len) & mask_e);
		}
		dp += dst->wbytes;
		sp += dst->wbytes;
	}
	return;
gen:
	bitblt_generic(dst,dx,dy,src,sx,sy,xsize,ysize,op);
}

static void dec16_bitblt_scroll_backward(struct screen *dst, int dx, int dy
	, struct screen *src, int sx, int sy, int xsize, int ysize, int op
	, struct draw_engine *self) {
	char *p,*dp,*sp;
	int i,len,d,e;
	int mask_d,mask_e;
	if ((dx - sx) & 0x1) goto gen;

	dx += dst->off_x;
	dy += dst->off_y;
	sx += src->off_x;
	sy += src->off_y;
	if (dst->type & ST_SUBSCREEN) {
		p = ((struct screen *)(dst->bitmap))->bitmap;
		dst->wbytes = 
			((struct screen *)(dst->bitmap))->wbytes;
	} else {
		p = (char *)dst->bitmap;
	}
	d = 0;
	if (dx & 1) {
		d = 2 - (dx & 1);
	}
	dx += d;
	sx += d;
	xsize -= d;
	e = (xsize & 1);
	xsize -= e;
	dy += ysize - 1;
	sy += ysize - 1;
//printf("bitblt_backward (%d,%d)-(%d,%d) [%d,%d] d = %d e = %d"
//		,dx,dy,sx,sy,xsize,ysize,d,e);
	dp = p + dy * dst->wbytes + (dx>>1);
	sp = p + sy * dst->wbytes + (sx>>1);
	len = xsize>>1;
	mask_d = mask_e = 0;
	for (i=0; i<d; i++) {
		mask_d |= (0xf<<4)>>(i*4);
	}
	for (i=0; i<e; i++) {
		mask_e |= 0xf<<(i*4);
	}
	for (i=0; i< ysize; i++) {
		if (e) {
		    *(dp+len) = (*(dp+len) & ~mask_e) | (*(sp+len) & mask_e);
		}
		memmove(dp,sp,len);
		if (d) {
		    *(dp-1) = (*(dp-1) & ~mask_d) | (*(sp-1) & mask_d);
		}
		dp -= dst->wbytes;
		sp -= dst->wbytes;
	}
	return;
gen:
	bitblt_generic(dst,dx,dy,src,sx,sy,xsize,ysize,op);
}

void dec16_init(int type) {
	dec16._create_subscreen = dec16_create_subscreen;
	dec16._free_screen = dec16_free_screen;
	dec16._put_pixel = dec16_put_pixel;
	dec16._get_pixel = dec16_get_pixel;

	dec16._set_color = dec16_set_color;
	dec16._draw_pixel = dec16_draw_pixel;

	//dec16._draw_line = dec16_draw_line;

	dec16._draw_line_vertical = dec16_draw_line_vertical;
	dec16._draw_line_horizontal = dec16_draw_line_horizontal;

	dec16._clear_screen = dec16_clear_screen;

	dec16._get_pixstream = dec16_get_pixstream;
	dec16._put_pixstream = dec16_put_pixstream;
	//dec16._put_pixstream_rect = dec16_put_pixstream_rect;

	//dec16._bitblt = dec16_bitblt;
	dec16._bitblt_scroll_forward = dec16_bitblt_scroll_forward;
	dec16._bitblt_scroll_backward = dec16_bitblt_scroll_backward;
	dec16._bitblt_copy = dec16_bitblt_copy;
	//dec16._bitblt_reserved_mask = dec16_bitblt_reserved_mask;
	//dec16._bitblt_reserved_masktile = dec16_bitblt_reserver_masktile;
	//dec16._bitblt_gen = dec16_bitblt_gen;
	setup_draw_engine(&dec16,0);
	_create_memscreen[type] = dec16_create_memscreen;
}
