/*
 * MGL -- MobileGear Graphic Library -
 * Copyright (C) 1998, 1999
 *      Koji Suzuki (suz@at.sakura.ne.jp)
 *      Yukihiko Sano (yukihiko@yk.rim.or.jp)
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY KOJI SUZUKI AND YUKIHIKO SANO ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE TERRENCE R. LAMBERT BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 */

#include <linux/keyboard.h>

static int alt_num;
static int shift_num;
static int ctrl_num;
static int menu_num;
static int esc_num;

static int
sym_convert(int sym) {
	int val = KVAL(sym);
	int ktype = KTYP(sym);

	if(ktype >= 0xf0){ ktype -= 0xf0; }

	// printf("symconvert(%02x, %02x) -> ", ktype, val);

	switch (ktype) {
	case KT_LATIN:
	case KT_LETTER:
	        // printf("letter %c\r\n", val);
		switch (val) {
		case '\e':
			esc_num++;
			if (esc_num == 1) return '\e';
			else return MKE_HANZEN;
		}
		return val;
	case KT_FN:
	        // printf("Fn %d\r\n", val);
		switch (val) {
		case  0: return MK_F1;
		case  1: return MK_F2;
		case  2: return MK_F3;
		case  3: return MK_F4;
		case  4: return MK_F5;
		case  5: return MK_F6;
		case  6: return MK_F7;
		case  7: return MK_F8;
		case  8: return MK_F9;
		case  9: return MK_F10;
		case 10: return MK_F11;
		case 11: return MK_F12;
		case 12: return MKE_F13;
		case 13: return MKE_F14;
		case 14: return MKE_F15;
		case 15: return MKE_F16;
		case 16: return MKE_F17;
		case 17: return MKE_F19;
		case 18: return MKE_F19;
		case 19: return MKE_F20;

		case KVAL(K_FIND):	return MK_HOME;
		case KVAL(K_INSERT):	return MK_INS;
		case KVAL(K_REMOVE):	return MK_DEL;
		case KVAL(K_SELECT):	return MK_END;
		case KVAL(K_PGUP): return MK_PAGE_UP;
		case KVAL(K_PGDN): return MK_PAGE_DOWN;
		case KVAL(K_MACRO): return MGL_SKM_MENU;

		/* case KVAL(K_HELP): return MKE_HELP; */
		case KVAL(K_DO): return MKE_EXECUTE;
		case KVAL(K_PAUSE): return MKE_PAUSE;
		}
		return MK_NONE;
	case KT_SPEC:
	        // printf("spec %d\r\n", val);
		switch (val) {
		case 0: return MK_NONE;
		case KVAL(K_ENTER): return '\r';
		case KVAL(K_BREAK): return MKE_BREAK;
		case KVAL(K_CAPS): return MGL_SKM_CAPS;
		case KVAL(K_NUM): return MK_NUMLOCK;
		case KVAL(K_HOLD): return MKE_SCRLOCK;
		/* case KVAL(K_COMPOSE): return MKE_MULTIKEY; */

		case 0x19: /* contrast++ for MobileGear */
			return MKE_CONTRAST_UP;
		case 0x1A: /* contrast-- for MobileGear */
			return MKE_CONTRAST_DOWN; 
		}
		return MK_NONE;
	case KT_PAD:
	        // printf("pad %d\r\n", val);
		switch (val) {
		case 0: return MKE_0;
		case 1: return MKE_1;
		case 2: return MKE_2;
		case 3: return MKE_3;
		case 4: return MKE_4;
		case 5: return MKE_5;
		case 6: return MKE_6;
		case 7: return MKE_7;
		case 8: return MKE_8;
		case 9: return MKE_9;
		case KVAL(K_PPLUS): return MKE_PLUS;
		case KVAL(K_PMINUS): return MKE_MINUS;
		case KVAL(K_PSTAR): return MKE_MULT;
		case KVAL(K_PSLASH): return MKE_DIV;
		case KVAL(K_PENTER): return '\r';
		/* case KVAL(K_PCOMMA): return MKE_RETURN; */
		/* case KVAL(K_PDOT): return MKE_RETURN; */
		}
		return MK_NONE;
	case KT_DEAD:
		/* not implemented yet.  */
		return MK_NONE;
	case KT_CONS:
#ifdef MD_PROC_CONSOLE
	        // printf("console %d\r\n", val);
		if(val < 10){
		    return (MKE_CONSOLE_1 + val);
		}
#endif
		return MK_NONE;
	case KT_CUR:
	        // printf("cur %d\r\n", val);
		switch (val) {
		case KVAL(K_DOWN): return MK_DOWN;
		case KVAL(K_LEFT): return MK_LEFT;
		case KVAL(K_RIGHT): return MK_RIGHT;
		case KVAL(K_UP): return MK_UP;
		}
		return MK_NONE;
	case KT_LOCK:
		/* not implemented yet, so treat as "shift".  */
	case KT_SLOCK:
		/* not implemented yet, so treat as "shift".  */
	case KT_SHIFT:
	        // printf("shift %d\r\n", val);
		switch (val) {
		case KVAL(K_SHIFT): 
			shift_num++;
			if (shift_num == 1)	return MGL_SKM_SHIFT;
			else 			return MGL_SKM_RSHIFT;
		case KVAL(K_SHIFTL): 		return MGL_SKM_SHIFT;
		case KVAL(K_SHIFTR): 		return MGL_SKM_RSHIFT; 
		case KVAL(K_CTRL): 
			ctrl_num++;
			if (ctrl_num == 1)	return MGL_SKM_CTRL;
			else 			return MGL_SKM_RCTRL;
		case KVAL(K_CTRLL): 		return MGL_SKM_CTRL;
		case KVAL(K_CTRLR): 		return MGL_SKM_RCTRL;
		case KVAL(K_ALT): 
			alt_num++;
			if (alt_num == 1)	return MGL_SKM_ALT;
			else 			return MGL_SKM_RALT;
		case KVAL(K_ALTGR):
			menu_num++;
			if (menu_num == 1)	return MGL_SKM_OPT;
			else 			return MGL_SKM_OPT;
		}
		return MK_NONE;
	case KT_META:
	case KT_ASCII:
		/* not implemented yet. */
		return MK_NONE;
	case 0x0f:  /* KT_ARCH. machine dependent functions. */
	        // printf("arch %d\r\n", val);
		switch (val) {
		case 1: /* SysREQ */
		case 2: /* OFF */
		case 6: /* debug 0 */
		case 7: /* debug 1 */
		case 8: /* debug 2 */
		case 9: /* debug 3 */
		case 10: /* debug 4 */
		case 11: /* debug 5 */
		default:
			return MK_NONE;
		case 3: /* backlight */
			return MKE_BACKLIGHT_TOGGLE;
		case 4: /* contrast-- */
			return MKE_CONTRAST_DOWN;
		case 5: /* contrast++ */
			return MKE_CONTRAST_UP;
		}
		return MK_NONE;
	}
	return MK_NONE;
}

static int create_keymap() {
	struct kbentry kbe;
	int ret;
	int i,j;
	int sym;

	mk_init();

#ifndef UNIFY_PLAIN_AND_ALTGR

	/* plain */
	alt_num = shift_num = ctrl_num = menu_num = esc_num = 0;
	for (i=0; i< 128; i++) {
		kbe.kb_index = i;
		kbe.kb_table = 0;
		ret = ioctl(mgl_keyboard_fd, KDGKBENT, &kbe);
		if (ret < 0) return (-1);
		sym = sym_convert(kbe.kb_value);
		mgl_keymap[i][0] = sym;
	}
	/* shift */
	alt_num = shift_num = ctrl_num = menu_num = esc_num = 0;
	for (i=0; i< 128; i++) {
		kbe.kb_index = i;
		kbe.kb_table = (1 << KG_SHIFT);
		ret = ioctl(mgl_keyboard_fd, KDGKBENT, &kbe);
		if (ret < 0) return (-1);
		sym = sym_convert(kbe.kb_value);
		mgl_keymap[i][1] = sym;
	}
	/* alt */
	alt_num = shift_num = ctrl_num = menu_num = esc_num = 0;
	for (i=0; i< 128; i++) {
		kbe.kb_index = i;
		kbe.kb_table = (1 << KG_ALT);
		ret = ioctl(mgl_keyboard_fd, KDGKBENT, &kbe);
		if (ret < 0) return (-1);
		sym = sym_convert(kbe.kb_value);

		mgl_keymap[i][2] = sym;
	}
	/*  104/106/109 keyboard patch */
	if (mgl_keymap[94][0] == ' ') {
		mgl_keymap[94][0] = MKE_HENKAN;
		mgl_keymap[94][1] = MKE_HENKAN;
	}
	if (mgl_keymap[92][0] == ' ') {
		mgl_keymap[92][0] = MKE_MUHENKAN;
		mgl_keymap[92][1] = MKE_MUHENKAN;
	}
	if (mgl_keymap[125][0] == MK_NONE) {
		mgl_keymap[125][0] = MGL_SKM_MENU;
		mgl_keymap[125][1] = MGL_SKM_MENU;
	}
	if (mgl_keymap[126][0] == MK_NONE) {
		mgl_keymap[126][0] = MGL_SKM_RMENU;
		mgl_keymap[126][1] = MGL_SKM_RMENU;
	}
#else /* defined UNIFY_PLAIN_AND_ALTGR */
	/* plain */
	alt_num = shift_num = ctrl_num = menu_num = esc_num = 0;
	for (i=0; i< 64; i++) {
	        int ktype;
		kbe.kb_index = i;
		kbe.kb_table = 0;
		ret = ioctl(mgl_keyboard_fd, KDGKBENT, &kbe);
		if (ret < 0) {
		    fprintf(stderr, "warning cannot get plainmap\r\n");
		    return (-1);
		}
		sym = sym_convert(kbe.kb_value);
		ktype = KTYP(kbe.kb_value) & 0x0f;
		if (ktype == KT_SHIFT || ktype == KT_LOCK || ktype == KT_SLOCK){
		    if (KVAL(kbe.kb_value) == KG_ALTGR) {
			if(altgr_code) {
			    fprintf(stderr,
				    "Warning. MGL2 overrides altgr(%d) by key(%d).\n",
				    altgr_code, i);
			}
			altgr_code = i;
		    }
		}
		mgl_keymap[i][0] = sym;
	}
	/* shift */
	alt_num = shift_num = ctrl_num = menu_num = esc_num = 0;
	for (i=0; i< 64; i++) {
		kbe.kb_index = i;
		kbe.kb_table = (1 << KG_SHIFT);
		ret = ioctl(mgl_keyboard_fd, KDGKBENT, &kbe);
		if (ret < 0) {
		    fprintf(stderr, "warning cannot get shiftmap\r\n");
		    return (-1);
		}
		sym = sym_convert(kbe.kb_value);
		mgl_keymap[i][1] = sym;
	}
	/* alt */
	alt_num = shift_num = ctrl_num = menu_num = esc_num = 0;
	for (i=0; i< 64; i++) {
		kbe.kb_index = i;
		kbe.kb_table = (1 << KG_ALT);
		ret = ioctl(mgl_keyboard_fd, KDGKBENT, &kbe);
		if (ret < 0) {
		    fprintf(stderr, "warning cannot get altmap\r\n");
		    return (-1);
		}
		sym = sym_convert(kbe.kb_value);

		mgl_keymap[i][2] = sym;
	}
	/* plain + altgr */
	alt_num = shift_num = ctrl_num = menu_num = esc_num = 0;
	for (i=0; i< 64; i++) {
		kbe.kb_index = i;
		kbe.kb_table = (1 << KG_ALTGR);
		ret = ioctl(mgl_keyboard_fd, KDGKBENT, &kbe);
		if (ret < 0) {
		    fprintf(stderr, "warning cannot get altgr map\r\n");
		    return (-1);
		}
		sym = sym_convert(kbe.kb_value);
		mgl_keymap[i+64][0] = sym;
	}
	/* shift + altgr */
	alt_num = shift_num = ctrl_num = menu_num = esc_num = 0;
	for (i=0; i< 64; i++) {
		kbe.kb_index = i;
		kbe.kb_table = (1 << KG_ALTGR) | ( 1 << KG_SHIFT) ;
		ret = ioctl(mgl_keyboard_fd, KDGKBENT, &kbe);
		if (ret < 0) {
		    fprintf(stderr, "warning cannot get shift+altgr map\r\n");
		    return (-1);
		}
		sym = sym_convert(kbe.kb_value);
		mgl_keymap[i+64][1] = sym;
	}
	/* alt + altgr */
	alt_num = shift_num = ctrl_num = menu_num = esc_num = 0;
	for (i=0; i< 64; i++) {
		kbe.kb_index = i;
		kbe.kb_table = (1 << KG_ALTGR) | ( 1 << KG_ALT);
		ret = ioctl(mgl_keyboard_fd, KDGKBENT, &kbe);
		if (ret < 0) {
		    fprintf(stderr, "warning cannot get alt+altgr map\r\n");
		    return (-1);
		}
		sym = sym_convert(kbe.kb_value);

		mgl_keymap[i+64][2] = sym;
	}
	altgr_unified = 1;
#endif

#if 0
mk_dump_keymap("keymap");
#endif
	mk_create_raw_keymap();
	modify_keymap_for_mglsvr();
#if 0
mk_dump_keymap("mglmap ");
#endif
}
