/* TECO for Ultrix   Copyright 1986 Matt Fichtenbaum						*/
/* This program and its components belong to GenRad Inc, Concord MA 01742	*/
/* They may be copied if this copyright notice is included					*/

/* te_chario.c   character I/O routines   02/05/88 */
/* mod to leave stdin modes unchanged if not a terminal */
/* mod 10/21/87 to not reset signals on TTY_SUSP */
/* version for multiple windows 04/13/89 15.54 */
/* mod 7/20/89 to declare signal handlers as "VOID" not "int" */
/* mod 12/14/90  09.31 for function keys */

#include <errno.h>
#include <sys/time.h>
#include "te_defs.h"


#ifdef SHRET_INT
#define SHRET int
#else
#define SHRET VOID
#endif

/* compiler supports void type, otherwise declare this to be int */
#define VOID void

#ifdef POSIX_TECO
#include <termios.h>
#endif

#include <unistd.h>

#if defined(USE_SELECT) && defined(_AIX)
#include <sys/select.h>
#endif

#ifdef USE_POLL
#include <stropts.h>
#include <poll.h>
#endif

#include <fcntl.h>
#ifndef DEBUG
#include <signal.h>
extern SHRET int_handler();
extern SHRET stp_handler();
extern SHRET hup_handler();
extern SHRET winch_handler();
#define SIGINTMASK 2
#endif

int lf_sw;									/* nonzero: make up a LF following an entered CR */
int ttyflags;								/* flags for (stdin) file descriptor */
#ifndef POSIX_TECO
struct tchars tc_orig, tc_new, tc_noint;	/* original, new, disabled intrpt tty special chars */
struct ltchars lc_orig, lc_new;				/* original and new local special chars */
struct sgttyb tty_orig, tty_new;			/* original and new tty flags */
int tty_local;								/* original tty local mode flags */
int lnoflsh = LNOFLSH;						/* bit to force "no flush on interrupt */
#else
/* original and new tty flags */
struct termios tty_orig, tty_new, tc_noint;
static char backpush = 0;	/* Emulate BSD push-back function */
#endif

#ifdef _AIX
int bsdld;
#endif

#ifndef DEBUG
#ifdef POSIX_TECO
#ifndef SA_INTERRUPT
/* this is really for linux */
#define SA_INTERRUPT 0
#endif
struct sigaction intsigstruc = { int_handler, 0, SA_INTERRUPT } ;		/* info structure for ^C interrupt		*/
struct sigaction stpsigstruc = { stp_handler, 0, SA_INTERRUPT } ;		/* info structure for "stop" signal		*/
struct sigaction hupsigstruc = { hup_handler, 0, SA_INTERRUPT } ;		/* info structure for "hangup" signal	*/
struct sigaction nosigstr = { SIG_DFL, 0, 0 };				/* default structure for signal			*/
struct sigaction sigwinchstruc = { winch_handler, 0, SA_INTERRUPT} ;
#else
struct sigvec intsigstruc = { int_handler, 0, 0 } ;		/* info structure for ^C interrupt		*/
struct sigvec stpsigstruc = { stp_handler, 0, 0 } ;		/* info structure for "stop" signal		*/
struct sigvec hupsigstruc = { hup_handler, 0, 0 } ;		/* info structure for "hangup" signal	*/
struct sigvec nosigstr = { SIG_DFL, 0, 0 };				/* default structure for signal			*/
struct sigvec sigwinchstruc = { winch_handler, 0, 0} ;
#endif
#endif

int inp_noterm;				/* nonzero if standard input is not a terminal */
int out_noterm;				/* nonzero if standard output is not a terminal */
/* set tty (stdin) mode.  TECO mode is CBREAK, no ECHO, sep CR & LF				*/
/* operation; normal mode is none of the above.  TTY_OFF and TTY_ON do this		*/
/* absolutely; TTY_SUSP and TTY_RESUME use saved signal status.					*/

#if defined(_AIX) && !defined(POSIX_TECO)
char ldbuf[TTNAMEMAX];
#endif

#ifndef POSIX_TECO

setup_tty(arg)
	int arg;
{
	extern int errno;
	int ioerr;
	struct sgttyb tmpbuf;

/* initial processing: set tty mode */

	if (arg == TTY_ON)
	{
		ioerr = ioctl(fileno(stdin), TIOCGETP, &tty_orig);	/* get std input characteristics */
		inp_noterm = (ioerr && (errno == ENOTTY));			/* nonzero if input not a terminal */
		ioerr = ioctl(fileno(stdout), TIOCGETP, &tmpbuf);	/* get std output characteristics */
		out_noterm = (ioerr && (errno == ENOTTY));			/* nonzero if output not a terminal */
		ioctl(fileno(stdout), TIOCLGET, &tty_local);		/* get current "local mode flags" word */

		ttybuf = tty_new = tty_orig;						/* make a copy of tty control structure */
		tty_new.sg_flags = (tty_new.sg_flags & ~ECHO & ~CRMOD) | CBREAK;	/* turn on teco modes */

		ioctl(fileno(stdin), TIOCGETC, &tc_orig);		/* read current tchars */
		tc_new = tc_orig;								/* make local copy */
		tc_new.t_quitc = tc_new.t_brkc = -1;			/* disable "quit" and "delimiter" chars */
		tc_noint = tc_new;
		tc_noint.t_intrc = -1;							/* disable the interrupt char in this one */

		ioctl(fileno(stdin), TIOCGLTC, &lc_orig);		/* read current ltchars */
		lc_new = lc_orig;								/* make local copy */
		lc_new.t_rprntc = lc_new.t_werasc = lc_new.t_lnextc = -1;	/* disable "reprint," "word erase," "lit next" */
	}

	if (!inp_noterm)		/* if std input is from terminal */
	{
		if ((arg == TTY_ON) || (arg == TTY_RESUME))
		{
#ifdef _AIX
/* bsd line discipline */
			ioctl(fileno(stdin), TXGETLD, ldbuf);
			ioctl(fileno(stdin), TXSETLD, "bsd");
			bsdld = NTTYDISC;
			ioctl(fileno(stdin), TIOCSETD, &bsdld);
#endif
			ioctl(fileno(stdin), TIOCSETN, &tty_new);		/* set flags for teco */
			ioctl(fileno(stdin), TIOCSETC, &tc_new);		/* update both */
			ioctl(fileno(stdin), TIOCSLTC, &lc_new);
			ioctl(fileno(stdout), TIOCLBIS, &lnoflsh);		/* disable "interrupt => flush buffers" */
#ifndef DEBUG
			if (arg == TTY_ON)
			{
				sigvec(SIGTSTP, &stpsigstruc, 0);			/* set up to trap "stop" signal */
				sigvec(SIGINT, &intsigstruc, 0);			/* and "interrupt" signal */
				sigvec(SIGWINCH, &sigwinchstruc, 0);

				sigvec(SIGHUP, &hupsigstruc, 0);			/* and "hangup" signal */
			}
#endif
		}

		else							/* argument is TTY_OFF or TTY_SUSP */
		{
#ifdef _AIX
				ioctl(fileno(stdin), TXSETLD, ldbuf);
#endif
			ioctl(fileno(stdin), TIOCSETC, &tc_orig);		/* put both back to orig states */
			ioctl(fileno(stdin), TIOCSLTC, &lc_orig);
			ioctl(fileno(stdout), TIOCLSET, &tty_local);	/* restore local mode flags to original states */
			ioctl(fileno(stdin), TIOCSETN, &tty_orig);		/* set flags back to original, but don't flush input */
#ifndef DEBUG
			if (arg == TTY_OFF)
			{
				sigvec(SIGTSTP, &nosigstr, 0);				/* restore default signal handling */
				sigvec(SIGINT, &nosigstr, 0);
				sigvec(SIGHUP, &nosigstr, 0);
				sigvec(SIGWINCH, &nosigstr, 0);
			}
#endif
		}
	}
}
#else   /* POSIX_TECO
/*
 * set tty (stdin) mode.  TECO mode is CBREAK, no ECHO, sep CR & LF
 * operation; normal mode is none of the above.  TTY_OFF and TTY_ON do this
 * absolutely; TTY_SUSP and TTY_RESUME use saved signal status.
 */
setup_tty(arg)
int arg;
{
	extern int errno;
	int ioerr;
	struct termios tmpbuf;

	/* initial processing: set tty mode */

	if (arg == TTY_ON)
	{

		ioerr = tcgetattr(fileno(stdin), &tty_orig);

		/* nonzero if input not a terminal */
		inp_noterm = (ioerr && (errno == ENOTTY));

		/* get std output characteristics */
		ioerr = tcgetattr(fileno(stdout), &tmpbuf);
		/* nonzero if output not a terminal */
		out_noterm = (ioerr && (errno == ENOTTY));

		/* make a copy of tty control structure */
		tty_new = tty_orig;

		/* turn on teco modes */
		tty_new.c_iflag &= ~(INLCR|IGNCR|ICRNL|IXON|IXOFF);
		tty_new.c_lflag &= ~(ICANON|ECHO);
		tty_new.c_lflag |= (ISIG);
		tty_new.c_oflag &= ~(ONLCR);
		tty_new.c_cc[VMIN] = 1;
		tty_new.c_cc[VTIME] = 0;

		/* disable the interrupt char in this one */
		tc_noint = tty_new;
		tc_noint.c_cc[VINTR] = -1;
	}

	if ((arg == TTY_ON) || (arg == TTY_RESUME))
	{
		/* Set up TTY for TECO */
		tcdrain(fileno(stdin));
		tcdrain(fileno(stdin)); tcsetattr(fileno(stdin), TCSANOW, &tty_new);
#ifndef DEBUG
		/* Handle signals */
		if (sigaction(SIGTSTP, &stpsigstruc, NULL) < 0) {
			fprintf(stderr, "could not install signal handler for SIGTSTP\n");
			exit(1);
	    }
		if (sigaction(SIGINT, &intsigstruc, NULL) < 0) {
			fprintf(stderr, "could not install signal handler for SIGINT\n");
			exit(1);
	    }
		if (sigaction(SIGHUP, &hupsigstruc, NULL) < 0) {
			fprintf(stderr, "could not install signal handler for SIGHUP\n");
			exit(1);
	    }
		if (sigaction(SIGWINCH, &sigwinchstruc, NULL) < 0) {
			fprintf(stderr, "could not install signal handler for SIGWINCH\n");
			exit(1);
	    }
#endif
	} else {
		/* Restore to original state */
		tcdrain(fileno(stdin)); tcsetattr(fileno(stdin), TCSANOW, &tty_orig);
#ifndef DEBUG
		sigaction(SIGTSTP, &nosigstr, 0);
		sigaction(SIGINT, &nosigstr, 0);
		sigaction(SIGHUP, &nosigstr, 0);
		sigaction(SIGWINCH, &nosigstr, 0);
#endif
	}
}
#endif
/* routines to handle keyboard input */

/* routine to get a character without waiting, used by ^T when ET & 64 is set	*/
/* if lf_sw is nonzero, return the LF; else use the FNDELAY fcntl to inquire of the input */
/* if input is not a terminal don't switch modes */

unsigned short get_kbd_enh();
#ifndef POSIX_TECO

int gettty_nowait()
{
	int c;

	if (lf_sw)
	{
		lf_sw = 0;
		return(LF);			/* LF to be sent: return it */
	}
	if (!inp_noterm) fcntl(fileno(stdin), F_SETFL, ttyflags | FNDELAY);		/* set to "no delay" mode */
	while (!(c = get_kbd_enh()));				/* read character, or -1, skip nulls */
	if (!inp_noterm) fcntl(fileno(stdin), F_SETFL, ttyflags);				/* reset to normal mode */
	if (c == CR) ++lf_sw;						/* CR: set switch to make up a LF */
	return(c);
}

#else /* POSIX_TECO */

/*
 * routine to get a character without waiting, used by ^T when ET & 64 is
 * set if lf_sw is nonzero, return the LF; else use the FNDELAY fcntl to
 * inquire of the input
 */
gettty_nowait()
{
	char c;
	int err, cnt;

again:
	if (lf_sw) {
		lf_sw = 0;
		return(LF);		/* LF to be sent: return it */
	}

	/* If have pushback, return it */
	if (backpush) {
		c = backpush;
		backpush = '\0';
		return(c);
	}

	/* Clear EOF if set */
	if (feof(stdin))
		clearerr(stdin);

	/* set to "no delay" mode */
	tty_new.c_cc[VMIN] = 0;
	tcdrain(fileno(stdin)); tcsetattr(fileno(stdin), TCSANOW, &tty_new);

	/* read character, or -1, skip nulls */
	do {
		cnt = read(fileno(stdin), &c, sizeof(c));
	} while ((cnt > 0) && (c == '\0'));
	err = errno;

	/* reset to normal mode */
	tty_new.c_cc[VMIN] = 1;
	tcdrain(fileno(stdin)); tcsetattr(fileno(stdin), TCSANOW, &tty_new);

	/* If interrupted, try again */
	if ((cnt < 0) && (err == EINTR))
		goto again;

	/* No data--return -1 */
	if (cnt == 0)
		return(-1);

	/* CR: set switch to make up a LF */
	if (c == CR)
		++lf_sw;
	return(c);
}
#endif


/* normal routine to get a character */

int in_read = 0;		/* flag for "read busy" (used by interrupt handler) */

int gettty()
{
	int c;

	if (lf_sw)
	{
		lf_sw = 0;
		return(LF);		/* if switch set, make up a line feed */
	}
	++in_read;								/* set "read busy" switch */
	while(!(c = get_kbd_enh()));			/* get character; skip nulls */
	in_read = 0;							/* clear switch */
	if (c == CR) ++lf_sw;					/* CR: set switch to make up a LF */
	if (c == EOF) ERROR(E_EFI);				/* end-of-file from standard input */
	return(c);
}
/* routine to read keyboard characters and interpret incoming escape sequences */

/* sequences accepted :
 * keypad 0		$Op			0400
 * through
 * keypad 9		$Oy			0411
 * cursor up	$OA, $[A	0412
 * cursor down	$OB, $[B	0413
 * cursor right	$OC, $[C	0414
 * cursor left	$OD, $[D	0415
 * PF1			$OP			0416
 * PF2			$OQ			0417
 * PF3			$OR			0420
 * PF4			$OS			0421
 * keypad ,		$Ol			0422
 * keypad -		$Om			0423
 * keypad .		$On			0424
 * keypad ENTER	$OM			0425
 */

struct trans_entry
{
	unsigned char i;
	unsigned char o;
};

static struct trans_entry table1[] =
{
	{	'p',	0	},
	{	'q',	1	},
	{	'r',	2	},
	{	's',	3	},
	{	't',	4	},
	{	'u',	5	},
	{	'v',	6	},
	{	'w',	7	},
	{	'x',	8	},
	{	'y',	9	},
	{	'P',	14	},
	{	'Q',	15	},
	{	'R',	16	},
	{	'S',	17	},
	{	'm',	18	},
	{	'l',	19	},
	{	'n',	20	},
	{	'M',	21	},
	{	'A',	10	},
	{	'B',	11	},
	{	'C',	13	},
	{	'D',	12	}
} ;

#define TABLE_SIZE (sizeof(table1)/sizeof(struct trans_entry))

#ifdef USE_SELECT
static struct timeval max_timeout;
#endif
unsigned short n_buff;
static unsigned char ca[3] = {0,0,0};
#define c1 ca[0]
#define c2 ca[1]
#define c3 ca[2]

int zread(fd,c)
int fd;
char *c;
{
	int i=read(fd,c,1);
	if (i<1)
		*c=0;
	return i;
}

#if defined(USE_SELECT)
int timed_read(fd,c)
int fd;
char *c;
{
	fd_set read_mask, write_mask, except_mask;
	if (esc_timeout > 0) {
		max_timeout.tv_sec = esc_timeout / 10;
		max_timeout.tv_usec = (esc_timeout % 10) * 100000;
		FD_ZERO(&read_mask); 
		FD_ZERO(&write_mask); 
		FD_ZERO(&except_mask); 
		FD_SET(fd, &read_mask);
		*c=0;
		if (select(fd+1, &read_mask, &write_mask, &except_mask, &max_timeout) > 0) 
			return zread(fd, c);
		else 
			return 0;
	}
	else
		return 0;
}
#endif

#if defined(USE_POLL)
int timed_read(fd,c)
int fd;
char *c;
{
	struct pollfd pfd[1];
	if (esc_timeout > 0) {
	pfd[0].fd = fd;
	pfd[0].events = POLLIN;
	pfd[0].revents = 0;
	if (poll(pfd, (unsigned long)1, esc_timeout * 100) > 0)
		return zread(fd, c);
	else
		return 0;
	}
	else
		return 0;
}
#endif

#if defined(USE_POSIX_C_CC) 
int timed_read(fd,c)
int fd;
char *c;
{
	struct termios old,new;
	int retval;
	if (esc_timeout > 0) {
	tcgetattr(fd,&old);
	memcpy(&new,&old,sizeof(struct termios));
	new.c_cc[VMIN]=0;
	new.c_cc[VTIME]=esc_timeout;
	tcdrain(fd); tcsetattr(fd,TCSANOW, &new);
	if (zread(fd, c) > 0)
		retval=1;
	else
		retval=0;
	tcdrain(fd); tcsetattr(fd,TCSANOW, &old);
	return retval;
	}
	else
		return 0;
}
#endif

unsigned short get_kbd_enh()
{
	unsigned short retval;
	struct trans_entry *pt;

#ifdef POSIX_TECO
	if (backpush) {
		retval=(unsigned short)backpush;
		backpush=(char)0;
		return (retval);
	}
#endif
	if (inp_noterm) return((unsigned short)getchar());
	else fflush(stdout);
	if (c2 != 0) {
		retval = c2;
		c2 = 0;
	}
	else if (c3 != 0) {
		retval = c3;
		c3 = 0;
	}
	else
	{
		c1=0;
		(VOID) zread(fileno(stdin), &c1);
		if (((retval = c1) == 033) && (esc_timeout > 0))				/* escape */
		{
			c2 = 0;
			if (timed_read(fileno(stdin), &c2) > 0) {
				if (c2 == '[' || c2 == 'O') {
					if (timed_read(fileno(stdin), &c3) > 0) {
						retval = c1;
						pt = (c2 == '[') ? &table1[TABLE_SIZE-4] : table1;
						for (; pt < &table1[TABLE_SIZE]; pt++) {
							if (c3 == pt->i) {
								retval = pt->o | 0400;
								c2 = c3 = 0;
								break;
							}
						}
					}
				}
			}
		}
	}
	return(retval);
}


#ifndef DEBUG

SHRET winch_handler()
{
	get_term_par();
	window(WIN_INIT);
	if (WN_scroll != 0)
	{
		window(WIN_REDRAW);		/* redraw full window */
		window(WIN_REFR);
	}
}

SHRET int_handler()
{

	if (exitflag <= 0)						/* if executing commands */
	{
		if (et_val & ET_CTRLC) et_val &= ~ET_CTRLC;		/* if "trap ^C" set, clear it and ignore */
		else exitflag = -2;								/* else set flag to stop execution */
	}
	if (in_read)							/* if interrupt happened in "getchar" pass a ^C to input */
	{
		in_read = 0;									/* clear "read" switch */
#ifdef POSIX_TECO
		tcdrain(fileno(stdin)); tcsetattr(fileno(stdin), TCSANOW, &tc_noint);
#else
		ioctl(fileno(stdin), TIOCSETC, &tc_noint);		/* disable interrupt char */
#endif
		qio_char(CTL('C'));								/* send a ^C to input stream */
#ifdef POSIX_TECO
		tcdrain(fileno(stdin)); tcsetattr(fileno(stdin), TCSANOW, &tty_new);
#else
		ioctl(fileno(stdin), TIOCSETC, &tc_new);		/* reenable interrupt char */
#endif
	}
}
#endif
/* routine to disable (1), enable (0) ^C interrupt, used to block interrupts during display update */

#ifdef POSIX_TECO
block_inter(func)
int func;
{
	sigset_t ss;
	(VOID)sigemptyset(&ss);
	(VOID)sigaddset(&ss,SIGINT);
	if (func)
		(VOID)sigprocmask(SIG_BLOCK, &ss, NULL);
	else
		(VOID)sigprocmask(SIG_UNBLOCK, &ss, NULL);
}
#else
int old_mask;				/* storage for previous signal mask */
#define INT_MASK 2
block_inter(func)
	int func;
{
#ifndef DEBUG
	if (func) old_mask = sigblock(INT_MASK);			/* if arg nonzero, block interrupt */
	else sigsetmask(old_mask);							/* otherwise restore old signal mask */
#endif
}
#endif



#ifndef DEBUG
/* routine to handle "stop" signal (^Y) */
#ifdef POSIX_TECO

SHRET stp_handler()
{
	sigset_t ss;
	sigemptyset(&ss);
	crlf();
	window(WIN_SUSP);				/* restore screen */
	setup_tty(TTY_SUSP);			/* put tty back to normal */
	sigaction(SIGTSTP, &nosigstr, 0); 	/* put default action back */
	sigprocmask(SIG_SETMASK, &ss, NULL);
	kill(0, SIGTSTP);				/* suspend this process */

/* ----- process gets suspended here ----- */

	sigaction(SIGTSTP, &stpsigstruc, 0);	/* restore local handling of "stop" signal */
	setup_tty(TTY_RESUME);				/* restore tty */
	pbuff->buff_mod = 0;				/* set whole screen modified */
	if (win_data[7])			/* redraw window */
	{
		window(WIN_RESUME);		/* re-enable window */
		window(WIN_REDRAW);		/* force complete redraw */
		window(WIN_REFR);		/* and refresh */
	}
	qio_char('\0');				/* wake up the input */
	if (exitflag) retype_cmdstr('*');	/* if not executing, prompt again and echo command string so far */
}

#else   /* BSD style signals */

SHRET stp_handler()
{
	crlf();
	window(WIN_SUSP);				/* restore screen */
	setup_tty(TTY_SUSP);			/* put tty back to normal */
	sigvec(SIGTSTP, &nosigstr, 0); 	/* put default action back */
	sigsetmask(0);					/* unblock "suspend" signal */
	kill(0, SIGTSTP);				/* suspend this process */

/* ----- process gets suspended here ----- */

	sigvec(SIGTSTP, &stpsigstruc, 0);	/* restore local handling of "stop" signal */
	setup_tty(TTY_RESUME);				/* restore tty */
	pbuff->buff_mod = 0;				/* set whole screen modified */
	if (win_data[7])			/* redraw window */
	{
		window(WIN_RESUME);		/* re-enable window */
		window(WIN_REDRAW);		/* force complete redraw */
		window(WIN_REFR);		/* and refresh */
	}
	qio_char('\0');				/* wake up the input */
	if (exitflag) retype_cmdstr('*');	/* if not executing, prompt again and echo command string so far */
}
#endif
#endif



/* simulate a character's having been typed on the keyboard */

qio_char(c)
	char c;
{
#ifdef POSIX_TECO
	backpush = c;
#else
	ioctl(fileno(stdin), TIOCSTI, &c);				/* send char to input stream */
#endif
}
/* routine to handle "hangup" signal */
#ifndef DEBUG

SHRET hup_handler()
{
	if (!exitflag) exitflag = -3;					/* if executing, set flag to terminate */
	else
	{
		panic();									/* dump buffer and close output files */
		exit(1);
	}
}
#endif



/* type a crlf */

crlf()
{
	type_char(CR);
	type_char(LF);
}

#ifndef POSIX_TECO
/* reset ^O status */

int lflusho = LFLUSHO;
int lfo;

reset_ctlo()
{
	ioctl(fileno(stdin), TIOCLGET, &lfo);		/* read flags */
	if (lfo & LFLUSHO)							/* if ^O was set */
	{
		ioctl(fileno(stdin), TIOCLBIC, &lflusho);	/* reset ^O */
		crlf();										/* type a crlf */
	}
}
#else
reset_ctlo() 
{
}

#endif
/* routine to type one character */

type_char(c)
	char c;
{

	if ((char_count >= WN_width) && (c != CR) && !(spec_chars[c] & A_L))	/* spacing char beyond end of line */
	{
		if (et_val & ET_TRUNC) return;		/* truncate output to line width */
		else crlf();						/* otherwise do automatic new line (note recursive call to type_char) */
	}

	if ((c & 0140) == 0)				/* control char? */
	{
		switch (c & 0177)
		{
			case CR:
				putchar(c);
				char_count = 0;
				break;

			case LF:
				putchar(c);
/*				scroll_dly();			/* filler chars in case VT-100 scrolls */
				break;

			case ESC:
				if ((et_val & ET_IMAGE) && !exitflag) putchar(c);
				else
				{
					putchar('$');
					char_count++;
				}
				break;

			case TAB:
				if ((et_val & ET_IMAGE) && !exitflag) putchar(c);
				else for (type_char(' '); (char_count & tabmask) != 0; type_char(' '));
				break;

			default:
				if ((et_val & ET_IMAGE) && !exitflag) putchar(c);
				else
				{
					putchar('^');
					putchar(c + 'A'-1);
					char_count += 2;
				}
				break;
		}
	}
	else
	{
		putchar(c);
		char_count++;
	}
}
