/***************************************************************************
	objects.c - object routines

    begin                : 03 Apr 2003
    copyright            : (C) 2003 by Paul Rahme
****************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
****************************************************************************/

#include <string.h>

#include <math.h>
#include <mikmod.h>

#include "main.h"

#include "sprites.h"
#include "spritesData.h"
#include "objects.h"
#include "backgrounds.h"
#include "tuning.h"
#include "timer.h"
#include "input.h"
#include "helpers.h"
#include "audio.h"

// -------------------------------------------------------------------------------------------------
// Structures & constants
// -------------------------------------------------------------------------------------------------

// Different types of animations for objects
enum OBJ_AnimTypes
{
	OBA_Frozen,
	OBA_Once,
	OBA_OnceReverse,
	OBA_Loop,
	OBA_LoopReverse,
	OBA_PingPong,
};

// Matrix used to look up directions
const int DirMatrix[9] =
{
	DirUpLeft,		DirUp,		DirUpRight,
	DirLeft,		-1,			DirRight,
	DirDownLeft,	DirDown,	DirDownRight
};

// -------------------------------------------------------------------------------------------------
// Object definitions
// -------------------------------------------------------------------------------------------------

#include "objectDefs.inc"

// -------------------------------------------------------------------------------------------------
// Object structure
// -------------------------------------------------------------------------------------------------

typedef struct objStruct *objPtr;	// eg. "objPtr parent" is equivalent to "objStruct *parent"

typedef struct objStruct
{
	bool			active;				// Whether or not the object is active
	enum OBJ_Names	name;				// Should match a name from enum list objectNames (from objectDefs.inc)
	enum OBJ_Types	type;				// OBT_... type from OBJ_Types (in objectDefs.inc)
	SPR_Sprite		*sprite;			// Pointer to SPR_Sprite structure
	Uint32			dir;				// Sprite's direction
	enum OBJ_AnimTypes	animType;		// OBA_... animation type (from OBJ_AnimTypes above)
	Uint32			frame;				// Sprite's current frame number
	Uint32			frameRate;			// Sprite's current frame's remaining time
	Uint32			frameTime;			// When elapsedTime (timer.c) passes this, change frame
	bool			animFinished;		// True when the animation has played through
	int				state;				// What this object is currently doing
	vec2di			pos;				// current position
	vec2di			speed;				// current moving speed
	bool			backwards;			// Whether or not the sprite's animation is playing backwards
	objPtr			parent;				// Another object this one depends on
	int				energy;				// Enemy's life, shot's strength, etc
	int				timer;				// Generic timer variable for each object to use as it likes
	HLP_Wave		wave;				// Embedded sin/cos curve structure
} OBJ_Object;

// Actual object array, segmented into individual arrays for each type
OBJ_Object OBJ_InterfaceObjects[MAX_INTERFACE_OBJECTS];
OBJ_Object OBJ_Blocks[MAX_BLOCKS_X][MAX_BLOCKS_Y];
OBJ_Object OBJ_BgObjects[MAX_BG_OBJECTS];

// -------------------------------------------------------------------------------------------------
// Constants and enums for various objects
// -------------------------------------------------------------------------------------------------

// -------------------------------------------------------------------------------------------------
// Global variables
// -------------------------------------------------------------------------------------------------
OBJ_Object	*objMainCharacter;					// Pointer to the main character object
int			OBJ_HeldKeyDir;						// direction being held down, used on interface screens
vec2di		OBJ_LevelSize;						// Size of current level
vec2di		OBJ_TopLeft;						// Top left corner of the playing board

// -------------------------------------------------------------------------------------------------
// Function prototypes
// -------------------------------------------------------------------------------------------------
static void OBJ_SetSprite(OBJ_Object *_obj, Uint32 _spriteName, int _animType);
static OBJ_Object* OBJ_CreateObject(enum OBJ_Names _name, int _x, int _y, OBJ_Object *_parent);
static void OBJ_DestroyObject(OBJ_Object *_obj);
static void OBJ_DisableEnableButtons(void);

// -------------------------------- Object function prototypes -------------------------------------

// Interface Objects
static void OBJ_FnInterface(OBJ_Object *_obj);

// Blocks
static void OBJ_FnMainChar(OBJ_Object *_obj);
static void OBJ_FnBlank(OBJ_Object *_obj);
static void OBJ_FnStandard(OBJ_Object *_obj);
static void OBJ_FnStuck(OBJ_Object *_obj);
static void OBJ_FnTeleport(OBJ_Object *_obj);
static void OBJ_FnExit(OBJ_Object *_obj);

// Bg Objects
static void OBJ_FnInterfaceButton(OBJ_Object *_obj);

// -------------------------------------------------------------------------------------------------
// Object function list
// -------------------------------------------------------------------------------------------------

typedef void (*objFunction)(OBJ_Object *_obj);
const objFunction objFunctions[OBJ_NO_OF_OBJECTS] =
{
	// Interface Object/s
	OBJ_FnInterface,

	// Blocks
	OBJ_FnMainChar,
	OBJ_FnBlank,
	OBJ_FnStandard,
	OBJ_FnStuck,
	OBJ_FnTeleport,
	OBJ_FnExit,

	// Bg Objects
	OBJ_FnInterfaceButton,
	OBJ_FnInterfaceButton,
	OBJ_FnInterfaceButton,
	OBJ_FnInterfaceButton,
};

// -------------------------------------------------------------------------------------------------
// Functions
// -------------------------------------------------------------------------------------------------

// Sets the sprite pointer to the sprite of this name (name comes from enum spriteNames in sprites.h)
static void OBJ_SetSprite(OBJ_Object *_obj, Uint32 _spriteName, int _animType)
{
	_obj->sprite = &(sprites[_spriteName]);
	if (!SPR_IsSpriteLoaded(_obj->sprite))
	{
		SPR_LoadSprite(_obj->sprite);
	}
	_obj->animType = _animType;
	switch (_animType)
	{
		case OBA_Loop:
		case OBA_LoopReverse:
		{
			_obj->frame = HLP_Rnd(_obj->sprite->totalFrames-1);
			_obj->frameTime = elapsedTime + HLP_Rnd(_obj->sprite->frameRate);
			break;
		}

		case OBA_OnceReverse:
		{
			_obj->frame = _obj->sprite->totalFrames-1;
			_obj->frameTime = elapsedTime + _obj->sprite->frameRate;
			break;
		}

		default:
		{
			_obj->frame = 0;
			_obj->frameTime = elapsedTime + _obj->sprite->frameRate;
			break;
		}
	}

	_obj->backwards = false;
	if (_obj->sprite->frameRate == -1)
	{
		_obj->animFinished = true;
	} else {
		_obj->animFinished = false;
	}
}

// Sets the parent to point to the specified address
static void OBJ_SetParent(OBJ_Object *_obj, OBJ_Object *_parent)
{
	_obj->parent = _parent;
}

// Sets the object's position & layer relative to those of its parent
static inline void OBJ_GetParentPos(OBJ_Object *_obj, float _xOffset, float _yOffset)
{
	#ifdef DEBUG_CHECKS
	if (_obj->parent == NULL)
	{
		Lockup("OBJ_GetParentPos : _obj->parent == NULL.");
	}
	#endif

	_obj->pos.x = _obj->parent->pos.x + _xOffset;
	_obj->pos.y = _obj->parent->pos.y + _yOffset;
}
/*
// Sets the object's position & layer relative to those of its parent
// This functions takes into account the scrolling direction. Pass it upward facing offsets.
static void OBJ_GetParentPosDir(OBJ_Object *_obj, float _xOffset, float _yOffset)
{
	#ifdef DEBUG_CHECKS
	if (_obj->parent == NULL)
	{
		Lockup("OBJ_GetParentPosDir : _obj->parent == NULL.");
	}
	#endif

	_obj->pos.x = _obj->parent->pos.x + _xOffset;
	_obj->pos.y = _obj->parent->pos.y + _yOffset;
}
*/
// Sets up anything specific to this object
static void OBJ_SetupSpecificStuff (OBJ_Object *_obj)
{
	// Put the object into its initial state
	_obj->state = OBJ_Defaults[_obj->name].initialState;

	// Put these in order of most frequently created in the game
	switch (_obj->name)
	{
		case OBJ_MainChar:
		{
			// Main Char global pointer
			objMainCharacter = _obj;

			break;
		}

		default:
		{
			break;
		}
	}
}

// Creates an object of the specified name, position & ptr to parent object.
// Returns a pointer to the object structure.
static OBJ_Object* OBJ_CreateObject(enum OBJ_Names _name, int _x,  int _y, OBJ_Object *_parent)
{
	int objCount;
	OBJ_Object *obj;
	OBJ_DefaultsStruct *defaults;

	#ifdef DEBUG_MODE1_OBJ_C
	printf("OBJ_CreateObject() called, name = %i, x = %i, y = %i\n", _name, _x, _y);
	#endif

	defaults = &OBJ_Defaults[_name];

	// Find the first inactive object of the required type
	objCount = 0;
	switch (defaults->objType)
	{
		case OBT_Interface:
			while (OBJ_InterfaceObjects[objCount].active)
			{
				objCount++;
				if (objCount >= MAX_INTERFACE_OBJECTS)
				{
					Lockup("OBJ_CreateObject : Too many Interface objects allocated");
				}
			}
			obj = &OBJ_InterfaceObjects[objCount];
			break;

		case OBT_Block:
			obj = &OBJ_Blocks[_x][_y];
			break;

		case OBT_BgObject:
			while (OBJ_BgObjects[objCount].active)
			{
				objCount++;
				if (objCount >= MAX_BG_OBJECTS)
				{
					Lockup("OBJ_CreateObject : Too many BgObject objects allocated");
				}
			}
			obj = &OBJ_BgObjects[objCount];
			break;

		default:
			Lockup("OBJ_CreateObject: Invalid object type");
			break;

	}

	obj->active = true;
	obj->name = _name;
	obj->type = defaults->objType;
	obj->state = defaults->initialState;
	obj->timer = 0;
	obj->pos.x = _x;
	obj->pos.y = _y;
	obj->speed.x = 0;
	obj->speed.y = 0;
	obj->dir = DirUp;

	// Load the object's sprite
	OBJ_SetSprite(obj, defaults->defaultSprite, defaults->animType);

	// Set up the parent pointer
	OBJ_SetParent(obj, _parent);

	// Any settings that need to be done specifically for this type of object
	OBJ_SetupSpecificStuff(obj);

	return obj;
}

// Deactivates an object, ready to be created again with OBJ_CreateObject
static void OBJ_DestroyObject(OBJ_Object *_obj)
{
	#ifdef DEBUG_MODE1_OBJ_D
	printf("Destroying object %i, x = %f, y = %f\n", _obj->name, _obj->pos.x, _obj->pos.y);
	#endif

	_obj->active = false;
}

const enum OBJ_Names OBJ_BlockCodes[] =
{
	OBJ_Blank,			// 0
	-1,					// 1
	OBJ_MainChar,		// 2
	-1,					// 3
	-1,					// 4
	OBJ_Teleport,		// 5
	-1,					// 6
	-1,					// 7
	-1,					// 8
	OBJ_Exit,			// 9
	OBJ_Standard,		// 10
	OBJ_Stuck,			// 11
};

// Reads the specified level no
static void OBJ_ReadLevel()
{
	FILE *levelFile;
	char title[50], fullTitle[100];
	int temp, i, j, levelNo;
//	bool oneTeleportFound = 0;

	// Open the level file
	levelFile = fopen("data/level.dat","ro");

	#ifdef DEBUG_MODE1_FILE
	printf("OBJ_ReadLevel : MN_CurrentLevel = %i\n", MN_CurrentLevel);
	#endif

	// Read levels up until this one
	for (levelNo=0; levelNo < MN_CurrentLevel; levelNo++)
	{
		// Read the title of the next level
		fgets(title, 50, levelFile);

		// If it reached the end of the file, quit
		if (strncmp(title, "END", 3) == 0)
		{
			printf("\n\n\n");
			printf("*************************************\n");
			printf("All levels complete, congratulations!\n");
			printf("Try make some of your own and send\n");
			printf("them to me.\n\n");
			printf("-Mr. Bogus (paulrahme@yahoo.com)\n");
			printf("*************************************\n\n");
			SDL_Quit();
			exit(0);
		}

		sprintf(fullTitle, "KaPooka level %d : %s", MN_CurrentLevel, title);

		// Read this level's width & height
		fscanf(levelFile, "%i", &temp);
		OBJ_LevelSize.x = temp;

		#ifdef DEBUG_MODE1_FILE
		printf("\nLevel size = %i, ", temp);
		#endif

		fscanf(levelFile, "%i", &temp);
		OBJ_LevelSize.y = temp;

		#ifdef DEBUG_MODE1_FILE
		printf("%i.", temp);
		#endif

		// Make sure the teleports are by default way off the screen in case there are none on this level
	//	block1->teleport_position[0][0]=500;
	//	block1->teleport_position[0][1]=500;
	//	block1->teleport_position[1][0]=500;
	//	block1->teleport_position[1][1]=500;

		// Read the level contents
	/*
		0 = blank
		2 = start (red block)
		5 = teleport
		9 = exit
		10 = normal (green) block
		11 = unmovable (grey/cement) block
	*/

		for (j=0; j < OBJ_LevelSize.y; j++)
		{
			#ifdef DEBUG_MODE1_FILE
			printf("\n");
			#endif

			for (i=0; i<OBJ_LevelSize.x; i++)
			{
				fscanf(levelFile,"%i", &temp);

				#ifdef DEBUG_MODE1_FILE
				printf("%i ",temp);
				#endif

				OBJ_CreateObject(OBJ_BlockCodes[temp], i, j, NULL);
			}
		}

		// Read the end of the last line and the blank line after the level
		fgetc(levelFile); fgetc(levelFile);

		#ifdef DEBUG_MODE1_FILE
		printf("\n");
		#endif
	}
	fclose(levelFile);

	SDL_WM_SetCaption(fullTitle, "");
}

// Creates all the interface objects
static void OBJ_CreateInterface()
{
	OBJ_CreateObject(OBJ_InterfaceBegin, 15, 15, NULL);
	OBJ_CreateObject(OBJ_InterfaceBack, 45, 15, NULL);
	OBJ_CreateObject(OBJ_InterfaceForward, 75, 15, NULL);
	OBJ_CreateObject(OBJ_InterfaceEnd, 105, 15, NULL);

	OBJ_DisableEnableButtons();
}

// Initialises all objects and sets their active flags to false.
void OBJ_Init()
{
	int objID, objID2;

	#ifdef DEBUG_MODE_OBJ
	printf("OBJ_Init\n");
	#endif

	// Clear & reset all objects
	for (objID=0; objID < MAX_INTERFACE_OBJECTS; objID++)
	{
		OBJ_DestroyObject(&OBJ_InterfaceObjects[objID]);
	}
	for (objID=0; objID < MAX_BLOCKS_X; objID++)
	{
		for (objID2=0; objID2 < MAX_BLOCKS_Y; objID2++)
		{
			OBJ_DestroyObject(&OBJ_Blocks[objID][objID2]);
		}
	}
	for (objID=0; objID < MAX_BG_OBJECTS; objID++)
	{
		OBJ_DestroyObject(&OBJ_BgObjects[objID]);
	}

	OBJ_CreateInterface();
	OBJ_ReadLevel();

	// Calculate the top left corner for drawing the board
	OBJ_TopLeft.x = (PLAY_AREA_WIDTH - (OBJ_LevelSize.x * OBJ_WIDTH))>>1;
	OBJ_TopLeft.y = INTERFACE_HEIGHT + ((PLAY_AREA_HEIGHT - (OBJ_LevelSize.y * OBJ_HEIGHT))>>1);

	#ifdef DEBUG_MODE1_OBJ
	printf("OBJ_TopLeft = %i, %i\n", OBJ_TopLeft.x, OBJ_TopLeft.y);
	#endif

	// Initialise variables
	OBJ_HeldKeyDir = -1;

}

// Updates the object's animation counters
void OBJ_UpdateAnim(OBJ_Object *_obj)
{
	// Only update anims if the game's not paused
	if (!gamePaused)
	{
		#ifdef DEBUG_CHECKS
		if (_obj->sprite->surface == NULL)
		{
			printf("OBJ_UpdateAnim: Object name %i\n", _obj->name);
			Lockup("OBJ_UpdateAnim: Trying to update object's anim but the sprite is not yet loaded");
		}
		#endif

		// Only update the anim if it's still being played
		if ((_obj->animType != OBA_Frozen) && (!_obj->animFinished))
		{
			if (elapsedTime > _obj->frameTime)
			{
				_obj->frameTime = elapsedTime + _obj->sprite->frameRate;
				if ((_obj->animType == OBA_OnceReverse) || (_obj->animType == OBA_LoopReverse))
				{
					_obj->frame--;
					while (_obj->frame <= 0)
					{
						if (_obj->animType == OBA_OnceReverse)
						{
							_obj->frame++;
							_obj->animFinished = true;
						} else {
							_obj->frame += _obj->sprite->totalFrames;
						}
					}
				} else {
					_obj->frame++;
					while (_obj->frame >= _obj->sprite->totalFrames)
					{
						if (_obj->animType == OBA_Once)
						{
							_obj->frame--;
							_obj->animFinished = true;
						} else {
							_obj->frame -= _obj->sprite->totalFrames;
						}
					}
				}
			}

			#ifdef DEBUG_CHECKS
			if (_obj->frame < 0)
			{
				Lockup("OBJ_UpdateAnim : frame < 0");
			}
			#endif
		}
	}
}

// Updates the position of the specified layer and copy it onto the main screen
static void OBJ_DrawObject(OBJ_Object* _obj)
{
	#ifdef DEBUG_MODE3_OBJ
	printf("OBJ_DrawObject(), x=%f, y=%f\n", _obj->pos.x, _obj->pos.y);
	#endif

	// Update the sprite's animation
	OBJ_UpdateAnim(_obj);

	// Draw it
	if (_obj->type == OBT_Block)
	{
		SPR_BlitSprite(_obj->sprite, OBJ_TopLeft.x + (_obj->pos.x*OBJ_WIDTH) + (OBJ_WIDTH/2),
					OBJ_TopLeft.y + (_obj->pos.y*OBJ_HEIGHT) + (OBJ_HEIGHT/2), _obj->dir, _obj->frame);
	} else {
		SPR_BlitSprite(_obj->sprite, _obj->pos.x, _obj->pos.y, _obj->dir, _obj->frame);
	}
}

// Gets the main character's x position
int OBJ_GetMainCharX()
{
	return objMainCharacter->pos.x;
}

// Gets the main character's y position
int OBJ_GetMainCharY()
{
	return objMainCharacter->pos.y;
}

// Sets the main character's x position
void OBJ_SetMainCharX(int x)
{
	objMainCharacter->pos.x = x;
}

// Sets the main character's y position
void OBJ_SetMainCharY(int y)
{
	objMainCharacter->pos.y = y;
}

// Updates & draws all active objects
void OBJ_UpdateObjects()
{
	uint objID, objID2;
	OBJ_Object *obj;

	#ifdef DEBUG_MODE3_OBJ
	printf("OBJ_DrawObjects()\n");
	#endif

	for (objID=0; objID < MAX_INTERFACE_OBJECTS; objID++)
	{
		obj = &(OBJ_InterfaceObjects[objID]);
		if (obj->active)
		{
			(objFunctions[obj->name])(obj);
			OBJ_DrawObject(obj);
		}
	}
	for (objID2=0; objID2 < OBJ_LevelSize.y; objID2++)
	{
		for (objID=0; objID < OBJ_LevelSize.x; objID++)
		{
			obj = &(OBJ_Blocks[objID][objID2]);
			if (obj->active)
			{
				(objFunctions[obj->name])(obj);
				OBJ_DrawObject(obj);
			}
		}
	}
	for (objID=0; objID < MAX_BG_OBJECTS; objID++)
	{
		obj = &(OBJ_BgObjects[objID]);
		if (obj->active)
		{
			(objFunctions[obj->name])(obj);
			OBJ_DrawObject(obj);
		}
	}
}

#ifdef DEBUG_PAUSE
void OBJ_PrintMainCharPos()
{
	printf("Main Character's position : x = %f, y = %f.\n", objMainCharacter->pos.x, objMainCharacter->pos.y);
}
#endif

// Counts down the timer, and if it's reached 0 changes the object's state to the specified one
static inline bool OBJ_CommonDelay(OBJ_Object *_obj, int _state)
{
	if (_obj->timer < 0)
	{
		_obj->timer = 0;
		_obj->state = _state;
		return true;
	}
	else
	{
		_obj->timer -= elapsedFrameTime;
		return false;
	}
}

// -------------------------------------------------------------------------------------------------
// Object-specific functions follow
// -------------------------------------------------------------------------------------------------

// ----------------------------------- Interface ---------------------------------------

// Actual interface "object"
static void OBJ_FnInterface(OBJ_Object *_obj)
{
	switch (_obj->state)
	{
		case OBJS_InterfaceCompanyLogo:
		{
			if (IN_Up)
			{
				// Prepate for title screen
				OBJ_SetSprite(_obj, SPR_Title, OBA_Frozen);
				IN_Up = false;
				_obj->state = OBJS_InterfaceTitle;
			}
			break;
		}

		case OBJS_InterfaceTitle:
		{
			if (IN_Up)
			{
				MN_CurrentLevel = 0;
				MN_InitLevel();
				IN_Up = false;
			}
			break;
		}

		default:
		{
			break;
		}
	}
};

// ---------------------------- Blocks -----------------------------------

// Calculates & returns the 4-way sprite direction from the given speeds
static enum MN_Dirs4 OBJ_GetSpriteDir4(int speedX, int speedY)
{
	enum MN_Dirs4 dir;

	if (speedX > 0)
	{
		dir = Dir4Right;
	} else {
		if (speedX < 0)
		{
			dir = Dir4Left;
		} else {
			if (speedY > 0)
			{
				dir = Dir4Down;
			} else {
				if (speedY < 0)
				{
					dir = Dir4Up;
				#ifdef DEBUG_CHECKS
				} else {
					Lockup("OBJ_GetSpriteDir4 : Both x & y speeds are 0, invalid direction.");
				#endif
					dir = 0;
				}
			}
		}
	}

	return dir;
}
/*
// Sets the object's speed depending on its direction
static void OBJ_GetMovementFromDir4(OBJ_Object *_obj)
{
	switch (_obj->dir)
	{
		case Dir4Up:
		{
			_obj->speed.x = 0;
			_obj->speed.y = -1;
			break;
		}

		case Dir4Right:
		{
			_obj->speed.x = 1;
			_obj->speed.y = 0;
			break;
		}

		case Dir4Down:
		{
			_obj->speed.x = 0;
			_obj->speed.y = 1;
			break;
		}

		case Dir4Left:
		{
			_obj->speed.x = -1;
			_obj->speed.y = 0;
			break;
		}

		default:
		{
			#ifdef DEBUG_CHECKS
			Lockup("OBJ_GetMovementFromDir: Unhandled/invalid direction.");
			_obj->speed.x = 0;
			_obj->speed.y = 0;
			#endif
			break;
		}
	}
}
*/
// OBJ_MainChar
static void OBJ_FnMainChar(OBJ_Object *_obj)
{
	switch (_obj->state)
	{
		case OBJS_MainCharMoveIn:
		{
			if (_obj->animFinished)
			{
				OBJ_SetSprite(_obj, SPR_MainCharIdle, OBA_Frozen);
				_obj->dir = Dir4Up;
				_obj->timer = MAINCHAR_PAUSE_TIME;
				_obj->state = OBJS_MainCharPause;
			}
			break;
		}

		case OBJS_MainCharTeleportIn:
		{
			if (_obj->animFinished)
			{
				_obj->backwards = false;
				OBJ_SetSprite(_obj, SPR_MainCharInTeleport, OBA_Frozen);
				_obj->dir = Dir4Up;
				_obj->state = OBJS_MainCharInTeleport;
			}
			break;
		}

		case OBJS_MainCharPause:
		{
			if (_obj->timer < 0)
			{
				_obj->state = OBJS_MainCharInPlay;
			} else {
				_obj->timer -= elapsedFrameTime;
			}
			break;
		}

		case OBJS_MainCharInPlay:
		case OBJS_MainCharInTeleport:
		{
			OBJ_Object *newObj;
			int newX, newY;

			// Process input
			if (IN_Up)
			{
				_obj->speed.y = -1;
			}
			if (IN_Down)
			{
				_obj->speed.y = 1;
			}
			if (IN_Left)
			{
				_obj->speed.x = -1;
			}
			if (IN_Right)
			{
				_obj->speed.x = 1;
			}

			newX = HLP_Limit(_obj->pos.x + _obj->speed.x, 0, OBJ_LevelSize.x-1);
			newY = HLP_Limit(_obj->pos.y + _obj->speed.y, 0, OBJ_LevelSize.y-1);

			// If the position has changed, try to move
			if ((newX != _obj->pos.x) || (newY != _obj->pos.y))
			{
				newObj = &(OBJ_Blocks[newX][newY]);
				switch (newObj->name)
				{
					case OBJ_Blank:
					{
						OBJ_Object *newMainChar;

						#ifdef DEBUG_MODE1_OBJ
						printf("MainChar moving from %i, %i to %i, %i speed %i, %i\n",
							_obj->pos.x, _obj->pos.y, newX, newY, _obj->speed.x, _obj->speed.y);
						#endif

						// Create a new main character object in the new location, and set it to move in
						newMainChar = OBJ_CreateObject(OBJ_MainChar, newX, newY, NULL);
						OBJ_SetSprite(newMainChar, SPR_MainCharMoveIn, OBA_Once);
						newMainChar->dir = OBJ_GetSpriteDir4(_obj->speed.x, _obj->speed.y);
						newMainChar->state = OBJS_MainCharMoveIn;

						// Set the main char to move out in the correct direction
						OBJ_SetSprite(_obj, SPR_MainCharMoveOut, OBA_Once);
						_obj->dir = OBJ_GetSpriteDir4(_obj->speed.x, _obj->speed.y);
						if (_obj->state == OBJS_MainCharInTeleport)
						{
							_obj->state = OBJS_MainCharMoveOutOfTeleport;
						} else {
							_obj->state = OBJS_MainCharMoveOut;
						}
						break;
					}

					case OBJ_Standard:
					{
						OBJ_Object *newObj2;
						int newX2, newY2;

						newX2 = HLP_Limit(newX + _obj->speed.x, 0, OBJ_LevelSize.x-1);
						newY2 = HLP_Limit(newY + _obj->speed.y, 0, OBJ_LevelSize.y-1);

						#ifdef DEBUG_MODE1_OBJ
						printf("MainChar at %i, %i trying to move block at %i, %i to %i, %i\n",
								_obj->pos.x, _obj->pos.y, newX, newY, newX2, newY2);
						#endif

						newObj2 = &(OBJ_Blocks[newX2][newY2]);
						if ((newObj2->active) && (newObj2->name == OBJ_Blank))
						{
							OBJ_Object *newStandard;

							// Set the new standard block to move in
							newStandard = OBJ_CreateObject(OBJ_Standard, newX2, newY2, NULL);
							OBJ_SetSprite(newStandard, SPR_StandardMoveIn, OBA_Once);
							newStandard->speed.x = _obj->speed.x;
							newStandard->speed.y = _obj->speed.y;
							newStandard->dir = OBJ_GetSpriteDir4(newStandard->speed.x, newStandard->speed.y);
							newStandard->state = OBJS_StandardMoveIn;

							// Set the existing standard block to move out
							OBJ_SetSprite(newObj, SPR_StandardMoveOut, OBA_Once);
							newObj->dir = newStandard->dir;
							newObj->state = OBJS_StandardMoveOut;
							if (_obj->state == OBJS_MainCharInTeleport)
							{
								_obj->state = OBJS_MainCharInTeleportWaitingForBlank;
							} else {
								_obj->state = OBJS_MainCharWaitingForBlank;
							}
						} else {
							// Set its speed back to 0
							_obj->speed.x = 0;
							_obj->speed.y = 0;
						}
						break;
					}

					case OBJ_Stuck:
					{
						// Do nothing, just set speed back to 0
						_obj->speed.x = 0;
						_obj->speed.y = 0;
						break;
					}

					case OBJ_Teleport:
					{
						bool found;
						OBJ_Object *newMainChar;

						// Create a new main character object in the new location, and set it to teleport
						newMainChar = OBJ_CreateObject(OBJ_MainChar, newX, newY, NULL);
						OBJ_SetSprite(newMainChar, SPR_MainCharTeleport, OBA_Once);
						newMainChar->dir = Dir4Up;
						newMainChar->state = OBJS_MainCharTeleportOut;

						// Find the other teleport and create a new main char there, teleporting in
						// (Keeping x & y order the same as old kpooka for "dummy teleporter" compatibility)
						newX = 0;
						newY = 0;
						found = false;
						while ((!found) && (newX < OBJ_LevelSize.x))
						{
							while ((!found) && (newY < OBJ_LevelSize.y))
							{
								if (((newX != _obj->pos.x) || (newY != _obj->pos.y)) &&
									(OBJ_Blocks[newX][newY].name == OBJ_Teleport))
								{
									found = true;
								} else {
									newY++;
								}
							}

							if (!found)
							{
								newY = 0;
								newX++;
							}
						}

						#ifdef DEBUG_CHECKS
						if (!found)
						{
							Lockup("OBJ_FnMainChar: other teleport not found.");
						}
						#endif

						newMainChar = OBJ_CreateObject(OBJ_MainChar, newX, newY, NULL);
						OBJ_SetSprite(newMainChar, SPR_MainCharTeleport, OBA_Once);
						newMainChar->backwards = true;
						newMainChar->dir = Dir4Up;
						newMainChar->state = OBJS_MainCharTeleportIn;

						// Set the main char to move out in the correct direction
						OBJ_SetSprite(_obj, SPR_MainCharMoveOut, OBA_Once);
						_obj->dir = OBJ_GetSpriteDir4(_obj->speed.x, _obj->speed.y);
						_obj->state = OBJS_MainCharMoveOut;

						break;
					}

					case OBJ_Exit:
					{
						OBJ_Object *newMainChar;

						newMainChar = OBJ_CreateObject(OBJ_MainChar, newX, newY, NULL);
						OBJ_CreateObject(OBJ_Blank, _obj->pos.x, _obj->pos.y, NULL);

						OBJ_SetSprite(newMainChar, SPR_MainCharExit, OBA_Once);
						newMainChar->state = OBJS_MainCharExiting;

						// Set its speed back to 0
						_obj->speed.x = 0;
						_obj->speed.y = 0;

						break;
					}

					default:
					{
						break;
					}
				}
			}

			break;
		}

		case OBJS_MainCharInTeleportWaitingForBlank:
		case OBJS_MainCharWaitingForBlank:
		{
			OBJ_Object *newObj;
			int newX, newY;

			newX = _obj->pos.x + _obj->speed.x;
			newY = _obj->pos.y + _obj->speed.y;
			newObj = &(OBJ_Blocks[newX][newY]);

			// Wait until the new position is blank before moving there
			if (newObj->name == OBJ_Blank)
			{
				OBJ_Object *newMainChar;

				#ifdef DEBUG_MODE1_OBJ
				printf("MainChar moving from %i, %i to %i, %i speed %i, %i\n",
					_obj->pos.x, _obj->pos.y, newX, newY, _obj->speed.x, _obj->speed.y);
				#endif

				// Create a new main character object in the new location, and set it to move in
				newMainChar = OBJ_CreateObject(OBJ_MainChar, newX, newY, NULL);
				OBJ_SetSprite(newMainChar, SPR_MainCharMoveIn, OBA_Once);
				newMainChar->dir = OBJ_GetSpriteDir4(_obj->speed.x, _obj->speed.y);
				newMainChar->state = OBJS_MainCharMoveIn;

				// Set the main char to move out in the correct direction
				OBJ_SetSprite(_obj, SPR_MainCharMoveOut, OBA_Once);
				_obj->dir = OBJ_GetSpriteDir4(_obj->speed.x, _obj->speed.y);
				if (_obj->state == OBJS_MainCharInTeleportWaitingForBlank)
				{
					_obj->state = OBJS_MainCharMoveOutOfTeleport;
				} else {
					_obj->state = OBJS_MainCharMoveOut;
				}
			}

			break;
		}

		case OBJS_MainCharMoveOut:
		{
			// If the move out anim is complete, create a blank where the object was
			if (_obj->animFinished)
			{
				OBJ_CreateObject(OBJ_Blank, _obj->pos.x, _obj->pos.y, NULL);
			}
			break;
		}

		case OBJS_MainCharTeleportOut:
		case OBJS_MainCharMoveOutOfTeleport:
		{
			// If the move out anim is complete, create a teleport where the object was
			if (_obj->animFinished)
			{
				OBJ_CreateObject(OBJ_Teleport, _obj->pos.x, _obj->pos.y, NULL);
			}
			break;
		}

		case OBJS_MainCharExiting:
		{
			if (_obj->animFinished)
			{
				MN_CurrentLevel++;
				if (MN_CurrentLevel > MN_HighestLevel)
				{
					MN_HighestLevel = MN_CurrentLevel;
				}
				MN_Reset();
			}
			break;
		}

		default:
		{
			#ifdef DEBUG_CHECKS
			Lockup("OBJ_FnMainChar: Unhandled/invalid state.");
			#endif

			break;
		}
	}
}

// OBJ_Blank
static void OBJ_FnBlank(OBJ_Object *_obj)
{
}

// OBJ_Standard
static void OBJ_FnStandard(OBJ_Object *_obj)
{
	switch (_obj->state)
	{
		case OBJS_StandardIdle:
		{
			break;
		}

		case OBJS_StandardMoveIn:
		{
			if (_obj->animFinished)
			{
				// Pause before trying to move again
				OBJ_SetSprite(_obj, SPR_StandardIdle, OBA_Once);
				_obj->dir = Dir4Up;
				_obj->timer = STANDARD_PAUSE_TIME;
				_obj->state = OBJS_StandardPause;
			}
			break;
		}

		case OBJS_StandardPause:
		{
			if (_obj->timer < 0)
			{
				OBJ_Object *newObj;
				int newX, newY;

				// Find new position to try move to
				newX = HLP_Limit(_obj->pos.x + _obj->speed.x, 0, OBJ_LevelSize.x-1);
				newY = HLP_Limit(_obj->pos.y + _obj->speed.y, 0, OBJ_LevelSize.y-1);
				newObj = &(OBJ_Blocks[newX][newY]);

				#ifdef DEBUG_MODE1_OBJ
				printf("Standard block trying to move from %i, %i to %i, %i speed %i, %i\n",
					_obj->pos.x, _obj->pos.y, newX, newY, _obj->speed.x, _obj->speed.y);
				#endif

				// If it's valid & blank, move there; else stop moving
				if ((newObj->active) && (newObj->name == OBJ_Blank))
				{
					OBJ_Object *newStandard;

					// Set the new standard block to move in
					newStandard = OBJ_CreateObject(OBJ_Standard, newX, newY, NULL);
					OBJ_SetSprite(newStandard, SPR_StandardMoveIn, OBA_Once);
					newStandard->speed.x = _obj->speed.x;
					newStandard->speed.y = _obj->speed.y;
					newStandard->dir = OBJ_GetSpriteDir4(newStandard->speed.x, newStandard->speed.y);
					newStandard->state = OBJS_StandardMoveIn;

					OBJ_SetSprite(_obj, SPR_StandardMoveOut, OBA_Once);
					_obj->dir = OBJ_GetSpriteDir4(_obj->speed.x, _obj->speed.y);
					_obj->state = OBJS_StandardMoveOut;
				} else {
					OBJ_SetSprite(_obj, SPR_StandardIdle, OBA_Frozen);
					_obj->dir = Dir4Up;
					_obj->speed.x = 0;
					_obj->speed.y = 0;
					_obj->state = OBJS_StandardIdle;
				}
			} else {
				_obj->timer -= elapsedFrameTime;
			}

			break;
		}

		case OBJS_StandardMoveOut:
		{
			// If the move out anim is complete, create a blank where the object was
			if (_obj->animFinished)
			{
				OBJ_CreateObject(OBJ_Blank, _obj->pos.x, _obj->pos.y, NULL);
			}
			break;
		}

		default:
		{
			#ifdef DEBUG_CHECKS
			Lockup("OBJ_FnStandard: Unhandled/invalid state.");
			#endif

			break;
		}
	}
}

// OBJ_Stuck
static void OBJ_FnStuck(OBJ_Object *_obj)
{
}

// OBJ_Teleport
static void OBJ_FnTeleport(OBJ_Object *_obj)
{
}

// OBJ_Exit
static void OBJ_FnExit(OBJ_Object *_obj)
{
}

// ---------------------------- Bg Objects -----------------------------------

// Returns true if the position pointed to by pos is inside the object, taking into account obj's width/height
static bool OBJ_IsPositionInObj(OBJ_Object *_obj, vec2di *pos)
{
	return ((pos->x > _obj->pos.x - (_obj->sprite->width>>1)) &&
			(pos->x < _obj->pos.x + (_obj->sprite->width>>1)) &&
			(pos->y > _obj->pos.y - (_obj->sprite->height>>1)) &&
			(pos->y < _obj->pos.y + (_obj->sprite->height>>1)));
}

static OBJ_Object *OBJ_FindButton(enum OBJ_Names _name)
{
	int objCount;

	objCount = 0;
	while (OBJ_BgObjects[objCount].name != _name)
	{
		objCount++;

		#ifdef DEBUG_CHECKS
		if (objCount >= MAX_BG_OBJECTS)
		{
			printf("Button type %i not found in BG_Objects! --- ", _name);
			Lockup("OBJ_FindButton : Could not find button");
		}
		#endif
	}
	return &OBJ_BgObjects[objCount];
}

// Disable / re-enable buttons depending on which level the player's currently on
static void OBJ_DisableEnableButtons()
{
	OBJ_Object *obj;

	if (MN_CurrentLevel == 1)
	{
		obj = OBJ_FindButton(OBJ_InterfaceBack);
		OBJ_SetSprite(obj, SPR_InterfaceBackDisabled, OBA_Frozen);
		obj->state = OBJS_InterfaceButtonDisabled;

		obj = OBJ_FindButton(OBJ_InterfaceBegin);
		OBJ_SetSprite(obj, SPR_InterfaceBeginDisabled, OBA_Frozen);
		obj->state = OBJS_InterfaceButtonDisabled;
	}

	if (MN_CurrentLevel == MN_HighestLevel)
	{
		obj = OBJ_FindButton(OBJ_InterfaceForward);
		OBJ_SetSprite(obj, SPR_InterfaceForwardDisabled, OBA_Frozen);
		obj->state = OBJS_InterfaceButtonDisabled;

		obj = OBJ_FindButton(OBJ_InterfaceEnd);
		OBJ_SetSprite(obj, SPR_InterfaceEndDisabled, OBA_Frozen);
		obj->state = OBJS_InterfaceButtonDisabled;
	}
}

static enum SPR_Names OBJ_GetInterfaceIdleSprite(OBJ_Object *_obj)
{
	enum SPR_Names ret_val;

	switch (_obj->name)
	{
		case OBJ_InterfaceBack:
		{
			ret_val = SPR_InterfaceBackIdle;
			break;
		}

		case OBJ_InterfaceForward:
		{
			ret_val = SPR_InterfaceForwardIdle;
			break;
		}

		case OBJ_InterfaceBegin:
		{
			ret_val = SPR_InterfaceBeginIdle;
			break;
		}

		case OBJ_InterfaceEnd:
		{
			ret_val = SPR_InterfaceEndIdle;
			break;
		}

		default:
		{
			Lockup("OBJ_GetInterfaceIdleSprite : Unhandled/invalid object name");
			ret_val = -1;
			break;
		}
	}

	return ret_val;
}

// Interface buttons
static void OBJ_FnInterfaceButton(OBJ_Object *_obj)
{
	switch (_obj->state)
	{
		case OBJS_InterfaceButtonIdle:
		{
			// Check if the mouse was clicked while over this button
			if (OBJ_IsPositionInObj(_obj, &IN_MousePos))
			{
				if (IN_MouseButtonState == IN_MB_CLICKED)
				{
					#ifdef DEBUG_MODE1_OBJ
					printf("OBJ_FnInterfaceButton : Mouse pressed over button type %i\n", _obj->name);
					#endif

					_obj->state = OBJS_InterfaceButtonPressed;
				} else {
					if (IN_MouseButtonState == IN_MB_UP)
					{
						OBJ_SetSprite(_obj, OBJ_GetInterfaceIdleSprite(_obj) + 2, OBA_Frozen);
					}
				}
			} else {
				OBJ_SetSprite(_obj, OBJ_GetInterfaceIdleSprite(_obj) + 0, OBA_Frozen);
			}
			break;
		}

		case OBJS_InterfaceButtonPressed:
		{
			// Check if the mouse button was released, and whether it's still over this button or not
			if (IN_MouseButtonState == IN_MB_UP)
			{
				OBJ_SetSprite(_obj, OBJ_GetInterfaceIdleSprite(_obj) + 0, OBA_Frozen);
				if (OBJ_IsPositionInObj(_obj, &IN_MousePos))
				{
					#ifdef DEBUG_MODE1_OBJ
					printf("OBJ_FnInterfaceButton : Mouse depressed over button type %i, performing action\n", _obj->name);
					#endif

					_obj->state = OBJS_InterfaceButtonDepressed;
				} else {
					_obj->state = OBJS_InterfaceButtonIdle;
				}
			} else {
				if (OBJ_IsPositionInObj(_obj, &IN_MousePos))
				{
					OBJ_SetSprite(_obj, OBJ_GetInterfaceIdleSprite(_obj) + 3, OBA_Frozen);
				} else {
					OBJ_SetSprite(_obj, OBJ_GetInterfaceIdleSprite(_obj) + 0, OBA_Frozen);
				}
			}
			break;
		}

		case OBJS_InterfaceButtonDepressed:
		{
			// Button properly clicked, perform its action
			switch (_obj->name)
			{
				case OBJ_InterfaceBack:
				{
					if (MN_CurrentLevel > 1)
					{
						MN_CurrentLevel--;
						MN_Reset();
					}
					break;
				}

				case OBJ_InterfaceForward:
				{
					if (MN_CurrentLevel < MN_HighestLevel)
					{
						MN_CurrentLevel++;
						MN_Reset();
					}
					break;
				}

				case OBJ_InterfaceBegin:
				{
					if (MN_CurrentLevel > 1)
					{
						MN_CurrentLevel = 1;
						MN_Reset();
					}
					break;
				}

				case OBJ_InterfaceEnd:
				{
					if (MN_CurrentLevel < MN_HighestLevel)
					{
						MN_CurrentLevel = MN_HighestLevel;
						MN_Reset();
					}
					break;
				}

				default:
				{
					#ifdef DEBUG_CHECKS
					Lockup("OBJ_FnInterfaceButton: Unhandled/invalid object");
					#endif

					break;
				}
			}

			OBJ_DisableEnableButtons();

			break;
		}

		case OBJS_InterfaceButtonDisabled:
		{
			break;
		}

		default:
		{
			#ifdef DEBUG_CHECKS
			Lockup("OBJ_FnInterfaceButton: Unhandled/invalid state.");
			#endif

			break;
		}
	}
}
