/***************************************************************************
	helpers.h - macros and other small routines - header

    begin                : 8 May 2003
    copyright            : (C) 2003 by Paul Rahme
****************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
****************************************************************************/

// -------------------------------------------------------------------------------------------------
// Consts and structures
// -------------------------------------------------------------------------------------------------

#define u8 unsigned char
#define s8 signed char

typedef struct
{
	float	fAngle;			// current angle value
	float	speed;			// current angle's speed (gets added to fAngle each frame)
	u8		angle;			// 8-bit integer part of fAngle is put into here
	float	centre;			// centre value around which the wave moves
	float	size;			// radius/amplitude/etc that gets multiplied by the sin/cos function
	float	value;			// current value (centre + (size*sin(angle))
} HLP_Wave;

// -------------------------------------------------------------------------------------------------
// Variables
// -------------------------------------------------------------------------------------------------

extern float HLP_SinCosTable[256+64]; // used for storing the sin & cos lookup values

// -------------------------------------------------------------------------------------------------
// Macros
// -------------------------------------------------------------------------------------------------

extern float HLP_SinCosTable[256+64]; // used for storing the sin & cos lookup values

// Calculates how many bytes in the given pixels
#define HLP_BPP(_a) \
	(_a)*BPP

// Given a surface, and x, y coordinates this macro returns the corresponding base address
#define HLP_GetPos(_surface, _x, _y) \
	(Uint32)(_surface->pixels) + HLP_BPP((_y * _surface->w) + _x)

// Given a certain speed, calculates movement since the last frame
#define HLP_FrameSpeed(_speed) \
	(_speed * elapsedFrameTime)

// Returns the sine of angle, where angle is a char (0 = 0 deg, 64 = 90 deg, 128 = 180 deg, 192 = 270 deg).
#define HLP_Sin(_angle) \
	(HLP_SinCosTable[_angle])

// Returns the cosine of angle, where angle is a char (0 = 0 deg, 64 = 90 deg, 128 = 180 deg, 192 = 270 deg).
#define HLP_Cos(_angle) \
	(HLP_SinCosTable[_angle+64])

// Returns true if pos1 and pos2 are within range (rangeX, rangeY) from each other.
#define HLP_CheckPosRange(pos1, pos2, rangeX, rangeY) \
	((fabsf(pos2.x - pos1.x) < rangeX) && (fabsf(pos2.y - pos1.y) < rangeY))

// Returns a random number between 0 and x (inclusive)
#define HLP_Rnd(x) \
	(float)(rand())/RAND_MAX*(float)(x)

// -------------------------------------------------------------------------------------------------
// Functions
// -------------------------------------------------------------------------------------------------

// Generates sin/cos table.
void HLP_Init(void);

// Sets the angle and speed in a wave structure (_speed should be in units per second, ie. 256 = 1 hz)
void HLP_SetWave(HLP_Wave *_wave, float _centre, float _size, u8 _angle, float _speed);

// Updates a sine wave structure and calculates its new value - call once per frame
void HLP_UpdateSinWave(HLP_Wave *_wave);

// Returns the value wrapped inside the range [min, max] inclusive
int HLP_Wrap(int x, int min, int max);

// Returns the value confined to the range [min, max] inclusive
int HLP_Limit(int x, int min, int max);
