/***************************************************************************
	helpers.c - macros and other small routines

    begin                : 8 May 2003
    copyright            : (C) 2003 by Paul Rahme
****************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
****************************************************************************/

#include <math.h>

#include "main.h"
#include "helpers.h"
#include "timer.h"

// -------------------------------------------------------------------------------------------------
// Variables
// -------------------------------------------------------------------------------------------------

float HLP_SinCosTable[256+64]; // used for storing the sin & cos lookup values

// -------------------------------------------------------------------------------------------------
// Functions
// -------------------------------------------------------------------------------------------------

// Generates sin/cos table.
void HLP_Init()
{
	int angle;

	for (angle=0; angle<256+64; angle++)
	{
		HLP_SinCosTable[angle] = sin((float)(angle)/128*PI); // 128 (180 degrees) is PI/2 radians
	}
}

// Sets the angle and speed in a wave structure (_speed should be in units per second, ie. 256 = 1 hz)
void HLP_SetWave(HLP_Wave *_wave, float _centre, float _size, u8 _angle, float _speed)
{
	_wave->fAngle = _angle;
	_wave->angle = _angle;
	_wave->speed = _speed/1000;
	_wave->centre = _centre;
	_wave->size = _size;
	_wave->value = _centre;
}

// Updates a wave structure - call once per frame
void HLP_UpdateSinWave(HLP_Wave *_wave)
{
	_wave->fAngle += HLP_FrameSpeed(_wave->speed);
	_wave->angle = (u8)(_wave->fAngle);
	_wave->value = _wave->centre + (_wave->size * HLP_Sin(_wave->angle));
}

// Returns the value wrapped inside the range [min, max] inclusive
int HLP_Wrap(int x, int min, int max)
{
	#ifdef DEBUG_CHECKS
	if (min >= max)
	{
		Lockup("HLP_Wrap: min >= max!");
	}
	#endif

	int diff = max-min;

	while (x < min)
	{
		x += diff;
	}

	while (x > max)
	{
		x -= diff;
	}

	return x;
}

// Returns the value confined to the range [min, max] inclusive
int HLP_Limit(int x, int min, int max)
{
	#ifdef DEBUG_CHECKS
	if (min >= max)
	{
		Lockup("HLP_Limit: min >= max!");
	}
	#endif

	if (x < min)
	{
		x = min;
	}

	if (x > max)
	{
		x = max;
	}

	return x;
}
