/***************************************************************************
	backgrounds.c - background layer routines

    begin                : 22 Feb 2003
    copyright            : (C) 2003 by Paul Rahme
****************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
****************************************************************************/

#include <string.h>
#include "SDL/SDL_image.h"

#include "main.h"
#include "backgrounds.h"
#include "backgroundsData.h"
#include "helpers.h"
#include "timer.h"
#include "objects.h"
#include "tuning.h"
#include "sprites.h"

// -------------------------------------------------------------------------------------------------
// Constants
// -------------------------------------------------------------------------------------------------

// -------------------------------------------------------------------------------------------------
// Variables
// -------------------------------------------------------------------------------------------------

BG_Layer	BG_Layers[BG_MAX_LAYERS];	// Array of all the background layer structures
BG_Layer	*BG_MainLayer;			// Pointer to the main layer

uchar		BG_MainLayerNo;			// Which layer is currently the main layer
float		BG_MainSpeedX;			// Horizontal speed of the main bg layer
float		BG_MainSpeedY;			// Vertical speed of the main bg layer
float		BG_MainFrameSpeedX;		// Horizontal speed of the main layer for this frame
float		BG_MainFrameSpeedY;		// Vertical speed of the main layer for this frame
float		BG_MainX, BG_MainY;		// main layer's top left corner's current coordinates
enum MN_Dirs4 BG_ScrollingDir;		// Direction the bg is scrolling (Dir4... from enum MN_Dirs4)

// -------------------------------------------------------------------------------------------------
// Functions
// -------------------------------------------------------------------------------------------------

// Loads a bitmap from the speciified filename and converts it to SDL-optimized format.
static SDL_Surface* BG_LoadConvert(const char *_fileName, Uint8 _alpha)
{
	SDL_Surface *tempSurface, *finalSurface;
	Uint32 colorKey;

	// Load the original image into the temp surface, and get colorkey value
	tempSurface = IMG_Load(_fileName);
	colorKey = tempSurface->format->colorkey;

	// Convert it into SDL-optimized format on the final surface, and free up the temp one
	finalSurface = SDL_DisplayFormat(tempSurface);
	SDL_FreeSurface(tempSurface);

	// Set up the correct alpha value on the final surface
	SDL_SetColorKey(finalSurface, SDL_SRCCOLORKEY, colorKey);

	// Prepare the per-surface alpha value on the final surface
	if (SDL_SetAlpha(finalSurface, SDL_SRCALPHA, _alpha) == -1)
	{
		Lockup("Failed to set alpha.");
	}

	return finalSurface;
}

// Loads the bitmap from the specified filename into the specified layer.
static void BG_CreateLayer(BG_Layer* _layer, const char *_fileName, Uint8 _alpha,
							int _startX, int _startY, float _relSpeedX, float _relSpeedY)
{
	#ifdef DEBUG_MODE1
	printf("BG_CreateLayer called with _fileName=%s, _alpha = %i\n", _fileName, _alpha);
	printf("_startX = %i, _startY=%i, _relSpeedX=%f, _relSpeedY=%f\n",
			_startX, _startY, _relSpeedX, _relSpeedY);
	#endif

	// Load the layer's data from the fileBG_UpdateLayer(%i)
	_layer->surface = BG_LoadConvert(_fileName, _alpha);

	// Set all the values of this layer
	_layer->x = _startX;
	_layer->y = _startY;
	_layer->relSpeedX = _relSpeedX;
	_layer->relSpeedY = _relSpeedY;
	_layer->wrapX = _layer->surface->w - SCREEN_WIDTH;
	_layer->wrapY = _layer->surface->h - SCREEN_HEIGHT;
}

// Sets all the layers to the correct position relative to the main layer
void BG_SetAllLayersPos()
{
	int layerNo;
	BG_Layer *layer;

	for (layerNo=0; layerNo < levelTotalLayers[MN_CurrentLevel]; layerNo++)
	{
		layer = &(BG_Layers[layerNo]);
		layer->x = BG_MainX * layer->relSpeedX;
		layer->y = BG_MainY * layer->relSpeedY;
	}
}

// Resets the main layer's position & speed to the start of the level
static void BG_ReloadInitialPos()
{
	BG_MainLayer->x = mainLayers[MN_CurrentLevel].startX;
	BG_MainLayer->y = mainLayers[MN_CurrentLevel].startY;
	BG_MainX = BG_MainLayer->x;
	BG_MainY = BG_MainLayer->y;

	BG_MainSpeedX = mainLayers[MN_CurrentLevel].speedX;
	BG_MainSpeedY = mainLayers[MN_CurrentLevel].speedY;
}

// (Re)sets the bg layers' initial position and sets the main extern globals.
void BG_Reset()
{
	BG_ReloadInitialPos();
	BG_SetAllLayersPos();

	// Set the scrolling direction from the speeds
	if (BG_MainSpeedX > 0)
	{
		BG_ScrollingDir = Dir4Right;
	} else {
		if (BG_MainSpeedX < 0)
		{
			BG_ScrollingDir = Dir4Left;
		} else {
			if (BG_MainSpeedY > 0)
			{
				BG_ScrollingDir = Dir4Down;
			} else {
				#ifdef DEBUG_MODE1_BG
				if (BG_MainSpeedY == 0)
				{
					printf("BG_Reset : Warning, no direction set for this level (both x and y = 0).\n");
				}
				#endif
				BG_ScrollingDir = Dir4Up;
			}
		}
	}
}

// Goes through all layers and frees the surfaces from memory
void BG_FreeSurfaces()
{
	int layerNo;

	#ifdef DEBUG_MODE1_BG
	printf("BG_Free()\n");
	#endif

	for (layerNo=0; layerNo < BG_MAX_LAYERS; layerNo++)
	{
		BG_Layers[layerNo].visible = false;
		if (BG_Layers[layerNo].surface != NULL)
		{
			#ifdef DEBUG_MODE1_BG
			printf("Layer %i : Freeing up surface.\n", layerNo);
			#endif
			SDL_FreeSurface(BG_Layers[layerNo].surface);
			BG_Layers[layerNo].surface = NULL;
		}
	}
}

// Initialiases the background module, allocates & loads bg layers.
void BG_Init()
{
	int layerNo;
	layerStruct* currLayerArray;
	char currLayerFileName[256];

	#ifdef DEBUG_MODE1_BG
	printf("BG_Init\n");
	#endif

	// Initialise and disable all the layers
	for (layerNo=0; layerNo < BG_MAX_LAYERS; layerNo++)
	{
		BG_Layers[layerNo].visible = false;
		if (BG_Layers[layerNo].surface != NULL)
		{
			printf("Layer %i: ", layerNo);
			Lockup("Surface for layer is not null. Possible memory leak.");
		}
	}

	#ifdef DEBUG_MODE1
	printf("levelTotalLayers[%i] = %i\n", MN_CurrentLevel, levelTotalLayers[MN_CurrentLevel]);
	#endif

	// Read in and enable the layers for this level
	for (layerNo=0; layerNo < levelTotalLayers[MN_CurrentLevel]; layerNo++)
	{
		#ifdef DEBUG_MODE1
		printf("BG_Init: Setting up layer no %i\n", layerNo);
		#endif

		// Find this layer's array
		currLayerArray = (layerStruct*)(&(levelLayerList[MN_CurrentLevel])[layerNo]);

		// Read the filename of this layer
		strcpy(currLayerFileName, BASE_DIR);		// copy it into a long string so that the filename can be appended
		strcat(currLayerFileName, currLayerArray->fileName);

		#ifdef DEBUG_MODE1
		printf("currLayerFileName = %s\n", currLayerFileName);
		#endif

		BG_CreateLayer(&BG_Layers[layerNo], currLayerFileName,
						currLayerArray->alpha,
						mainLayers[MN_CurrentLevel].startX * currLayerArray->relSpeedX,
						mainLayers[MN_CurrentLevel].startY * currLayerArray->relSpeedY,
						currLayerArray->relSpeedX, currLayerArray->relSpeedY);

		if (BG_Layers[layerNo].surface == NULL)
		{
			Lockup("Could not load layer from file.\n");
		}
		BG_Layers[layerNo].visible = currLayerArray->visible;
	}

	// Any variables that need setting after all the layers have loaded
	BG_MainLayerNo = mainLayers[MN_CurrentLevel].layerNo;
	BG_MainLayer = &BG_Layers[BG_MainLayerNo];
	BG_MainLayer->x = mainLayers[MN_CurrentLevel].startX;
	BG_MainX = BG_MainLayer->x;
	BG_MainLayer->y = mainLayers[MN_CurrentLevel].startY;
	BG_MainY = BG_MainLayer->y;
	BG_MainSpeedX = mainLayers[MN_CurrentLevel].speedX;
	BG_MainSpeedY = mainLayers[MN_CurrentLevel].speedY;
}

// Updates and draws all layers at once
void BG_UpdateAllLayers()
{
	uint layerNo, lastLayer;
	BG_Layer *layer;
	int srcCount[BG_MAX_LAYERS], destCount, lineCount, skipPixels[BG_MAX_LAYERS], srcBase[BG_MAX_LAYERS], destBase;
	Uint32 *destPos;
	Uint32 pixel;
	SDL_Surface *src[BG_MAX_LAYERS];
	bool visible[BG_MAX_LAYERS];

	#ifdef DEBUG_MODE2_BG
	printf("BG_UpdateAllLayers()\n");
	#endif

	// ----- STAGE 1 : Update layer positions -----

	for (layerNo = 0; layerNo < levelTotalLayers[MN_CurrentLevel]; layerNo++)
	{
		layer = &(BG_Layers[layerNo]);

		// Set this layer's position relative to the main layer
		layer->x = (BG_MainX * layer->relSpeedX);
		layer->y = (BG_MainY * layer->relSpeedY);

		// Check if the layer needs to wrap around
		if (layer->x < 0)
		{
			layer->x += layer->wrapX;
		}
		if (layer->x > layer->wrapX)
		{
			layer->x -= layer->wrapX;
		}
		if (layer->y < 0)
		{
			layer->y += layer->wrapY;
		}
		if (layer->y > layer->wrapY)
		{
			layer->y -= layer->wrapY;
		}

		// Update the integer positions from the float values
		layer->pos.x = (Sint16)(layer->x);
		layer->pos.y = (Sint16)(layer->y);

		#ifdef DEBUG_MODE3_BG
		printf("Layer %i's position is now (%f, %f)\n", layerNo, layer->x, layer->y);
		#endif

		// If it's the main layer, update the global coordinates
		if (layer == BG_MainLayer)
		{
			// Update the main layer's coordinates
			BG_MainX = layer->pos.x;
			BG_MainY = layer->pos.y;

			#ifdef DEBUG_MODE2_BG
			printf("Main layer's position is now (%f, %f)\n", BG_MainX, BG_MainY);
			#endif
		}
	}

	// ----- STAGE 2 : Draw layers -----

	// Lock & prepare src surfaces
	for (layerNo = 0; layerNo < levelTotalLayers[MN_CurrentLevel]; layerNo++)
	{
		layer = &(BG_Layers[layerNo]);
		src[layerNo] = layer->surface;
		if (SDL_MUSTLOCK(src[layerNo]))
		{
			SDL_LockSurface(src[layerNo]);
		}

		// Calculate how many pixels to skip at the end of each line
		skipPixels[layerNo] = HLP_BPP(src[layerNo]->w - SCREEN_WIDTH);

		// set initial counter values
		srcCount[layerNo] = 0;

		// Calculate starting positions
		srcBase[layerNo] = HLP_GetPos(src[layerNo], layer->pos.x, layer->pos.y);

		// Cache visible flags
		visible[layerNo] = layer->visible;
	}

	// Lock & prepare vScreen
	if (SDL_MUSTLOCK(vScreen))
	{
		SDL_LockSurface(vScreen);
	}
	destCount = 0;

	lineCount = SCREEN_MAX_X;

	destBase = (Uint32)(vScreen->pixels);

	lastLayer = levelTotalLayers[MN_CurrentLevel]-1;

	// Actual blit loop
	while (destCount < (HLP_BPP(SCREEN_WIDTH*SCREEN_HEIGHT)))
	{
		// Update dest position
		destPos = (Uint32*)(destBase + destCount);

		// Progress down the layers finding the first non-transparent pixel in this position
		pixel = 0;
		layerNo = lastLayer;
		while (layerNo != -1)
		{
			if (pixel == 0)
			{
				if (visible[layerNo])
				{
					pixel = *(Uint32*)(srcBase[layerNo] + srcCount[layerNo]);
				}
			}

			// Increment this layer's position
			srcCount[layerNo] += BPP;

			layerNo--;
		}

		// Update the dest surface and increment the dest position counter
		pixel |= 0xFF000000;
		*(Uint32*)(destPos) = pixel;
		destCount += BPP;

		// Check if it's reached the end of the screen, if so wrap around
		if (lineCount != 0)
		{
			lineCount--;
		} else {
			for (layerNo = 0; layerNo <= lastLayer; layerNo++)
			{
				srcCount[layerNo] += skipPixels[layerNo];
			}
			lineCount = SCREEN_MAX_X;
		}
	}

	// Unlock surfaces if necessary
	if (SDL_MUSTLOCK(vScreen))
	{
		SDL_UnlockSurface(vScreen);
	}
	for (layerNo = 0; layerNo <= lastLayer; layerNo++)
	{
		if (SDL_MUSTLOCK(src[layerNo]))
		{
			SDL_UnlockSurface(src[layerNo]);
		}
	}

	// For some reason it still works without this, even though it shouldn't...
//	SDL_UpdateRect(vScreen, 0, 0, vScreen->w, vScreen->h);
}

// Blits vScreen to screen, similar to SDL_BlitSurface but with lots of hardcoded bodges for more speed
void BG_BlitScreen()
{
	int count, lineCount, srcBase, destBase;

	// Lock surfaces if necessary
	if (SDL_MUSTLOCK(vScreen))
	{
		SDL_LockSurface(vScreen);
	}
	if (SDL_MUSTLOCK(screen))
	{
		SDL_LockSurface(screen);
	}

	// set initial counter values
	count = 0;
	lineCount = SCREEN_MAX_X;

	// Calculate starting positions
	srcBase = (Uint32)(vScreen->pixels);
	destBase = (Uint32)(screen->pixels);

	// Actual blit loop
	while (count < (HLP_BPP(SCREEN_WIDTH*SCREEN_HEIGHT)))
	{
		*(Uint32*)(destBase + count) = *(Uint32*)(srcBase + count);
		count += BPP;
		if (lineCount != 0)
		{
			lineCount--;
		} else {
			lineCount = SCREEN_MAX_X;
		}
	}

	// Unlock surfaces if necessary
	if (SDL_MUSTLOCK(screen))
	{
		SDL_UnlockSurface(screen);
	}
	if (SDL_MUSTLOCK(vScreen))
	{
		SDL_UnlockSurface(vScreen);
	}

	SDL_UpdateRect(screen, 0, 0, SCREEN_WIDTH, SCREEN_HEIGHT);
}

// Goes through all the layers updating them and their objects
void BG_UpdateLayers()
{
	// Calculate the main layer's speed from the previous frame's time
	BG_MainFrameSpeedX = HLP_FrameSpeed(BG_MainSpeedX);
	BG_MainFrameSpeedY = HLP_FrameSpeed(BG_MainSpeedY);

	#ifdef DEBUG_MODE2_BG
	printf("BG_UpdateLayers : BG_MainFrameSpeedX = %f, BG_MainFrameSpeedY = %f\n",
		BG_MainFrameSpeedX, BG_MainFrameSpeedY);
	#endif

	// Update all the bg layers at once
	BG_UpdateAllLayers();
}

// Given left, top, width & height, checks if the rectangle is clear on the main layer
bool BG_IsSpaceClear(int _left, int _top, int _width, int _height)
{
	int bgCount, lineCount, skipBytes, bgBase;
	Uint32 *bgPos;
	Uint32 pixel;
	SDL_Surface* bgSurface;
	bool spaceIsClear;

	#ifdef DEBUG_MODE3_BG
	printf("BG_IsSpaceClear(_left=%i, _top=%i, _width=%i, _height=%i)\n", _left, _top, _width, _height);
	#endif

	spaceIsClear = true;

	// First check that some part of this rectangle's on the screen
	if ((_left < BG_MainX) || (_left + _width >= (int)(BG_MainX + SCREEN_WIDTH))
	 || (_top < BG_MainY) || (_top + _height >= (int)(BG_MainY + SCREEN_HEIGHT)))
	{
		spaceIsClear = false;
	} else {
		// Cache a pointer to the bg surface
		bgSurface = BG_MainLayer->surface;

		// Lock surface if necessary
		if (SDL_MUSTLOCK(bgSurface))
		{
			SDL_LockSurface(bgSurface);
		}

		// Calculate how many pixels to skip at the end of each line
		skipBytes = HLP_BPP(bgSurface->w - _width);

		// set initial counter values
		bgCount = 0;
		lineCount = _width-1; // how many pixels to check per row

		// Calculate starting positions
		bgBase = HLP_GetPos(bgSurface, (int)(_left), (int)(_top));

		#ifdef DEBUG_POINTS_BG
		// Draw the bounding box points in purple
		*(Uint32*)(bgBase) = 0xffff00ff;
		*(Uint32*)(bgBase+(int)(HLP_BPP(_width))) = 0xffff00ff;
		*(Uint32*)(bgBase+(int)(HLP_BPP(bgSurface->w*_height))) = 0xffff00ff;
		*(Uint32*)(bgBase+(int)(HLP_BPP(bgSurface->w*_height+_width))) = 0xffff00ff;
		#endif

		// Actual blit loop
		while ((bgCount < HLP_BPP(_height*bgSurface->w)) && (spaceIsClear))
		{
			bgPos = (Uint32*)(bgBase + bgCount);

			pixel = *(Uint32*)(bgPos);
			#ifdef DEBUG_POINTS_BG
			if ((pixel != 0) && (pixel != 0xffff00ff)) // ignore purple
			#else
			if (pixel != 0) // TODO: change this to colorkey - might not always be 0
			#endif
			{
				spaceIsClear = false;
				#ifdef DEBUG_POINTS_BG
				*(Uint32*)(bgPos) = 0xffffffff; // Make the collision pixel white
				#endif
			}
			bgCount += BPP;
			if (lineCount != 0)
			{
				lineCount--;
			} else {
				bgCount += skipBytes;
				lineCount = _width-1;
			}
		}

		// Unlock surfaces if necessary
		if (SDL_MUSTLOCK(bgSurface))
		{
			SDL_UnlockSurface(bgSurface);
		}
	}

	return spaceIsClear;
}
