/***************************************************************************
	audio.c - music & sound code

    begin                : 20 June 2004
    copyright            : (C) 2004 by Paul Rahme
****************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
****************************************************************************/

#include <mikmod.h>
#include <string.h>

#include "main.h"
#include "audio.h"
#include "backgrounds.h"
#include "input.h"

// -------------------------------------------------------------------------------------------------
// Constants
// -------------------------------------------------------------------------------------------------
const char *AUD_SongNames[AUD_NO_OF_SONGS] =
{
	"music/dummy.xm",				// SONG_DUMMY
};

const char *AUD_SoundNames[AUD_NO_OF_SOUNDS] =
{
	"sound/weak_shot.wav",			// SND_WEAK_SHOT
};

// -------------------------------------------------------------------------------------------------
// Variables
// -------------------------------------------------------------------------------------------------
MODULE *AUD_Module;
SAMPLE *AUD_Wavs[AUD_NO_OF_SOUNDS];
enum AUD_Songs AUD_CurrentSong;

// -------------------------------------------------------------------------------------------------
// Functions
// -------------------------------------------------------------------------------------------------

#ifdef SOUND_ENABLED

// Initialises audio
void AUD_Init()
{
	int wavID;
	char soundFileName[256];

	#ifdef DEBUG_MODE1_MUS
	printf("AUD_Init()\n");
	#endif

	AUD_CurrentSong = -1;

	// Prepare mikmod
	MikMod_RegisterDriver(&drv_oss);
	MikMod_RegisterAllLoaders();
	md_device = 1;

	// Initialise mikmod
	if (MikMod_Init("buffer=10"))
	{
		fprintf(stderr, "Could not initialize mikmod, reason: %s\n", MikMod_strerror(MikMod_errno));
		Lockup("AUD_Init() : Could not initialize audio.");
	}

	// Load all samples into memory
	for (wavID=0; wavID < AUD_NO_OF_SOUNDS; wavID++)
	{
		strcpy(soundFileName, BASE_DIR);		// copy it into a long string so that the filename can be appended
		strcat(soundFileName, AUD_SoundNames[wavID]);

		#ifdef DEBUG_MODE1_SND
		printf("AUD_Init : Loading sample %s...", soundFileName);
		#endif
		AUD_Wavs[wavID] = Sample_Load(soundFileName);
		if (AUD_Wavs[wavID] == NULL)
		{
			int i;

			// Free the samples loaded so far and shut down mikmod
			for (i=0; i<wavID; i++)
			{
				Sample_Free(AUD_Wavs[i]);
			}
			MikMod_Exit();

			// Lockup
			fprintf(stderr, "Could not load sample %i, reason: %s\n", wavID, MikMod_strerror(MikMod_errno));
			Lockup("AUD_Init : Could not load sample.");
		}
		#ifdef DEBUG_MODE1_SND
		else
		{
			printf("ok\n");
		}
		#endif
	}

	MikMod_SetNumVoices(-1, 8);
	MikMod_EnableOutput();
}

// Starts the specified sound effect playing, and calculates stereo from the object's x position (relative to the screen pos).
// _volume can be -1 for default or 0-255, _freq can be -1 for default or actual hz.
void AUD_PlaySound(enum AUD_Sounds _soundID, int _xPos, int _volume, int _freq)
{
	SBYTE sound;

	sound = Sample_Play(AUD_Wavs[_soundID], 0, 0);
	Voice_SetPanning(sound, ((_xPos-BG_MainX)*255)/639);
	if (_volume != -1)
	{
		Voice_SetVolume(sound, _volume);
	}
	if (_freq != -1)
	{
		Voice_SetFrequency(sound, _freq);
	}
}

// Loads the specified module and starts it playing, forcing it to restart the song if required
void AUD_StartSong(enum AUD_Songs _song, int _forceRestart)
{
	if ((_forceRestart) || (AUD_CurrentSong != _song))
	{
		AUD_CurrentSong = _song;

		char songFileName[256];

		#ifdef DEBUG_MODE1_MUS
		printf("AUD_StartSong(%i)\n", _song);
		#endif

		strcpy(songFileName, BASE_DIR);		// copy it into a long string so that the filename can be appended
		strcat(songFileName, AUD_SongNames[_song]);

		AUD_Module = Player_Load(songFileName, 64, 0);
		if (AUD_Module)
		{
			AUD_Module->wrap = 1;
			Player_Start(AUD_Module);
		}
		else
		{
			fprintf(stderr, "Could not load module, reason: %s\n", MikMod_strerror(MikMod_errno));
			Lockup("AUD_StartSong : Could not load module.");
		}
	}
}

// Updates the audio - call once per frame
void AUD_Update()
{
	if (!gamePaused)
	{
		MikMod_Update();
	}
}

// Stops the current song
void AUD_HaltSong()
{
	#ifdef DEBUG_CHECKS
	if (!AUD_Module)
	{
		Lockup("AUD_HaltSong : Song is not active.");
	}
	#endif
	Player_Stop();
	Player_Free(AUD_Module);
	AUD_CurrentSong = -1;
}

// Shuts down audio
void AUD_Kill()
{
	int sound;

	// Free the samples loaded
	for (sound=0; sound < AUD_NO_OF_SOUNDS; sound++)
	{
		Sample_Free(AUD_Wavs[sound]);
	}

	MikMod_Exit();
}

#else	// !SOUND_ENABLED
void AUD_Init() {}
void AUD_PlaySound(enum AUD_Sounds _soundID, int _xPos, int _volume, int _freq) {}
void AUD_StartSong(enum AUD_Songs _song, int _forceRestart) {}
void AUD_Update() {}
void AUD_HaltSong() {}
void AUD_Kill() {}
#endif
