// Copyright 2015 PingCAP, Inc.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//     http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// See the License for the specific language governing permissions and
// limitations under the License.

package mysql

// ErrMessage is a error message with the format specifier.
type ErrMessage struct {
	Raw          string
	RedactArgPos []int
}

// Message creates a error message with the format specifier.
func Message(message string, redactArgs []int) *ErrMessage {
	return &ErrMessage{Raw: message, RedactArgPos: redactArgs}
}

// MySQLErrName maps error code to MySQL error messages.
var MySQLErrName = map[uint16]*ErrMessage{
	ErrHashchk:                                  Message("hashchk", nil),
	ErrNisamchk:                                 Message("isamchk", nil),
	ErrNo:                                       Message("NO", nil),
	ErrYes:                                      Message("YES", nil),
	ErrCantCreateFile:                           Message("Can't create file '%-.200s' (errno: %d - %s)", nil),
	ErrCantCreateTable:                          Message("Can't create table '%-.200s' (errno: %d)", nil),
	ErrCantCreateDB:                             Message("Can't create database '%-.192s' (errno: %d)", nil),
	ErrDBCreateExists:                           Message("Can't create database '%-.192s'; database exists", nil),
	ErrDBDropExists:                             Message("Can't drop database '%-.192s'; database doesn't exist", nil),
	ErrDBDropDelete:                             Message("Error dropping database (can't delete '%-.192s', errno: %d)", nil),
	ErrDBDropRmdir:                              Message("Error dropping database (can't rmdir '%-.192s', errno: %d)", nil),
	ErrCantDeleteFile:                           Message("Error on delete of '%-.192s' (errno: %d - %s)", nil),
	ErrCantFindSystemRec:                        Message("Can't read record in system table", nil),
	ErrCantGetStat:                              Message("Can't get status of '%-.200s' (errno: %d - %s)", nil),
	ErrCantGetWd:                                Message("Can't get working directory (errno: %d - %s)", nil),
	ErrCantLock:                                 Message("Can't lock file (errno: %d - %s)", nil),
	ErrCantOpenFile:                             Message("Can't open file: '%-.200s' (errno: %d - %s)", nil),
	ErrFileNotFound:                             Message("Can't find file: '%-.200s' (errno: %d - %s)", nil),
	ErrCantReadDir:                              Message("Can't read dir of '%-.192s' (errno: %d - %s)", nil),
	ErrCantSetWd:                                Message("Can't change dir to '%-.192s' (errno: %d - %s)", nil),
	ErrCheckread:                                Message("Record has changed since last read in table '%-.192s'", nil),
	ErrDiskFull:                                 Message("Disk full (%s); waiting for someone to free some space... (errno: %d - %s)", nil),
	ErrDupKey:                                   Message("Can't write; duplicate key in table '%-.192s'", nil),
	ErrErrorOnClose:                             Message("Error on close of '%-.192s' (errno: %d - %s)", nil),
	ErrErrorOnRead:                              Message("Error reading file '%-.200s' (errno: %d - %s)", nil),
	ErrErrorOnRename:                            Message("Error on rename of '%-.210s' to '%-.210s' (errno: %d - %s)", nil),
	ErrErrorOnWrite:                             Message("Error writing file '%-.200s' (errno: %d - %s)", nil),
	ErrFileUsed:                                 Message("'%-.192s' is locked against change", nil),
	ErrFilsortAbort:                             Message("Sort aborted", nil),
	ErrFormNotFound:                             Message("View '%-.192s' doesn't exist for '%-.192s'", nil),
	ErrGetErrno:                                 Message("Got error %d from storage engine", nil),
	ErrIllegalHa:                                Message("Table storage engine for '%-.192s' doesn't have this option", nil),
	ErrKeyNotFound:                              Message("Can't find record in '%-.192s'", nil),
	ErrNotFormFile:                              Message("Incorrect information in file: '%-.200s'", nil),
	ErrNotKeyFile:                               Message("Incorrect key file for table '%-.200s'; try to repair it", nil),
	ErrOldKeyFile:                               Message("Old key file for table '%-.192s'; repair it!", nil),
	ErrOpenAsReadonly:                           Message("Table '%-.192s' is read only", nil),
	ErrOutofMemory:                              Message("Out of memory; restart server and try again (needed %d bytes)", nil),
	ErrOutOfSortMemory:                          Message("Out of sort memory, consider increasing server sort buffer size", nil),
	ErrUnexpectedEOF:                            Message("Unexpected EOF found when reading file '%-.192s' (errno: %d - %s)", nil),
	ErrConCount:                                 Message("Too many connections", nil),
	ErrOutOfResources:                           Message("Out of memory; check if mysqld or some other process uses all available memory; if not, you may have to use 'ulimit' to allow mysqld to use more memory or you can add more swap space", nil),
	ErrBadHost:                                  Message("Can't get hostname for your address", nil),
	ErrHandshake:                                Message("Bad handshake", nil),
	ErrDBaccessDenied:                           Message("Access denied for user '%-.48s'@'%-.64s' to database '%-.192s'", nil),
	ErrAccessDenied:                             Message("Access denied for user '%-.48s'@'%-.64s' (using password: %s)", nil),
	ErrNoDB:                                     Message("No database selected", nil),
	ErrUnknownCom:                               Message("Unknown command", nil),
	ErrBadNull:                                  Message("Column '%-.192s' cannot be null", nil),
	ErrBadDB:                                    Message("Unknown database '%-.192s'", nil),
	ErrTableExists:                              Message("Table '%-.192s' already exists", nil),
	ErrBadTable:                                 Message("Unknown table '%-.100s'", nil),
	ErrNonUniq:                                  Message("Column '%-.192s' in %-.192s is ambiguous", nil),
	ErrServerShutdown:                           Message("Server shutdown in progress", nil),
	ErrBadField:                                 Message("Unknown column '%-.192s' in '%-.192s'", nil),
	ErrFieldNotInGroupBy:                        Message("Expression #%d of %s is not in GROUP BY clause and contains nonaggregated column '%s' which is not functionally dependent on columns in GROUP BY clause; this is incompatible with sql_mode=only_full_group_by", nil),
	ErrWrongGroupField:                          Message("Can't group on '%-.192s'", nil),
	ErrWrongSumSelect:                           Message("Statement has sum functions and columns in same statement", nil),
	ErrWrongValueCount:                          Message("Column count doesn't match value count", nil),
	ErrTooLongIdent:                             Message("Identifier name '%-.100s' is too long", nil),
	ErrDupFieldName:                             Message("Duplicate column name '%-.192s'", nil),
	ErrDupKeyName:                               Message("Duplicate key name '%-.192s'", nil),
	ErrDupEntry:                                 Message("Duplicate entry '%-.64s' for key '%-.192s'", nil),
	ErrWrongFieldSpec:                           Message("Incorrect column specifier for column '%-.192s'", nil),
	ErrParse:                                    Message("%s %s", nil),
	ErrEmptyQuery:                               Message("Query was empty", nil),
	ErrNonuniqTable:                             Message("Not unique table/alias: '%-.192s'", nil),
	ErrInvalidDefault:                           Message("Invalid default value for '%-.192s'", nil),
	ErrMultiplePriKey:                           Message("Multiple primary key defined", nil),
	ErrTooManyKeys:                              Message("Too many keys specified; max %d keys allowed", nil),
	ErrTooManyKeyParts:                          Message("Too many key parts specified; max %d parts allowed", nil),
	ErrTooLongKey:                               Message("Specified key was too long (%d bytes); max key length is %d bytes", nil),
	ErrKeyColumnDoesNotExits:                    Message("Key column '%-.192s' doesn't exist in table", nil),
	ErrBlobUsedAsKey:                            Message("BLOB column '%-.192s' can't be used in key specification with the used table type", nil),
	ErrTooBigFieldlength:                        Message("Column length too big for column '%-.192s' (max = %d); use BLOB or TEXT instead", nil),
	ErrWrongAutoKey:                             Message("Incorrect table definition; there can be only one auto column and it must be defined as a key", nil),
	ErrReady:                                    Message("%s: ready for connections.\nVersion: '%s'  socket: '%s'  port: %d", nil),
	ErrNormalShutdown:                           Message("%s: Normal shutdown\n", nil),
	ErrGotSignal:                                Message("%s: Got signal %d. Aborting!\n", nil),
	ErrShutdownComplete:                         Message("%s: Shutdown complete\n", nil),
	ErrForcingClose:                             Message("%s: Forcing close of thread %d  user: '%-.48s'\n", nil),
	ErrIpsock:                                   Message("Can't create IP socket", nil),
	ErrNoSuchIndex:                              Message("Table '%-.192s' has no index like the one used in CREATE INDEX; recreate the table", nil),
	ErrWrongFieldTerminators:                    Message("Field separator argument is not what is expected; check the manual", nil),
	ErrBlobsAndNoTerminated:                     Message("You can't use fixed rowlength with BLOBs; please use 'fields terminated by'", nil),
	ErrTextFileNotReadable:                      Message("The file '%-.128s' must be in the database directory or be readable by all", nil),
	ErrFileExists:                               Message("File '%-.200s' already exists", nil),
	ErrLoadInfo:                                 Message("Records: %d  Deleted: %d  Skipped: %d  Warnings: %d", nil),
	ErrAlterInfo:                                Message("Records: %d  Duplicates: %d", nil),
	ErrWrongSubKey:                              Message("Incorrect prefix key; the used key part isn't a string, the used length is longer than the key part, or the storage engine doesn't support unique prefix keys", nil),
	ErrCantRemoveAllFields:                      Message("You can't delete all columns with ALTER TABLE; use DROP TABLE instead", nil),
	ErrCantDropFieldOrKey:                       Message("Can't DROP '%-.192s'; check that column/key exists", nil),
	ErrInsertInfo:                               Message("Records: %d  Duplicates: %d  Warnings: %d", nil),
	ErrUpdateTableUsed:                          Message("You can't specify target table '%-.192s' for update in FROM clause", nil),
	ErrNoSuchThread:                             Message("Unknown thread id: %d", nil),
	ErrKillDenied:                               Message("You are not owner of thread %d", nil),
	ErrNoTablesUsed:                             Message("No tables used", nil),
	ErrTooBigSet:                                Message("Too many strings for column %-.192s and SET", nil),
	ErrNoUniqueLogFile:                          Message("Can't generate a unique log-filename %-.200s.(1-999)\n", nil),
	ErrTableNotLockedForWrite:                   Message("Table '%-.192s' was locked with a READ lock and can't be updated", nil),
	ErrTableNotLocked:                           Message("Table '%-.192s' was not locked with LOCK TABLES", nil),
	ErrBlobCantHaveDefault:                      Message("BLOB/TEXT/JSON column '%-.192s' can't have a default value", nil),
	ErrWrongDBName:                              Message("Incorrect database name '%-.100s'", nil),
	ErrWrongTableName:                           Message("Incorrect table name '%-.100s'", nil),
	ErrTooBigSelect:                             Message("The SELECT would examine more than MAXJOINSIZE rows; check your WHERE and use SET SQLBIGSELECTS=1 or SET MAXJOINSIZE=# if the SELECT is okay", nil),
	ErrUnknown:                                  Message("Unknown error", nil),
	ErrUnknownProcedure:                         Message("Unknown procedure '%-.192s'", nil),
	ErrWrongParamcountToProcedure:               Message("Incorrect parameter count to procedure '%-.192s'", nil),
	ErrWrongParametersToProcedure:               Message("Incorrect parameters to procedure '%-.192s'", nil),
	ErrUnknownTable:                             Message("Unknown table '%-.192s' in %-.32s", nil),
	ErrFieldSpecifiedTwice:                      Message("Column '%-.192s' specified twice", nil),
	ErrInvalidGroupFuncUse:                      Message("Invalid use of group function", nil),
	ErrUnsupportedExtension:                     Message("Table '%-.192s' uses an extension that doesn't exist in this MySQL version", nil),
	ErrTableMustHaveColumns:                     Message("A table must have at least 1 column", nil),
	ErrRecordFileFull:                           Message("The table '%-.192s' is full", nil),
	ErrUnknownCharacterSet:                      Message("Unknown character set: '%-.64s'", nil),
	ErrTooManyTables:                            Message("Too many tables; MySQL can only use %d tables in a join", nil),
	ErrTooManyFields:                            Message("Too many columns", nil),
	ErrTooBigRowsize:                            Message("Row size too large. The maximum row size for the used table type, not counting BLOBs, is %d. This includes storage overhead, check the manual. You have to change some columns to TEXT or BLOBs", nil),
	ErrStackOverrun:                             Message("Thread stack overrun:  Used: %d of a %d stack.  Use 'mysqld --threadStack=#' to specify a bigger stack if needed", nil),
	ErrWrongOuterJoin:                           Message("Cross dependency found in OUTER JOIN; examine your ON conditions", nil),
	ErrNullColumnInIndex:                        Message("Table handler doesn't support NULL in given index. Please change column '%-.192s' to be NOT NULL or use another handler", nil),
	ErrCantFindUdf:                              Message("Can't load function '%-.192s'", nil),
	ErrCantInitializeUdf:                        Message("Can't initialize function '%-.192s'; %-.80s", nil),
	ErrUdfNoPaths:                               Message("No paths allowed for shared library", nil),
	ErrUdfExists:                                Message("Function '%-.192s' already exists", nil),
	ErrCantOpenLibrary:                          Message("Can't open shared library '%-.192s' (errno: %d %-.128s)", nil),
	ErrCantFindDlEntry:                          Message("Can't find symbol '%-.128s' in library", nil),
	ErrFunctionNotDefined:                       Message("Function '%-.192s' is not defined", nil),
	ErrHostIsBlocked:                            Message("Host '%-.64s' is blocked because of many connection errors; unblock with 'mysqladmin flush-hosts'", nil),
	ErrHostNotPrivileged:                        Message("Host '%-.64s' is not allowed to connect to this MySQL server", nil),
	ErrPasswordAnonymousUser:                    Message("You are using MySQL as an anonymous user and anonymous users are not allowed to change passwords", nil),
	ErrPasswordNotAllowed:                       Message("You must have privileges to update tables in the mysql database to be able to change passwords for others", nil),
	ErrPasswordNoMatch:                          Message("Can't find any matching row in the user table", nil),
	ErrUpdateInfo:                               Message("Rows matched: %d  Changed: %d  Warnings: %d", nil),
	ErrCantCreateThread:                         Message("Can't create a new thread (errno %d); if you are not out of available memory, you can consult the manual for a possible OS-dependent bug", nil),
	ErrWrongValueCountOnRow:                     Message("Column count doesn't match value count at row %d", nil),
	ErrCantReopenTable:                          Message("Can't reopen table: '%-.192s'", nil),
	ErrInvalidUseOfNull:                         Message("Invalid use of NULL value", nil),
	ErrRegexp:                                   Message("Got error '%-.64s' from regexp", nil),
	ErrMixOfGroupFuncAndFields:                  Message("Mixing of GROUP columns (MIN(),MAX(),COUNT(),...) with no GROUP columns is illegal if there is no GROUP BY clause", nil),
	ErrNonexistingGrant:                         Message("There is no such grant defined for user '%-.48s' on host '%-.64s'", nil),
	ErrTableaccessDenied:                        Message("%-.128s command denied to user '%-.48s'@'%-.64s' for table '%-.64s'", nil),
	ErrColumnaccessDenied:                       Message("%-.16s command denied to user '%-.48s'@'%-.64s' for column '%-.192s' in table '%-.192s'", nil),
	ErrIllegalGrantForTable:                     Message("Illegal GRANT/REVOKE command; please consult the manual to see which privileges can be used", nil),
	ErrGrantWrongHostOrUser:                     Message("The host or user argument to GRANT is too long", nil),
	ErrNoSuchTable:                              Message("Table '%-.192s.%-.192s' doesn't exist", nil),
	ErrNonexistingTableGrant:                    Message("There is no such grant defined for user '%-.48s' on host '%-.64s' on table '%-.192s'", nil),
	ErrNotAllowedCommand:                        Message("The used command is not allowed with this MySQL version", nil),
	ErrSyntax:                                   Message("You have an error in your SQL syntax; check the manual that corresponds to your MySQL server version for the right syntax to use", nil),
	ErrDelayedCantChangeLock:                    Message("Delayed insert thread couldn't get requested lock for table %-.192s", nil),
	ErrTooManyDelayedThreads:                    Message("Too many delayed threads in use", nil),
	ErrAbortingConnection:                       Message("Aborted connection %d to db: '%-.192s' user: '%-.48s' (%-.64s)", nil),
	ErrNetPacketTooLarge:                        Message("Got a packet bigger than 'max_allowed_packet' bytes", nil),
	ErrNetReadErrorFromPipe:                     Message("Got a read error from the connection pipe", nil),
	ErrNetFcntl:                                 Message("Got an error from fcntl()", nil),
	ErrNetPacketsOutOfOrder:                     Message("Got packets out of order", nil),
	ErrNetUncompress:                            Message("Couldn't uncompress communication packet", nil),
	ErrNetRead:                                  Message("Got an error reading communication packets", nil),
	ErrNetReadInterrupted:                       Message("Got timeout reading communication packets", nil),
	ErrNetErrorOnWrite:                          Message("Got an error writing communication packets", nil),
	ErrNetWriteInterrupted:                      Message("Got timeout writing communication packets", nil),
	ErrTooLongString:                            Message("Result string is longer than 'maxAllowedPacket' bytes", nil),
	ErrTableCantHandleBlob:                      Message("The used table type doesn't support BLOB/TEXT columns", nil),
	ErrTableCantHandleAutoIncrement:             Message("The used table type doesn't support AUTOINCREMENT columns", nil),
	ErrDelayedInsertTableLocked:                 Message("INSERT DELAYED can't be used with table '%-.192s' because it is locked with LOCK TABLES", nil),
	ErrWrongColumnName:                          Message("Incorrect column name '%-.100s'", nil),
	ErrWrongKeyColumn:                           Message("The used storage engine can't index column '%-.192s'", nil),
	ErrWrongMrgTable:                            Message("Unable to open underlying table which is differently defined or of non-MyISAM type or doesn't exist", nil),
	ErrDupUnique:                                Message("Can't write, because of unique constraint, to table '%-.192s'", nil),
	ErrBlobKeyWithoutLength:                     Message("BLOB/TEXT column '%-.192s' used in key specification without a key length", nil),
	ErrPrimaryCantHaveNull:                      Message("All parts of a PRIMARY KEY must be NOT NULL; if you need NULL in a key, use UNIQUE instead", nil),
	ErrTooManyRows:                              Message("Result consisted of more than one row", nil),
	ErrRequiresPrimaryKey:                       Message("This table type requires a primary key", nil),
	ErrNoRaidCompiled:                           Message("This version of MySQL is not compiled with RAID support", nil),
	ErrUpdateWithoutKeyInSafeMode:               Message("You are using safe update mode and you tried to update a table without a WHERE that uses a KEY column", nil),
	ErrKeyDoesNotExist:                          Message("Key '%-.192s' doesn't exist in table '%-.192s'", nil),
	ErrCheckNoSuchTable:                         Message("Can't open table", nil),
	ErrCheckNotImplemented:                      Message("The storage engine for the table doesn't support %s", nil),
	ErrCantDoThisDuringAnTransaction:            Message("You are not allowed to execute this command in a transaction", nil),
	ErrErrorDuringCommit:                        Message("Got error %d during COMMIT", nil),
	ErrErrorDuringRollback:                      Message("Got error %d during ROLLBACK", nil),
	ErrErrorDuringFlushLogs:                     Message("Got error %d during FLUSHLOGS", nil),
	ErrErrorDuringCheckpoint:                    Message("Got error %d during CHECKPOINT", nil),
	ErrNewAbortingConnection:                    Message("Aborted connection %d to db: '%-.192s' user: '%-.48s' host: '%-.64s' (%-.64s)", nil),
	ErrDumpNotImplemented:                       Message("The storage engine for the table does not support binary table dump", nil),
	ErrFlushMasterBinlogClosed:                  Message("Binlog closed, cannot RESET MASTER", nil),
	ErrIndexRebuild:                             Message("Failed rebuilding the index of  dumped table '%-.192s'", nil),
	ErrMaster:                                   Message("Error from master: '%-.64s'", nil),
	ErrMasterNetRead:                            Message("Net error reading from master", nil),
	ErrMasterNetWrite:                           Message("Net error writing to master", nil),
	ErrFtMatchingKeyNotFound:                    Message("Can't find FULLTEXT index matching the column list", nil),
	ErrLockOrActiveTransaction:                  Message("Can't execute the given command because you have active locked tables or an active transaction", nil),
	ErrUnknownSystemVariable:                    Message("Unknown system variable '%-.64s'", nil),
	ErrCrashedOnUsage:                           Message("Table '%-.192s' is marked as crashed and should be repaired", nil),
	ErrCrashedOnRepair:                          Message("Table '%-.192s' is marked as crashed and last (automatic?) repair failed", nil),
	ErrWarningNotCompleteRollback:               Message("Some non-transactional changed tables couldn't be rolled back", nil),
	ErrTransCacheFull:                           Message("Multi-statement transaction required more than 'maxBinlogCacheSize' bytes of storage; increase this mysqld variable and try again", nil),
	ErrSlaveMustStop:                            Message("This operation cannot be performed with a running slave; run STOP SLAVE first", nil),
	ErrSlaveNotRunning:                          Message("This operation requires a running slave; configure slave and do START SLAVE", nil),
	ErrBadSlave:                                 Message("The server is not configured as slave; fix in config file or with CHANGE MASTER TO", nil),
	ErrMasterInfo:                               Message("Could not initialize master info structure; more error messages can be found in the MySQL error log", nil),
	ErrSlaveThread:                              Message("Could not create slave thread; check system resources", nil),
	ErrTooManyUserConnections:                   Message("User %-.64s already has more than 'maxUserConnections' active connections", nil),
	ErrSetConstantsOnly:                         Message("You may only use constant expressions with SET", nil),
	ErrLockWaitTimeout:                          Message("Lock wait timeout exceeded; try restarting transaction", nil),
	ErrLockTableFull:                            Message("The total number of locks exceeds the lock table size", nil),
	ErrReadOnlyTransaction:                      Message("Update locks cannot be acquired during a READ UNCOMMITTED transaction", nil),
	ErrDropDBWithReadLock:                       Message("DROP DATABASE not allowed while thread is holding global read lock", nil),
	ErrCreateDBWithReadLock:                     Message("CREATE DATABASE not allowed while thread is holding global read lock", nil),
	ErrWrongArguments:                           Message("Incorrect arguments to %s", nil),
	ErrNoPermissionToCreateUser:                 Message("'%-.48s'@'%-.64s' is not allowed to create new users", nil),
	ErrUnionTablesInDifferentDir:                Message("Incorrect table definition; all MERGE tables must be in the same database", nil),
	ErrLockDeadlock:                             Message("Deadlock found when trying to get lock; try restarting transaction", nil),
	ErrTableCantHandleFt:                        Message("The used table type doesn't support FULLTEXT indexes", nil),
	ErrCannotAddForeign:                         Message("Cannot add foreign key constraint", nil),
	ErrNoReferencedRow:                          Message("Cannot add or update a child row: a foreign key constraint fails", nil),
	ErrRowIsReferenced:                          Message("Cannot delete or update a parent row: a foreign key constraint fails", nil),
	ErrConnectToMaster:                          Message("Error connecting to master: %-.128s", nil),
	ErrQueryOnMaster:                            Message("Error running query on master: %-.128s", nil),
	ErrErrorWhenExecutingCommand:                Message("Error when executing command %s: %-.128s", nil),
	ErrWrongUsage:                               Message("Incorrect usage of %s and %s", nil),
	ErrWrongNumberOfColumnsInSelect:             Message("The used SELECT statements have a different number of columns", nil),
	ErrCantUpdateWithReadlock:                   Message("Can't execute the query because you have a conflicting read lock", nil),
	ErrMixingNotAllowed:                         Message("Mixing of transactional and non-transactional tables is disabled", nil),
	ErrDupArgument:                              Message("Option '%s' used twice in statement", nil),
	ErrUserLimitReached:                         Message("User '%-.64s' has exceeded the '%s' resource (current value: %d)", nil),
	ErrSpecificAccessDenied:                     Message("Access denied; you need (at least one of) the %-.128s privilege(s) for this operation", nil),
	ErrLocalVariable:                            Message("Variable '%-.64s' is a SESSION variable and can't be used with SET GLOBAL", nil),
	ErrGlobalVariable:                           Message("Variable '%-.64s' is a GLOBAL variable and should be set with SET GLOBAL", nil),
	ErrNoDefault:                                Message("Variable '%-.64s' doesn't have a default value", nil),
	ErrWrongValueForVar:                         Message("Variable '%-.64s' can't be set to the value of '%-.200s'", nil),
	ErrWrongTypeForVar:                          Message("Incorrect argument type to variable '%-.64s'", nil),
	ErrVarCantBeRead:                            Message("Variable '%-.64s' can only be set, not read", nil),
	ErrCantUseOptionHere:                        Message("Incorrect usage/placement of '%s'", nil),
	ErrNotSupportedYet:                          Message("This version of TiDB doesn't yet support '%s'", nil),
	ErrMasterFatalErrorReadingBinlog:            Message("Got fatal error %d from master when reading data from binary log: '%-.320s'", nil),
	ErrSlaveIgnoredTable:                        Message("Slave SQL thread ignored the query because of replicate-*-table rules", nil),
	ErrIncorrectGlobalLocalVar:                  Message("Variable '%-.192s' is a %s variable", nil),
	ErrWrongFkDef:                               Message("Incorrect foreign key definition for '%-.192s': %s", nil),
	ErrKeyRefDoNotMatchTableRef:                 Message("Key reference and table reference don't match", nil),
	ErrOperandColumns:                           Message("Operand should contain %d column(s)", nil),
	ErrSubqueryNo1Row:                           Message("Subquery returns more than 1 row", nil),
	ErrUnknownStmtHandler:                       Message("Unknown prepared statement handler %s given to %s", nil),
	ErrCorruptHelpDB:                            Message("Help database is corrupt or does not exist", nil),
	ErrCyclicReference:                          Message("Cyclic reference on subqueries", nil),
	ErrAutoConvert:                              Message("Converting column '%s' from %s to %s", nil),
	ErrIllegalReference:                         Message("Reference '%-.64s' not supported (%s)", nil),
	ErrDerivedMustHaveAlias:                     Message("Every derived table must have its own alias", nil),
	ErrSelectReduced:                            Message("Select %d was reduced during optimization", nil),
	ErrTablenameNotAllowedHere:                  Message("Table '%s' from one of the %ss cannot be used in %s", nil),
	ErrNotSupportedAuthMode:                     Message("Client does not support authentication protocol requested by server; consider upgrading MySQL client", nil),
	ErrSpatialCantHaveNull:                      Message("All parts of a SPATIAL index must be NOT NULL", nil),
	ErrCollationCharsetMismatch:                 Message("COLLATION '%s' is not valid for CHARACTER SET '%s'", nil),
	ErrSlaveWasRunning:                          Message("Slave is already running", nil),
	ErrSlaveWasNotRunning:                       Message("Slave already has been stopped", nil),
	ErrTooBigForUncompress:                      Message("Uncompressed data size too large; the maximum size is %d (probably, length of uncompressed data was corrupted)", nil),
	ErrZlibZMem:                                 Message("ZLIB: Not enough memory", nil),
	ErrZlibZBuf:                                 Message("ZLIB: Not enough room in the output buffer (probably, length of uncompressed data was corrupted)", nil),
	ErrZlibZData:                                Message("ZLIB: Input data corrupted", nil),
	ErrCutValueGroupConcat:                      Message("Some rows were cut by GROUPCONCAT(%s)", nil),
	ErrWarnTooFewRecords:                        Message("Row %d doesn't contain data for all columns", nil),
	ErrWarnTooManyRecords:                       Message("Row %d was truncated; it contained more data than there were input columns", nil),
	ErrWarnNullToNotnull:                        Message("Column set to default value; NULL supplied to NOT NULL column '%s' at row %d", nil),
	ErrWarnDataOutOfRange:                       Message("Out of range value for column '%s' at row %d", nil),
	WarnDataTruncated:                           Message("Data truncated for column '%s' at row %d", nil),
	ErrWarnUsingOtherHandler:                    Message("Using storage engine %s for table '%s'", nil),
	ErrCantAggregate2collations:                 Message("Illegal mix of collations (%s,%s) and (%s,%s) for operation '%s'", nil),
	ErrDropUser:                                 Message("Cannot drop one or more of the requested users", nil),
	ErrRevokeGrants:                             Message("Can't revoke all privileges for one or more of the requested users", nil),
	ErrCantAggregate3collations:                 Message("Illegal mix of collations (%s,%s), (%s,%s), (%s,%s) for operation '%s'", nil),
	ErrCantAggregateNcollations:                 Message("Illegal mix of collations for operation '%s'", nil),
	ErrVariableIsNotStruct:                      Message("Variable '%-.64s' is not a variable component (can't be used as XXXX.variableName)", nil),
	ErrUnknownCollation:                         Message("Unknown collation: '%-.64s'", nil),
	ErrSlaveIgnoredSslParams:                    Message("SSL parameters in CHANGE MASTER are ignored because this MySQL slave was compiled without SSL support; they can be used later if MySQL slave with SSL is started", nil),
	ErrServerIsInSecureAuthMode:                 Message("Server is running in --secure-auth mode, but '%s'@'%s' has a password in the old format; please change the password to the new format", nil),
	ErrWarnFieldResolved:                        Message("Field or reference '%-.192s%s%-.192s%s%-.192s' of SELECT #%d was resolved in SELECT #%d", nil),
	ErrBadSlaveUntilCond:                        Message("Incorrect parameter or combination of parameters for START SLAVE UNTIL", nil),
	ErrMissingSkipSlave:                         Message("It is recommended to use --skip-slave-start when doing step-by-step replication with START SLAVE UNTIL; otherwise, you will get problems if you get an unexpected slave's mysqld restart", nil),
	ErrUntilCondIgnored:                         Message("SQL thread is not to be started so UNTIL options are ignored", nil),
	ErrWrongNameForIndex:                        Message("Incorrect index name '%-.100s'", nil),
	ErrWrongNameForCatalog:                      Message("Incorrect catalog name '%-.100s'", nil),
	ErrWarnQcResize:                             Message("Query cache failed to set size %d; new query cache size is %d", nil),
	ErrBadFtColumn:                              Message("Column '%-.192s' cannot be part of FULLTEXT index", nil),
	ErrUnknownKeyCache:                          Message("Unknown key cache '%-.100s'", nil),
	ErrWarnHostnameWontWork:                     Message("MySQL is started in --skip-name-resolve mode; you must restart it without this switch for this grant to work", nil),
	ErrUnknownStorageEngine:                     Message("Unknown storage engine '%s'", nil),
	ErrWarnDeprecatedSyntax:                     Message("'%s' is deprecated and will be removed in a future release. Please use %s instead", nil),
	ErrNonUpdatableTable:                        Message("The target table %-.100s of the %s is not updatable", nil),
	ErrFeatureDisabled:                          Message("The '%s' feature is disabled; you need MySQL built with '%s' to have it working", nil),
	ErrOptionPreventsStatement:                  Message("The MySQL server is running with the %s option so it cannot execute this statement", nil),
	ErrDuplicatedValueInType:                    Message("Column '%-.100s' has duplicated value '%-.64s' in %s", nil),
	ErrTruncatedWrongValue:                      Message("Truncated incorrect %-.64s value: '%-.128s'", nil),
	ErrTooMuchAutoTimestampCols:                 Message("Incorrect table definition; there can be only one TIMESTAMP column with CURRENTTIMESTAMP in DEFAULT or ON UPDATE clause", nil),
	ErrInvalidOnUpdate:                          Message("Invalid ON UPDATE clause for '%-.192s' column", nil),
	ErrUnsupportedPs:                            Message("This command is not supported in the prepared statement protocol yet", nil),
	ErrGetErrmsg:                                Message("Got error %d '%-.100s' from %s", nil),
	ErrGetTemporaryErrmsg:                       Message("Got temporary error %d '%-.100s' from %s", nil),
	ErrUnknownTimeZone:                          Message("Unknown or incorrect time zone: '%-.64s'", nil),
	ErrWarnInvalidTimestamp:                     Message("Invalid TIMESTAMP value in column '%s' at row %d", nil),
	ErrInvalidCharacterString:                   Message("Invalid %s character string: '%.64s'", nil),
	ErrWarnAllowedPacketOverflowed:              Message("Result of %s() was larger than max_allowed_packet (%d) - truncated", nil),
	ErrConflictingDeclarations:                  Message("Conflicting declarations: '%s%s' and '%s%s'", nil),
	ErrSpNoRecursiveCreate:                      Message("Can't create a %s from within another stored routine", nil),
	ErrSpAlreadyExists:                          Message("%s %s already exists", nil),
	ErrSpDoesNotExist:                           Message("%s %s does not exist", nil),
	ErrSpDropFailed:                             Message("Failed to DROP %s %s", nil),
	ErrSpStoreFailed:                            Message("Failed to CREATE %s %s", nil),
	ErrSpLilabelMismatch:                        Message("%s with no matching label: %s", nil),
	ErrSpLabelRedefine:                          Message("Redefining label %s", nil),
	ErrSpLabelMismatch:                          Message("End-label %s without match", nil),
	ErrSpUninitVar:                              Message("Referring to uninitialized variable %s", nil),
	ErrSpBadselect:                              Message("PROCEDURE %s can't return a result set in the given context", nil),
	ErrSpBadreturn:                              Message("RETURN is only allowed in a FUNCTION", nil),
	ErrSpBadstatement:                           Message("%s is not allowed in stored procedures", nil),
	ErrUpdateLogDeprecatedIgnored:               Message("The update log is deprecated and replaced by the binary log; SET SQLLOGUPDATE has been ignored.", nil),
	ErrUpdateLogDeprecatedTranslated:            Message("The update log is deprecated and replaced by the binary log; SET SQLLOGUPDATE has been translated to SET SQLLOGBIN.", nil),
	ErrQueryInterrupted:                         Message("Query execution was interrupted", nil),
	ErrSpWrongNoOfArgs:                          Message("Incorrect number of arguments for %s %s; expected %d, got %d", nil),
	ErrSpCondMismatch:                           Message("Undefined CONDITION: %s", nil),
	ErrSpNoreturn:                               Message("No RETURN found in FUNCTION %s", nil),
	ErrSpNoreturnend:                            Message("FUNCTION %s ended without RETURN", nil),
	ErrSpBadCursorQuery:                         Message("Cursor statement must be a SELECT", nil),
	ErrSpBadCursorSelect:                        Message("Cursor SELECT must not have INTO", nil),
	ErrSpCursorMismatch:                         Message("Undefined CURSOR: %s", nil),
	ErrSpCursorAlreadyOpen:                      Message("Cursor is already open", nil),
	ErrSpCursorNotOpen:                          Message("Cursor is not open", nil),
	ErrSpUndeclaredVar:                          Message("Undeclared variable: %s", nil),
	ErrSpWrongNoOfFetchArgs:                     Message("Incorrect number of FETCH variables", nil),
	ErrSpFetchNoData:                            Message("No data - zero rows fetched, selected, or processed", nil),
	ErrSpDupParam:                               Message("Duplicate parameter: %s", nil),
	ErrSpDupVar:                                 Message("Duplicate variable: %s", nil),
	ErrSpDupCond:                                Message("Duplicate condition: %s", nil),
	ErrSpDupCurs:                                Message("Duplicate cursor: %s", nil),
	ErrSpCantAlter:                              Message("Failed to ALTER %s %s", nil),
	ErrSpSubselectNyi:                           Message("Subquery value not supported", nil),
	ErrStmtNotAllowedInSfOrTrg:                  Message("%s is not allowed in stored function or trigger", nil),
	ErrSpVarcondAfterCurshndlr:                  Message("Variable or condition declaration after cursor or handler declaration", nil),
	ErrSpCursorAfterHandler:                     Message("Cursor declaration after handler declaration", nil),
	ErrSpCaseNotFound:                           Message("Case not found for CASE statement", nil),
	ErrFparserTooBigFile:                        Message("Configuration file '%-.192s' is too big", nil),
	ErrFparserBadHeader:                         Message("Malformed file type header in file '%-.192s'", nil),
	ErrFparserEOFInComment:                      Message("Unexpected end of file while parsing comment '%-.200s'", nil),
	ErrFparserErrorInParameter:                  Message("Error while parsing parameter '%-.192s' (line: '%-.192s')", nil),
	ErrFparserEOFInUnknownParameter:             Message("Unexpected end of file while skipping unknown parameter '%-.192s'", nil),
	ErrViewNoExplain:                            Message("EXPLAIN/SHOW can not be issued; lacking privileges for underlying table", nil),
	ErrFrmUnknownType:                           Message("File '%-.192s' has unknown type '%-.64s' in its header", nil),
	ErrWrongObject:                              Message("'%-.192s.%-.192s' is not %s", nil),
	ErrNonupdateableColumn:                      Message("Column '%-.192s' is not updatable", nil),
	ErrViewSelectDerived:                        Message("View's SELECT contains a subquery in the FROM clause", nil),
	ErrViewSelectClause:                         Message("View's SELECT contains a '%s' clause", nil),
	ErrViewSelectVariable:                       Message("View's SELECT contains a variable or parameter", nil),
	ErrViewSelectTmptable:                       Message("View's SELECT refers to a temporary table '%-.192s'", nil),
	ErrViewWrongList:                            Message("View's SELECT and view's field list have different column counts", nil),
	ErrWarnViewMerge:                            Message("View merge algorithm can't be used here for now (assumed undefined algorithm)", nil),
	ErrWarnViewWithoutKey:                       Message("View being updated does not have complete key of underlying table in it", nil),
	ErrViewInvalid:                              Message("View '%-.192s.%-.192s' references invalid table(s) or column(s) or function(s) or definer/invoker of view lack rights to use them", nil),
	ErrSpNoDropSp:                               Message("Can't drop or alter a %s from within another stored routine", nil),
	ErrSpGotoInHndlr:                            Message("GOTO is not allowed in a stored procedure handler", nil),
	ErrTrgAlreadyExists:                         Message("Trigger already exists", nil),
	ErrTrgDoesNotExist:                          Message("Trigger does not exist", nil),
	ErrTrgOnViewOrTempTable:                     Message("Trigger's '%-.192s' is view or temporary table", nil),
	ErrTrgCantChangeRow:                         Message("Updating of %s row is not allowed in %strigger", nil),
	ErrTrgNoSuchRowInTrg:                        Message("There is no %s row in %s trigger", nil),
	ErrNoDefaultForField:                        Message("Field '%-.192s' doesn't have a default value", nil),
	ErrDivisionByZero:                           Message("Division by 0", nil),
	ErrTruncatedWrongValueForField:              Message("Incorrect %-.32s value: '%-.128s' for column '%.192s' at row %d", nil),
	ErrIllegalValueForType:                      Message("Illegal %s '%-.192s' value found during parsing", nil),
	ErrViewNonupdCheck:                          Message("CHECK OPTION on non-updatable view '%-.192s.%-.192s'", nil),
	ErrViewCheckFailed:                          Message("CHECK OPTION failed '%-.192s.%-.192s'", nil),
	ErrProcaccessDenied:                         Message("%-.16s command denied to user '%-.48s'@'%-.64s' for routine '%-.192s'", nil),
	ErrRelayLogFail:                             Message("Failed purging old relay logs: %s", nil),
	ErrPasswdLength:                             Message("Password hash should be a %d-digit hexadecimal number", nil),
	ErrUnknownTargetBinlog:                      Message("Target log not found in binlog index", nil),
	ErrIoErrLogIndexRead:                        Message("I/O error reading log index file", nil),
	ErrBinlogPurgeProhibited:                    Message("Server configuration does not permit binlog purge", nil),
	ErrFseekFail:                                Message("Failed on fseek()", nil),
	ErrBinlogPurgeFatalErr:                      Message("Fatal error during log purge", nil),
	ErrLogInUse:                                 Message("A purgeable log is in use, will not purge", nil),
	ErrLogPurgeUnknownErr:                       Message("Unknown error during log purge", nil),
	ErrRelayLogInit:                             Message("Failed initializing relay log position: %s", nil),
	ErrNoBinaryLogging:                          Message("You are not using binary logging", nil),
	ErrReservedSyntax:                           Message("The '%-.64s' syntax is reserved for purposes internal to the MySQL server", nil),
	ErrWsasFailed:                               Message("WSAStartup Failed", nil),
	ErrDiffGroupsProc:                           Message("Can't handle procedures with different groups yet", nil),
	ErrNoGroupForProc:                           Message("Select must have a group with this procedure", nil),
	ErrOrderWithProc:                            Message("Can't use ORDER clause with this procedure", nil),
	ErrLoggingProhibitChangingOf:                Message("Binary logging and replication forbid changing the global server %s", nil),
	ErrNoFileMapping:                            Message("Can't map file: %-.200s, errno: %d", nil),
	ErrWrongMagic:                               Message("Wrong magic in %-.64s", nil),
	ErrPsManyParam:                              Message("Prepared statement contains too many placeholders", nil),
	ErrKeyPart0:                                 Message("Key part '%-.192s' length cannot be 0", nil),
	ErrViewChecksum:                             Message("View text checksum failed", nil),
	ErrViewMultiupdate:                          Message("Can not modify more than one base table through a join view '%-.192s.%-.192s'", nil),
	ErrViewNoInsertFieldList:                    Message("Can not insert into join view '%-.192s.%-.192s' without fields list", nil),
	ErrViewDeleteMergeView:                      Message("Can not delete from join view '%-.192s.%-.192s'", nil),
	ErrCannotUser:                               Message("Operation %s failed for %.256s", nil),
	ErrXaerNota:                                 Message("XAERNOTA: Unknown XID", nil),
	ErrXaerInval:                                Message("XAERINVAL: Invalid arguments (or unsupported command)", nil),
	ErrXaerRmfail:                               Message("XAERRMFAIL: The command cannot be executed when global transaction is in the  %.64s state", nil),
	ErrXaerOutside:                              Message("XAEROUTSIDE: Some work is done outside global transaction", nil),
	ErrXaerRmerr:                                Message("XAERRMERR: Fatal error occurred in the transaction branch - check your data for consistency", nil),
	ErrXaRbrollback:                             Message("XARBROLLBACK: Transaction branch was rolled back", nil),
	ErrNonexistingProcGrant:                     Message("There is no such grant defined for user '%-.48s' on host '%-.64s' on routine '%-.192s'", nil),
	ErrProcAutoGrantFail:                        Message("Failed to grant EXECUTE and ALTER ROUTINE privileges", nil),
	ErrProcAutoRevokeFail:                       Message("Failed to revoke all privileges to dropped routine", nil),
	ErrDataTooLong:                              Message("Data too long for column '%s' at row %d", nil),
	ErrSpBadSQLstate:                            Message("Bad SQLSTATE: '%s'", nil),
	ErrStartup:                                  Message("%s: ready for connections.\nVersion: '%s'  socket: '%s'  port: %d  %s", nil),
	ErrLoadFromFixedSizeRowsToVar:               Message("Can't load value from file with fixed size rows to variable", nil),
	ErrCantCreateUserWithGrant:                  Message("You are not allowed to create a user with GRANT", nil),
	ErrWrongValueForType:                        Message("Incorrect %-.32s value: '%-.128s' for function %-.32s", nil),
	ErrTableDefChanged:                          Message("Table definition has changed, please retry transaction", nil),
	ErrSpDupHandler:                             Message("Duplicate handler declared in the same block", nil),
	ErrSpNotVarArg:                              Message("OUT or INOUT argument %d for routine %s is not a variable or NEW pseudo-variable in BEFORE trigger", nil),
	ErrSpNoRetset:                               Message("Not allowed to return a result set from a %s", nil),
	ErrCantCreateGeometryObject:                 Message("Cannot get geometry object from data you send to the GEOMETRY field", nil),
	ErrFailedRoutineBreakBinlog:                 Message("A routine failed and has neither NO SQL nor READS SQL DATA in its declaration and binary logging is enabled; if non-transactional tables were updated, the binary log will miss their changes", nil),
	ErrBinlogUnsafeRoutine:                      Message("This function has none of DETERMINISTIC, NO SQL, or READS SQL DATA in its declaration and binary logging is enabled (you *might* want to use the less safe logBinTrustFunctionCreators variable)", nil),
	ErrBinlogCreateRoutineNeedSuper:             Message("You do not have the SUPER privilege and binary logging is enabled (you *might* want to use the less safe logBinTrustFunctionCreators variable)", nil),
	ErrExecStmtWithOpenCursor:                   Message("You can't execute a prepared statement which has an open cursor associated with it. Reset the statement to re-execute it.", nil),
	ErrStmtHasNoOpenCursor:                      Message("The statement (%d) has no open cursor.", nil),
	ErrCommitNotAllowedInSfOrTrg:                Message("Explicit or implicit commit is not allowed in stored function or trigger.", nil),
	ErrNoDefaultForViewField:                    Message("Field of view '%-.192s.%-.192s' underlying table doesn't have a default value", nil),
	ErrSpNoRecursion:                            Message("Recursive stored functions and triggers are not allowed.", nil),
	ErrTooBigScale:                              Message("Too big scale %d specified for column '%-.192s'. Maximum is %d.", nil),
	ErrTooBigPrecision:                          Message("Too big precision %d specified for column '%-.192s'. Maximum is %d.", nil),
	ErrMBiggerThanD:                             Message("For float(M,D), double(M,D) or decimal(M,D), M must be >= D (column '%-.192s').", nil),
	ErrWrongLockOfSystemTable:                   Message("You can't combine write-locking of system tables with other tables or lock types", nil),
	ErrConnectToForeignDataSource:               Message("Unable to connect to foreign data source: %.64s", nil),
	ErrQueryOnForeignDataSource:                 Message("There was a problem processing the query on the foreign data source. Data source : %-.64s", nil),
	ErrForeignDataSourceDoesntExist:             Message("The foreign data source you are trying to reference does not exist. Data source :  %-.64s", nil),
	ErrForeignDataStringInvalidCantCreate:       Message("Can't create federated table. The data source connection string '%-.64s' is not in the correct format", nil),
	ErrForeignDataStringInvalid:                 Message("The data source connection string '%-.64s' is not in the correct format", nil),
	ErrCantCreateFederatedTable:                 Message("Can't create federated table. Foreign data src :  %-.64s", nil),
	ErrTrgInWrongSchema:                         Message("Trigger in wrong schema", nil),
	ErrStackOverrunNeedMore:                     Message("Thread stack overrun:  %d bytes used of a %d byte stack, and %d bytes needed.  Use 'mysqld --threadStack=#' to specify a bigger stack.", nil),
	ErrTooLongBody:                              Message("Routine body for '%-.100s' is too long", nil),
	ErrWarnCantDropDefaultKeycache:              Message("Cannot drop default keycache", nil),
	ErrTooBigDisplaywidth:                       Message("Display width out of range for column '%-.192s' (max = %d)", nil),
	ErrXaerDupid:                                Message("XAERDUPID: The XID already exists", nil),
	ErrDatetimeFunctionOverflow:                 Message("Datetime function: %-.32s field overflow", nil),
	ErrCantUpdateUsedTableInSfOrTrg:             Message("Can't update table '%-.192s' in stored function/trigger because it is already used by statement which invoked this stored function/trigger.", nil),
	ErrViewPreventUpdate:                        Message("The definition of table '%-.192s' prevents operation %.192s on table '%-.192s'.", nil),
	ErrPsNoRecursion:                            Message("The prepared statement contains a stored routine call that refers to that same statement. It's not allowed to execute a prepared statement in such a recursive manner", nil),
	ErrSpCantSetAutocommit:                      Message("Not allowed to set autocommit from a stored function or trigger", nil),
	ErrMalformedDefiner:                         Message("Definer is not fully qualified", nil),
	ErrViewFrmNoUser:                            Message("View '%-.192s'.'%-.192s' has no definer information (old table format). Current user is used as definer. Please recreate the view!", nil),
	ErrViewOtherUser:                            Message("You need the SUPER privilege for creation view with '%-.192s'@'%-.192s' definer", nil),
	ErrNoSuchUser:                               Message("The user specified as a definer ('%-.64s'@'%-.64s') does not exist", nil),
	ErrForbidSchemaChange:                       Message("Changing schema from '%-.192s' to '%-.192s' is not allowed.", nil),
	ErrRowIsReferenced2:                         Message("Cannot delete or update a parent row: a foreign key constraint fails (%.192s)", nil),
	ErrNoReferencedRow2:                         Message("Cannot add or update a child row: a foreign key constraint fails (%.192s)", nil),
	ErrSpBadVarShadow:                           Message("Variable '%-.64s' must be quoted with `...`, or renamed", nil),
	ErrTrgNoDefiner:                             Message("No definer attribute for trigger '%-.192s'.'%-.192s'. The trigger will be activated under the authorization of the invoker, which may have insufficient privileges. Please recreate the trigger.", nil),
	ErrOldFileFormat:                            Message("'%-.192s' has an old format, you should re-create the '%s' object(s)", nil),
	ErrSpRecursionLimit:                         Message("Recursive limit %d (as set by the maxSpRecursionDepth variable) was exceeded for routine %.192s", nil),
	ErrSpProcTableCorrupt:                       Message("Failed to load routine %-.192s. The table mysql.proc is missing, corrupt, or contains bad data (internal code %d)", nil),
	ErrSpWrongName:                              Message("Incorrect routine name '%-.192s'", nil),
	ErrTableNeedsUpgrade:                        Message("Table upgrade required. Please do \"REPAIR TABLE `%-.32s`\"", nil),
	ErrSpNoAggregate:                            Message("AGGREGATE is not supported for stored functions", nil),
	ErrMaxPreparedStmtCountReached:              Message("Can't create more than maxPreparedStmtCount statements (current value: %d)", nil),
	ErrViewRecursive:                            Message("`%-.192s`.`%-.192s` contains view recursion", nil),
	ErrNonGroupingFieldUsed:                     Message("Non-grouping field '%-.192s' is used in %-.64s clause", nil),
	ErrTableCantHandleSpkeys:                    Message("The used table type doesn't support SPATIAL indexes", nil),
	ErrNoTriggersOnSystemSchema:                 Message("Triggers can not be created on system tables", nil),
	ErrRemovedSpaces:                            Message("Leading spaces are removed from name '%s'", nil),
	ErrAutoincReadFailed:                        Message("Failed to read auto-increment value from storage engine", nil),
	ErrUsername:                                 Message("user name", nil),
	ErrHostname:                                 Message("host name", nil),
	ErrWrongStringLength:                        Message("String '%-.70s' is too long for %s (should be no longer than %d)", nil),
	ErrNonInsertableTable:                       Message("The target table %-.100s of the %s is not insertable-into", nil),
	ErrAdminWrongMrgTable:                       Message("Table '%-.64s' is differently defined or of non-MyISAM type or doesn't exist", nil),
	ErrTooHighLevelOfNestingForSelect:           Message("Too high level of nesting for select", nil),
	ErrNameBecomesEmpty:                         Message("Name '%-.64s' has become ''", nil),
	ErrAmbiguousFieldTerm:                       Message("First character of the FIELDS TERMINATED string is ambiguous; please use non-optional and non-empty FIELDS ENCLOSED BY", nil),
	ErrForeignServerExists:                      Message("The foreign server, %s, you are trying to create already exists.", nil),
	ErrForeignServerDoesntExist:                 Message("The foreign server name you are trying to reference does not exist. Data source :  %-.64s", nil),
	ErrIllegalHaCreateOption:                    Message("Table storage engine '%-.64s' does not support the create option '%.64s'", nil),
	ErrPartitionRequiresValues:                  Message("Syntax : %-.64s PARTITIONING requires definition of VALUES %-.64s for each partition", nil),
	ErrPartitionWrongValues:                     Message("Only %-.64s PARTITIONING can use VALUES %-.64s in partition definition", nil),
	ErrPartitionMaxvalue:                        Message("MAXVALUE can only be used in last partition definition", nil),
	ErrPartitionSubpartition:                    Message("Subpartitions can only be hash partitions and by key", nil),
	ErrPartitionSubpartMix:                      Message("Must define subpartitions on all partitions if on one partition", nil),
	ErrPartitionWrongNoPart:                     Message("Wrong number of partitions defined, mismatch with previous setting", nil),
	ErrPartitionWrongNoSubpart:                  Message("Wrong number of subpartitions defined, mismatch with previous setting", nil),
	ErrWrongExprInPartitionFunc:                 Message("Constant, random or timezone-dependent expressions in (sub)partitioning function are not allowed", nil),
	ErrNoConstExprInRangeOrList:                 Message("Expression in RANGE/LIST VALUES must be constant", nil),
	ErrFieldNotFoundPart:                        Message("Field in list of fields for partition function not found in table", nil),
	ErrListOfFieldsOnlyInHash:                   Message("List of fields is only allowed in KEY partitions", nil),
	ErrInconsistentPartitionInfo:                Message("The partition info in the frm file is not consistent with what can be written into the frm file", nil),
	ErrPartitionFuncNotAllowed:                  Message("The %-.192s function returns the wrong type", nil),
	ErrPartitionsMustBeDefined:                  Message("For %-.64s partitions each partition must be defined", nil),
	ErrRangeNotIncreasing:                       Message("VALUES LESS THAN value must be strictly increasing for each partition", nil),
	ErrInconsistentTypeOfFunctions:              Message("VALUES value must be of same type as partition function", nil),
	ErrMultipleDefConstInListPart:               Message("Multiple definition of same constant in list partitioning", nil),
	ErrPartitionEntry:                           Message("Partitioning can not be used stand-alone in query", nil),
	ErrMixHandler:                               Message("The mix of handlers in the partitions is not allowed in this version of MySQL", nil),
	ErrPartitionNotDefined:                      Message("For the partitioned engine it is necessary to define all %-.64s", nil),
	ErrTooManyPartitions:                        Message("Too many partitions (including subpartitions) were defined", nil),
	ErrSubpartition:                             Message("It is only possible to mix RANGE/LIST partitioning with HASH/KEY partitioning for subpartitioning", nil),
	ErrCantCreateHandlerFile:                    Message("Failed to create specific handler file", nil),
	ErrBlobFieldInPartFunc:                      Message("A BLOB field is not allowed in partition function", nil),
	ErrUniqueKeyNeedAllFieldsInPf:               Message("A %-.192s must include all columns in the table's partitioning function", nil),
	ErrNoParts:                                  Message("Number of %-.64s = 0 is not an allowed value", nil),
	ErrPartitionMgmtOnNonpartitioned:            Message("Partition management on a not partitioned table is not possible", nil),
	ErrForeignKeyOnPartitioned:                  Message("Foreign key clause is not yet supported in conjunction with partitioning", nil),
	ErrDropPartitionNonExistent:                 Message("Error in list of partitions to %-.64s", nil),
	ErrDropLastPartition:                        Message("Cannot remove all partitions, use DROP TABLE instead", nil),
	ErrCoalesceOnlyOnHashPartition:              Message("COALESCE PARTITION can only be used on HASH/KEY partitions", nil),
	ErrReorgHashOnlyOnSameNo:                    Message("REORGANIZE PARTITION can only be used to reorganize partitions not to change their numbers", nil),
	ErrReorgNoParam:                             Message("REORGANIZE PARTITION without parameters can only be used on auto-partitioned tables using HASH PARTITIONs", nil),
	ErrOnlyOnRangeListPartition:                 Message("%-.64s PARTITION can only be used on RANGE/LIST partitions", nil),
	ErrAddPartitionSubpart:                      Message("Trying to Add partition(s) with wrong number of subpartitions", nil),
	ErrAddPartitionNoNewPartition:               Message("At least one partition must be added", nil),
	ErrCoalescePartitionNoPartition:             Message("At least one partition must be coalesced", nil),
	ErrReorgPartitionNotExist:                   Message("More partitions to reorganize than there are partitions", nil),
	ErrSameNamePartition:                        Message("Duplicate partition name %-.192s", nil),
	ErrNoBinlog:                                 Message("It is not allowed to shut off binlog on this command", nil),
	ErrConsecutiveReorgPartitions:               Message("When reorganizing a set of partitions they must be in consecutive order", nil),
	ErrReorgOutsideRange:                        Message("Reorganize of range partitions cannot change total ranges except for last partition where it can extend the range", nil),
	ErrPartitionFunctionFailure:                 Message("Partition function not supported in this version for this handler", nil),
	ErrPartState:                                Message("Partition state cannot be defined from CREATE/ALTER TABLE", nil),
	ErrLimitedPartRange:                         Message("The %-.64s handler only supports 32 bit integers in VALUES", nil),
	ErrPluginIsNotLoaded:                        Message("Plugin '%-.192s' is not loaded", nil),
	ErrWrongValue:                               Message("Incorrect %-.32s value: '%-.128s'", nil),
	ErrNoPartitionForGivenValue:                 Message("Table has no partition for value %-.64s", nil),
	ErrFilegroupOptionOnlyOnce:                  Message("It is not allowed to specify %s more than once", nil),
	ErrCreateFilegroupFailed:                    Message("Failed to create %s", nil),
	ErrDropFilegroupFailed:                      Message("Failed to drop %s", nil),
	ErrTablespaceAutoExtend:                     Message("The handler doesn't support autoextend of tablespaces", nil),
	ErrWrongSizeNumber:                          Message("A size parameter was incorrectly specified, either number or on the form 10M", nil),
	ErrSizeOverflow:                             Message("The size number was correct but we don't allow the digit part to be more than 2 billion", nil),
	ErrAlterFilegroupFailed:                     Message("Failed to alter: %s", nil),
	ErrBinlogRowLoggingFailed:                   Message("Writing one row to the row-based binary log failed", nil),
	ErrBinlogRowWrongTableDef:                   Message("Table definition on master and slave does not match: %s", nil),
	ErrBinlogRowRbrToSbr:                        Message("Slave running with --log-slave-updates must use row-based binary logging to be able to replicate row-based binary log events", nil),
	ErrEventAlreadyExists:                       Message("Event '%-.192s' already exists", nil),
	ErrEventStoreFailed:                         Message("Failed to store event %s. Error code %d from storage engine.", nil),
	ErrEventDoesNotExist:                        Message("Unknown event '%-.192s'", nil),
	ErrEventCantAlter:                           Message("Failed to alter event '%-.192s'", nil),
	ErrEventDropFailed:                          Message("Failed to drop %s", nil),
	ErrEventIntervalNotPositiveOrTooBig:         Message("INTERVAL is either not positive or too big", nil),
	ErrEventEndsBeforeStarts:                    Message("ENDS is either invalid or before STARTS", nil),
	ErrEventExecTimeInThePast:                   Message("Event execution time is in the past. Event has been disabled", nil),
	ErrEventOpenTableFailed:                     Message("Failed to open mysql.event", nil),
	ErrEventNeitherMExprNorMAt:                  Message("No datetime expression provided", nil),
	ErrObsoleteColCountDoesntMatchCorrupted:     Message("Column count of mysql.%s is wrong. Expected %d, found %d. The table is probably corrupted", nil),
	ErrObsoleteCannotLoadFromTable:              Message("Cannot load from mysql.%s. The table is probably corrupted", nil),
	ErrEventCannotDelete:                        Message("Failed to delete the event from mysql.event", nil),
	ErrEventCompile:                             Message("Error during compilation of event's body", nil),
	ErrEventSameName:                            Message("Same old and new event name", nil),
	ErrEventDataTooLong:                         Message("Data for column '%s' too long", nil),
	ErrDropIndexNeededInForeignKey:              Message("Cannot drop index '%-.192s': needed in a foreign key constraint", nil),
	ErrWarnDeprecatedSyntaxWithVer:              Message("The syntax '%s' is deprecated and will be removed in MySQL %s. Please use %s instead", nil),
	ErrCantWriteLockLogTable:                    Message("You can't write-lock a log table. Only read access is possible", nil),
	ErrCantLockLogTable:                         Message("You can't use locks with log tables.", nil),
	ErrForeignDuplicateKeyOldUnused:             Message("Upholding foreign key constraints for table '%.192s', entry '%-.192s', key %d would lead to a duplicate entry", nil),
	ErrColCountDoesntMatchPleaseUpdate:          Message("Column count of mysql.%s is wrong. Expected %d, found %d. Created with MySQL %d, now running %d. Please use mysqlUpgrade to fix this error.", nil),
	ErrTempTablePreventsSwitchOutOfRbr:          Message("Cannot switch out of the row-based binary log format when the session has open temporary tables", nil),
	ErrStoredFunctionPreventsSwitchBinlogFormat: Message("Cannot change the binary logging format inside a stored function or trigger", nil),
	ErrNdbCantSwitchBinlogFormat:                Message("The NDB cluster engine does not support changing the binlog format on the fly yet", nil),
	ErrPartitionNoTemporary:                     Message("Cannot create temporary table with partitions", nil),
	ErrPartitionConstDomain:                     Message("Partition constant is out of partition function domain", nil),
	ErrPartitionFunctionIsNotAllowed:            Message("This partition function is not allowed", nil),
	ErrDdlLog:                                   Message("Error in DDL log", nil),
	ErrNullInValuesLessThan:                     Message("Not allowed to use NULL value in VALUES LESS THAN", nil),
	ErrWrongPartitionName:                       Message("Incorrect partition name", nil),
	ErrCantChangeTxCharacteristics:              Message("Transaction characteristics can't be changed while a transaction is in progress", nil),
	ErrDupEntryAutoincrementCase:                Message("ALTER TABLE causes autoIncrement resequencing, resulting in duplicate entry '%-.192s' for key '%-.192s'", nil),
	ErrEventModifyQueue:                         Message("Internal scheduler error %d", nil),
	ErrEventSetVar:                              Message("Error during starting/stopping of the scheduler. Error code %d", nil),
	ErrPartitionMerge:                           Message("Engine cannot be used in partitioned tables", nil),
	ErrCantActivateLog:                          Message("Cannot activate '%-.64s' log", nil),
	ErrRbrNotAvailable:                          Message("The server was not built with row-based replication", nil),
	ErrBase64Decode:                             Message("Decoding of base64 string failed", nil),
	ErrEventRecursionForbidden:                  Message("Recursion of EVENT DDL statements is forbidden when body is present", nil),
	ErrEventsDB:                                 Message("Cannot proceed because system tables used by Event Scheduler were found damaged at server start", nil),
	ErrOnlyIntegersAllowed:                      Message("Only integers allowed as number here", nil),
	ErrUnsuportedLogEngine:                      Message("This storage engine cannot be used for log tables\"", nil),
	ErrBadLogStatement:                          Message("You cannot '%s' a log table if logging is enabled", nil),
	ErrCantRenameLogTable:                       Message("Cannot rename '%s'. When logging enabled, rename to/from log table must rename two tables: the log table to an archive table and another table back to '%s'", nil),
	ErrWrongParamcountToNativeFct:               Message("Incorrect parameter count in the call to native function '%-.192s'", nil),
	ErrWrongParametersToNativeFct:               Message("Incorrect parameters in the call to native function '%-.192s'", nil),
	ErrWrongParametersToStoredFct:               Message("Incorrect parameters in the call to stored function '%-.192s'", nil),
	ErrNativeFctNameCollision:                   Message("This function '%-.192s' has the same name as a native function", nil),
	ErrDupEntryWithKeyName:                      Message("Duplicate entry '%-.64s' for key '%-.192s'", nil),
	ErrBinlogPurgeEmFile:                        Message("Too many files opened, please execute the command again", nil),
	ErrEventCannotCreateInThePast:               Message("Event execution time is in the past and ON COMPLETION NOT PRESERVE is set. The event was dropped immediately after creation.", nil),
	ErrEventCannotAlterInThePast:                Message("Event execution time is in the past and ON COMPLETION NOT PRESERVE is set. The event was not changed. Specify a time in the future.", nil),
	ErrSlaveIncident:                            Message("The incident %s occurred on the master. Message: %-.64s", nil),
	ErrNoPartitionForGivenValueSilent:           Message("Table has no partition for some existing values", nil),
	ErrBinlogUnsafeStatement:                    Message("Unsafe statement written to the binary log using statement format since BINLOGFORMAT = STATEMENT. %s", nil),
	ErrSlaveFatal:                               Message("Fatal : %s", nil),
	ErrSlaveRelayLogReadFailure:                 Message("Relay log read failure: %s", nil),
	ErrSlaveRelayLogWriteFailure:                Message("Relay log write failure: %s", nil),
	ErrSlaveCreateEventFailure:                  Message("Failed to create %s", nil),
	ErrSlaveMasterComFailure:                    Message("Master command %s failed: %s", nil),
	ErrBinlogLoggingImpossible:                  Message("Binary logging not possible. Message: %s", nil),
	ErrViewNoCreationCtx:                        Message("View `%-.64s`.`%-.64s` has no creation context", nil),
	ErrViewInvalidCreationCtx:                   Message("Creation context of view `%-.64s`.`%-.64s' is invalid", nil),
	ErrSrInvalidCreationCtx:                     Message("Creation context of stored routine `%-.64s`.`%-.64s` is invalid", nil),
	ErrTrgCorruptedFile:                         Message("Corrupted TRG file for table `%-.64s`.`%-.64s`", nil),
	ErrTrgNoCreationCtx:                         Message("Triggers for table `%-.64s`.`%-.64s` have no creation context", nil),
	ErrTrgInvalidCreationCtx:                    Message("Trigger creation context of table `%-.64s`.`%-.64s` is invalid", nil),
	ErrEventInvalidCreationCtx:                  Message("Creation context of event `%-.64s`.`%-.64s` is invalid", nil),
	ErrTrgCantOpenTable:                         Message("Cannot open table for trigger `%-.64s`.`%-.64s`", nil),
	ErrCantCreateSroutine:                       Message("Cannot create stored routine `%-.64s`. Check warnings", nil),
	ErrNeverUsed:                                Message("Ambiguous slave modes combination. %s", nil),
	ErrNoFormatDescriptionEventBeforeBinlogStatement:         Message("The BINLOG statement of type `%s` was not preceded by a format description BINLOG statement.", nil),
	ErrSlaveCorruptEvent:                                     Message("Corrupted replication event was detected", nil),
	ErrLoadDataInvalidColumn:                                 Message("Invalid column reference (%-.64s) in LOAD DATA", nil),
	ErrLogPurgeNoFile:                                        Message("Being purged log %s was not found", nil),
	ErrXaRbtimeout:                                           Message("XARBTIMEOUT: Transaction branch was rolled back: took too long", nil),
	ErrXaRbdeadlock:                                          Message("XARBDEADLOCK: Transaction branch was rolled back: deadlock was detected", nil),
	ErrNeedReprepare:                                         Message("Prepared statement needs to be re-prepared", nil),
	ErrDelayedNotSupported:                                   Message("DELAYED option not supported for table '%-.192s'", nil),
	WarnNoMasterInfo:                                         Message("The master info structure does not exist", nil),
	WarnOptionIgnored:                                        Message("<%-.64s> option ignored", nil),
	WarnPluginDeleteBuiltin:                                  Message("Built-in plugins cannot be deleted", nil),
	WarnPluginBusy:                                           Message("Plugin is busy and will be uninstalled on shutdown", nil),
	ErrVariableIsReadonly:                                    Message("%s variable '%s' is read-only. Use SET %s to assign the value", nil),
	ErrWarnEngineTransactionRollback:                         Message("Storage engine %s does not support rollback for this statement. Transaction rolled back and must be restarted", nil),
	ErrSlaveHeartbeatFailure:                                 Message("Unexpected master's heartbeat data: %s", nil),
	ErrSlaveHeartbeatValueOutOfRange:                         Message("The requested value for the heartbeat period is either negative or exceeds the maximum allowed (%s seconds).", nil),
	ErrNdbReplicationSchema:                                  Message("Bad schema for mysql.ndbReplication table. Message: %-.64s", nil),
	ErrConflictFnParse:                                       Message("Error in parsing conflict function. Message: %-.64s", nil),
	ErrExceptionsWrite:                                       Message("Write to exceptions table failed. Message: %-.128s\"", nil),
	ErrTooLongTableComment:                                   Message("Comment for table '%-.64s' is too long (max = %d)", nil),
	ErrTooLongFieldComment:                                   Message("Comment for field '%-.64s' is too long (max = %d)", nil),
	ErrFuncInexistentNameCollision:                           Message("FUNCTION %s does not exist. Check the 'Function Name Parsing and Resolution' section in the Reference Manual", nil),
	ErrDatabaseName:                                          Message("Database", nil),
	ErrTableName:                                             Message("Table", nil),
	ErrPartitionName:                                         Message("Partition", nil),
	ErrSubpartitionName:                                      Message("Subpartition", nil),
	ErrTemporaryName:                                         Message("Temporary", nil),
	ErrRenamedName:                                           Message("Renamed", nil),
	ErrTooManyConcurrentTrxs:                                 Message("Too many active concurrent transactions", nil),
	WarnNonASCIISeparatorNotImplemented:                      Message("Non-ASCII separator arguments are not fully supported", nil),
	ErrDebugSyncTimeout:                                      Message("debug sync point wait timed out", nil),
	ErrDebugSyncHitLimit:                                     Message("debug sync point hit limit reached", nil),
	ErrDupSignalSet:                                          Message("Duplicate condition information item '%s'", nil),
	ErrSignalWarn:                                            Message("Unhandled user-defined warning condition", nil),
	ErrSignalNotFound:                                        Message("Unhandled user-defined not found condition", nil),
	ErrSignalException:                                       Message("Unhandled user-defined exception condition", nil),
	ErrResignalWithoutActiveHandler:                          Message("RESIGNAL when handler not active", nil),
	ErrSignalBadConditionType:                                Message("SIGNAL/RESIGNAL can only use a CONDITION defined with SQLSTATE", nil),
	WarnCondItemTruncated:                                    Message("Data truncated for condition item '%s'", nil),
	ErrCondItemTooLong:                                       Message("Data too long for condition item '%s'", nil),
	ErrUnknownLocale:                                         Message("Unknown locale: '%-.64s'", nil),
	ErrSlaveIgnoreServerIds:                                  Message("The requested server id %d clashes with the slave startup option --replicate-same-server-id", nil),
	ErrQueryCacheDisabled:                                    Message("Query cache is disabled; restart the server with queryCacheType=1 to enable it", nil),
	ErrSameNamePartitionField:                                Message("Duplicate partition field name '%-.192s'", nil),
	ErrPartitionColumnList:                                   Message("Inconsistency in usage of column lists for partitioning", nil),
	ErrWrongTypeColumnValue:                                  Message("Partition column values of incorrect type", nil),
	ErrTooManyPartitionFuncFields:                            Message("Too many fields in '%-.192s'", nil),
	ErrMaxvalueInValuesIn:                                    Message("Cannot use MAXVALUE as value in VALUES IN", nil),
	ErrTooManyValues:                                         Message("Cannot have more than one value for this type of %-.64s partitioning", nil),
	ErrRowSinglePartitionField:                               Message("Row expressions in VALUES IN only allowed for multi-field column partitioning", nil),
	ErrFieldTypeNotAllowedAsPartitionField:                   Message("Field '%-.192s' is of a not allowed type for this type of partitioning", nil),
	ErrPartitionFieldsTooLong:                                Message("The total length of the partitioning fields is too large", nil),
	ErrBinlogRowEngineAndStmtEngine:                          Message("Cannot execute statement: impossible to write to binary log since both row-incapable engines and statement-incapable engines are involved.", nil),
	ErrBinlogRowModeAndStmtEngine:                            Message("Cannot execute statement: impossible to write to binary log since BINLOGFORMAT = ROW and at least one table uses a storage engine limited to statement-based logging.", nil),
	ErrBinlogUnsafeAndStmtEngine:                             Message("Cannot execute statement: impossible to write to binary log since statement is unsafe, storage engine is limited to statement-based logging, and BINLOGFORMAT = MIXED. %s", nil),
	ErrBinlogRowInjectionAndStmtEngine:                       Message("Cannot execute statement: impossible to write to binary log since statement is in row format and at least one table uses a storage engine limited to statement-based logging.", nil),
	ErrBinlogStmtModeAndRowEngine:                            Message("Cannot execute statement: impossible to write to binary log since BINLOGFORMAT = STATEMENT and at least one table uses a storage engine limited to row-based logging.%s", nil),
	ErrBinlogRowInjectionAndStmtMode:                         Message("Cannot execute statement: impossible to write to binary log since statement is in row format and BINLOGFORMAT = STATEMENT.", nil),
	ErrBinlogMultipleEnginesAndSelfLoggingEngine:             Message("Cannot execute statement: impossible to write to binary log since more than one engine is involved and at least one engine is self-logging.", nil),
	ErrBinlogUnsafeLimit:                                     Message("The statement is unsafe because it uses a LIMIT clause. This is unsafe because the set of rows included cannot be predicted.", nil),
	ErrBinlogUnsafeInsertDelayed:                             Message("The statement is unsafe because it uses INSERT DELAYED. This is unsafe because the times when rows are inserted cannot be predicted.", nil),
	ErrBinlogUnsafeSystemTable:                               Message("The statement is unsafe because it uses the general log, slow query log, or performanceSchema table(s). This is unsafe because system tables may differ on slaves.", nil),
	ErrBinlogUnsafeAutoincColumns:                            Message("Statement is unsafe because it invokes a trigger or a stored function that inserts into an AUTOINCREMENT column. Inserted values cannot be logged correctly.", nil),
	ErrBinlogUnsafeUdf:                                       Message("Statement is unsafe because it uses a UDF which may not return the same value on the slave.", nil),
	ErrBinlogUnsafeSystemVariable:                            Message("Statement is unsafe because it uses a system variable that may have a different value on the slave.", nil),
	ErrBinlogUnsafeSystemFunction:                            Message("Statement is unsafe because it uses a system function that may return a different value on the slave.", nil),
	ErrBinlogUnsafeNontransAfterTrans:                        Message("Statement is unsafe because it accesses a non-transactional table after accessing a transactional table within the same transaction.", nil),
	ErrMessageAndStatement:                                   Message("%s Statement: %s", nil),
	ErrSlaveConversionFailed:                                 Message("Column %d of table '%-.192s.%-.192s' cannot be converted from type '%-.32s' to type '%-.32s'", nil),
	ErrSlaveCantCreateConversion:                             Message("Can't create conversion table for table '%-.192s.%-.192s'", nil),
	ErrInsideTransactionPreventsSwitchBinlogFormat:           Message("Cannot modify @@session.binlogFormat inside a transaction", nil),
	ErrPathLength:                                            Message("The path specified for %.64s is too long.", nil),
	ErrWarnDeprecatedSyntaxNoReplacement:                     Message("%s is deprecated and will be removed in a future release.%s", nil),
	ErrWrongNativeTableStructure:                             Message("Native table '%-.64s'.'%-.64s' has the wrong structure", nil),
	ErrWrongPerfSchemaUsage:                                  Message("Invalid performanceSchema usage.", nil),
	ErrWarnISSkippedTable:                                    Message("Table '%s'.'%s' was skipped since its definition is being modified by concurrent DDL statement", nil),
	ErrInsideTransactionPreventsSwitchBinlogDirect:           Message("Cannot modify @@session.binlogDirectNonTransactionalUpdates inside a transaction", nil),
	ErrStoredFunctionPreventsSwitchBinlogDirect:              Message("Cannot change the binlog direct flag inside a stored function or trigger", nil),
	ErrSpatialMustHaveGeomCol:                                Message("A SPATIAL index may only contain a geometrical type column", nil),
	ErrTooLongIndexComment:                                   Message("Comment for index '%-.64s' is too long (max = %d)", nil),
	ErrLockAborted:                                           Message("Wait on a lock was aborted due to a pending exclusive lock", nil),
	ErrDataOutOfRange:                                        Message("%s value is out of range in '%s'", nil),
	ErrWrongSpvarTypeInLimit:                                 Message("A variable of a non-integer based type in LIMIT clause", nil),
	ErrBinlogUnsafeMultipleEnginesAndSelfLoggingEngine:       Message("Mixing self-logging and non-self-logging engines in a statement is unsafe.", nil),
	ErrBinlogUnsafeMixedStatement:                            Message("Statement accesses nontransactional table as well as transactional or temporary table, and writes to any of them.", nil),
	ErrInsideTransactionPreventsSwitchSQLLogBin:              Message("Cannot modify @@session.sqlLogBin inside a transaction", nil),
	ErrStoredFunctionPreventsSwitchSQLLogBin:                 Message("Cannot change the sqlLogBin inside a stored function or trigger", nil),
	ErrFailedReadFromParFile:                                 Message("Failed to read from the .par file", nil),
	ErrValuesIsNotIntType:                                    Message("VALUES value for partition '%-.64s' must have type INT", nil),
	ErrAccessDeniedNoPassword:                                Message("Access denied for user '%-.48s'@'%-.64s'", nil),
	ErrSetPasswordAuthPlugin:                                 Message("SET PASSWORD has no significance for user '%-.48s'@'%-.255s' as authentication plugin does not support it.", nil),
	ErrGrantPluginUserExists:                                 Message("GRANT with IDENTIFIED WITH is illegal because the user %-.*s already exists", nil),
	ErrTruncateIllegalForeignKey:                             Message("Cannot truncate a table referenced in a foreign key constraint (%.192s)", nil),
	ErrPluginIsPermanent:                                     Message("Plugin '%s' is forcePlusPermanent and can not be unloaded", nil),
	ErrSlaveHeartbeatValueOutOfRangeMin:                      Message("The requested value for the heartbeat period is less than 1 millisecond. The value is reset to 0, meaning that heartbeating will effectively be disabled.", nil),
	ErrSlaveHeartbeatValueOutOfRangeMax:                      Message("The requested value for the heartbeat period exceeds the value of `slaveNetTimeout' seconds. A sensible value for the period should be less than the timeout.", nil),
	ErrStmtCacheFull:                                         Message("Multi-row statements required more than 'maxBinlogStmtCacheSize' bytes of storage; increase this mysqld variable and try again", nil),
	ErrMultiUpdateKeyConflict:                                Message("Primary key/partition key update is not allowed since the table is updated both as '%-.192s' and '%-.192s'.", nil),
	ErrTableNeedsRebuild:                                     Message("Table rebuild required. Please do \"ALTER TABLE `%-.32s` FORCE\" or dump/reload to fix it!", nil),
	WarnOptionBelowLimit:                                     Message("The value of '%s' should be no less than the value of '%s'", nil),
	ErrIndexColumnTooLong:                                    Message("Index column size too large. The maximum column size is %d bytes.", nil),
	ErrErrorInTriggerBody:                                    Message("Trigger '%-.64s' has an error in its body: '%-.256s'", nil),
	ErrErrorInUnknownTriggerBody:                             Message("Unknown trigger has an error in its body: '%-.256s'", nil),
	ErrIndexCorrupt:                                          Message("Index %s is corrupted", nil),
	ErrUndoRecordTooBig:                                      Message("Undo log record is too big.", nil),
	ErrBinlogUnsafeInsertIgnoreSelect:                        Message("INSERT IGNORE... SELECT is unsafe because the order in which rows are retrieved by the SELECT determines which (if any) rows are ignored. This order cannot be predicted and may differ on master and the slave.", nil),
	ErrBinlogUnsafeInsertSelectUpdate:                        Message("INSERT... SELECT... ON DUPLICATE KEY UPDATE is unsafe because the order in which rows are retrieved by the SELECT determines which (if any) rows are updated. This order cannot be predicted and may differ on master and the slave.", nil),
	ErrBinlogUnsafeReplaceSelect:                             Message("REPLACE... SELECT is unsafe because the order in which rows are retrieved by the SELECT determines which (if any) rows are replaced. This order cannot be predicted and may differ on master and the slave.", nil),
	ErrBinlogUnsafeCreateIgnoreSelect:                        Message("CREATE... IGNORE SELECT is unsafe because the order in which rows are retrieved by the SELECT determines which (if any) rows are ignored. This order cannot be predicted and may differ on master and the slave.", nil),
	ErrBinlogUnsafeCreateReplaceSelect:                       Message("CREATE... REPLACE SELECT is unsafe because the order in which rows are retrieved by the SELECT determines which (if any) rows are replaced. This order cannot be predicted and may differ on master and the slave.", nil),
	ErrBinlogUnsafeUpdateIgnore:                              Message("UPDATE IGNORE is unsafe because the order in which rows are updated determines which (if any) rows are ignored. This order cannot be predicted and may differ on master and the slave.", nil),
	ErrPluginNoUninstall:                                     Message("Plugin '%s' is marked as not dynamically uninstallable. You have to stop the server to uninstall it.", nil),
	ErrPluginNoInstall:                                       Message("Plugin '%s' is marked as not dynamically installable. You have to stop the server to install it.", nil),
	ErrBinlogUnsafeWriteAutoincSelect:                        Message("Statements writing to a table with an auto-increment column after selecting from another table are unsafe because the order in which rows are retrieved determines what (if any) rows will be written. This order cannot be predicted and may differ on master and the slave.", nil),
	ErrBinlogUnsafeCreateSelectAutoinc:                       Message("CREATE TABLE... SELECT...  on a table with an auto-increment column is unsafe because the order in which rows are retrieved by the SELECT determines which (if any) rows are inserted. This order cannot be predicted and may differ on master and the slave.", nil),
	ErrBinlogUnsafeInsertTwoKeys:                             Message("INSERT... ON DUPLICATE KEY UPDATE  on a table with more than one UNIQUE KEY is unsafe", nil),
	ErrTableInFkCheck:                                        Message("Table is being used in foreign key check.", nil),
	ErrUnsupportedEngine:                                     Message("Storage engine '%s' does not support system tables. [%s.%s]", nil),
	ErrBinlogUnsafeAutoincNotFirst:                           Message("INSERT into autoincrement field which is not the first part in the composed primary key is unsafe.", nil),
	ErrCannotLoadFromTableV2:                                 Message("Cannot load from %s.%s. The table is probably corrupted", nil),
	ErrMasterDelayValueOutOfRange:                            Message("The requested value %d for the master delay exceeds the maximum %d", nil),
	ErrOnlyFdAndRbrEventsAllowedInBinlogStatement:            Message("Only FormatDescriptionLogEvent and row events are allowed in BINLOG statements (but %s was provided)", nil),
	ErrPartitionExchangeDifferentOption:                      Message("Non matching attribute '%-.64s' between partition and table", nil),
	ErrPartitionExchangePartTable:                            Message("Table to exchange with partition is partitioned: '%-.64s'", nil),
	ErrPartitionExchangeTempTable:                            Message("Table to exchange with partition is temporary: '%-.64s'", nil),
	ErrPartitionInsteadOfSubpartition:                        Message("Subpartitioned table, use subpartition instead of partition", nil),
	ErrUnknownPartition:                                      Message("Unknown partition '%-.64s' in table '%-.64s'", nil),
	ErrTablesDifferentMetadata:                               Message("Tables have different definitions", nil),
	ErrRowDoesNotMatchPartition:                              Message("Found a row that does not match the partition", nil),
	ErrBinlogCacheSizeGreaterThanMax:                         Message("Option binlogCacheSize (%d) is greater than maxBinlogCacheSize (%d); setting binlogCacheSize equal to maxBinlogCacheSize.", nil),
	ErrWarnIndexNotApplicable:                                Message("Cannot use %-.64s access on index '%-.64s' due to type or collation conversion on field '%-.64s'", nil),
	ErrPartitionExchangeForeignKey:                           Message("Table to exchange with partition has foreign key references: '%-.64s'", nil),
	ErrNoSuchKeyValue:                                        Message("Key value '%-.192s' was not found in table '%-.192s.%-.192s'", nil),
	ErrRplInfoDataTooLong:                                    Message("Data for column '%s' too long", nil),
	ErrNetworkReadEventChecksumFailure:                       Message("Replication event checksum verification failed while reading from network.", nil),
	ErrBinlogReadEventChecksumFailure:                        Message("Replication event checksum verification failed while reading from a log file.", nil),
	ErrBinlogStmtCacheSizeGreaterThanMax:                     Message("Option binlogStmtCacheSize (%d) is greater than maxBinlogStmtCacheSize (%d); setting binlogStmtCacheSize equal to maxBinlogStmtCacheSize.", nil),
	ErrCantUpdateTableInCreateTableSelect:                    Message("Can't update table '%-.192s' while '%-.192s' is being created.", nil),
	ErrPartitionClauseOnNonpartitioned:                       Message("PARTITION () clause on non partitioned table", nil),
	ErrRowDoesNotMatchGivenPartitionSet:                      Message("Found a row not matching the given partition set", nil),
	ErrNoSuchPartitionunused:                                 Message("partition '%-.64s' doesn't exist", nil),
	ErrChangeRplInfoRepositoryFailure:                        Message("Failure while changing the type of replication repository: %s.", nil),
	ErrWarningNotCompleteRollbackWithCreatedTempTable:        Message("The creation of some temporary tables could not be rolled back.", nil),
	ErrWarningNotCompleteRollbackWithDroppedTempTable:        Message("Some temporary tables were dropped, but these operations could not be rolled back.", nil),
	ErrMtsFeatureIsNotSupported:                              Message("%s is not supported in multi-threaded slave mode. %s", nil),
	ErrMtsUpdatedDBsGreaterMax:                               Message("The number of modified databases exceeds the maximum %d; the database names will not be included in the replication event metadata.", nil),
	ErrMtsCantParallel:                                       Message("Cannot execute the current event group in the parallel mode. Encountered event %s, relay-log name %s, position %s which prevents execution of this event group in parallel mode. Reason: %s.", nil),
	ErrMtsInconsistentData:                                   Message("%s", nil),
	ErrFulltextNotSupportedWithPartitioning:                  Message("FULLTEXT index is not supported for partitioned tables.", nil),
	ErrDaInvalidConditionNumber:                              Message("Invalid condition number", nil),
	ErrInsecurePlainText:                                     Message("Sending passwords in plain text without SSL/TLS is extremely insecure.", nil),
	ErrInsecureChangeMaster:                                  Message("Storing MySQL user name or password information in the master.info repository is not secure and is therefore not recommended. Please see the MySQL Manual for more about this issue and possible alternatives.", nil),
	ErrForeignDuplicateKeyWithChildInfo:                      Message("Foreign key constraint for table '%.192s', record '%-.192s' would lead to a duplicate entry in table '%.192s', key '%.192s'", nil),
	ErrForeignDuplicateKeyWithoutChildInfo:                   Message("Foreign key constraint for table '%.192s', record '%-.192s' would lead to a duplicate entry in a child table", nil),
	ErrSQLthreadWithSecureSlave:                              Message("Setting authentication options is not possible when only the Slave SQL Thread is being started.", nil),
	ErrTableHasNoFt:                                          Message("The table does not have FULLTEXT index to support this query", nil),
	ErrVariableNotSettableInSfOrTrigger:                      Message("The system variable %.200s cannot be set in stored functions or triggers.", nil),
	ErrVariableNotSettableInTransaction:                      Message("The system variable %.200s cannot be set when there is an ongoing transaction.", nil),
	ErrGtidNextIsNotInGtidNextList:                           Message("The system variable @@SESSION.GTIDNEXT has the value %.200s, which is not listed in @@SESSION.GTIDNEXTLIST.", nil),
	ErrCantChangeGtidNextInTransactionWhenGtidNextListIsNull: Message("When @@SESSION.GTIDNEXTLIST == NULL, the system variable @@SESSION.GTIDNEXT cannot change inside a transaction.", nil),
	ErrSetStatementCannotInvokeFunction:                      Message("The statement 'SET %.200s' cannot invoke a stored function.", nil),
	ErrGtidNextCantBeAutomaticIfGtidNextListIsNonNull:        Message("The system variable @@SESSION.GTIDNEXT cannot be 'AUTOMATIC' when @@SESSION.GTIDNEXTLIST is non-NULL.", nil),
	ErrSkippingLoggedTransaction:                             Message("Skipping transaction %.200s because it has already been executed and logged.", nil),
	ErrMalformedGtidSetSpecification:                         Message("Malformed GTID set specification '%.200s'.", nil),
	ErrMalformedGtidSetEncoding:                              Message("Malformed GTID set encoding.", nil),
	ErrMalformedGtidSpecification:                            Message("Malformed GTID specification '%.200s'.", nil),
	ErrGnoExhausted:                                          Message("Impossible to generate Global Transaction Identifier: the integer component reached the maximal value. Restart the server with a new serverUuid.", nil),
	ErrBadSlaveAutoPosition:                                  Message("Parameters MASTERLOGFILE, MASTERLOGPOS, RELAYLOGFILE and RELAYLOGPOS cannot be set when MASTERAUTOPOSITION is active.", nil),
	ErrAutoPositionRequiresGtidModeOn:                        Message("CHANGE MASTER TO MASTERAUTOPOSITION = 1 can only be executed when @@GLOBAL.GTIDMODE = ON.", nil),
	ErrCantDoImplicitCommitInTrxWhenGtidNextIsSet:            Message("Cannot execute statements with implicit commit inside a transaction when @@SESSION.GTIDNEXT != AUTOMATIC or @@SESSION.GTIDNEXTLIST != NULL.", nil),
	ErrGtidMode2Or3RequiresEnforceGtidConsistencyOn:          Message("@@GLOBAL.GTIDMODE = ON or UPGRADESTEP2 requires @@GLOBAL.ENFORCEGTIDCONSISTENCY = 1.", nil),
	ErrGtidModeRequiresBinlog:                                Message("@@GLOBAL.GTIDMODE = ON or UPGRADESTEP1 or UPGRADESTEP2 requires --log-bin and --log-slave-updates.", nil),
	ErrCantSetGtidNextToGtidWhenGtidModeIsOff:                Message("@@SESSION.GTIDNEXT cannot be set to UUID:NUMBER when @@GLOBAL.GTIDMODE = OFF.", nil),
	ErrCantSetGtidNextToAnonymousWhenGtidModeIsOn:            Message("@@SESSION.GTIDNEXT cannot be set to ANONYMOUS when @@GLOBAL.GTIDMODE = ON.", nil),
	ErrCantSetGtidNextListToNonNullWhenGtidModeIsOff:         Message("@@SESSION.GTIDNEXTLIST cannot be set to a non-NULL value when @@GLOBAL.GTIDMODE = OFF.", nil),
	ErrFoundGtidEventWhenGtidModeIsOff:                       Message("Found a GtidLogEvent or PreviousGtidsLogEvent when @@GLOBAL.GTIDMODE = OFF.", nil),
	ErrGtidUnsafeNonTransactionalTable:                       Message("When @@GLOBAL.ENFORCEGTIDCONSISTENCY = 1, updates to non-transactional tables can only be done in either autocommitted statements or single-statement transactions, and never in the same statement as updates to transactional tables.", nil),
	ErrGtidUnsafeCreateSelect:                                Message("CREATE TABLE ... SELECT is forbidden when @@GLOBAL.ENFORCEGTIDCONSISTENCY = 1.", nil),
	ErrGtidUnsafeCreateDropTemporaryTableInTransaction:       Message("When @@GLOBAL.ENFORCEGTIDCONSISTENCY = 1, the statements CREATE TEMPORARY TABLE and DROP TEMPORARY TABLE can be executed in a non-transactional context only, and require that AUTOCOMMIT = 1.", nil),
	ErrGtidModeCanOnlyChangeOneStepAtATime:                   Message("The value of @@GLOBAL.GTIDMODE can only change one step at a time: OFF <-> UPGRADESTEP1 <-> UPGRADESTEP2 <-> ON. Also note that this value must be stepped up or down simultaneously on all servers; see the Manual for instructions.", nil),
	ErrMasterHasPurgedRequiredGtids:                          Message("The slave is connecting using CHANGE MASTER TO MASTERAUTOPOSITION = 1, but the master has purged binary logs containing GTIDs that the slave requires.", nil),
	ErrCantSetGtidNextWhenOwningGtid:                         Message("@@SESSION.GTIDNEXT cannot be changed by a client that owns a GTID. The client owns %s. Ownership is released on COMMIT or ROLLBACK.", nil),
	ErrUnknownExplainFormat:                                  Message("Unknown EXPLAIN format name: '%s'", nil),
	ErrCantExecuteInReadOnlyTransaction:                      Message("Cannot execute statement in a READ ONLY transaction.", nil),
	ErrTooLongTablePartitionComment:                          Message("Comment for table partition '%-.64s' is too long (max = %d)", nil),
	ErrSlaveConfiguration:                                    Message("Slave is not configured or failed to initialize properly. You must at least set --server-id to enable either a master or a slave. Additional error messages can be found in the MySQL error log.", nil),
	ErrInnodbFtLimit:                                         Message("InnoDB presently supports one FULLTEXT index creation at a time", nil),
	ErrInnodbNoFtTempTable:                                   Message("Cannot create FULLTEXT index on temporary InnoDB table", nil),
	ErrInnodbFtWrongDocidColumn:                              Message("Column '%-.192s' is of wrong type for an InnoDB FULLTEXT index", nil),
	ErrInnodbFtWrongDocidIndex:                               Message("Index '%-.192s' is of wrong type for an InnoDB FULLTEXT index", nil),
	ErrInnodbOnlineLogTooBig:                                 Message("Creating index '%-.192s' required more than 'innodbOnlineAlterLogMaxSize' bytes of modification log. Please try again.", nil),
	ErrUnknownAlterAlgorithm:                                 Message("Unknown ALGORITHM '%s'", nil),
	ErrUnknownAlterLock:                                      Message("Unknown LOCK type '%s'", nil),
	ErrMtsChangeMasterCantRunWithGaps:                        Message("CHANGE MASTER cannot be executed when the slave was stopped with an error or killed in MTS mode. Consider using RESET SLAVE or START SLAVE UNTIL.", nil),
	ErrMtsRecoveryFailure:                                    Message("Cannot recover after SLAVE errored out in parallel execution mode. Additional error messages can be found in the MySQL error log.", nil),
	ErrMtsResetWorkers:                                       Message("Cannot clean up worker info tables. Additional error messages can be found in the MySQL error log.", nil),
	ErrColCountDoesntMatchCorruptedV2:                        Message("Column count of %s.%s is wrong. Expected %d, found %d. The table is probably corrupted", nil),
	ErrSlaveSilentRetryTransaction:                           Message("Slave must silently retry current transaction", nil),
	ErrDiscardFkChecksRunning:                                Message("There is a foreign key check running on table '%-.192s'. Cannot discard the table.", nil),
	ErrTableSchemaMismatch:                                   Message("Schema mismatch (%s)", nil),
	ErrTableInSystemTablespace:                               Message("Table '%-.192s' in system tablespace", nil),
	ErrIoRead:                                                Message("IO Read : (%d, %s) %s", nil),
	ErrIoWrite:                                               Message("IO Write : (%d, %s) %s", nil),
	ErrTablespaceMissing:                                     Message("Tablespace is missing for table '%-.192s'", nil),
	ErrTablespaceExists:                                      Message("Tablespace for table '%-.192s' exists. Please DISCARD the tablespace before IMPORT.", nil),
	ErrTablespaceDiscarded:                                   Message("Tablespace has been discarded for table '%-.192s'", nil),
	ErrInternal:                                              Message("Internal : %s", nil),
	ErrInnodbImport:                                          Message("ALTER TABLE '%-.192s' IMPORT TABLESPACE failed with error %d : '%s'", nil),
	ErrInnodbIndexCorrupt:                                    Message("Index corrupt: %s", nil),
	ErrInvalidYearColumnLength:                               Message("Supports only YEAR or YEAR(4) column", nil),
	ErrNotValidPassword:                                      Message("Your password does not satisfy the current policy requirements (%s)", nil),
	ErrMustChangePassword:                                    Message("You must SET PASSWORD before executing this statement", nil),
	ErrFkNoIndexChild:                                        Message("Failed to add the foreign key constraint. Missing index for constraint '%s' in the foreign table '%s'", nil),
	ErrForeignKeyNoIndexInParent:                             Message("Failed to add the foreign key constraint. Missing index for constraint '%s' in the referenced table '%s'", nil),
	ErrFkFailAddSystem:                                       Message("Failed to add the foreign key constraint '%s' to system tables", nil),
	ErrForeignKeyCannotOpenParent:                            Message("Failed to open the referenced table '%s'", nil),
	ErrFkIncorrectOption:                                     Message("Failed to add the foreign key constraint on table '%s'. Incorrect options in FOREIGN KEY constraint '%s'", nil),
	ErrFkDupName:                                             Message("Duplicate foreign key constraint name '%s'", nil),
	ErrPasswordFormat:                                        Message("The password hash doesn't have the expected format. Check if the correct password algorithm is being used with the PASSWORD() function.", nil),
	ErrFkColumnCannotDrop:                                    Message("Cannot drop column '%-.192s': needed in a foreign key constraint '%-.192s'", nil),
	ErrFkColumnCannotDropChild:                               Message("Cannot drop column '%-.192s': needed in a foreign key constraint '%-.192s' of table '%-.192s'", nil),
	ErrForeignKeyColumnNotNull:                               Message("Column '%-.192s' cannot be NOT NULL: needed in a foreign key constraint '%-.192s' SET NULL", nil),
	ErrDupIndex:                                              Message("Duplicate index '%-.64s' defined on the table '%-.64s.%-.64s'. This is deprecated and will be disallowed in a future release.", nil),
	ErrForeignKeyColumnCannotChange:                          Message("Cannot change column '%-.192s': used in a foreign key constraint '%-.192s'", nil),
	ErrForeignKeyColumnCannotChangeChild:                     Message("Cannot change column '%-.192s': used in a foreign key constraint '%-.192s' of table '%-.192s'", nil),
	ErrFkCannotDeleteParent:                                  Message("Cannot delete rows from table which is parent in a foreign key constraint '%-.192s' of table '%-.192s'", nil),
	ErrMalformedPacket:                                       Message("Malformed communication packet.", nil),
	ErrReadOnlyMode:                                          Message("Running in read-only mode", nil),
	ErrGtidNextTypeUndefinedGroup:                            Message("When @@SESSION.GTIDNEXT is set to a GTID, you must explicitly set it again after a COMMIT or ROLLBACK. If you see this error message in the slave SQL thread, it means that a table in the current transaction is transactional on the master and non-transactional on the slave. In a client connection, it means that you executed SET @@SESSION.GTIDNEXT before a transaction and forgot to set @@SESSION.GTIDNEXT to a different identifier or to 'AUTOMATIC' after COMMIT or ROLLBACK. Current @@SESSION.GTIDNEXT is '%s'.", nil),
	ErrVariableNotSettableInSp:                               Message("The system variable %.200s cannot be set in stored procedures.", nil),
	ErrCantSetGtidPurgedWhenGtidModeIsOff:                    Message("@@GLOBAL.GTIDPURGED can only be set when @@GLOBAL.GTIDMODE = ON.", nil),
	ErrCantSetGtidPurgedWhenGtidExecutedIsNotEmpty:           Message("@@GLOBAL.GTIDPURGED can only be set when @@GLOBAL.GTIDEXECUTED is empty.", nil),
	ErrCantSetGtidPurgedWhenOwnedGtidsIsNotEmpty:             Message("@@GLOBAL.GTIDPURGED can only be set when there are no ongoing transactions (not even in other clients).", nil),
	ErrGtidPurgedWasChanged:                                  Message("@@GLOBAL.GTIDPURGED was changed from '%s' to '%s'.", nil),
	ErrGtidExecutedWasChanged:                                Message("@@GLOBAL.GTIDEXECUTED was changed from '%s' to '%s'.", nil),
	ErrBinlogStmtModeAndNoReplTables:                         Message("Cannot execute statement: impossible to write to binary log since BINLOGFORMAT = STATEMENT, and both replicated and non replicated tables are written to.", nil),
	ErrAlterOperationNotSupported:                            Message("%s is not supported for this operation. Try %s.", nil),
	ErrAlterOperationNotSupportedReason:                      Message("%s is not supported. Reason: %s. Try %s.", nil),
	ErrAlterOperationNotSupportedReasonCopy:                  Message("COPY algorithm requires a lock", nil),
	ErrAlterOperationNotSupportedReasonPartition:             Message("Partition specific operations do not yet support LOCK/ALGORITHM", nil),
	ErrAlterOperationNotSupportedReasonFkRename:              Message("Columns participating in a foreign key are renamed", nil),
	ErrAlterOperationNotSupportedReasonColumnType:            Message("Cannot change column type INPLACE", nil),
	ErrAlterOperationNotSupportedReasonFkCheck:               Message("Adding foreign keys needs foreignKeyChecks=OFF", nil),
	ErrAlterOperationNotSupportedReasonIgnore:                Message("Creating unique indexes with IGNORE requires COPY algorithm to remove duplicate rows", nil),
	ErrAlterOperationNotSupportedReasonNopk:                  Message("Dropping a primary key is not allowed without also adding a new primary key", nil),
	ErrAlterOperationNotSupportedReasonAutoinc:               Message("Adding an auto-increment column requires a lock", nil),
	ErrAlterOperationNotSupportedReasonHiddenFts:             Message("Cannot replace hidden FTSDOCID with a user-visible one", nil),
	ErrAlterOperationNotSupportedReasonChangeFts:             Message("Cannot drop or rename FTSDOCID", nil),
	ErrAlterOperationNotSupportedReasonFts:                   Message("Fulltext index creation requires a lock", nil),
	ErrSQLSlaveSkipCounterNotSettableInGtidMode:              Message("sqlSlaveSkipCounter can not be set when the server is running with @@GLOBAL.GTIDMODE = ON. Instead, for each transaction that you want to skip, generate an empty transaction with the same GTID as the transaction", nil),
	ErrDupUnknownInIndex:                                     Message("Duplicate entry for key '%-.192s'", nil),
	ErrIdentCausesTooLongPath:                                Message("Long database name and identifier for object resulted in path length exceeding %d characters. Path: '%s'.", nil),
	ErrAlterOperationNotSupportedReasonNotNull:               Message("cannot silently convert NULL values, as required in this SQLMODE", nil),
	ErrMustChangePasswordLogin:                               Message("Your password has expired. To log in you must change it using a client that supports expired passwords.", nil),
	ErrRowInWrongPartition:                                   Message("Found a row in wrong partition %s", nil),
	ErrGeneratedColumnFunctionIsNotAllowed:                   Message("Expression of generated column '%s' contains a disallowed function.", nil),
	ErrUnsupportedAlterInplaceOnVirtualColumn:                Message("INPLACE ADD or DROP of virtual columns cannot be combined with other ALTER TABLE actions.", nil),
	ErrWrongFKOptionForGeneratedColumn:                       Message("Cannot define foreign key with %s clause on a generated column.", nil),
	ErrBadGeneratedColumn:                                    Message("The value specified for generated column '%s' in table '%s' is not allowed.", nil),
	ErrUnsupportedOnGeneratedColumn:                          Message("'%s' is not supported for generated columns.", nil),
	ErrGeneratedColumnNonPrior:                               Message("Generated column can refer only to generated columns defined prior to it.", nil),
	ErrDependentByGeneratedColumn:                            Message("Column '%s' has a generated column dependency.", nil),
	ErrGeneratedColumnRefAutoInc:                             Message("Generated column '%s' cannot refer to auto-increment column.", nil),
	ErrInvalidFieldSize:                                      Message("Invalid size for column '%s'.", nil),
	ErrPasswordExpireAnonymousUser:                           Message("The password for anonymous user cannot be expired.", nil),
	ErrIncorrectType:                                         Message("Incorrect type for argument %s in function %s.", nil),
	ErrInvalidJSONData:                                       Message("Invalid JSON data provided to function %s: %s", nil),
	ErrInvalidJSONText:                                       Message("Invalid JSON text: %-.192s", nil),
	ErrInvalidJSONPath:                                       Message("Invalid JSON path expression %s.", nil),
	ErrInvalidTypeForJSON:                                    Message("Invalid data type for JSON data in argument %d to function %s; a JSON string or JSON type is required.", nil),
	ErrInvalidJSONPathWildcard:                               Message("In this situation, path expressions may not contain the * and ** tokens.", nil),
	ErrInvalidJSONContainsPathType:                           Message("The second argument can only be either 'one' or 'all'.", nil),
	ErrJSONUsedAsKey:                                         Message("JSON column '%-.192s' cannot be used in key specification.", nil),
	ErrJSONDocumentNULLKey:                                   Message("JSON documents may not contain NULL member names.", nil),
	ErrBadUser:                                               Message("User %s does not exist.", nil),
	ErrUserAlreadyExists:                                     Message("User %s already exists.", nil),
	ErrInvalidJSONPathArrayCell:                              Message("A path expression is not a path to a cell in an array.", nil),
	ErrInvalidEncryptionOption:                               Message("Invalid encryption option.", nil),
	ErrWindowNoSuchWindow:                                    Message("Window name '%s' is not defined.", nil),
	ErrWindowCircularityInWindowGraph:                        Message("There is a circularity in the window dependency graph.", nil),
	ErrWindowNoChildPartitioning:                             Message("A window which depends on another cannot define partitioning.", nil),
	ErrWindowNoInherentFrame:                                 Message("Window '%s' has a frame definition, so cannot be referenced by another window.", nil),
	ErrWindowNoRedefineOrderBy:                               Message("Window '%s' cannot inherit '%s' since both contain an ORDER BY clause.", nil),
	ErrWindowFrameStartIllegal:                               Message("Window '%s': frame start cannot be UNBOUNDED FOLLOWING.", nil),
	ErrWindowFrameEndIllegal:                                 Message("Window '%s': frame end cannot be UNBOUNDED PRECEDING.", nil),
	ErrWindowFrameIllegal:                                    Message("Window '%s': frame start or end is negative, NULL or of non-integral type", nil),
	ErrWindowRangeFrameOrderType:                             Message("Window '%s' with RANGE N PRECEDING/FOLLOWING frame requires exactly one ORDER BY expression, of numeric or temporal type", nil),
	ErrWindowRangeFrameTemporalType:                          Message("Window '%s' with RANGE frame has ORDER BY expression of datetime type. Only INTERVAL bound value allowed.", nil),
	ErrWindowRangeFrameNumericType:                           Message("Window '%s' with RANGE frame has ORDER BY expression of numeric type, INTERVAL bound value not allowed.", nil),
	ErrWindowRangeBoundNotConstant:                           Message("Window '%s' has a non-constant frame bound.", nil),
	ErrWindowDuplicateName:                                   Message("Window '%s' is defined twice.", nil),
	ErrWindowIllegalOrderBy:                                  Message("Window '%s': ORDER BY or PARTITION BY uses legacy position indication which is not supported, use expression.", nil),
	ErrWindowInvalidWindowFuncUse:                            Message("You cannot use the window function '%s' in this context.'", nil),
	ErrWindowInvalidWindowFuncAliasUse:                       Message("You cannot use the alias '%s' of an expression containing a window function in this context.'", nil),
	ErrWindowNestedWindowFuncUseInWindowSpec:                 Message("You cannot nest a window function in the specification of window '%s'.", nil),
	ErrWindowRowsIntervalUse:                                 Message("Window '%s': INTERVAL can only be used with RANGE frames.", nil),
	ErrWindowNoGroupOrderUnused:                              Message("ASC or DESC with GROUP BY isn't allowed with window functions; put ASC or DESC in ORDER BY", nil),
	ErrWindowExplainJson:                                     Message("To get information about window functions use EXPLAIN FORMAT=JSON", nil),
	ErrWindowFunctionIgnoresFrame:                            Message("Window function '%s' ignores the frame clause of window '%s' and aggregates over the whole partition", nil),
	ErrRoleNotGranted:                                        Message("%s is not granted to %s", nil),
	ErrMaxExecTimeExceeded:                                   Message("Query execution was interrupted, maximum statement execution time exceeded", nil),
	ErrLockAcquireFailAndNoWaitSet:                           Message("Statement aborted because lock(s) could not be acquired immediately and NOWAIT is set.", nil),
	ErrDataTruncatedFunctionalIndex:                          Message("Data truncated for functional index '%s' at row %d", nil),
	ErrDataOutOfRangeFunctionalIndex:                         Message("Value is out of range for functional index '%s' at row %d", nil),
	ErrFunctionalIndexOnJsonOrGeometryFunction:               Message("Cannot create a functional index on a function that returns a JSON or GEOMETRY value", nil),
	ErrFunctionalIndexRefAutoIncrement:                       Message("Functional index '%s' cannot refer to an auto-increment column", nil),
	ErrCannotDropColumnFunctionalIndex:                       Message("Cannot drop column '%s' because it is used by a functional index. In order to drop the column, you must remove the functional index", nil),
	ErrFunctionalIndexPrimaryKey:                             Message("The primary key cannot be a functional index", nil),
	ErrFunctionalIndexOnLob:                                  Message("Cannot create a functional index on an expression that returns a BLOB or TEXT. Please consider using CAST", nil),
	ErrFunctionalIndexFunctionIsNotAllowed:                   Message("Expression of functional index '%s' contains a disallowed function", nil),
	ErrFulltextFunctionalIndex:                               Message("Fulltext functional index is not supported", nil),
	ErrSpatialFunctionalIndex:                                Message("Spatial functional index is not supported", nil),
	ErrWrongKeyColumnFunctionalIndex:                         Message("The used storage engine cannot index the expression '%s'", nil),
	ErrFunctionalIndexOnField:                                Message("Functional index on a column is not supported. Consider using a regular index instead", nil),
	ErrFKIncompatibleColumns:                                 Message("Referencing column '%s' and referenced column '%s' in foreign key constraint '%s' are incompatible.", nil),
	ErrFunctionalIndexRowValueIsNotAllowed:                   Message("Expression of functional index '%s' cannot refer to a row value", nil),
	ErrDependentByFunctionalIndex:                            Message("Column '%s' has a functional index dependency and cannot be dropped or renamed", nil),
	ErrInvalidJsonValueForFuncIndex:                          Message("Invalid JSON value for CAST for functional index '%s'", nil),
	ErrJsonValueOutOfRangeForFuncIndex:                       Message("Out of range JSON value for CAST for functional index '%s'", nil),
	ErrFunctionalIndexDataIsTooLong:                          Message("Data too long for functional index '%s'", nil),
	ErrFunctionalIndexNotApplicable:                          Message("Cannot use functional index '%s' due to type or collation conversion", nil),

	// MariaDB errors.
	ErrOnlyOneDefaultPartionAllowed:         Message("Only one DEFAULT partition allowed", nil),
	ErrWrongPartitionTypeExpectedSystemTime: Message("Wrong partitioning type, expected type: `SYSTEM_TIME`", nil),
	ErrSystemVersioningWrongPartitions:      Message("Wrong Partitions: must have at least one HISTORY and exactly one last CURRENT", nil),
	ErrSequenceRunOut:                       Message("Sequence '%-.64s.%-.64s' has run out", nil),
	ErrSequenceInvalidData:                  Message("Sequence '%-.64s.%-.64s' values are conflicting", nil),
	ErrSequenceAccessFail:                   Message("Sequence '%-.64s.%-.64s' access error", nil),
	ErrNotSequence:                          Message("'%-.64s.%-.64s' is not a SEQUENCE", nil),
	ErrUnknownSequence:                      Message("Unknown SEQUENCE: '%-.300s'", nil),
	ErrWrongInsertIntoSequence:              Message("Wrong INSERT into a SEQUENCE. One can only do single table INSERT into a sequence object (like with mysqldump).  If you want to change the SEQUENCE, use ALTER SEQUENCE instead.", nil),
	ErrSequenceInvalidTableStructure:        Message("Sequence '%-.64s.%-.64s' table structure is invalid (%s)", nil),

	// TiDB errors.
	ErrWarnOptimizerHintInvalidInteger:  Message("integer value is out of range in '%s'", nil),
	ErrWarnOptimizerHintUnsupportedHint: Message("Optimizer hint %s is not supported by TiDB and is ignored", nil),
	ErrWarnOptimizerHintInvalidToken:    Message("Cannot use %s '%s' (tok = %d) in an optimizer hint", nil),
	ErrWarnMemoryQuotaOverflow:          Message("Max value of MEMORY_QUOTA is %d bytes, ignore this invalid limit", nil),
	ErrWarnOptimizerHintParseError:      Message("Optimizer hint syntax error at %v", nil),
	ErrWarnOptimizerHintWrongPos:        Message("Optimizer hint can only be followed by certain keywords like SELECT, INSERT, etc.", nil),
}
