package constants

type CipherSuite struct {
	IANAName     string     `json:"iana_name"`
	GnuTLSName   string     `json:"gnutls_name"`
	NSSName      string     `json:"nss_name"`
	Protocol     string     `json:"protocol"`
	ProtocolCode uint64     `json:"protocol_code"`
	Kx           string     `json:"kx"`
	Au           string     `json:"au"`
	Enc          Encryption `json:"encryption"`
	Mac          string     `json:"mac"`
	Code         uint64     `json:"code"`
}

type Encryption struct {
	Cipher string `json:"cipher"`
	Bits   int    `json:"bits"`
}

var CipherSuites = map[string]CipherSuite{
	"AES128-GCM-SHA256": CipherSuite{
		IANAName:   "TLS_RSA_WITH_AES_128_GCM_SHA256",
		GnuTLSName: "TLS_RSA_AES_128_GCM_SHA256",
		NSSName:    "TLS_RSA_WITH_AES_128_GCM_SHA256",
		Protocol:   "TLSv1.2", ProtocolCode: 771,
		Kx: "RSA",
		Au: "RSA",
		Enc: Encryption{
			Cipher: "AESGCM",
			Bits:   128,
		},
		Mac:  "AEAD",
		Code: 156,
	},
	"AES128-SHA": CipherSuite{
		IANAName:   "TLS_RSA_WITH_AES_128_CBC_SHA",
		GnuTLSName: "TLS_RSA_AES_128_CBC_SHA1",
		NSSName:    "TLS_RSA_WITH_AES_128_CBC_SHA",
		Protocol:   "SSLv3", ProtocolCode: 768,
		Kx: "RSA",
		Au: "RSA",
		Enc: Encryption{
			Cipher: "AES",
			Bits:   128,
		},
		Mac:  "SHA1",
		Code: 47,
	},
	"AES128-SHA256": CipherSuite{
		IANAName:   "TLS_RSA_WITH_AES_128_CBC_SHA256",
		GnuTLSName: "TLS_RSA_AES_128_CBC_SHA256",
		NSSName:    "TLS_RSA_WITH_AES_128_CBC_SHA256",
		Protocol:   "TLSv1.2", ProtocolCode: 771,
		Kx: "RSA",
		Au: "RSA",
		Enc: Encryption{
			Cipher: "AES",
			Bits:   128,
		},
		Mac:  "SHA256",
		Code: 60,
	},
	"AES256-GCM-SHA384": CipherSuite{
		IANAName:   "TLS_RSA_WITH_AES_256_GCM_SHA384",
		GnuTLSName: "TLS_RSA_AES_256_GCM_SHA384",
		NSSName:    "TLS_RSA_WITH_AES_256_GCM_SHA384",
		Protocol:   "TLSv1.2", ProtocolCode: 771,
		Kx: "RSA",
		Au: "RSA",
		Enc: Encryption{
			Cipher: "AESGCM",
			Bits:   256,
		},
		Mac:  "AEAD",
		Code: 157,
	},
	"AES256-SHA": CipherSuite{
		IANAName:   "TLS_RSA_WITH_AES_256_CBC_SHA",
		GnuTLSName: "TLS_RSA_AES_256_CBC_SHA1",
		NSSName:    "TLS_RSA_WITH_AES_256_CBC_SHA",
		Protocol:   "SSLv3", ProtocolCode: 768,
		Kx: "RSA",
		Au: "RSA",
		Enc: Encryption{
			Cipher: "AES",
			Bits:   256,
		},
		Mac:  "SHA1",
		Code: 53,
	},
	"AES256-SHA256": CipherSuite{
		IANAName:   "TLS_RSA_WITH_AES_256_CBC_SHA256",
		GnuTLSName: "TLS_RSA_AES_256_CBC_SHA256",
		NSSName:    "TLS_RSA_WITH_AES_256_CBC_SHA256",
		Protocol:   "TLSv1.2", ProtocolCode: 771,
		Kx: "RSA",
		Au: "RSA",
		Enc: Encryption{
			Cipher: "AES",
			Bits:   256,
		},
		Mac:  "SHA256",
		Code: 61,
	},
	"CAMELLIA128-SHA": CipherSuite{
		IANAName:   "TLS_RSA_WITH_CAMELLIA_128_CBC_SHA",
		GnuTLSName: "TLS_RSA_CAMELLIA_128_CBC_SHA1",
		NSSName:    "TLS_RSA_WITH_CAMELLIA_128_CBC_SHA",
		Protocol:   "SSLv3", ProtocolCode: 768,
		Kx: "RSA",
		Au: "RSA",
		Enc: Encryption{
			Cipher: "Camellia",
			Bits:   128,
		},
		Mac:  "SHA1",
		Code: 65,
	},
	"CAMELLIA128-SHA256": CipherSuite{
		IANAName:   "TLS_RSA_WITH_CAMELLIA_128_CBC_SHA256",
		GnuTLSName: "TLS_RSA_CAMELLIA_128_CBC_SHA256",
		NSSName:    "",
		Protocol:   "TLSv1.2", ProtocolCode: 771,
		Kx: "RSA",
		Au: "RSA",
		Enc: Encryption{
			Cipher: "Camellia",
			Bits:   128,
		},
		Mac:  "SHA256",
		Code: 186,
	},
	"CAMELLIA256-SHA": CipherSuite{
		IANAName:   "TLS_RSA_WITH_CAMELLIA_256_CBC_SHA",
		GnuTLSName: "TLS_RSA_CAMELLIA_256_CBC_SHA1",
		NSSName:    "TLS_RSA_WITH_CAMELLIA_256_CBC_SHA",
		Protocol:   "SSLv3", ProtocolCode: 768,
		Kx: "RSA",
		Au: "RSA",
		Enc: Encryption{
			Cipher: "Camellia",
			Bits:   256,
		},
		Mac:  "SHA1",
		Code: 132,
	},
	"CAMELLIA256-SHA256": CipherSuite{
		IANAName:   "TLS_RSA_WITH_CAMELLIA_256_CBC_SHA256",
		GnuTLSName: "TLS_RSA_CAMELLIA_256_CBC_SHA256",
		NSSName:    "",
		Protocol:   "TLSv1.2", ProtocolCode: 771,
		Kx: "RSA",
		Au: "RSA",
		Enc: Encryption{
			Cipher: "Camellia",
			Bits:   256,
		},
		Mac:  "SHA256",
		Code: 192,
	},
	"DES-CBC3-SHA": CipherSuite{
		IANAName:   "TLS_RSA_WITH_3DES_EDE_CBC_SHA",
		GnuTLSName: "TLS_RSA_3DES_EDE_CBC_SHA1",
		NSSName:    "TLS_RSA_WITH_3DES_EDE_CBC_SHA",
		Protocol:   "SSLv3", ProtocolCode: 768,
		Kx: "RSA",
		Au: "RSA",
		Enc: Encryption{
			Cipher: "3DES",
			Bits:   168,
		},
		Mac:  "SHA1",
		Code: 10,
	},
	"DH-DSS-AES128-GCM-SHA256": CipherSuite{
		IANAName:   "TLS_DH_DSS_WITH_AES_128_GCM_SHA256",
		GnuTLSName: "",
		NSSName:    "",
		Protocol:   "TLSv1.2", ProtocolCode: 771,
		Kx: "DH/DSS",
		Au: "DH",
		Enc: Encryption{
			Cipher: "AESGCM",
			Bits:   128,
		},
		Mac:  "AEAD",
		Code: 164,
	},
	"DH-DSS-AES128-SHA": CipherSuite{
		IANAName:   "TLS_DH_DSS_WITH_AES_128_CBC_SHA",
		GnuTLSName: "",
		NSSName:    "TLS_DH_DSS_WITH_AES_128_CBC_SHA",
		Protocol:   "SSLv3", ProtocolCode: 768,
		Kx: "DH/DSS",
		Au: "DH",
		Enc: Encryption{
			Cipher: "AES",
			Bits:   128,
		},
		Mac:  "SHA1",
		Code: 48,
	},
	"DH-DSS-AES128-SHA256": CipherSuite{
		IANAName:   "TLS_DH_DSS_WITH_AES_128_CBC_SHA256",
		GnuTLSName: "",
		NSSName:    "",
		Protocol:   "TLSv1.2", ProtocolCode: 771,
		Kx: "DH/DSS",
		Au: "DH",
		Enc: Encryption{
			Cipher: "AES",
			Bits:   128,
		},
		Mac:  "SHA256",
		Code: 62,
	},
	"DH-DSS-AES256-GCM-SHA384": CipherSuite{
		IANAName:   "TLS_DH_DSS_WITH_AES_256_GCM_SHA384",
		GnuTLSName: "",
		NSSName:    "",
		Protocol:   "TLSv1.2", ProtocolCode: 771,
		Kx: "DH/DSS",
		Au: "DH",
		Enc: Encryption{
			Cipher: "AESGCM",
			Bits:   256,
		},
		Mac:  "AEAD",
		Code: 165,
	},
	"DH-DSS-AES256-SHA": CipherSuite{
		IANAName:   "TLS_DH_DSS_WITH_AES_256_CBC_SHA",
		GnuTLSName: "",
		NSSName:    "TLS_DH_DSS_WITH_AES_256_CBC_SHA",
		Protocol:   "SSLv3", ProtocolCode: 768,
		Kx: "DH/DSS",
		Au: "DH",
		Enc: Encryption{
			Cipher: "AES",
			Bits:   256,
		},
		Mac:  "SHA1",
		Code: 54,
	},
	"DH-DSS-AES256-SHA256": CipherSuite{
		IANAName:   "TLS_DH_DSS_WITH_AES_256_CBC_SHA256",
		GnuTLSName: "",
		NSSName:    "",
		Protocol:   "TLSv1.2", ProtocolCode: 771,
		Kx: "DH/DSS",
		Au: "DH",
		Enc: Encryption{
			Cipher: "AES",
			Bits:   256,
		},
		Mac:  "SHA256",
		Code: 104,
	},
	"DH-DSS-CAMELLIA128-SHA": CipherSuite{
		IANAName:   "TLS_DH_DSS_WITH_CAMELLIA_128_CBC_SHA",
		GnuTLSName: "",
		NSSName:    "TLS_DH_DSS_WITH_CAMELLIA_128_CBC_SHA",
		Protocol:   "SSLv3", ProtocolCode: 768,
		Kx: "DH/DSS",
		Au: "DH",
		Enc: Encryption{
			Cipher: "Camellia",
			Bits:   128,
		},
		Mac:  "SHA1",
		Code: 66,
	},
	"DH-DSS-CAMELLIA128-SHA256": CipherSuite{
		IANAName:   "TLS_DH_DSS_WITH_CAMELLIA_128_CBC_SHA256",
		GnuTLSName: "",
		NSSName:    "",
		Protocol:   "TLSv1.2", ProtocolCode: 771,
		Kx: "DH/DSS",
		Au: "DH",
		Enc: Encryption{
			Cipher: "Camellia",
			Bits:   128,
		},
		Mac:  "SHA256",
		Code: 187,
	},
	"DH-DSS-CAMELLIA256-SHA": CipherSuite{
		IANAName:   "TLS_DH_DSS_WITH_CAMELLIA_256_CBC_SHA",
		GnuTLSName: "",
		NSSName:    "TLS_DH_DSS_WITH_CAMELLIA_256_CBC_SHA",
		Protocol:   "SSLv3", ProtocolCode: 768,
		Kx: "DH/DSS",
		Au: "DH",
		Enc: Encryption{
			Cipher: "Camellia",
			Bits:   256,
		},
		Mac:  "SHA1",
		Code: 133,
	},
	"DH-DSS-CAMELLIA256-SHA256": CipherSuite{
		IANAName:   "TLS_DH_DSS_WITH_CAMELLIA_256_CBC_SHA256",
		GnuTLSName: "",
		NSSName:    "",
		Protocol:   "TLSv1.2", ProtocolCode: 771,
		Kx: "DH/DSS",
		Au: "DH",
		Enc: Encryption{
			Cipher: "Camellia",
			Bits:   256,
		},
		Mac:  "SHA256",
		Code: 193,
	},
	"DH-DSS-DES-CBC3-SHA": CipherSuite{
		IANAName:   "TLS_DH_DSS_WITH_3DES_EDE_CBC_SHA",
		GnuTLSName: "",
		NSSName:    "TLS_DH_DSS_WITH_3DES_EDE_CBC_SHA",
		Protocol:   "SSLv3", ProtocolCode: 768,
		Kx: "DH/DSS",
		Au: "DH",
		Enc: Encryption{
			Cipher: "3DES",
			Bits:   168,
		},
		Mac:  "SHA1",
		Code: 13,
	},
	"DH-DSS-SEED-SHA": CipherSuite{
		IANAName:   "TLS_DH_DSS_WITH_SEED_CBC_SHA",
		GnuTLSName: "",
		NSSName:    "",
		Protocol:   "SSLv3", ProtocolCode: 768,
		Kx: "DH/DSS",
		Au: "DH",
		Enc: Encryption{
			Cipher: "SEED",
			Bits:   128,
		},
		Mac:  "SHA1",
		Code: 151,
	},
	"DH-RSA-AES128-GCM-SHA256": CipherSuite{
		IANAName:   "TLS_DH_RSA_WITH_AES_128_GCM_SHA256",
		GnuTLSName: "",
		NSSName:    "",
		Protocol:   "TLSv1.2", ProtocolCode: 771,
		Kx: "DH/RSA",
		Au: "DH",
		Enc: Encryption{
			Cipher: "AESGCM",
			Bits:   128,
		},
		Mac:  "AEAD",
		Code: 160,
	},
	"DH-RSA-AES128-SHA": CipherSuite{
		IANAName:   "TLS_DH_RSA_WITH_AES_128_CBC_SHA",
		GnuTLSName: "",
		NSSName:    "TLS_DH_RSA_WITH_AES_128_CBC_SHA",
		Protocol:   "SSLv3", ProtocolCode: 768,
		Kx: "DH/RSA",
		Au: "DH",
		Enc: Encryption{
			Cipher: "AES",
			Bits:   128,
		},
		Mac:  "SHA1",
		Code: 49,
	},
	"DH-RSA-AES128-SHA256": CipherSuite{
		IANAName:   "TLS_DH_RSA_WITH_AES_128_CBC_SHA256",
		GnuTLSName: "",
		NSSName:    "",
		Protocol:   "TLSv1.2", ProtocolCode: 771,
		Kx: "DH/RSA",
		Au: "DH",
		Enc: Encryption{
			Cipher: "AES",
			Bits:   128,
		},
		Mac:  "SHA256",
		Code: 63,
	},
	"DH-RSA-AES256-GCM-SHA384": CipherSuite{
		IANAName:   "TLS_DH_RSA_WITH_AES_256_GCM_SHA384",
		GnuTLSName: "",
		NSSName:    "",
		Protocol:   "TLSv1.2", ProtocolCode: 771,
		Kx: "DH/RSA",
		Au: "DH",
		Enc: Encryption{
			Cipher: "AESGCM",
			Bits:   256,
		},
		Mac:  "AEAD",
		Code: 161,
	},
	"DH-RSA-AES256-SHA": CipherSuite{
		IANAName:   "TLS_DH_RSA_WITH_AES_256_CBC_SHA",
		GnuTLSName: "",
		NSSName:    "TLS_DH_RSA_WITH_AES_256_CBC_SHA",
		Protocol:   "SSLv3", ProtocolCode: 768,
		Kx: "DH/RSA",
		Au: "DH",
		Enc: Encryption{
			Cipher: "AES",
			Bits:   256,
		},
		Mac:  "SHA1",
		Code: 55,
	},
	"DH-RSA-AES256-SHA256": CipherSuite{
		IANAName:   "TLS_DH_RSA_WITH_AES_256_CBC_SHA256",
		GnuTLSName: "",
		NSSName:    "",
		Protocol:   "TLSv1.2", ProtocolCode: 771,
		Kx: "DH/RSA",
		Au: "DH",
		Enc: Encryption{
			Cipher: "AES",
			Bits:   256,
		},
		Mac:  "SHA256",
		Code: 105,
	},
	"DH-RSA-CAMELLIA128-SHA": CipherSuite{
		IANAName:   "TLS_DH_RSA_WITH_CAMELLIA_128_CBC_SHA",
		GnuTLSName: "",
		NSSName:    "TLS_DH_RSA_WITH_CAMELLIA_128_CBC_SHA",
		Protocol:   "SSLv3", ProtocolCode: 768,
		Kx: "DH/RSA",
		Au: "DH",
		Enc: Encryption{
			Cipher: "Camellia",
			Bits:   128,
		},
		Mac:  "SHA1",
		Code: 67,
	},
	"DH-RSA-CAMELLIA128-SHA256": CipherSuite{
		IANAName:   "TLS_DH_RSA_WITH_CAMELLIA_128_CBC_SHA256",
		GnuTLSName: "",
		NSSName:    "",
		Protocol:   "TLSv1.2", ProtocolCode: 771,
		Kx: "DH/RSA",
		Au: "DH",
		Enc: Encryption{
			Cipher: "Camellia",
			Bits:   128,
		},
		Mac:  "SHA256",
		Code: 188,
	},
	"DH-RSA-CAMELLIA256-SHA": CipherSuite{
		IANAName:   "TLS_DH_RSA_WITH_CAMELLIA_256_CBC_SHA",
		GnuTLSName: "",
		NSSName:    "TLS_DH_RSA_WITH_CAMELLIA_256_CBC_SHA",
		Protocol:   "SSLv3", ProtocolCode: 768,
		Kx: "DH/RSA",
		Au: "DH",
		Enc: Encryption{
			Cipher: "Camellia",
			Bits:   256,
		},
		Mac:  "SHA1",
		Code: 134,
	},
	"DH-RSA-CAMELLIA256-SHA256": CipherSuite{
		IANAName:   "TLS_DH_RSA_WITH_CAMELLIA_256_CBC_SHA256",
		GnuTLSName: "",
		NSSName:    "",
		Protocol:   "TLSv1.2", ProtocolCode: 771,
		Kx: "DH/RSA",
		Au: "DH",
		Enc: Encryption{
			Cipher: "Camellia",
			Bits:   256,
		},
		Mac:  "SHA256",
		Code: 194,
	},
	"DH-RSA-DES-CBC3-SHA": CipherSuite{
		IANAName:   "TLS_DH_RSA_WITH_3DES_EDE_CBC_SHA",
		GnuTLSName: "",
		NSSName:    "TLS_DH_RSA_WITH_3DES_EDE_CBC_SHA",
		Protocol:   "SSLv3", ProtocolCode: 768,
		Kx: "DH/RSA",
		Au: "DH",
		Enc: Encryption{
			Cipher: "3DES",
			Bits:   168,
		},
		Mac:  "SHA1",
		Code: 16,
	},
	"DH-RSA-SEED-SHA": CipherSuite{
		IANAName:   "TLS_DH_RSA_WITH_SEED_CBC_SHA",
		GnuTLSName: "",
		NSSName:    "",
		Protocol:   "SSLv3", ProtocolCode: 768,
		Kx: "DH/RSA",
		Au: "DH",
		Enc: Encryption{
			Cipher: "SEED",
			Bits:   128,
		},
		Mac:  "SHA1",
		Code: 152,
	},
	"DHE-DSS-AES128-GCM-SHA256": CipherSuite{
		IANAName:   "TLS_DHE_DSS_WITH_AES_128_GCM_SHA256",
		GnuTLSName: "TLS_DHE_DSS_AES_128_GCM_SHA256",
		NSSName:    "TLS_DHE_DSS_WITH_AES_128_GCM_SHA256",
		Protocol:   "TLSv1.2", ProtocolCode: 771,
		Kx: "DH",
		Au: "DSS",
		Enc: Encryption{
			Cipher: "AESGCM",
			Bits:   128,
		},
		Mac:  "AEAD",
		Code: 162,
	},
	"DHE-DSS-AES128-SHA": CipherSuite{
		IANAName:   "TLS_DHE_DSS_WITH_AES_128_CBC_SHA",
		GnuTLSName: "TLS_DHE_DSS_AES_128_CBC_SHA1",
		NSSName:    "TLS_DHE_DSS_WITH_AES_128_CBC_SHA",
		Protocol:   "SSLv3", ProtocolCode: 768,
		Kx: "DH",
		Au: "DSS",
		Enc: Encryption{
			Cipher: "AES",
			Bits:   128,
		},
		Mac:  "SHA1",
		Code: 50,
	},
	"DHE-DSS-AES128-SHA256": CipherSuite{
		IANAName:   "TLS_DHE_DSS_WITH_AES_128_CBC_SHA256",
		GnuTLSName: "TLS_DHE_DSS_AES_128_CBC_SHA256",
		NSSName:    "TLS_DHE_DSS_WITH_AES_128_CBC_SHA256",
		Protocol:   "TLSv1.2", ProtocolCode: 771,
		Kx: "DH",
		Au: "DSS",
		Enc: Encryption{
			Cipher: "AES",
			Bits:   128,
		},
		Mac:  "SHA256",
		Code: 64,
	},
	"DHE-DSS-AES256-GCM-SHA384": CipherSuite{
		IANAName:   "TLS_DHE_DSS_WITH_AES_256_GCM_SHA384",
		GnuTLSName: "TLS_DHE_DSS_AES_256_GCM_SHA384",
		NSSName:    "TLS_DHE_DSS_WITH_AES_256_GCM_SHA384",
		Protocol:   "TLSv1.2", ProtocolCode: 771,
		Kx: "DH",
		Au: "DSS",
		Enc: Encryption{
			Cipher: "AESGCM",
			Bits:   256,
		},
		Mac:  "AEAD",
		Code: 163,
	},
	"DHE-DSS-AES256-SHA": CipherSuite{
		IANAName:   "TLS_DHE_DSS_WITH_AES_256_CBC_SHA",
		GnuTLSName: "TLS_DHE_DSS_AES_256_CBC_SHA1",
		NSSName:    "TLS_DHE_DSS_WITH_AES_256_CBC_SHA",
		Protocol:   "SSLv3", ProtocolCode: 768,
		Kx: "DH",
		Au: "DSS",
		Enc: Encryption{
			Cipher: "AES",
			Bits:   256,
		},
		Mac:  "SHA1",
		Code: 56,
	},
	"DHE-DSS-AES256-SHA256": CipherSuite{
		IANAName:   "TLS_DHE_DSS_WITH_AES_256_CBC_SHA256",
		GnuTLSName: "TLS_DHE_DSS_AES_256_CBC_SHA256",
		NSSName:    "TLS_DHE_DSS_WITH_AES_256_CBC_SHA256",
		Protocol:   "TLSv1.2", ProtocolCode: 771,
		Kx: "DH",
		Au: "DSS",
		Enc: Encryption{
			Cipher: "AES",
			Bits:   256,
		},
		Mac:  "SHA256",
		Code: 106,
	},
	"DHE-DSS-CAMELLIA128-SHA": CipherSuite{
		IANAName:   "TLS_DHE_DSS_WITH_CAMELLIA_128_CBC_SHA",
		GnuTLSName: "TLS_DHE_DSS_CAMELLIA_128_CBC_SHA1",
		NSSName:    "TLS_DHE_DSS_WITH_CAMELLIA_128_CBC_SHA",
		Protocol:   "SSLv3", ProtocolCode: 768,
		Kx: "DH",
		Au: "DSS",
		Enc: Encryption{
			Cipher: "Camellia",
			Bits:   128,
		},
		Mac:  "SHA1",
		Code: 68,
	},
	"DHE-DSS-CAMELLIA128-SHA256": CipherSuite{
		IANAName:   "TLS_DHE_DSS_WITH_CAMELLIA_128_CBC_SHA256",
		GnuTLSName: "TLS_DHE_DSS_CAMELLIA_128_CBC_SHA256",
		NSSName:    "",
		Protocol:   "TLSv1.2", ProtocolCode: 771,
		Kx: "DH",
		Au: "DSS",
		Enc: Encryption{
			Cipher: "Camellia",
			Bits:   128,
		},
		Mac:  "SHA256",
		Code: 189,
	},
	"DHE-DSS-CAMELLIA256-SHA": CipherSuite{
		IANAName:   "TLS_DHE_DSS_WITH_CAMELLIA_256_CBC_SHA",
		GnuTLSName: "TLS_DHE_DSS_CAMELLIA_256_CBC_SHA1",
		NSSName:    "TLS_DHE_DSS_WITH_CAMELLIA_256_CBC_SHA",
		Protocol:   "SSLv3", ProtocolCode: 768,
		Kx: "DH",
		Au: "DSS",
		Enc: Encryption{
			Cipher: "Camellia",
			Bits:   256,
		},
		Mac:  "SHA1",
		Code: 135,
	},
	"DHE-DSS-CAMELLIA256-SHA256": CipherSuite{
		IANAName:   "TLS_DHE_DSS_WITH_CAMELLIA_256_CBC_SHA256",
		GnuTLSName: "TLS_DHE_DSS_CAMELLIA_256_CBC_SHA256",
		NSSName:    "",
		Protocol:   "TLSv1.2", ProtocolCode: 771,
		Kx: "DH",
		Au: "DSS",
		Enc: Encryption{
			Cipher: "Camellia",
			Bits:   256,
		},
		Mac:  "SHA256",
		Code: 195,
	},
	"DHE-DSS-RC4-SHA": CipherSuite{
		IANAName:   "",
		GnuTLSName: "",
		NSSName:    "",
		Protocol:   "SSLv3", ProtocolCode: 768,
		Kx: "DH",
		Au: "DSS",
		Enc: Encryption{
			Cipher: "RC4",
			Bits:   128,
		},
		Mac:  "SHA1",
		Code: 102,
	},
	"DHE-DSS-SEED-SHA": CipherSuite{
		IANAName:   "TLS_DHE_DSS_WITH_SEED_CBC_SHA",
		GnuTLSName: "",
		NSSName:    "",
		Protocol:   "SSLv3", ProtocolCode: 768,
		Kx: "DH",
		Au: "DSS",
		Enc: Encryption{
			Cipher: "SEED",
			Bits:   128,
		},
		Mac:  "SHA1",
		Code: 153,
	},
	"DHE-RSA-AES128-GCM-SHA256": CipherSuite{
		IANAName:   "TLS_DHE_RSA_WITH_AES_128_GCM_SHA256",
		GnuTLSName: "TLS_DHE_RSA_AES_128_GCM_SHA256",
		NSSName:    "TLS_DHE_RSA_WITH_AES_128_GCM_SHA256",
		Protocol:   "TLSv1.2", ProtocolCode: 771,
		Kx: "DH",
		Au: "RSA",
		Enc: Encryption{
			Cipher: "AESGCM",
			Bits:   128,
		},
		Mac:  "AEAD",
		Code: 158,
	},
	"DHE-RSA-AES128-SHA": CipherSuite{
		IANAName:   "TLS_DHE_RSA_WITH_AES_128_CBC_SHA",
		GnuTLSName: "TLS_DHE_RSA_AES_128_CBC_SHA1",
		NSSName:    "TLS_DHE_RSA_WITH_AES_128_CBC_SHA",
		Protocol:   "SSLv3", ProtocolCode: 768,
		Kx: "DH",
		Au: "RSA",
		Enc: Encryption{
			Cipher: "AES",
			Bits:   128,
		},
		Mac:  "SHA1",
		Code: 51,
	},
	"DHE-RSA-AES128-SHA256": CipherSuite{
		IANAName:   "TLS_DHE_RSA_WITH_AES_128_CBC_SHA256",
		GnuTLSName: "TLS_DHE_RSA_AES_128_CBC_SHA256",
		NSSName:    "TLS_DHE_RSA_WITH_AES_128_CBC_SHA256",
		Protocol:   "TLSv1.2", ProtocolCode: 771,
		Kx: "DH",
		Au: "RSA",
		Enc: Encryption{
			Cipher: "AES",
			Bits:   128,
		},
		Mac:  "SHA256",
		Code: 103,
	},
	"DHE-RSA-AES256-GCM-SHA384": CipherSuite{
		IANAName:   "TLS_DHE_RSA_WITH_AES_256_GCM_SHA384",
		GnuTLSName: "TLS_DHE_RSA_AES_256_GCM_SHA384",
		NSSName:    "TLS_DHE_RSA_WITH_AES_256_GCM_SHA384",
		Protocol:   "TLSv1.2", ProtocolCode: 771,
		Kx: "DH",
		Au: "RSA",
		Enc: Encryption{
			Cipher: "AESGCM",
			Bits:   256,
		},
		Mac:  "AEAD",
		Code: 159,
	},
	"DHE-RSA-AES256-SHA": CipherSuite{
		IANAName:   "TLS_DHE_RSA_WITH_AES_256_CBC_SHA",
		GnuTLSName: "TLS_DHE_RSA_AES_256_CBC_SHA1",
		NSSName:    "TLS_DHE_RSA_WITH_AES_256_CBC_SHA",
		Protocol:   "SSLv3", ProtocolCode: 768,
		Kx: "DH",
		Au: "RSA",
		Enc: Encryption{
			Cipher: "AES",
			Bits:   256,
		},
		Mac:  "SHA1",
		Code: 57,
	},
	"DHE-RSA-AES256-SHA256": CipherSuite{
		IANAName:   "TLS_DHE_RSA_WITH_AES_256_CBC_SHA256",
		GnuTLSName: "TLS_DHE_RSA_AES_256_CBC_SHA256",
		NSSName:    "TLS_DHE_RSA_WITH_AES_256_CBC_SHA256",
		Protocol:   "TLSv1.2", ProtocolCode: 771,
		Kx: "DH",
		Au: "RSA",
		Enc: Encryption{
			Cipher: "AES",
			Bits:   256,
		},
		Mac:  "SHA256",
		Code: 107,
	},
	"DHE-RSA-CAMELLIA128-SHA": CipherSuite{
		IANAName:   "TLS_DHE_RSA_WITH_CAMELLIA_128_CBC_SHA",
		GnuTLSName: "TLS_DHE_RSA_CAMELLIA_128_CBC_SHA1",
		NSSName:    "TLS_DHE_RSA_WITH_CAMELLIA_128_CBC_SHA",
		Protocol:   "SSLv3", ProtocolCode: 768,
		Kx: "DH",
		Au: "RSA",
		Enc: Encryption{
			Cipher: "Camellia",
			Bits:   128,
		},
		Mac:  "SHA1",
		Code: 69,
	},
	"DHE-RSA-CAMELLIA128-SHA256": CipherSuite{
		IANAName:   "TLS_DHE_RSA_WITH_CAMELLIA_128_CBC_SHA256",
		GnuTLSName: "TLS_DHE_RSA_CAMELLIA_128_CBC_SHA256",
		NSSName:    "",
		Protocol:   "TLSv1.2", ProtocolCode: 771,
		Kx: "DH",
		Au: "RSA",
		Enc: Encryption{
			Cipher: "Camellia",
			Bits:   128,
		},
		Mac:  "SHA256",
		Code: 190,
	},
	"DHE-RSA-CAMELLIA256-SHA": CipherSuite{
		IANAName:   "TLS_DHE_RSA_WITH_CAMELLIA_256_CBC_SHA",
		GnuTLSName: "TLS_DHE_RSA_CAMELLIA_256_CBC_SHA1",
		NSSName:    "TLS_DHE_RSA_WITH_CAMELLIA_256_CBC_SHA",
		Protocol:   "SSLv3", ProtocolCode: 768,
		Kx: "DH",
		Au: "RSA",
		Enc: Encryption{
			Cipher: "Camellia",
			Bits:   256,
		},
		Mac:  "SHA1",
		Code: 136,
	},
	"DHE-RSA-CAMELLIA256-SHA256": CipherSuite{
		IANAName:   "TLS_DHE_RSA_WITH_CAMELLIA_256_CBC_SHA256",
		GnuTLSName: "TLS_DHE_RSA_CAMELLIA_256_CBC_SHA256",
		NSSName:    "",
		Protocol:   "TLSv1.2", ProtocolCode: 771,
		Kx: "DH",
		Au: "RSA",
		Enc: Encryption{
			Cipher: "Camellia",
			Bits:   256,
		},
		Mac:  "SHA256",
		Code: 196,
	},
	"DHE-RSA-CHACHA20-POLY1305-OLD": CipherSuite{
		IANAName:   "",
		GnuTLSName: "",
		NSSName:    "",
		Protocol:   "TLSv1.2", ProtocolCode: 771,
		Kx: "DH",
		Au: "RSA",
		Enc: Encryption{
			Cipher: "ChaCha20",
			Bits:   256,
		},
		Mac:  "AEAD",
		Code: 52245,
	},
	"DHE-RSA-CHACHA20-POLY1305": CipherSuite{
		IANAName:   "",
		GnuTLSName: "TLS_DHE_RSA_CHACHA20_POLY1305",
		NSSName:    "TLS_DHE_RSA_WITH_CHACHA20_POLY1305_SHA256,",
		Protocol:   "TLSv1.2", ProtocolCode: 771,
		Kx: "DH",
		Au: "RSA",
		Enc: Encryption{
			Cipher: "ChaCha20",
			Bits:   256,
		},
		Mac:  "AEAD",
		Code: 52394,
	},
	"DHE-RSA-SEED-SHA": CipherSuite{
		IANAName:   "TLS_DHE_RSA_WITH_SEED_CBC_SHA",
		GnuTLSName: "",
		NSSName:    "",
		Protocol:   "SSLv3", ProtocolCode: 768,
		Kx: "DH",
		Au: "RSA",
		Enc: Encryption{
			Cipher: "SEED",
			Bits:   128,
		},
		Mac:  "SHA1",
		Code: 154,
	},
	"ECDH-ECDSA-AES128-GCM-SHA256": CipherSuite{
		IANAName:   "TLS_ECDH_ECDSA_WITH_AES_128_GCM_SHA256",
		GnuTLSName: "",
		NSSName:    "TLS_ECDH_ECDSA_WITH_AES_128_GCM_SHA256",
		Protocol:   "TLSv1.2", ProtocolCode: 771,
		Kx: "ECDH/ECDSA",
		Au: "ECDH",
		Enc: Encryption{
			Cipher: "AESGCM",
			Bits:   128,
		},
		Mac:  "AEAD",
		Code: 49197,
	},
	"ECDH-ECDSA-AES128-SHA": CipherSuite{
		IANAName:   "TLS_ECDH_ECDSA_WITH_AES_128_CBC_SHA",
		GnuTLSName: "",
		NSSName:    "TLS_ECDH_ECDSA_WITH_AES_128_CBC_SHA",
		Protocol:   "SSLv3", ProtocolCode: 768,
		Kx: "ECDH/ECDSA",
		Au: "ECDH",
		Enc: Encryption{
			Cipher: "AES",
			Bits:   128,
		},
		Mac:  "SHA1",
		Code: 49156,
	},
	"ECDH-ECDSA-AES128-SHA256": CipherSuite{
		IANAName:   "TLS_ECDH_ECDSA_WITH_AES_128_CBC_SHA256",
		GnuTLSName: "",
		NSSName:    "",
		Protocol:   "TLSv1.2", ProtocolCode: 771,
		Kx: "ECDH/ECDSA",
		Au: "ECDH",
		Enc: Encryption{
			Cipher: "AES",
			Bits:   128,
		},
		Mac:  "SHA256",
		Code: 49189,
	},
	"ECDH-ECDSA-AES256-GCM-SHA384": CipherSuite{
		IANAName:   "TLS_ECDH_ECDSA_WITH_AES_256_GCM_SHA384",
		GnuTLSName: "",
		NSSName:    "",
		Protocol:   "TLSv1.2", ProtocolCode: 771,
		Kx: "ECDH/ECDSA",
		Au: "ECDH",
		Enc: Encryption{
			Cipher: "AESGCM",
			Bits:   256,
		},
		Mac:  "AEAD",
		Code: 49198,
	},
	"ECDH-ECDSA-AES256-SHA": CipherSuite{
		IANAName:   "TLS_ECDH_ECDSA_WITH_AES_256_CBC_SHA",
		GnuTLSName: "",
		NSSName:    "TLS_ECDH_ECDSA_WITH_AES_256_CBC_SHA",
		Protocol:   "SSLv3", ProtocolCode: 768,
		Kx: "ECDH/ECDSA",
		Au: "ECDH",
		Enc: Encryption{
			Cipher: "AES",
			Bits:   256,
		},
		Mac:  "SHA1",
		Code: 49157,
	},
	"ECDH-ECDSA-AES256-SHA384": CipherSuite{
		IANAName:   "TLS_ECDH_ECDSA_WITH_AES_256_CBC_SHA384",
		GnuTLSName: "",
		NSSName:    "",
		Protocol:   "TLSv1.2", ProtocolCode: 771,
		Kx: "ECDH/ECDSA",
		Au: "ECDH",
		Enc: Encryption{
			Cipher: "AES",
			Bits:   256,
		},
		Mac:  "SHA384",
		Code: 49190,
	},
	"ECDH-ECDSA-CAMELLIA128-SHA256": CipherSuite{
		IANAName:   "TLS_ECDH_ECDSA_WITH_CAMELLIA_128_CBC_SHA256",
		GnuTLSName: "",
		NSSName:    "",
		Protocol:   "TLSv1.2", ProtocolCode: 771,
		Kx: "ECDH/ECDSA",
		Au: "ECDH",
		Enc: Encryption{
			Cipher: "Camellia",
			Bits:   128,
		},
		Mac:  "SHA256",
		Code: 49268,
	},
	"ECDH-ECDSA-CAMELLIA256-SHA384": CipherSuite{
		IANAName:   "TLS_ECDH_ECDSA_WITH_CAMELLIA_256_CBC_SHA384",
		GnuTLSName: "",
		NSSName:    "",
		Protocol:   "TLSv1.2", ProtocolCode: 771,
		Kx: "ECDH/ECDSA",
		Au: "ECDH",
		Enc: Encryption{
			Cipher: "Camellia",
			Bits:   256,
		},
		Mac:  "SHA384",
		Code: 49269,
	},
	"ECDH-ECDSA-DES-CBC3-SHA": CipherSuite{
		IANAName:   "TLS_ECDH_ECDSA_WITH_3DES_EDE_CBC_SHA",
		GnuTLSName: "",
		NSSName:    "TLS_ECDH_ECDSA_WITH_3DES_EDE_CBC_SHA",
		Protocol:   "SSLv3", ProtocolCode: 768,
		Kx: "ECDH/ECDSA",
		Au: "ECDH",
		Enc: Encryption{
			Cipher: "3DES",
			Bits:   168,
		},
		Mac:  "SHA1",
		Code: 49155,
	},
	"ECDH-ECDSA-RC4-SHA": CipherSuite{
		IANAName:   "TLS_ECDH_ECDSA_WITH_RC4_128_SHA",
		GnuTLSName: "",
		NSSName:    "TLS_ECDH_ECDSA_WITH_RC4_128_SHA",
		Protocol:   "SSLv3", ProtocolCode: 768,
		Kx: "ECDH/ECDSA",
		Au: "ECDH",
		Enc: Encryption{
			Cipher: "RC4",
			Bits:   128,
		},
		Mac:  "SHA1",
		Code: 49154,
	},
	"ECDH-RSA-AES128-GCM-SHA256": CipherSuite{
		IANAName:   "TLS_ECDH_RSA_WITH_AES_128_GCM_SHA256",
		GnuTLSName: "",
		NSSName:    "TLS_ECDH_RSA_WITH_AES_128_GCM_SHA256",
		Protocol:   "TLSv1.2", ProtocolCode: 771,
		Kx: "ECDH/RSA",
		Au: "ECDH",
		Enc: Encryption{
			Cipher: "AESGCM",
			Bits:   128,
		},
		Mac:  "AEAD",
		Code: 49201,
	},
	"ECDH-RSA-AES128-SHA": CipherSuite{
		IANAName:   "TLS_ECDH_RSA_WITH_AES_128_CBC_SHA",
		GnuTLSName: "",
		NSSName:    "TLS_ECDH_RSA_WITH_AES_128_CBC_SHA",
		Protocol:   "SSLv3", ProtocolCode: 768,
		Kx: "ECDH/RSA",
		Au: "ECDH",
		Enc: Encryption{
			Cipher: "AES",
			Bits:   128,
		},
		Mac:  "SHA1",
		Code: 49166,
	},
	"ECDH-RSA-AES128-SHA256": CipherSuite{
		IANAName:   "TLS_ECDH_RSA_WITH_AES_128_CBC_SHA256",
		GnuTLSName: "",
		NSSName:    "",
		Protocol:   "TLSv1.2", ProtocolCode: 771,
		Kx: "ECDH/RSA",
		Au: "ECDH",
		Enc: Encryption{
			Cipher: "AES",
			Bits:   128,
		},
		Mac:  "SHA256",
		Code: 49193,
	},
	"ECDH-RSA-AES256-GCM-SHA384": CipherSuite{
		IANAName:   "TLS_ECDH_RSA_WITH_AES_256_GCM_SHA384",
		GnuTLSName: "",
		NSSName:    "",
		Protocol:   "TLSv1.2", ProtocolCode: 771,
		Kx: "ECDH/RSA",
		Au: "ECDH",
		Enc: Encryption{
			Cipher: "AESGCM",
			Bits:   256,
		},
		Mac:  "AEAD",
		Code: 49202,
	},
	"ECDH-RSA-AES256-SHA": CipherSuite{
		IANAName:   "TLS_ECDH_RSA_WITH_AES_256_CBC_SHA",
		GnuTLSName: "",
		NSSName:    "TLS_ECDH_RSA_WITH_AES_256_CBC_SHA",
		Protocol:   "SSLv3", ProtocolCode: 768,
		Kx: "ECDH/RSA",
		Au: "ECDH",
		Enc: Encryption{
			Cipher: "AES",
			Bits:   256,
		},
		Mac:  "SHA1",
		Code: 49167,
	},
	"ECDH-RSA-AES256-SHA384": CipherSuite{
		IANAName:   "TLS_ECDH_RSA_WITH_AES_256_CBC_SHA384",
		GnuTLSName: "",
		NSSName:    "",
		Protocol:   "TLSv1.2", ProtocolCode: 771,
		Kx: "ECDH/RSA",
		Au: "ECDH",
		Enc: Encryption{
			Cipher: "AES",
			Bits:   256,
		},
		Mac:  "SHA384",
		Code: 49194,
	},
	"ECDH-RSA-CAMELLIA128-SHA256": CipherSuite{
		IANAName:   "TLS_ECDH_RSA_WITH_CAMELLIA_128_CBC_SHA256",
		GnuTLSName: "",
		NSSName:    "",
		Protocol:   "TLSv1.2", ProtocolCode: 771,
		Kx: "ECDH/RSA",
		Au: "ECDH",
		Enc: Encryption{
			Cipher: "Camellia",
			Bits:   128,
		},
		Mac:  "SHA256",
		Code: 49272,
	},
	"ECDH-RSA-CAMELLIA256-SHA384": CipherSuite{
		IANAName:   "TLS_ECDH_RSA_WITH_CAMELLIA_256_CBC_SHA384",
		GnuTLSName: "",
		NSSName:    "",
		Protocol:   "TLSv1.2", ProtocolCode: 771,
		Kx: "ECDH/RSA",
		Au: "ECDH",
		Enc: Encryption{
			Cipher: "Camellia",
			Bits:   256,
		},
		Mac:  "SHA384",
		Code: 49273,
	},
	"ECDH-RSA-DES-CBC3-SHA": CipherSuite{
		IANAName:   "TLS_ECDH_RSA_WITH_3DES_EDE_CBC_SHA",
		GnuTLSName: "",
		NSSName:    "TLS_ECDH_RSA_WITH_3DES_EDE_CBC_SHA",
		Protocol:   "SSLv3", ProtocolCode: 768,
		Kx: "ECDH/RSA",
		Au: "ECDH",
		Enc: Encryption{
			Cipher: "3DES",
			Bits:   168,
		},
		Mac:  "SHA1",
		Code: 49165,
	},
	"ECDH-RSA-RC4-SHA": CipherSuite{
		IANAName:   "TLS_ECDH_RSA_WITH_RC4_128_SHA",
		GnuTLSName: "",
		NSSName:    "TLS_ECDH_RSA_WITH_RC4_128_SHA",
		Protocol:   "SSLv3", ProtocolCode: 768,
		Kx: "ECDH/RSA",
		Au: "ECDH",
		Enc: Encryption{
			Cipher: "RC4",
			Bits:   128,
		},
		Mac:  "SHA1",
		Code: 49164,
	},
	"ECDHE-ECDSA-AES128-GCM-SHA256": CipherSuite{
		IANAName:   "TLS_ECDHE_ECDSA_WITH_AES_128_GCM_SHA256",
		GnuTLSName: "TLS_ECDHE_ECDSA_AES_128_GCM_SHA256",
		NSSName:    "TLS_ECDHE_ECDSA_WITH_AES_128_GCM_SHA256",
		Protocol:   "TLSv1.2", ProtocolCode: 771,
		Kx: "ECDH",
		Au: "ECDSA",
		Enc: Encryption{
			Cipher: "AESGCM",
			Bits:   128,
		},
		Mac:  "AEAD",
		Code: 49195,
	},
	"ECDHE-ECDSA-AES128-SHA": CipherSuite{
		IANAName:   "TLS_ECDHE_ECDSA_WITH_AES_128_CBC_SHA",
		GnuTLSName: "TLS_ECDHE_ECDSA_AES_128_CBC_SHA1",
		NSSName:    "TLS_ECDHE_ECDSA_WITH_AES_128_CBC_SHA",
		Protocol:   "SSLv3", ProtocolCode: 768,
		Kx: "ECDH",
		Au: "ECDSA",
		Enc: Encryption{
			Cipher: "AES",
			Bits:   128,
		},
		Mac:  "SHA1",
		Code: 49161,
	},
	"ECDHE-ECDSA-AES128-SHA256": CipherSuite{
		IANAName:   "TLS_ECDHE_ECDSA_WITH_AES_128_CBC_SHA256",
		GnuTLSName: "TLS_ECDHE_ECDSA_AES_128_CBC_SHA256",
		NSSName:    "TLS_ECDHE_ECDSA_WITH_AES_128_CBC_SHA256",
		Protocol:   "TLSv1.2", ProtocolCode: 771,
		Kx: "ECDH",
		Au: "ECDSA",
		Enc: Encryption{
			Cipher: "AES",
			Bits:   128,
		},
		Mac:  "SHA256",
		Code: 49187,
	},
	"ECDHE-ECDSA-AES256-GCM-SHA384": CipherSuite{
		IANAName:   "TLS_ECDHE_ECDSA_WITH_AES_256_GCM_SHA384",
		GnuTLSName: "TLS_ECDHE_ECDSA_AES_256_GCM_SHA384",
		NSSName:    "TLS_ECDHE_ECDSA_WITH_AES_256_GCM_SHA384",
		Protocol:   "TLSv1.2", ProtocolCode: 771,
		Kx: "ECDH",
		Au: "ECDSA",
		Enc: Encryption{
			Cipher: "AESGCM",
			Bits:   256,
		},
		Mac:  "AEAD",
		Code: 49196,
	},
	"ECDHE-ECDSA-AES256-SHA": CipherSuite{
		IANAName:   "TLS_ECDHE_ECDSA_WITH_AES_256_CBC_SHA",
		GnuTLSName: "TLS_ECDHE_ECDSA_AES_256_CBC_SHA1",
		NSSName:    "TLS_ECDHE_ECDSA_WITH_AES_256_CBC_SHA",
		Protocol:   "SSLv3", ProtocolCode: 768,
		Kx: "ECDH",
		Au: "ECDSA",
		Enc: Encryption{
			Cipher: "AES",
			Bits:   256,
		},
		Mac:  "SHA1",
		Code: 49162,
	},
	"ECDHE-ECDSA-AES256-SHA384": CipherSuite{
		IANAName:   "TLS_ECDHE_ECDSA_WITH_AES_256_CBC_SHA384",
		GnuTLSName: "TLS_ECDHE_ECDSA_AES_256_CBC_SHA384",
		NSSName:    "TLS_ECDHE_ECDSA_WITH_AES_256_CBC_SHA384",
		Protocol:   "TLSv1.2", ProtocolCode: 771,
		Kx: "ECDH",
		Au: "ECDSA",
		Enc: Encryption{
			Cipher: "AES",
			Bits:   256,
		},
		Mac:  "SHA384",
		Code: 49188,
	},
	"ECDHE-ECDSA-CAMELLIA128-SHA256": CipherSuite{
		IANAName:   "TLS_ECDHE_ECDSA_WITH_CAMELLIA_128_CBC_SHA256",
		GnuTLSName: "TLS_ECDHE_ECDSA_CAMELLIA_128_CBC_SHA256",
		NSSName:    "",
		Protocol:   "TLSv1.2", ProtocolCode: 771,
		Kx: "ECDH",
		Au: "ECDSA",
		Enc: Encryption{
			Cipher: "Camellia",
			Bits:   128,
		},
		Mac:  "SHA256",
		Code: 49266,
	},
	"ECDHE-ECDSA-CAMELLIA256-SHA384": CipherSuite{
		IANAName:   "TLS_ECDHE_ECDSA_WITH_CAMELLIA_256_CBC_SHA384",
		GnuTLSName: "TLS_ECDHE_ECDSA_CAMELLIA_256_CBC_SHA384",
		NSSName:    "",
		Protocol:   "TLSv1.2", ProtocolCode: 771,
		Kx: "ECDH",
		Au: "ECDSA",
		Enc: Encryption{
			Cipher: "Camellia",
			Bits:   256,
		},
		Mac:  "SHA384",
		Code: 49267,
	},
	"ECDHE-ECDSA-CHACHA20-POLY1305-OLD": CipherSuite{
		IANAName:   "",
		GnuTLSName: "",
		NSSName:    "",
		Protocol:   "TLSv1.2", ProtocolCode: 771,
		Kx: "ECDH",
		Au: "ECDSA",
		Enc: Encryption{
			Cipher: "ChaCha20",
			Bits:   256,
		},
		Mac:  "AEAD",
		Code: 52244,
	},
	"ECDHE-ECDSA-CHACHA20-POLY1305": CipherSuite{
		IANAName:   "TLS_ECDHE_ECDSA_WITH_CHACHA20_POLY1305",
		GnuTLSName: "TLS_ECDHE_ECDSA_CHACHA20_POLY1305",
		NSSName:    "TLS_ECDHE_ECDSA_WITH_CHACHA20_POLY1305_SHA256",
		Protocol:   "TLSv1.2", ProtocolCode: 771,
		Kx: "ECDH",
		Au: "ECDSA",
		Enc: Encryption{
			Cipher: "ChaCha20",
			Bits:   256,
		},
		Mac:  "AEAD",
		Code: 52393,
	},
	"ECDHE-ECDSA-DES-CBC3-SHA": CipherSuite{
		IANAName:   "TLS_ECDHE_ECDSA_WITH_3DES_EDE_CBC_SHA",
		GnuTLSName: "TLS_ECDHE_ECDSA_3DES_EDE_CBC_SHA1",
		NSSName:    "TLS_ECDHE_ECDSA_WITH_3DES_EDE_CBC_SHA",
		Protocol:   "SSLv3", ProtocolCode: 768,
		Kx: "ECDH",
		Au: "ECDSA",
		Enc: Encryption{
			Cipher: "3DES",
			Bits:   168,
		},
		Mac:  "SHA1",
		Code: 49160,
	},
	"ECDHE-ECDSA-RC4-SHA": CipherSuite{
		IANAName:   "TLS_ECDHE_ECDSA_WITH_RC4_128_SHA",
		GnuTLSName: "TLS_ECDHE_ECDSA_ARCFOUR_128_SHA1",
		NSSName:    "TLS_ECDHE_ECDSA_WITH_RC4_128_SHA",
		Protocol:   "SSLv3", ProtocolCode: 768,
		Kx: "ECDH",
		Au: "ECDSA",
		Enc: Encryption{
			Cipher: "RC4",
			Bits:   128,
		},
		Mac:  "SHA1",
		Code: 49159,
	},
	"ECDHE-RSA-AES128-GCM-SHA256": CipherSuite{
		IANAName:   "TLS_ECDHE_RSA_WITH_AES_128_GCM_SHA256",
		GnuTLSName: "TLS_ECDHE_RSA_AES_128_GCM_SHA256",
		NSSName:    "TLS_ECDHE_RSA_WITH_AES_128_GCM_SHA256",
		Protocol:   "TLSv1.2", ProtocolCode: 771,
		Kx: "ECDH",
		Au: "RSA",
		Enc: Encryption{
			Cipher: "AESGCM",
			Bits:   128,
		},
		Mac:  "AEAD",
		Code: 49199,
	},
	"ECDHE-RSA-AES128-SHA": CipherSuite{
		IANAName:   "TLS_ECDHE_RSA_WITH_AES_128_CBC_SHA",
		GnuTLSName: "TLS_ECDHE_RSA_AES_128_CBC_SHA1",
		NSSName:    "TLS_ECDHE_RSA_WITH_AES_128_CBC_SHA",
		Protocol:   "SSLv3", ProtocolCode: 768,
		Kx: "ECDH",
		Au: "RSA",
		Enc: Encryption{
			Cipher: "AES",
			Bits:   128,
		},
		Mac:  "SHA1",
		Code: 49171,
	},
	"ECDHE-RSA-AES128-SHA256": CipherSuite{
		IANAName:   "TLS_ECDHE_RSA_WITH_AES_128_CBC_SHA256",
		GnuTLSName: "TLS_ECDHE_RSA_AES_128_CBC_SHA256",
		NSSName:    "TLS_ECDHE_RSA_WITH_AES_128_CBC_SHA256",
		Protocol:   "TLSv1.2", ProtocolCode: 771,
		Kx: "ECDH",
		Au: "RSA",
		Enc: Encryption{
			Cipher: "AES",
			Bits:   128,
		},
		Mac:  "SHA256",
		Code: 49191,
	},
	"ECDHE-RSA-AES256-GCM-SHA384": CipherSuite{
		IANAName:   "TLS_ECDHE_RSA_WITH_AES_256_GCM_SHA384",
		GnuTLSName: "TLS_ECDHE_RSA_AES_256_GCM_SHA384",
		NSSName:    "TLS_ECDHE_RSA_WITH_AES_256_GCM_SHA384",
		Protocol:   "TLSv1.2", ProtocolCode: 771,
		Kx: "ECDH",
		Au: "RSA",
		Enc: Encryption{
			Cipher: "AESGCM",
			Bits:   256,
		},
		Mac:  "AEAD",
		Code: 49200,
	},
	"ECDHE-RSA-AES256-SHA": CipherSuite{
		IANAName:   "TLS_ECDHE_RSA_WITH_AES_256_CBC_SHA",
		GnuTLSName: "TLS_ECDHE_RSA_AES_256_CBC_SHA1",
		NSSName:    "TLS_ECDHE_RSA_WITH_AES_256_CBC_SHA",
		Protocol:   "SSLv3", ProtocolCode: 768,
		Kx: "ECDH",
		Au: "RSA",
		Enc: Encryption{
			Cipher: "AES",
			Bits:   256,
		},
		Mac:  "SHA1",
		Code: 49172,
	},
	"ECDHE-RSA-AES256-SHA384": CipherSuite{
		IANAName:   "TLS_ECDHE_RSA_WITH_AES_256_CBC_SHA384",
		GnuTLSName: "TLS_ECDHE_RSA_AES_256_CBC_SHA384",
		NSSName:    "TLS_ECDHE_RSA_WITH_AES_256_CBC_SHA384",
		Protocol:   "TLSv1.2", ProtocolCode: 771,
		Kx: "ECDH",
		Au: "RSA",
		Enc: Encryption{
			Cipher: "AES",
			Bits:   256,
		},
		Mac:  "SHA384",
		Code: 49192,
	},
	"ECDHE-RSA-CAMELLIA128-SHA256": CipherSuite{
		IANAName:   "TLS_ECDHE_RSA_WITH_CAMELLIA_128_CBC_SHA256",
		GnuTLSName: "TLS_ECDHE_RSA_CAMELLIA_128_CBC_SHA256",
		NSSName:    "",
		Protocol:   "TLSv1.2", ProtocolCode: 771,
		Kx: "ECDH",
		Au: "RSA",
		Enc: Encryption{
			Cipher: "Camellia",
			Bits:   128,
		},
		Mac:  "SHA256",
		Code: 49270,
	},
	"ECDHE-RSA-CAMELLIA256-SHA384": CipherSuite{
		IANAName:   "TLS_ECDHE_RSA_WITH_CAMELLIA_256_CBC_SHA384",
		GnuTLSName: "TLS_ECDHE_RSA_CAMELLIA_256_CBC_SHA384",
		NSSName:    "",
		Protocol:   "TLSv1.2", ProtocolCode: 771,
		Kx: "ECDH",
		Au: "RSA",
		Enc: Encryption{
			Cipher: "Camellia",
			Bits:   256,
		},
		Mac:  "SHA384",
		Code: 49271,
	},
	"ECDHE-RSA-CHACHA20-POLY1305-OLD": CipherSuite{
		IANAName:   "",
		GnuTLSName: "",
		NSSName:    "",
		Protocol:   "TLSv1.2", ProtocolCode: 771,
		Kx: "ECDH",
		Au: "RSA",
		Enc: Encryption{
			Cipher: "ChaCha20",
			Bits:   256,
		},
		Mac:  "AEAD",
		Code: 52243,
	},
	"ECDHE-RSA-CHACHA20-POLY1305": CipherSuite{
		IANAName:   "TLS_ECDHE_RSA_WITH_CHACHA20_POLY1305",
		GnuTLSName: "TLS_ECDHE_RSA_CHACHA20_POLY1305",
		NSSName:    "TLS_ECDHE_RSA_WITH_CHACHA20_POLY1305_SHA256",
		Protocol:   "TLSv1.2", ProtocolCode: 771,
		Kx: "ECDH",
		Au: "RSA",
		Enc: Encryption{
			Cipher: "ChaCha20",
			Bits:   256,
		},
		Mac:  "AEAD",
		Code: 52392,
	},
	"ECDHE-RSA-DES-CBC3-SHA": CipherSuite{
		IANAName:   "TLS_ECDHE_RSA_WITH_3DES_EDE_CBC_SHA",
		GnuTLSName: "TLS_ECDHE_RSA_3DES_EDE_CBC_SHA1",
		NSSName:    "TLS_ECDHE_RSA_WITH_3DES_EDE_CBC_SHA",
		Protocol:   "SSLv3", ProtocolCode: 768,
		Kx: "ECDH",
		Au: "RSA",
		Enc: Encryption{
			Cipher: "3DES",
			Bits:   168,
		},
		Mac:  "SHA1",
		Code: 49170,
	},
	"ECDHE-RSA-RC4-SHA": CipherSuite{
		IANAName:   "TLS_ECDHE_RSA_WITH_RC4_128_SHA",
		GnuTLSName: "TLS_ECDHE_RSA_ARCFOUR_128_SHA1",
		NSSName:    "TLS_ECDHE_RSA_WITH_RC4_128_SHA",
		Protocol:   "SSLv3", ProtocolCode: 768,
		Kx: "ECDH",
		Au: "RSA",
		Enc: Encryption{
			Cipher: "RC4",
			Bits:   128,
		},
		Mac:  "SHA1",
		Code: 49169,
	},
	"EDH-DSS-DES-CBC3-SHA": CipherSuite{
		IANAName:   "TLS_DHE_DSS_WITH_3DES_EDE_CBC_SHA",
		GnuTLSName: "TLS_DHE_DSS_3DES_EDE_CBC_SHA1",
		NSSName:    "TLS_DHE_DSS_WITH_3DES_EDE_CBC_SHA",
		Protocol:   "SSLv3", ProtocolCode: 768,
		Kx: "DH",
		Au: "DSS",
		Enc: Encryption{
			Cipher: "3DES",
			Bits:   168,
		},
		Mac:  "SHA1",
		Code: 19,
	},
	"EDH-RSA-DES-CBC3-SHA": CipherSuite{
		IANAName:   "TLS_DHE_RSA_WITH_3DES_EDE_CBC_SHA",
		GnuTLSName: "TLS_DHE_RSA_3DES_EDE_CBC_SHA1",
		NSSName:    "TLS_DHE_RSA_WITH_3DES_EDE_CBC_SHA",
		Protocol:   "SSLv3", ProtocolCode: 768,
		Kx: "DH",
		Au: "RSA",
		Enc: Encryption{
			Cipher: "3DES",
			Bits:   168,
		},
		Mac:  "SHA1",
		Code: 22,
	},
	"IDEA-CBC-SHA": CipherSuite{
		IANAName:   "TLS_RSA_WITH_IDEA_CBC_SHA",
		GnuTLSName: "",
		NSSName:    "TLS_RSA_WITH_IDEA_CBC_SHA",
		Protocol:   "SSLv3", ProtocolCode: 768,
		Kx: "RSA",
		Au: "RSA",
		Enc: Encryption{
			Cipher: "IDEA",
			Bits:   128,
		},
		Mac:  "SHA1",
		Code: 7,
	},
	"PSK-3DES-EDE-CBC-SHA": CipherSuite{
		IANAName:   "TLS_PSK_WITH_3DES_EDE_CBC_SHA",
		GnuTLSName: "TLS_PSK_3DES_EDE_CBC_SHA1",
		NSSName:    "",
		Protocol:   "SSLv3", ProtocolCode: 768,
		Kx: "PSK",
		Au: "PSK",
		Enc: Encryption{
			Cipher: "3DES",
			Bits:   168,
		},
		Mac:  "SHA1",
		Code: 139,
	},
	"PSK-AES128-CBC-SHA": CipherSuite{
		IANAName:   "TLS_PSK_WITH_AES_128_CBC_SHA",
		GnuTLSName: "TLS_PSK_AES_128_CBC_SHA1",
		NSSName:    "",
		Protocol:   "SSLv3", ProtocolCode: 768,
		Kx: "PSK",
		Au: "PSK",
		Enc: Encryption{
			Cipher: "AES",
			Bits:   128,
		},
		Mac:  "SHA1",
		Code: 140,
	},
	"PSK-AES256-CBC-SHA": CipherSuite{
		IANAName:   "TLS_PSK_WITH_AES_256_CBC_SHA",
		GnuTLSName: "TLS_PSK_AES_256_CBC_SHA1",
		NSSName:    "",
		Protocol:   "SSLv3", ProtocolCode: 768,
		Kx: "PSK",
		Au: "PSK",
		Enc: Encryption{
			Cipher: "AES",
			Bits:   256,
		},
		Mac:  "SHA1",
		Code: 141,
	},
	"PSK-RC4-SHA": CipherSuite{
		IANAName:   "TLS_PSK_WITH_RC4_128_SHA",
		GnuTLSName: "TLS_PSK_ARCFOUR_128_SHA1",
		NSSName:    "",
		Protocol:   "SSLv3", ProtocolCode: 768,
		Kx: "PSK",
		Au: "PSK",
		Enc: Encryption{
			Cipher: "RC4",
			Bits:   128,
		},
		Mac:  "SHA1",
		Code: 138,
	},
	"RC4-MD5": CipherSuite{
		IANAName:   "TLS_RSA_WITH_RC4_128_MD5",
		GnuTLSName: "TLS_RSA_ARCFOUR_128_MD5",
		NSSName:    "TLS_RSA_WITH_RC4_128_MD5",
		Protocol:   "SSLv3", ProtocolCode: 768,
		Kx: "RSA",
		Au: "RSA",
		Enc: Encryption{
			Cipher: "RC4",
			Bits:   128,
		},
		Mac:  "MD5",
		Code: 4,
	},
	"RC4-SHA": CipherSuite{
		IANAName:   "TLS_RSA_WITH_RC4_128_SHA",
		GnuTLSName: "TLS_RSA_ARCFOUR_128_SHA1",
		NSSName:    "TLS_RSA_WITH_RC4_128_SHA",
		Protocol:   "SSLv3", ProtocolCode: 768,
		Kx: "RSA",
		Au: "RSA",
		Enc: Encryption{
			Cipher: "RC4",
			Bits:   128,
		},
		Mac:  "SHA1",
		Code: 5,
	},
	"RSA-PSK-3DES-EDE-CBC-SHA": CipherSuite{
		IANAName:   "TLS_RSA_PSK_WITH_3DES_EDE_CBC_SHA",
		GnuTLSName: "TLS_RSA_PSK_3DES_EDE_CBC_SHA1",
		NSSName:    "",
		Protocol:   "SSLv3", ProtocolCode: 768,
		Kx: "RSAPSK",
		Au: "RSA",
		Enc: Encryption{
			Cipher: "3DES",
			Bits:   168,
		},
		Mac:  "SHA1",
		Code: 147,
	},
	"RSA-PSK-AES128-CBC-SHA": CipherSuite{
		IANAName:   "TLS_RSA_PSK_WITH_AES_128_CBC_SHA",
		GnuTLSName: "TLS_RSA_PSK_AES_128_CBC_SHA1",
		NSSName:    "",
		Protocol:   "SSLv3", ProtocolCode: 768,
		Kx: "RSAPSK",
		Au: "RSA",
		Enc: Encryption{
			Cipher: "AES",
			Bits:   128,
		},
		Mac:  "SHA1",
		Code: 148,
	},
	"RSA-PSK-AES256-CBC-SHA": CipherSuite{
		IANAName:   "TLS_RSA_PSK_WITH_AES_256_CBC_SHA",
		GnuTLSName: "TLS_RSA_PSK_AES_256_CBC_SHA1",
		NSSName:    "",
		Protocol:   "SSLv3", ProtocolCode: 768,
		Kx: "RSAPSK",
		Au: "RSA",
		Enc: Encryption{
			Cipher: "AES",
			Bits:   256,
		},
		Mac:  "SHA1",
		Code: 149,
	},
	"RSA-PSK-RC4-SHA": CipherSuite{
		IANAName:   "TLS_RSA_PSK_WITH_RC4_128_SHA",
		GnuTLSName: "TLS_RSA_PSK_ARCFOUR_128_SHA1",
		NSSName:    "",
		Protocol:   "SSLv3", ProtocolCode: 768,
		Kx: "RSAPSK",
		Au: "RSA",
		Enc: Encryption{
			Cipher: "RC4",
			Bits:   128,
		},
		Mac:  "SHA1",
		Code: 146,
	},
	"SEED-SHA": CipherSuite{
		IANAName:   "TLS_RSA_WITH_SEED_CBC_SHA",
		GnuTLSName: "",
		NSSName:    "TLS_RSA_WITH_SEED_CBC_SHA",
		Protocol:   "SSLv3", ProtocolCode: 768,
		Kx: "RSA",
		Au: "RSA",
		Enc: Encryption{
			Cipher: "SEED",
			Bits:   128,
		},
		Mac:  "SHA1",
		Code: 150,
	},
	"SRP-3DES-EDE-CBC-SHA": CipherSuite{
		IANAName:   "TLS_SRP_SHA_WITH_3DES_EDE_CBC_SHA",
		GnuTLSName: "TLS_SRP_SHA_3DES_EDE_CBC_SHA1",
		NSSName:    "",
		Protocol:   "SSLv3", ProtocolCode: 768,
		Kx: "SRP",
		Au: "SRP",
		Enc: Encryption{
			Cipher: "3DES",
			Bits:   168,
		},
		Mac:  "SHA1",
		Code: 49178,
	},
	"SRP-AES-128-CBC-SHA": CipherSuite{
		IANAName:   "TLS_SRP_SHA_WITH_AES_128_CBC_SHA",
		GnuTLSName: "TLS_SRP_SHA_AES_128_CBC_SHA1",
		NSSName:    "",
		Protocol:   "SSLv3", ProtocolCode: 768,
		Kx: "SRP",
		Au: "SRP",
		Enc: Encryption{
			Cipher: "AES",
			Bits:   128,
		},
		Mac:  "SHA1",
		Code: 49181,
	},
	"SRP-AES-256-CBC-SHA": CipherSuite{
		IANAName:   "TLS_SRP_SHA_WITH_AES_256_CBC_SHA",
		GnuTLSName: "TLS_SRP_SHA_AES_256_CBC_SHA1",
		NSSName:    "",
		Protocol:   "SSLv3", ProtocolCode: 768,
		Kx: "SRP",
		Au: "SRP",
		Enc: Encryption{
			Cipher: "AES",
			Bits:   256,
		},
		Mac:  "SHA1",
		Code: 49184,
	},
	"SRP-DSS-3DES-EDE-CBC-SHA": CipherSuite{
		IANAName:   "TLS_SRP_SHA_DSS_WITH_3DES_EDE_CBC_SHA",
		GnuTLSName: "TLS_SRP_SHA_DSS_3DES_EDE_CBC_SHA1",
		NSSName:    "",
		Protocol:   "SSLv3", ProtocolCode: 768,
		Kx: "SRP",
		Au: "DSS",
		Enc: Encryption{
			Cipher: "3DES",
			Bits:   168,
		},
		Mac:  "SHA1",
		Code: 49180,
	},
	"SRP-DSS-AES-128-CBC-SHA": CipherSuite{
		IANAName:   "TLS_SRP_SHA_DSS_WITH_AES_128_CBC_SHA",
		GnuTLSName: "TLS_SRP_SHA_DSS_AES_128_CBC_SHA1",
		NSSName:    "",
		Protocol:   "SSLv3", ProtocolCode: 768,
		Kx: "SRP",
		Au: "DSS",
		Enc: Encryption{
			Cipher: "AES",
			Bits:   128,
		},
		Mac:  "SHA1",
		Code: 49183,
	},
	"SRP-DSS-AES-256-CBC-SHA": CipherSuite{
		IANAName:   "TLS_SRP_SHA_DSS_WITH_AES_256_CBC_SHA",
		GnuTLSName: "TLS_SRP_SHA_DSS_AES_256_CBC_SHA1",
		NSSName:    "",
		Protocol:   "SSLv3", ProtocolCode: 768,
		Kx: "SRP",
		Au: "DSS",
		Enc: Encryption{
			Cipher: "AES",
			Bits:   256,
		},
		Mac:  "SHA1",
		Code: 49186,
	},
	"SRP-RSA-3DES-EDE-CBC-SHA": CipherSuite{
		IANAName:   "TLS_SRP_SHA_RSA_WITH_3DES_EDE_CBC_SHA",
		GnuTLSName: "TLS_SRP_SHA_RSA_3DES_EDE_CBC_SHA1",
		NSSName:    "",
		Protocol:   "SSLv3", ProtocolCode: 768,
		Kx: "SRP",
		Au: "RSA",
		Enc: Encryption{
			Cipher: "3DES",
			Bits:   168,
		},
		Mac:  "SHA1",
		Code: 49179,
	},
	"SRP-RSA-AES-128-CBC-SHA": CipherSuite{
		IANAName:   "TLS_SRP_SHA_RSA_WITH_AES_128_CBC_SHA",
		GnuTLSName: "TLS_SRP_SHA_RSA_AES_128_CBC_SHA1",
		NSSName:    "",
		Protocol:   "SSLv3", ProtocolCode: 768,
		Kx: "SRP",
		Au: "RSA",
		Enc: Encryption{
			Cipher: "AES",
			Bits:   128,
		},
		Mac:  "SHA1",
		Code: 49182,
	},
	"SRP-RSA-AES-256-CBC-SHA": CipherSuite{
		IANAName:   "TLS_SRP_SHA_RSA_WITH_AES_256_CBC_SHA",
		GnuTLSName: "TLS_SRP_SHA_RSA_AES_256_CBC_SHA1",
		NSSName:    "",
		Protocol:   "SSLv3", ProtocolCode: 768,
		Kx: "SRP",
		Au: "RSA",
		Enc: Encryption{
			Cipher: "AES",
			Bits:   256,
		},
		Mac:  "SHA1",
		Code: 49185,
	},
}
