//  Copyright (c) 2014 Couchbase, Inc.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
// 		http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

package regexp

import (
	"fmt"
	"regexp"

	"github.com/blevesearch/bleve/v2/analysis"
	"github.com/blevesearch/bleve/v2/registry"
)

const Name = "regexp"

type CharFilter struct {
	r           *regexp.Regexp
	replacement []byte
}

func New(r *regexp.Regexp, replacement []byte) *CharFilter {
	return &CharFilter{
		r:           r,
		replacement: replacement,
	}
}

func (s *CharFilter) Filter(input []byte) []byte {
	return s.r.ReplaceAll(input, s.replacement)
}

func CharFilterConstructor(config map[string]interface{}, cache *registry.Cache) (analysis.CharFilter, error) {
	regexpStr, ok := config["regexp"].(string)
	if !ok {
		return nil, fmt.Errorf("must specify regexp")
	}
	r, err := regexp.Compile(regexpStr)
	if err != nil {
		return nil, fmt.Errorf("unable to build regexp char filter: %v", err)
	}
	replaceBytes := []byte(" ")
	replaceStr, ok := config["replace"].(string)
	if ok {
		replaceBytes = []byte(replaceStr)
	}
	return New(r, replaceBytes), nil
}

func init() {
	registry.RegisterCharFilter(Name, CharFilterConstructor)
}
