/*****************************************************************************
 *  FTools - Freenet Client Protocol Tools
 *
 *  Copyright (C) 2002 Juergen Buchmueller <juergen@pullmoll.de>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 *	$Id: ftsite.c,v 1.66 2005/07/19 13:52:42 pullmoll Exp $
 *****************************************************************************/
#include "ftsite.h"
#include "ftmime.h"
#include "ftlog.h"

#define	MAXBUFF	32768

int sort_name(const void *p1, const void *p2)
{
	fileinfo_t *fi1 = *(fileinfo_t **)p1;
	fileinfo_t *fi2 = *(fileinfo_t **)p2;
	return strcasecmp(fi1->name, fi2->name);
}

static char *time_str(time_t t)
{
	static char buff[4][64];
	static int which = 0;
	struct tm *tm;

	tm = gmtime(&t);
	which = (which + 1) % 4;
	strftime(buff[which], sizeof(buff[which]), "%Y-%m-%d %H:%M:%S GMT", tm);

	return buff[which];
}

int put_site(fcp_t **pf, const char *uri, int htl, int wiggle, const char *path)
{
	fcp_t *f = *pf;
	char **paths = NULL;
	char *cmdline = NULL, *chk = NULL, *ssk = NULL;
	char *map_uri = NULL, *dst, *slash;
	char *meta = NULL;
	size_t path0, paths_max = 0, paths_idx = 0, paths_cnt = 0;
	fileinfo_t **fileinfo = NULL, *fi;
	size_t i, fileinfo_max = 0, fileinfo_idx = 0, fileinfo_cnt = 0;
	size_t metasize;
	DIR *dir = NULL;
	struct dirent *de;
	struct stat st;
	unsigned long long total;
	int fd, nfds, pending, minfd, maxfd;
	fd_set rdfds, exfds;
	struct timeval tv;
	struct tm *tm;
	time_t date0 = 0, date1 = 0;
	int rc = -1;

	cmdline = calloc(MAXBUFF, 1);
	if (NULL == cmdline) {
		LOG(L_ERROR,("barf: calloc(%d,%d) failed (%s)\n",
			MAXBUFF, 1, strerror(errno)));
		goto bailout;
	}

	if (0 != stat(path, &st)) {
		LOG(L_ERROR,("cannot stat \"%s\" (%s)\n",
			path, strerror(errno)));
		goto bailout;
	}

	if (!S_ISDIR(st.st_mode)) {
		LOG(L_ERROR,("\"%s\" is not a directory\n",
			path));
		goto bailout;
	}

	path0 = strlen(path);

	paths_max = 32;
	paths_cnt = 0;
	paths = calloc(sizeof(char *), paths_max);
	if (NULL == paths) {
		LOG(L_ERROR,("barf: calloc(%d,%d) failed (%s)\n",
			sizeof(char *), paths_max, strerror(errno)));
		goto bailout;
	}

	fileinfo_max = 256;
	fileinfo_cnt = 0;
	fileinfo = calloc(sizeof(fileinfo_t *), fileinfo_max);
	if (NULL == fileinfo) {
		LOG(L_ERROR,("barf: calloc(%d,%d) failed (%s)\n",
			sizeof(fileinfo_t *), fileinfo_max, strerror(errno)));
		goto bailout;
	}

	paths[paths_cnt] = strdup(path);
	if (NULL == paths[paths_cnt]) {
		LOG(L_ERROR,("barf: strdup(\"%s\") failed (%s)\n",
			path, strerror(errno)));
		goto bailout;
	}

	paths_cnt++;
	paths_idx = 0;
	fileinfo_idx = 0;
	while (paths_idx < paths_cnt) {
		dir = opendir(paths[paths_idx]);
		if (NULL == dir) {
			LOG(L_ERROR,("opendir(\"%s\") failed (%s)\n",
				path, strerror(errno)));
			goto bailout;
		}
		while (NULL != (de = readdir(dir))) {
			size_t len;
			char *filename;

			if (0 == strcmp(de->d_name, "."))
				continue;
			if (0 == strcmp(de->d_name, ".."))
				continue;
			len = strlen(paths[paths_idx]) + 1 + strlen(de->d_name) + 1;
			filename = calloc(len, 1);
			if (NULL == filename) {
				LOG(L_ERROR,("barf: calloc(%d,%d) failed (%s)\n",
					sizeof(char *), paths_max, strerror(errno)));
				goto bailout;
			}
			snprintf(filename, len, "%s/%s",
				paths[paths_idx], de->d_name);
			if (0 != stat(filename, &st)) {
				LOG(L_ERROR,("cannot stat \"%s\"\n", filename));
				free(filename);
				continue;
			}
			if (S_ISDIR(st.st_mode)) {
				if (paths_cnt >= paths_max) {
					paths_max += 32;
					paths = realloc(paths, paths_max * sizeof(char *));
					if (NULL == paths) {
						LOG(L_ERROR,("barf: realloc(...,%d) failed (%s)\n",
							paths_max * sizeof(char *), strerror(errno)));
						goto bailout;
					}
				}
				paths[paths_cnt++] = filename;

			} else {
				if (fileinfo_cnt >= fileinfo_max) {
					fileinfo_max += 256;
					fileinfo = realloc(fileinfo,
						fileinfo_max * sizeof(fileinfo_t *));
					if (NULL == fileinfo) {
						LOG(L_ERROR,("barf: realloc(...,%d) failed (%s)\n",
							fileinfo_max * sizeof(fileinfo_t *),
							strerror(errno)));
						goto bailout;
					}
				}
				fi = fileinfo[fileinfo_cnt] = calloc(sizeof(fileinfo_t), 1);
				if (NULL == fi) {
					LOG(L_ERROR,("barf: calloc(%d,%d) failed (%s)\n",
						sizeof(fileinfo_t), 1, strerror(errno)));
					goto bailout;
				}
				fi->size = st.st_size;
				fi->name = filename;
				fi->mimetype = strdup(mimetype_guess(filename));
				fileinfo_cnt++;
			}
		}
		closedir(dir);
		dir = NULL;
		paths_idx++;
	}

	qsort(fileinfo, fileinfo_cnt, sizeof(fileinfo_t *), sort_name);

	total = 0;
	for (i = 0; i < fileinfo_cnt; i++) {
		fileinfo_t *fi = fileinfo[i];
		total += fi->size;
	}
	printf("Inserting %llu bytes in %d files\n",
		total, fileinfo_cnt);
	if (0 == date_incr) {
		printf("Not inserting a date based redirect\n");
	} else {
		gettimeofday(&tv, NULL);
		tm = gmtime((const time_t *)&tv.tv_sec);
		date0 = timegm(tm);
		if (0 == date_incr) {
			date_incr = 24 * 60 * 60;
		}
		date0 -= date0 % date_incr;
		date0 += date_future * date_incr;
		date1 = date0 + date_incr - 1;

		printf("Date range %s to %s\n",
			time_str(date0), time_str(date1));
	}
	printf("Base URI %s\n",
		uri);

	for (;;) {

		/* construct file descriptor sets for pending inserts */
		FD_ZERO(&rdfds);
		FD_ZERO(&exfds);
		pending = 0;
		nfds = 0;
		minfd = 32767;
		maxfd = 0;
		LOG(L_DEBUGX,("pending:"));
		for (i = 0; i < fileinfo_cnt; i++) {
			fileinfo_t *fi = fileinfo[i];
			if (NULL != fi && NULL != fi->fp && NULL == fi->chk) {
				fd = fileno(fi->fp);
				FD_SET(fd, &rdfds);
				FD_SET(fd, &exfds);
				if (fd < minfd)
					minfd = fd;
				if (fd > maxfd)
					maxfd = fd;
				pending += 1;
				nfds += 1;
				LOG(L_DEBUGX,(" #%d (fd:%d)", i+1, fd));
			}
		}
		LOG(L_DEBUGX,("\n"));

		/* add another insert thread? */
		while (nfds < in_threads) {
			for (fi = NULL, i = 0; i < fileinfo_cnt; i++) {
				fi = fileinfo[i];
				if (NULL != fi && NULL == fi->fp && NULL == fi->chk)
					break;
			}
			if (NULL == fi || i == fileinfo_cnt)
				break;
			pending += 1;
			fi->htl = htl;
			if (wiggle > 0) {
				fi->htl += (rand() % wiggle) - wiggle / 2;
				if (fi->htl < 1) {
					fi->htl = 1;
				}
			}
			snprintf(cmdline, MAXBUFF,
				"\"%s\" %s -n%s -p%d -l%d -i%d -t%d -s%u -v%d %s%s%sCHK@ \"%s\"",
				program, "put", fcphost, fcpport, fi->htl, in_threads, sf_threads,
				chunksize, verbose,
				delete ? "--delete " : "", 
				meta_only ? "--meta-only " : "", 
				meta_none ? "--meta-none " : "", 
				fi->name);
			fi->fp = popen(cmdline, "r");
			if (NULL == fi->fp) {
				LOG(L_ERROR,("popen(\"%s\",\"r\") failed (%s)\n",
					cmdline, strerror(errno)));
			} else {
				fd = fileno(fi->fp);
				FD_SET(fd, &rdfds);
				FD_SET(fd, &exfds);
				if (fd < minfd)
					minfd = fd;
				if (fd > maxfd)
					maxfd = fd;
				nfds += 1;
				LOG(L_NORMAL,("Inserting file #%d/%d %s (%d, %s) with HTL:%d\n",
					i+1, fileinfo_cnt, fi->name + path0 + 1,
					fi->size, fi->mimetype, fi->htl));
				LOG(L_DEBUG,("%s\n", cmdline));
			}
		}

		/* bail out once all files were inserted */
		if (0 == pending) {
			break;
		}

		if (nfds > 0) {
			struct timeval tv = {1,0};

			fd = select(maxfd+1, &rdfds, NULL, &exfds, &tv);
			if (-1 == fd) {
				LOG(L_ERROR,("select(%d,...) failed (%s)\n",
					maxfd+1, strerror(errno)));
				goto bailout;
			}
			for (fd = minfd; fd < maxfd+1; fd++) {
				fi = NULL;
				for (i = 0; i < fileinfo_cnt; i++) {
					fi = fileinfo[i];
					if (NULL != fi &&
						NULL != fi->fp &&
						fd == fileno(fi->fp))
						break;
				}
				if (FD_ISSET(fd, &rdfds)) {
					char *reply = calloc(MAXBUFF, 1);
					char *line, *result, *chk;

					if (NULL == reply) {
						LOG(L_ERROR,("barf: calloc(%d,%d) failed (%s)\n",
							MAXBUFF, 1, strerror(errno)));
						goto bailout;
					}
					fgets(reply, MAXBUFF, fi->fp);
					line = strtok(reply, "\r\n");
					if (NULL == line) {
						result = NULL;
						chk = NULL;
					} else {
						LOG(L_DEBUG,("reply: \"%s\"\n", line));
						result = strtok(line, ":\r\n");
						chk = strtok(NULL, "\r\n");
						while (chk && *chk == ' ')
							chk++;
						LOG(L_NORMAL,("%s: file #%d/%d %s (%d, %s)\n",
							result, i+1, fileinfo_cnt, fi->name + path0 + 1,
							fi->size, fi->mimetype));
						LOG(L_MINOR,("%s\n", chk));
					}
					pclose(fi->fp);
					fi->fp = NULL;
					if (NULL != chk && NULL != result &&
						(0 == strcasecmp(result, SUCCESS) ||
						0 == strcasecmp(result, KEYCOLLISION))) {
						fi->chk = strdup(chk);
					}
					free(reply);
				} else if (FD_ISSET(fd, &exfds)) {
					if (NULL != fi->fp) {
						LOG(L_MINOR,("closing #%d/%d %s\n",
							i+1, fileinfo_cnt,
							fi->name + path0 + 1));
						pclose(fi->fp);
						fi->fp = NULL;
					}
				}
			}
		}
	}

	metasize = 0;
	metasize += strlen("Version\n");
	metasize += strlen("Revision=xxxxxx\n");
	metasize += strlen("EndPart\n");
	for (i = 0; i < fileinfo_cnt; i++) {
		fi = fileinfo[i];
		if (0 == strcasecmp(default_file, fi->name + path0 + 1)) {
			metasize += strlen("Document\n");
			metasize += strlen("Info.Format=\n") +
				strlen(mimetype_guess(fi->name));
			metasize += strlen("Redirect.Target=\n") +
				strlen(fi->chk);
			metasize += strlen("EndPart\n");
		}
		metasize += strlen("Document\n");
		metasize += strlen("Name=\n") + 
			strlen(fi->name + path0 + 1);
		metasize += strlen("Info.Format=\n") +
			strlen(mimetype_guess(fi->name));
		metasize += strlen("Redirect.Target=\n") +
			strlen(fi->chk);
		if (i + 1 < fileinfo_cnt) {
			metasize += strlen("EndPart\n");
		} else {
			metasize += strlen("End\n");
		}
	}

	dst = meta = calloc(metasize, 1);
	if (NULL == meta) {
		LOG(L_ERROR,("barf: calloc(%d,%d) failed (%s)\n",
			metasize, 1, strerror(errno)));
		goto bailout;
	}
	dst += sprintf(dst, "Version\n");
	dst += sprintf(dst, "Revision=%d\n", revision);
	dst += sprintf(dst, "EndPart\n");
	for (i = 0; i < fileinfo_cnt; i++) {
		fi = fileinfo[i];
		if (0 == strcasecmp(default_file, fi->name + path0 + 1)) {
			dst += sprintf(dst, "Document\n");
			dst += sprintf(dst, "Info.Format=%s\n",
				mimetype_guess(fi->name));
			dst += sprintf(dst, "Redirect.Target=%s\n",
				fi->chk);
			dst += sprintf(dst, "EndPart\n");
		}
		dst += sprintf(dst, "Document\n");
		dst += sprintf(dst, "Name=%s\n",
			fi->name + path0 + 1);
		dst += sprintf(dst, "Info.Format=%s\n",
			mimetype_guess(fi->name));
		dst += sprintf(dst, "Redirect.Target=%s\n",
			fi->chk);
		if (i + 1 < fileinfo_cnt) {
			dst += sprintf(dst, "EndPart\n");
		} else {
			dst += sprintf(dst, "End\n");
		}
	}

	if (NULL != map_uri) {
		free(map_uri);
		map_uri = NULL;
	}
	if (0 == date_incr) {
		/* if this is a non-DBR insert, put the map space to the URI */
		map_uri = strdup(uri);
		if (NULL == map_uri) {
			LOG(L_ERROR,("barf: strdup(\"%s\") failed (%s)\n",
				uri, strerror(errno)));
			goto bailout;
		}
	} else {
		size_t size = MAXPATHLEN;
		char *slash;

		map_uri = calloc(size, 1);
		if (NULL == map_uri) {
			LOG(L_ERROR,("barf: calloc(%d,%d) failed (%s)\n",
				size, 1, strerror(errno)));
			goto bailout;
		}
		strcpy(map_uri, uri);
		slash = strchr(map_uri, '/');
		if (NULL == slash) {
			LOG(L_ERROR,("FATAL: no slash found in URI (%s)\n",
				uri));
			goto bailout;
		}
		/* make room for 8 hex digits and a dash */
		memmove(slash + 8 + 1, slash, strlen(slash) + 2);
		snprintf(slash + 1, 8 + 1, "%08x", (unsigned)date0);
		slash[8 + 1] = '-';
	}
	metasize = (size_t)(dst - meta);

	for (;;) {
		int htl0 = htl;

		if (wiggle > 0) {
			htl0 += (rand() % wiggle) - wiggle / 2;
			if (htl0 < 1) {
				htl0 = 1;
			}
		}

		fcp_free(pf);
		if (0 != (rc = fcp_new(pf, fcphost, fcpport))) {
			LOG(L_ERROR,("ERROR: cannot connect to %s:%d\n",
				fcphost, fcpport));
			goto bailout;
		}
		f = *pf;
		f->meta = (char *)calloc(metasize + 1, 1);
		if (NULL == f->meta) {
			LOG(L_ERROR,("barf: calloc(%d,%d) failed (%s)\n",
				metasize, 1, strerror(errno)));
			goto bailout;
		}
		memcpy(f->meta, meta, metasize);
		f->metasize = metasize;

		if (0 != limit32k && f->metasize > 32700) {
			LOG(L_ERROR,("**** INSERTING CHK@ FOR MAP SPACE > 32K (%d BYTES)\n",
				f->metasize));
			if (0 == (rc = fcp_put(f, "CHK@", htl0))) {
				if (NULL == (chk = strdup(f->uri))) {
					LOG(L_ERROR,("barf: strdup(\"%s\") failed (%s)\n",
						f->uri, strerror(errno)));
					goto bailout;
				}
				fcp_free(pf);
				if (0 != (rc = fcp_new(pf, fcphost, fcpport))) {
					LOG(L_ERROR,("ERROR: cannot connect to %s:%d\n",
						fcphost, fcpport));
					goto bailout;
				}
				f = *pf;
				dst = f->meta = calloc(512, 1);
				if (NULL == f->meta) {
					LOG(L_ERROR,("barf: calloc(%d,%d) failed (%s)\n",
						512, 1, strerror(errno)));
					goto bailout;
				}
				dst += sprintf(dst, "Version\n");
				dst += sprintf(dst, "Revision=%d\n", revision);
				dst += sprintf(dst, "EndPart\n");
				dst += sprintf(dst, "Document\n");
				dst += sprintf(dst, "Redirect.Target=%s\n", chk);
				dst += sprintf(dst, "End\n");
				f->metasize = (size_t)(dst - f->meta);
				if (0 == (rc = fcp_put(f, map_uri, htl0))) {
					LOG(L_MINOR,("%s\n", f->chk ? f->chk : f->uri));
					/* we're done... */
					goto insert_dbr;
				}
			}
			LOG(L_ERROR,("FAILED: retrying to insert map space\n"));
		} else {
			LOG(L_NORMAL,("**** INSERTING METADATA FOR MAP SPACE (%d BYTES)\n",
				f->metasize));
			/* insert map space directly to map_uri */
			if (0 == (rc = fcp_put(f, map_uri, htl0))) {
				LOG(L_MINOR,("%s\n", f->chk ? f->chk : f->uri));
				/* we're done... */
				goto insert_dbr;
			}
			LOG(L_ERROR,("FAILED: retrying to insert map space\n"));
		}

		if (NULL != map_uri) {
			free(map_uri);
			map_uri = NULL;
		}
		if (NULL != chk) {
			free(chk);
			chk = NULL;
		}
	}

insert_dbr:
	if (0 != date_incr) {
		if (NULL == (ssk = strdup(f->uri))) {
			LOG(L_ERROR,("barf: strdup(\"%s\") failed (%s)\n",
				f->uri, strerror(errno)));
			goto bailout;
		}

		slash = strchr(ssk, '/');
		if (NULL == slash) {
			LOG(L_ERROR,("FATAL: no slash found in URI (%s)\n",
				ssk));
			goto bailout;
		}
		strcpy(slash + 1, slash + 1 + 8 + 1);

		LOG(L_NORMAL,("**** INSERTING DATE BASED REDIRECT METADATA (%s)\n",
			map_uri));
		for (;;) {
			fcp_free(pf);
			if (0 != (rc = fcp_new(pf, fcphost, fcpport))) {
				LOG(L_ERROR,("ERROR: cannot connect to %s:%d\n",
					fcphost, fcpport));
				goto bailout;
			}
			/* get new fcp pointer */
			dst = f->meta = calloc(512, 1);
			if (NULL == f->meta) {
				LOG(L_ERROR,("barf: calloc(%d,%d) failed (%s)\n",
					512, 1, strerror(errno)));
				goto bailout;
			}
			dst += sprintf(dst, "Version\n");
			dst += sprintf(dst, "Revision=%d\n", revision);
			dst += sprintf(dst, "EndPart\n");
			dst += sprintf(dst, "Document\n");
			dst += sprintf(dst, "DateRedirect.Target=%s\n", ssk);
			if (0 != date_offs) {
				dst += sprintf(dst, "DateRedirect.Offset=%x\n", date_offs);
			}
			if (24 * 60 * 60 != date_incr) {
				dst += sprintf(dst, "DateRedirect.Increment=%x\n", date_incr);
			}
			dst += sprintf(dst, "End\n");
			f->metasize = (size_t)(dst - f->meta);
			if (0 == (rc = fcp_put(f, uri, htl))) {
				/* we're done... */
				break;
			}
			LOG(L_ERROR,("FAILED: retrying to insert date based redirect\n"));
			fcp_free(pf);
			if (0 != (rc = fcp_new(pf, fcphost, fcpport))) {
				LOG(L_ERROR,("ERROR: cannot connect to %s:%d\n",
					fcphost, fcpport));
				goto bailout;
			}
			f = *pf;
		}
	}

bailout:
	if (NULL != dir) {
		closedir(dir);
		dir = NULL;
	}
	if (NULL != cmdline) {
		free(cmdline);
		cmdline = NULL;
	}
	if (NULL != chk) {
		free(chk);
		chk = NULL;
	}
	if (NULL != ssk) {
		free(ssk);
		ssk = NULL;
	}
	if (NULL != map_uri) {
		free(map_uri);
		map_uri = NULL;
	}
	if (NULL != meta) {
		free(meta);
		meta = NULL;
	}
	if (NULL != paths) {
		size_t i;
		for (i = 0; i < paths_cnt; i++) {
			if (NULL != paths[i]) {
				free(paths[i]);
				paths[i] = NULL;
			}
		}
		free(paths);
		paths = NULL;
	}
	if (NULL != fileinfo) {
		size_t i;
		for (i = 0; i < fileinfo_cnt; i++) {
			if (NULL != fileinfo[i]) {
				fileinfo_t *fi = fileinfo[i];
				if (NULL != fi->name) {
					free(fi->name);
					fi->name = NULL;
				}
				if (NULL != fi->chk) {
					free(fi->chk);
					fi->chk = NULL;
				}
				if (NULL != fi->mimetype) {
					free(fi->mimetype);
					fi->mimetype = NULL;
				}
				if (NULL != fi->fp) {
					pclose(fi->fp);
					fi->fp = NULL;
				}
			}
		}
		free(fileinfo);
		fileinfo = NULL;
	}
	return rc;	
}

int mkdir_parent(const char *path, const char *dir)
{
	size_t len;
	char *pathname = calloc(MAXPATHLEN, 1);
	struct stat st;
	int rc = 0;

	if (NULL == pathname) {
		LOG(L_ERROR,("calloc(%d,%d) failed (%s)\n",
			MAXPATHLEN, 1));
		rc = -1;
		goto bailout;
	}
	if (strlen(path) > 0) {
		len = snprintf(pathname, MAXPATHLEN, "%s/", path);
	} else {
		len = 0;
		pathname[len] = '\0';
	}
	
	while (*dir && len < MAXPATHLEN) {
		if (*dir == '/') {
			pathname[len] = '\0';
			if (0 == stat(pathname,&st)) {
				if (0 == S_ISDIR(st.st_mode)) {
					LOG(L_ERROR,("\"%s\" is not a directory\n", pathname));
					rc = -1;
					goto bailout;
				}
			} else if (-1 == (rc = mkdir(pathname, 0755))) {
				LOG(L_ERROR,("mkdir(\"%s\", 0755) failed (%s)\n",
					pathname, strerror(errno)));
				goto bailout;
			}
		}
		pathname[len++] = *dir++;
	}

	pathname[len] = '\0';
	if (0 == stat(pathname,&st)) {
		if (0 == S_ISDIR(st.st_mode)) {
			LOG(L_ERROR,("\"%s\" is not a directory\n", pathname));
			rc = -1;
			goto bailout;
		}
	} else if (-1 == (rc = mkdir(pathname, 0755))) {
		LOG(L_ERROR,("mkdir(\"%s\", 0755) failed (%s)\n",
			pathname, strerror(errno)));
		goto bailout;
	}

bailout:
	if (NULL != pathname) {
		free(pathname);
		pathname = NULL;
	}
	return rc;
}

int get_site(fcp_t **pf, const char *uri, int htl, int wiggle, const char *path)
{
	char get_filename[MAXPATHLEN+1];
	fcp_t *f = *pf;
	char *dst, *src, *slash, *var = NULL, *val = NULL, *ssk = NULL;
	fileinfo_t **fileinfo = NULL, *fi = NULL;
	size_t i, start = 0, size, split_size = 0, chunk_size = 0;
	size_t fileinfo_cnt = 0, fileinfo_max = 0;
	char *cmdline = NULL;
	fd_set rdfds, exfds;
	int fd, nfds, minfd, maxfd;
	struct stat st;
	struct timeval tv;
	struct tm *tm;
	time_t date;
	int htl0 = htl;
	int mode = MODE_NONE, rc = 0;

	if (wiggle > 0) {
		htl0 += (rand() % wiggle) - wiggle / 2;
		if (htl0 < 1) {
			htl0 = 1;
		}
	}

	LOG(L_NORMAL,("Retrieving map space %s\n", uri));

	cmdline = calloc(MAXBUFF, 1);
	if (NULL == cmdline) {
		LOG(L_ERROR,("barf: calloc(%d,%d) failed (%s)\n",
			MAXBUFF, 1, strerror(errno)));
		rc = -1;
		goto bailout;
	}

	if (0 != strcmp(path, "/dev/null")) {
		if (0 != stat(path, &st)) {
			mkdir_parent("", path);
			if (0 != stat(path, &st)) {
				LOG(L_ERROR,("cannot stat \"%s\" (%s)\n",
					path, strerror(errno)));
				rc = -1;
				goto bailout;
			}
		}

		if (!S_ISDIR(st.st_mode)) {
			LOG(L_ERROR,("\"%s\" is not a directory\n",
				path));
			rc = -1;
			goto bailout;
		}
	}

	if (NULL == f->meta) {
		LOG(L_ERROR,("No metadata found\n"));
		rc = -1;
		goto bailout;
	}

	var = calloc(MAXBUFF, 1);
	if (NULL == var) {
		LOG(L_ERROR,("barf: calloc(%d,%d) failed (%s)\n",
			MAXBUFF, 1, strerror(errno)));
		rc = -1;
		goto bailout;
	}

	val = calloc(MAXBUFF, 1);
	if (NULL == val) {
		LOG(L_ERROR,("barf: calloc(%d,%d) failed (%s)\n",
			MAXBUFF, 1, strerror(errno)));
		rc = -1;
		goto bailout;
	}

	src = f->meta;
	while (*src) {
		dst = var;
		while (*src != '\0' && *src != '\r' && *src != '\n' && *src != '=') {
			*dst++ = *src++;
		}
		*dst = '\0';
		dst = val;
		if (*src == '=') {
			src++;
			while (*src != '\0' && *src != '\r' && *src != '\n') {
				*dst++ = *src++;
			}
		}
		*dst = '\0';
		while (*src == '\r' || *src == '\n') {
			src++;
		}
		if (0 == strcasecmp(var, "Version")) {
			LOG(L_MINOR,("meta: Version\n"));
		} else if (0 == strcasecmp(var, "Revision")) {
			LOG(L_MINOR,("meta: Revision=%s\n", val));
		} else if (0 == strcasecmp(var, "Document")) {
			LOG(L_MINOR,("meta: Document\n"));
			chunk_size = 256 * 1024;	/* Default is 256KB */
		} else if (0 == strcasecmp(var, "Name")) {
			LOG(L_MINOR,("meta: Name=%s\n", val));
			if (0 == fileinfo_max) {
				fileinfo_max = 1;
				fileinfo = calloc(sizeof(fileinfo_t *), fileinfo_max);
			} else if (fileinfo_cnt >= fileinfo_max) {
				fileinfo_max += 256;
				fileinfo = realloc(fileinfo,
					sizeof(fileinfo_t *) * fileinfo_max);
				memset(&fileinfo[fileinfo_cnt], 0,
					sizeof(fileinfo_t *) * (fileinfo_max - fileinfo_cnt));
			}
			if (NULL == (fi = fileinfo[fileinfo_cnt])) {
				fi = fileinfo[fileinfo_cnt] = calloc(sizeof(fileinfo_t), 1);
			}
			fi->name = strdup(val);
			fi->htl = htl;
		} else if (0 == strcasecmp(var, "Info.Format")) {
			LOG(L_MINOR,("meta: Info.Format=%s\n", val));
			if (0 == fileinfo_max) {
				fileinfo_max = 1;
				fileinfo = calloc(sizeof(fileinfo_t *), fileinfo_max);
			} else if (fileinfo_cnt >= fileinfo_max) {
				fileinfo_max += 256;
				fileinfo = realloc(fileinfo,
					sizeof(fileinfo_t *) * fileinfo_max);
				memset(&fileinfo[fileinfo_cnt], 0,
					sizeof(fileinfo_t *) * (fileinfo_max - fileinfo_cnt));
			}
			if (NULL == (fi = fileinfo[fileinfo_cnt])) {
				fi = fileinfo[fileinfo_cnt] = calloc(sizeof(fileinfo_t), 1);
			}
			fi->mimetype = strdup(val);
			fi->htl = htl;
			if (wiggle > 0) {
				fi->htl += (rand() % wiggle) - wiggle / 2;
				if (fi->htl < 1) {
					fi->htl = 1;
				}
			}
		} else if (0 == strcasecmp(var, "Info.Description")) {
			LOG(L_MINOR,("meta: Info.Description=%s\n", val));
		} else if (0 == strcasecmp(var, "Redirect.Target")) {
			if (MODE_NONE == mode) {
				mode = MODE_REDIR;
			} else {
				mode = MODE_MAPSPACE;
			}
			LOG(L_MINOR,("meta: Redirect.Target=%s\n", val));
			if (0 == fileinfo_max) {
				fileinfo_max = 1;
				fileinfo = calloc(sizeof(fileinfo_t *), fileinfo_max);
			} else if (fileinfo_cnt >= fileinfo_max) {
				fileinfo_max += 256;
				fileinfo = realloc(fileinfo,
					sizeof(fileinfo_t *) * fileinfo_max);
				memset(&fileinfo[fileinfo_cnt], 0,
					sizeof(fileinfo_t *) * (fileinfo_max - fileinfo_cnt));
			}
			if (NULL == (fi = fileinfo[fileinfo_cnt])) {
				fi = fileinfo[fileinfo_cnt] = calloc(sizeof(fileinfo_t), 1);
			}
			fi->chk = strdup(val);
			fi->htl = htl;
			if (wiggle > 0) {
				fi->htl += (rand() % wiggle) - wiggle / 2;
				if (fi->htl < 1) {
					fi->htl = 1;
				}
			}
		} else if (0 == strcasecmp(var, "DateRedirect.Target")) {
			LOG(L_MINOR,("meta: DateRedirect.Target=%s\n", val));
			mode = MODE_DATEBASEDREDIR;
			if (0 == fileinfo_max) {
				fileinfo_max = 1;
				fileinfo = calloc(sizeof(fileinfo_t *), fileinfo_max);
			} else if (fileinfo_cnt >= fileinfo_max) {
				fileinfo_max += 256;
				fileinfo = realloc(fileinfo,
					sizeof(fileinfo_t *) * fileinfo_max);
				memset(&fileinfo[fileinfo_cnt], 0,
					sizeof(fileinfo_t *) * (fileinfo_max - fileinfo_cnt));
			}
			if (NULL == (fi = fileinfo[fileinfo_cnt])) {
				fi = fileinfo[fileinfo_cnt] = calloc(sizeof(fileinfo_t), 1);
			}
			fi->chk = strdup(val);
			fi->htl = htl;
			if (wiggle > 0) {
				fi->htl += (rand() % wiggle) - wiggle / 2;
				if (fi->htl < 1) {
					fi->htl = 1;
				}
			}
		} else if (0 == strcasecmp(var, "DateRedirect.Offset")) {
			LOG(L_MINOR,("meta: DateRedirect.Offset=%s\n", val));
			date_offs = strtoul(val, NULL, 16);
		} else if (0 == strcasecmp(var, "DateRedirect.Increment")) {
			LOG(L_MINOR,("meta: DateRedirect.Increment=%s\n", val));
			date_incr = strtoul(val, NULL, 16);
		} else if (0 == strcasecmp(var, "SplitFile.Size")) {
			LOG(L_MINOR,("meta: SplitFile.Size=%s\n", val));
			split_size = strtoul(val, NULL, 16);
		} else if (0 == strcasecmp(var, "SplitFile.Blockcount")) {
			mode = MODE_SPLITFILE;
			LOG(L_MINOR,("meta: SplitFile.Blockcount=%s\n", val));
			if (0 == fileinfo_max) {
				fileinfo_max = strtoul(val, NULL, 16);
				fileinfo = calloc(sizeof(fileinfo_t *), fileinfo_max);
			} else {
				fileinfo_max += strtoul(val, NULL, 16);
				fileinfo = realloc(fileinfo,
					sizeof(fileinfo_t *) * fileinfo_max);
				memset(&fileinfo[fileinfo_cnt], 0,
					sizeof(fileinfo_t *) * (fileinfo_max - fileinfo_cnt));
			}
		} else if (0 == strcasecmp(var, "SplitFile.Blocksize")) {
			LOG(L_MINOR,("meta: SplitFile.Blocksize=%s\n", val));
			chunk_size = strtoul(val, NULL, 16);
		} else if (0 == strncmp(var, "SplitFile.Block.", 16)) {
			LOG(L_MINOR,("meta: %s=%s\n", var, val));
			i = strtoul(var + 16, NULL, 16);
			if (0 == fileinfo_max) {
				fileinfo_max = 1;
				fileinfo = calloc(sizeof(fileinfo_t *), fileinfo_max);
			} else if (fileinfo_cnt >= fileinfo_max) {
				fileinfo_max += 256;
				fileinfo = realloc(fileinfo,
					sizeof(fileinfo_t *) * fileinfo_max);
				memset(&fileinfo[fileinfo_cnt], 0,
					sizeof(fileinfo_t *) * (fileinfo_max - fileinfo_cnt));
			}
			if (NULL == (fi = fileinfo[fileinfo_cnt])) {
				fi = fileinfo[fileinfo_cnt] = calloc(sizeof(fileinfo_t), 1);
			}
			if (i == fileinfo_max) {	/* last chunk? */
				fi->size = split_size - (fileinfo_max - 1) * chunk_size;
			} else {
				fi->size = chunk_size;
			}
			fi->chk = strdup(val);
			fi->htl = htl;
			if (wiggle > 0) {
				fi->htl += (rand() % wiggle) - wiggle / 2;
				if (fi->htl < 1) {
					fi->htl = 1;
				}
			}
			if (NULL == fi->mimetype) {
				fi->mimetype = strdup("application/octet-stream");
			}
			if (fileinfo_cnt + 1 < fileinfo_max) {
				fileinfo_cnt++;
			}
		} else if (0 == strcasecmp(var, "EndPart")) {
			LOG(L_MINOR,("meta: EndPart\n"));
			if (NULL != fileinfo && NULL != fileinfo[fileinfo_cnt]) {
				fi = fileinfo[fileinfo_cnt];
				if (NULL == fi->mimetype) {
					fi->mimetype = strdup("application/octet-stream");
				}
				fileinfo_cnt++;
			}
		} else if (0 == strcasecmp(var, "End")) {
			LOG(L_MINOR,("meta: End\n"));
			if (NULL != fileinfo && NULL != fileinfo[fileinfo_cnt]) {
				fi = fileinfo[fileinfo_cnt];
				if (NULL == fi->mimetype) {
					fi->mimetype = strdup("application/octet-stream");
				}
				fileinfo_cnt++;
			}
		} else {
			LOG(L_ERROR,("Unhandled metadata tag: %s\n", var));
		}
	}

	switch (mode) {
	case MODE_REDIR:
		fcp_free(pf);
		if (0 != (rc = fcp_new(pf, fcphost, fcpport))) {
			printf("ERROR: connecting to %s:%d\n", fcphost, fcpport);
			goto bailout;
		}
		f = *pf;
		LOG(L_NORMAL,("Following redirect to:\n%s\n", fileinfo[0]->chk));
		rc = fcp_get(f, fileinfo[0]->chk, 0, htl0);
		if (0 == rc) {
			rc = get_site(pf, fileinfo[0]->chk, fileinfo[0]->htl,
				wiggle, path);
		}
		break;

	case MODE_MAPSPACE:
		start = fileinfo_cnt - 1;
		for (;;) {
			FD_ZERO(&rdfds);
			FD_ZERO(&exfds);
			nfds = 0;
			minfd = 32767;
			maxfd = 0;
			for (i = 0; i < fileinfo_cnt; i++) {
				fi = fileinfo[i];
				if (NULL != fi->fp) {
					fd = fileno(fi->fp);
					FD_SET(fd, &rdfds);
					FD_SET(fd, &exfds);
					if (fd < minfd)
						minfd = fd;
					if (fd > maxfd)
						maxfd = fd;
					nfds += 1;
				}
			}
			for (i = (start + 1) % fileinfo_cnt;
				nfds < in_threads && i != start;
				i = (i + 1) % fileinfo_cnt) {
				fi = fileinfo[i];
				if (NULL != fi->fp)
					continue;
				if (NULL == fi->name)
					continue;
				if (NULL != fi->mimetype &&
					NULL != strstr(mimeignore, fi->mimetype)) {
					LOG(L_NORMAL,("Ignoring #%d/%d %s (%s)\n",
						i+1, fileinfo_cnt, fi->name, fi->mimetype));
					free(fi->name);
					fi->name = NULL;
					continue;
				}
				if (0 == strcmp(path, "/dev/null")) {
					snprintf(cmdline, MAXBUFF,
						"\"%s\" %s -n%s -p%d -l%d -i%d -t%d -m%d -v%d %s%s %s \"%s\"",
						program, "get", fcphost, fcpport, fi->htl,
						in_threads, sf_threads, slimit, verbose,
						delete ? "--delete " : "",
						fi->chk, path, fi->name);
					fi->fp = popen(cmdline, "r");
					if (NULL == fi->fp) {
						LOG(L_ERROR,("popen(\"%s\",\"r\") failed (%s)\n",
							cmdline, strerror(errno)));
						rc = -1;
						goto bailout;
					}
					LOG(L_NORMAL,("Requesting #%d/%d %s (%s) HTL:%d\n",
						i+1, fileinfo_cnt, fi->name,
						fi->mimetype, fi->htl));
					fd = fileno(fi->fp);
					FD_SET(fd, &rdfds);
					FD_SET(fd, &exfds);
					if (fd < minfd)
						minfd = fd;
					if (fd > maxfd)
						maxfd = fd;
					nfds += 1;
				} else {
					if (NULL != (slash = strrchr(fi->name, '/'))) {
						*slash = '\0';
						mkdir_parent(path, fi->name);
						*slash = '/';
					}
					snprintf(get_filename, MAXPATHLEN, "%s/%s", path, fi->name);
					if (0 == stat(get_filename, &st)) {
						LOG(L_MINOR,("Already have #%d/%d %s\n",
							i+1, fileinfo_cnt, fi->name));
						fi->fp = NULL;
						free(fi->name);
						fi->name = NULL;
					} else {
						unlink(get_filename);
						snprintf(cmdline, MAXBUFF,
							"\"%s\" %s -n%s -p%d -l%d -i%d -t%d -m%d -v%d %s%s \"%s\" \"%s\"",
							program, "get", fcphost, fcpport, fi->htl,
							in_threads, sf_threads, slimit, verbose,
							delete ? "--delete " : "",
							fi->chk, get_filename, fi->name);
						fi->fp = popen(cmdline, "r");
						if (NULL == fi->fp) {
							LOG(L_ERROR,("popen(\"%s\",\"r\") failed (%s)\n",
								cmdline, strerror(errno)));
							rc = -1;
							goto bailout;
						}
						LOG(L_NORMAL,("Requesting #%d/%d %s (%s) HTL:%d\n",
							i+1, fileinfo_cnt, fi->name,
							fi->mimetype, fi->htl));
						fd = fileno(fi->fp);
						FD_SET(fd, &rdfds);
						FD_SET(fd, &exfds);
						if (fd < minfd)
							minfd = fd;
						if (fd > maxfd)
							maxfd = fd;
						nfds += 1;
					}
				}
				start = i;
			}

			if (0 == nfds) {
				break;
			}

			if (nfds > 0) {
				struct timeval tv = {1,0};
				if (-1 == select(maxfd + 1, &rdfds, NULL, &exfds, &tv)) {
					LOG(L_ERROR,("select(%d,...) failed (%s)\n",
						maxfd+1, strerror(errno)));
					rc = -1;
					goto bailout;
				}
				for (fd = minfd; fd < maxfd+1; fd++) {
					for (i = 0; i < fileinfo_cnt; i++) {
						fi = fileinfo[i];
						if (NULL != fi &&
							NULL != fi->fp &&
							fd == fileno(fi->fp))
							break;
					}
					if (FD_ISSET(fd, &rdfds)) {
						char *reply = calloc(MAXBUFF, 1);
						char *eol, *result, *chk;
						if (NULL == reply) {
							LOG(L_ERROR,("calloc(%d,%d) failed (%s)\n",
								MAXBUFF, 1, strerror(errno)));
							rc = -1;
							goto bailout;
						}
						fgets(reply, MAXBUFF, fi->fp);
						LOG(L_DEBUG,("reply: \"%s\"\n", reply));
						eol = strtok(reply, "\r\n");
						if (NULL != eol) {
							result = strtok(eol, ":\r\n");
							chk = strtok(NULL, "\r\n");
							while (chk && *chk == ' ')
							chk++;
						} else {
							result = NULL;
							chk = NULL;
						}
						if (NULL == result ||
							NULL == chk ||
							(0 != strcasecmp(result, SUCCESS) &&
							0 != strcasecmp(result, KEYCOLLISION))) {
							if (NULL == result) {
								result = GENERALERROR;
							}
							LOG(L_NORMAL,("%s: #%d/%d %s\n",
								result, i+1, fileinfo_cnt, fi->name));
							if (fi->htl < MAX_HTL) {
								fi->htl += 1;
							}
							fi->retry += 1;
						} else {
							LOG(L_NORMAL,("%s: #%d/%d %s\n",
								result, i+1, fileinfo_cnt, fi->name));
							LOG(L_MINOR,("%s\n",
								chk));
							free(fi->name);
							fi->name = NULL;
						}
						pclose(fi->fp);
						fi->fp = NULL;
						free(reply);
					}
					if (FD_ISSET(fd, &exfds)) {
						if (NULL != fi->fp) {
							LOG(L_MINOR,("closing #%d %s\n",
								i+1, fi->name));
							pclose(fi->fp);
							fi->fp = NULL;
						}
						if (fi->htl < MAX_HTL) {
							fi->htl += 1;
						}
						fi->retry += 1;
					}
				}
			}
		}
		break;

	case MODE_SPLITFILE:
		LOG(L_ERROR,("Unexpected split file metadata\n"));
		rc = -1;
		break;

	case MODE_DATEBASEDREDIR:
		if (0 == date_incr) {
			date_incr = 86400;
		}
		gettimeofday(&tv, NULL);
		tm = localtime((const time_t *)&tv.tv_sec);
		date = timegm(tm);
		date = date - (date % date_incr);
		size = MAXPATHLEN;
		if (NULL == (ssk = calloc(size, 1))) {
			LOG(L_ERROR,("barf: calloc(%d,%d) failed (%s)\n",
				size, 1, strerror(errno)));
			rc = -1;
			goto bailout;
		}
		strcpy(ssk, fileinfo[0]->chk);
		slash = strchr(ssk, '/');
		if (NULL == slash) {
			LOG(L_ERROR,("FATAL: no slash in date based redirect (%s)\n",
				ssk));
			goto bailout;
		}
		/* make room for 8 hex digits and a dash */
		memmove(slash + 8 + 1, slash, strlen(slash) + 2);
		snprintf(slash + 1, 8 + 1, "%08x", (unsigned)date);
		slash[8 + 1] = '-';

		printf("**** %s\n", ssk);

		fcp_free(pf);
		if (0 != (rc = fcp_new(pf, fcphost, fcpport))) {
			printf("ERROR: connecting to %s:%d\n", fcphost, fcpport);
			goto bailout;
		}
		f = *pf;
		LOG(L_NORMAL,("Following date based redirect to:\n%s\n", ssk));
		rc = fcp_get(f, ssk, 0, htl0);
		/* successful but again metadata only? */
		if (0 == rc && NULL == f->data) {
			rc = get_site(pf, ssk, htl, wiggle, path);
		}
		break;
	default:
		LOG(L_NORMAL,("invalid metadata?\n"));
		rc = -1;
	}

bailout:
	if (NULL != cmdline) {
		free(cmdline);
		cmdline = NULL;
	}
	if (NULL != var) {
		free(var);
		var = NULL;
	}
	if (NULL != val) {
		free(val);
		val = NULL;
	}
	if (NULL != ssk) {
		free(ssk);
		ssk = NULL;
	}
	if (NULL != fileinfo) {
		size_t i;
		for (i = 0; i < fileinfo_cnt; i++) {
			if (NULL != fileinfo[i]) {
				fileinfo_t *fi = fileinfo[i];
				if (NULL != fi->name) {
					free(fi->name);
					fi->name = NULL;
				}
				if (NULL != fi->chk) {
					free(fi->chk);
					fi->chk = NULL;
				}
				if (NULL != fi->mimetype) {
					free(fi->mimetype);
					fi->mimetype = NULL;
				}
				if (NULL != fi->fp) {
					pclose(fi->fp);
					fi->fp = NULL;
				}
			}
		}
		free(fileinfo);
		fileinfo = NULL;
	}
	return rc;	
}
