/***********************************************************
        Copyright 1991,1994 by Carnegie Mellon University

                      All Rights Reserved

Permission to use, copy, modify, and distribute this software and its
documentation for any purpose and without fee is hereby granted,
provided that the above copyright notice appear in all copies and that
both that copyright notice and this permission notice appear in
supporting documentation, and that the name of CMU not be
used in advertising or publicity pertaining to distribution of the
software without specific, written prior permission.

CMU DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING
ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT SHALL
CMU BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR
ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS,
WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,
ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
SOFTWARE.
******************************************************************/


#ifndef SABER
#ifndef LINT
static char rcs_id[] = "$Id: string.c,v 1.7 1994/10/12 17:12:35 ww0r Exp $";
#endif /* LINT */
#endif /* SABER */

/*
 * Author: Sohan C. Ramakrishna Pillai
 */

#include "depotlib.h"

#include "util.h"
#include "Error.h"


/* $$$UNDUPLICATE$$$
 * Combine StringSet_AddString(), StringSet_Member() and StringSet_Index()
 * into one function
 */


/********************************\
*	String routines          *
\********************************/

/*
 * Allocate and copy a new string
 */
char *
String(str, len)
     char *str;
     int len;
{
  char *newstr;

  if (str == NULL) {
    FatalError(E_BADSTRING,
	       "Attempt to create copy of NULL string\n");
  }
  if (PROGRAM_ErrorNo == E_NULL) {
    newstr = (char *) emalloc((len + 1) * sizeof(char *));
    newstr = strncpy(newstr, str, len);
    newstr[len] = '\0';
  }
  return ((PROGRAM_ErrorNo == E_NULL) ? newstr : NULL);
}


/*
 * Free space allocated for a string
 */
void 
String_Free(str)
     char *str;
{
  if (str == NULL) {
    FatalError(E_BADSTRING,
	       "Attempt to free NULL string.\n");
  } 

  (void) free(str);
  return;
}


/*
 * Compare possibly NULL strings
 */
int 
String_Comparator(str1, str2)
     char *str1, *str2;
{
  int StringComparison;


  if (str1 == NULL) {
    if (str2 == NULL) {
      StringComparison = 0;
    } else {
      StringComparison = -1;
    }
  } else {
    if (str2 == NULL) {
      StringComparison = 1;
    } else {
      StringComparison = strcmp(str1, str2);
      if (StringComparison > 0)
	StringComparison = 1;
      else if (StringComparison < 0)
	StringComparison = -1;
    }
  }
  
  return StringComparison;
}


/*
 * Remove quoting characters from a string
 */
char *
String_RemoveQuotes(string, quotechar)
     char *string;
     int quotechar;
{
  register char *fp, *tp;
  char *buffer, *newstr;

  buffer = String(string, strlen(string));
  if (PROGRAM_ErrorNo == E_NULL) {
    fp = string;
    tp = buffer;
    while (*fp != '\0') {
      if (*fp == quotechar)
	fp++;
      *tp++ = *fp++;
    }
    *tp = '\0';

    newstr = String(buffer, strlen(buffer));
    (void) free(buffer);
  }
  return ((PROGRAM_ErrorNo == E_NULL) ? newstr : NULL);
}


void 
String_Write(fp, string, quotechar, delim, specialcharlist, nspecialchars)
     FILE *fp;
     char *string;
     int quotechar, delim;
     char *specialcharlist;
     int nspecialchars;
{
  register char *cp;
  register char *qp;
  register int i;
  Boolean IsASplChar;

  if (string != NULL) {
    for (cp = string; *cp != '\0'; cp++) {

      IsASplChar = FALSE;
      if ((*cp == quotechar) || (*cp == delim))
	IsASplChar = TRUE;
      i = 0;
      qp = specialcharlist;
      while (!IsASplChar && (i < nspecialchars)) {
	if (*cp == *qp)
	  IsASplChar = TRUE;
	else
	  qp++, i++;
      }
      if (IsASplChar) {
	(void) putc((char) quotechar, fp);
      }
      (void) putc(*cp, fp);
    }
  }
  return;
}



/*
 * Allocate and copy a new string array
 */
STRINGARRAY *
StringArray(strarr, len)
     STRINGARRAY *strarr;
     int len;
{
  register int i;
  register char **fp, **tp;
  STRINGARRAY *newstrarr;

  if (strarr == NULL) {
    FatalError(E_BADSTRINGARRAY,
	       "Attempt to create copy of NULL string array\n");
  }
  if (PROGRAM_ErrorNo == E_NULL) {
    newstrarr = (STRINGARRAY *) emalloc(sizeof(STRINGARRAY));
  }
  if (PROGRAM_ErrorNo == E_NULL) {
    if (len <= 0) {
      STRINGARRAY_Size(newstrarr) = 0;
      STRINGARRAY_Values(newstrarr) = NULL;
    } else {			/* len > 0 */
      STRINGARRAY_Size(newstrarr) = len;
      STRINGARRAY_Values(newstrarr) =
	(char **) emalloc((len + 1) * sizeof(char *));
      if (STRINGARRAY_Values(strarr) == NULL) {
	FatalError(E_BADSTRINGARRAY,
		"Attempt to copy values of string array with no values.\n");
      }
      if (PROGRAM_ErrorNo == E_NULL) {
	i = 0;
	fp = STRINGARRAY_Values(strarr);
	tp = STRINGARRAY_Values(newstrarr);
	while ((PROGRAM_ErrorNo == E_NULL) && (i < len)) {
	  if (*fp == NULL)
	    *tp = NULL;
	  else
	    *tp = String(*fp, strlen(*fp));
	  i++, fp++, tp++;
	}
	*tp = NULL;
      }
    }
  }
  return newstrarr;
}


/*
 * Free space allocated for a string array
 */
void 
StringArray_Free(strarr)
     STRINGARRAY *strarr;
{
  register int i;

  if (strarr == NULL)
    return;

  if (PROGRAM_ErrorNo == E_NULL) {
    for (i = 0;
	 (PROGRAM_ErrorNo == E_NULL) && (i < STRINGARRAY_Size(strarr));
	 i++) {
      if (STRINGARRAY_String(strarr, i) != NULL)
	String_Free(STRINGARRAY_String(strarr, i));
    }
  }
  if (PROGRAM_ErrorNo == E_NULL) {
    if (STRINGARRAY_Values(strarr) != NULL)
      (void) free((void *) STRINGARRAY_Values(strarr));
    (void) free((void *) strarr);
  }
  return;
}


/*
 * Compute length of string array
 */
int 
StringArray_Size(strarr)
     STRINGARRAY *strarr;
{
  int size;

  if (strarr == NULL)
    size = 0;
  else {
    size = STRINGARRAY_Size(strarr);
  }

  return size;
}



/*
 * Compare two string arrays in strcmp(3) style
 */
int 
StringArray_Comparator(strarr1, strarr2)
     STRINGARRAY *strarr1, *strarr2;
{
  register char **sp1, **sp2;
  int ArrayComparison;
  Boolean eostrarray;

  if (strarr1 == NULL) {
    if (strarr2 == NULL) {
      ArrayComparison = 0;
    } else {
      ArrayComparison = -1;
    }
  } else {
    if (strarr2 == NULL) {
      ArrayComparison = 1;
    } else {
      sp1 = STRINGARRAY_Values(strarr1);
      sp2 = STRINGARRAY_Values(strarr2);
      if (sp1 == NULL) {
	if (sp2 == NULL) {
	  ArrayComparison = 0;
	} else {
	  ArrayComparison = -1;
	}
      } else {
	if (sp2 == NULL) {
	  ArrayComparison = 1;
	} else {
	  ArrayComparison = 0;
	  eostrarray = FALSE;
	  while ((ArrayComparison == 0) && !eostrarray) {
	    ArrayComparison = String_Comparator(*sp1, *sp2);
	    if ((*sp1 == NULL) || (*sp2 == NULL))
	      eostrarray = TRUE;
	    else
	      sp1++, sp2++;
	  }
	}
      }
    }
  }

  return ArrayComparison;
}



/*
 * Append a string to a NULL-terminated string array
 */
STRINGARRAY *
StringArray_AppendString(strarr, str)
     STRINGARRAY *strarr;
     char *str;
{
  STRINGARRAY *newstrarr;

  if (str == NULL) {
    FatalError(E_BADSTRING,
	       "Attempt to insert NULL string into array of strings.\n");
  }
  if (PROGRAM_ErrorNo == E_NULL) {
    newstrarr = strarr;
    if (newstrarr == NULL) {
      newstrarr = (STRINGARRAY *) emalloc(sizeof(STRINGARRAY));
      if (PROGRAM_ErrorNo == E_NULL) {
	STRINGARRAY_Size(newstrarr) = 0;
	STRINGARRAY_Values(newstrarr) = NULL;
      }
    }
  }
  if (PROGRAM_ErrorNo == E_NULL) {
    if ((STRINGARRAY_Size(newstrarr) == 0)
	&& (STRINGARRAY_Values(newstrarr) == NULL)) {	/* allocate array of
							 * length 1 */
      STRINGARRAY_Values(newstrarr) = (char **) emalloc(sizeof(char *));
      if (PROGRAM_ErrorNo == E_NULL) {
	*STRINGARRAY_Values(newstrarr) = NULL;
      }
    }
  }
  if (PROGRAM_ErrorNo == E_NULL) {
    STRINGARRAY_Size(newstrarr)++;
    STRINGARRAY_Values(newstrarr) =
      (char **) erealloc((void *) (STRINGARRAY_Values(newstrarr)),
			 (unsigned) (STRINGARRAY_Size(newstrarr) + 1)
			 * sizeof(char *));
    if (PROGRAM_ErrorNo == E_NULL) {
      STRINGARRAY_String(newstrarr, STRINGARRAY_Size(newstrarr) - 1) =
	String(str, strlen(str));
      STRINGARRAY_String(newstrarr, STRINGARRAY_Size(newstrarr)) =
	NULL;
    }
  }
  return (PROGRAM_ErrorNo == E_NULL) ? newstrarr : NULL;
}



void 
StringArray_Write(fp, strarr, quotechar, delim, specialcharlist, nspecialchars)
     FILE *fp;
     STRINGARRAY *strarr;
     int quotechar, delim;
     char *specialcharlist;
     int nspecialchars;
{
  register char **sp;

  if ((strarr != NULL) && (STRINGARRAY_Size(strarr) > 0)) {
    sp = STRINGARRAY_Values(strarr);
    if (*sp != NULL)
      String_Write(fp, *sp++, quotechar, delim,
		   specialcharlist, nspecialchars);
    while (*sp != NULL) {
      putc((char) delim, fp);
      String_Write(fp, *sp++, quotechar, delim,
		   specialcharlist, nspecialchars);
    }
  }
  return;
}


/*
 * char **StringToStringArray(string, delimiter, quotechar)
 *      char *string;
 *      int delimiter, quotechar;
 * generates a NULL-terminated string array consisting of the list of strings
 * in the given string separated by the specified delimiter.
 */
STRINGARRAY *
StringToStringArray(string, delimiter, quotechar)
     char *string;
     int delimiter, quotechar;
{
  register char *cp, *bufp;

  STRINGARRAY *strarr;
  char *buffer;
  Boolean eostring;

  strarr = NULL;

  if (string == NULL) {
    FatalError(E_BADSTRING,
	       "Attempt to generate string array from NULL string.\n");
  }
  /* allocate space in buffer to hold strings to copy to strarr */
  if (PROGRAM_ErrorNo == E_NULL)
    buffer = (char *) emalloc(strlen(string) + 1);	/* worst case size */

  /* harvest the delimited strings and append them to strarr */
  cp = string;
  eostring = FALSE;
  while ((PROGRAM_ErrorNo == E_NULL) && !eostring) {
    while (*cp == delimiter)
      cp++;
    if (*cp == '\0')
      eostring = TRUE;
    else {
      bufp = buffer;
      while ((PROGRAM_ErrorNo == E_NULL)
	     && (*cp != delimiter) && (*cp != '\0')) {
	if (*cp == quotechar)
	  *bufp++ = *cp++;
	if (*cp != '\0')
	  *bufp++ = *cp++;
      }
      if (PROGRAM_ErrorNo == E_NULL) {
	*bufp = '\0';
	strarr = StringArray_AppendString(strarr, buffer);
      }
    }
  }

  if (PROGRAM_ErrorNo == E_NULL)
    (void) free((void *) buffer);
  return (PROGRAM_ErrorNo == E_NULL) ? strarr : NULL;
}



/******************************************************************************\
 *	StringSet routines						      *
 * A string set is a NULL-terminated sorted array of strings		      *
\******************************************************************************/


/*
 * Add a string to a set of strings
 */
STRINGSET *
StringSet_AddString(strset, str)
     STRINGSET *strset;
     char *str;
{
  register char **fp, **tp;
  register char *sp;
  register int i;

  int StringComparison, InsertionPoint;
  int hi, lo;
  Boolean DuplicateString, FoundInsertionPoint;
  STRINGSET *newstrset;

  if (str == NULL) {
    FatalError(E_BADSTRING,
	       "Attempt to add NULL string to set of strings.\n");
  }
  if (PROGRAM_ErrorNo == E_NULL) {
    newstrset = strset;
    if (newstrset == NULL) {
      newstrset = (STRINGSET *) emalloc(sizeof(STRINGSET));
      if (PROGRAM_ErrorNo == E_NULL) {
	STRINGSET_Size(newstrset) = 0;
	STRINGSET_Values(newstrset) = NULL;
      }
    }
  }
  if (PROGRAM_ErrorNo == E_NULL) {
    if ((STRINGSET_Size(newstrset) == 0)
	&& (STRINGSET_Values(newstrset) == NULL)) {	/* allocate array of
							 * length 1 as empty
							 * set */
      STRINGSET_Values(newstrset) = (char **) emalloc(sizeof(char *));
      if (PROGRAM_ErrorNo == E_NULL) {
	STRINGSET_String(newstrset, 0) = NULL;
      }
    }
  }

  if (PROGRAM_ErrorNo == E_NULL) {
    lo = 0;
    hi = STRINGSET_Size(newstrset);
    DuplicateString = FALSE;
    FoundInsertionPoint = FALSE;
    while ((PROGRAM_ErrorNo == E_NULL)
	   && !FoundInsertionPoint && !DuplicateString) {
      if (hi != lo) {
	i = (hi + lo) / 2;
	sp = STRINGSET_String(newstrset, i);
	StringComparison = String_Comparator(sp, str);
	if (StringComparison == 0) {
	  DuplicateString = TRUE;
	} else if (StringComparison > 0) {
	  hi = i;
	} else {		/* StringComparison < 0 */
	  lo = i + 1;
	}
      } else {			/* hi == lo */
	sp = STRINGSET_String(newstrset, hi);
	if (sp == NULL)
	  StringComparison = 1;
	else
	  StringComparison = String_Comparator(sp, str);
	if (StringComparison == 0) {
	  DuplicateString = TRUE;
	} else if (StringComparison > 0) {
	  FoundInsertionPoint = TRUE;
	  InsertionPoint = hi;
	} else {		/* StringComparison < 0 - should never
				 * happen! */
	  FatalError(E_BADSTRINGSET,
		     "Error adding string %s into string set.\n", str);
	}
      }
    }

    if ((PROGRAM_ErrorNo == E_NULL) && !DuplicateString) {
      if (!FoundInsertionPoint) {
	FatalError(E_BADSTRINGSET,
		   "Error adding string %s into string set.\n, str");
      }
      STRINGSET_Size(newstrset)++;
      STRINGSET_Values(newstrset) =
	(char **) erealloc((void *) (STRINGSET_Values(newstrset)),
			   (unsigned) (STRINGSET_Size(newstrset) + 1)
			   * sizeof(char *));
      if (PROGRAM_ErrorNo == E_NULL) {
	i = STRINGSET_Size(newstrset);
	fp = STRINGSET_Values(newstrset) + i - 1;
	tp = fp + 1;
	while (i > InsertionPoint) {
	  *tp-- = *fp--;
	  i--;
	}
	STRINGSET_String(newstrset, InsertionPoint) = String(str,
							     strlen(str));
      }
    }
  }

  return (PROGRAM_ErrorNo == E_NULL) ? newstrset : NULL;
}


/*
 * Check membership of a string in a string set
 */
Boolean 
StringSet_Member(strset, str)
     STRINGSET *strset;
     char *str;
{
  register char *sp;
  register int i;
  register int hi, lo;
  Boolean FoundString, FoundPotentialLocation;

  if ((str == NULL) || (PROGRAM_ErrorNo != E_NULL)
      || StringSet_Empty(strset)) {
    return FALSE;
  }
  FoundString = FALSE;
  FoundPotentialLocation = FALSE;
  lo = 0;
  hi = STRINGSET_Size(strset);
  while ((PROGRAM_ErrorNo == E_NULL)
	 && !FoundPotentialLocation && !FoundString) {
    int StringComparison;
    
    if (hi != lo) {
      i = (hi + lo) / 2;
      sp = STRINGSET_String(strset, i);
      StringComparison = String_Comparator(sp, str);
      if (StringComparison == 0) {
	FoundString = TRUE;
      } else if (StringComparison > 0) {
	hi = i;
      } else {		/* StringComparison < 0 */
	lo = i + 1;
      }
    } else {		/* hi == lo */
      sp = STRINGSET_String(strset, hi);
      if (sp == NULL)
	StringComparison = 1;
      else
	StringComparison = String_Comparator(sp, str);
      if (StringComparison == 0) {
	FoundString = TRUE;
      } else if (StringComparison > 0) {
	FoundPotentialLocation = TRUE;
      } else {		/* StringComparison < 0 - should never happen! */
	FatalError(E_BADSTRINGSET,
		   "Error checking membership of string %s in string set.\n",
		   str);
      }
    }
  }
  return (PROGRAM_ErrorNo == E_NULL) ? FoundString : FALSE;
}



/*
 * Find index of a string in a string set (-1 if not found)
 */
int 
StringSet_Index(strset, str)
     STRINGSET *strset;
     char *str;
{
  register char *sp;
  register int i;

  int StringComparison;
  int hi, lo;
  Boolean FoundString, FoundPotentialLocation;
  int StringLocation;

  if (str == NULL) {
    FatalError(E_BADSTRING,
	       "Attempt to check membership of NULL in set of strings.\n");
  }
  if (PROGRAM_ErrorNo == E_NULL) {
    StringLocation = -1;
    FoundString = FALSE;
    FoundPotentialLocation = FALSE;
    if (!StringSet_Empty(strset)) {
      lo = 0;
      hi = STRINGSET_Size(strset);
      while ((PROGRAM_ErrorNo == E_NULL)
	     && !FoundPotentialLocation && !FoundString) {
	if (hi != lo) {
	  i = (hi + lo) / 2;
	  sp = STRINGSET_String(strset, i);
	  StringComparison = String_Comparator(sp, str);
	  if (StringComparison == 0) {
	    FoundString = TRUE;
	    StringLocation = i;
	  } else if (StringComparison > 0) {
	    hi = i;
	  } else {		/* StringComparison < 0 */
	    lo = i + 1;
	  }
	} else {		/* hi == lo */
	  sp = STRINGSET_String(strset, hi);
	  if (sp == NULL)
	    StringComparison = 1;
	  else
	    StringComparison = String_Comparator(sp, str);
	  if (StringComparison == 0) {
	    FoundString = TRUE;
	    StringLocation = hi;
	  } else if (StringComparison > 0) {
	    FoundPotentialLocation = TRUE;
	  } else {		/* StringComparison < 0 - should never
				 * happen! */
	    FatalError(E_BADSTRINGSET,
		  "Error checking membership of string %s in string set.\n",
		       str);
	  }
	}
      }
    }
  }
  return ((PROGRAM_ErrorNo == E_NULL) && FoundString) ? StringLocation : -1;
}



/*
 * Merge two string sets
 *      just use the cheap way now instead of doing a fast
 *      and more sophisticated merge
 *      $$$OPTIMIZABLE$$$
 *
 * NOTE: StringSet_Merge will merge the values in strset2 to what is in
 *       strset1 and return the pointer to the head of strset1. This is done to
 *       avoid core leaks as a result of strset1 getting stomped on.
 * Passing in NULL as the value of strset1 will result in a new stringset being created 
 */
STRINGSET *
StringSet_Merge(strset1, strset2)
     STRINGSET *strset1, *strset2;
{
  register char **ssp;

  if (strset1 == NULL) { /* allocate a new stringset */
    strset1 = (STRINGSET *) emalloc(sizeof(STRINGSET));
    if (PROGRAM_ErrorNo == E_NULL) {
      STRINGSET_Size(strset1) = 0;
      STRINGSET_Values(strset1) = (char **) emalloc(sizeof(char *));
      *STRINGSET_Values(strset1) = NULL;
    } 
  } 
  if (strset2 != NULL) {
    for (ssp = STRINGSET_Values(strset2);
	 (*ssp != NULL) && (PROGRAM_ErrorNo == E_NULL);
	 ssp++) {
      strset1 = StringSet_AddString(strset1, *ssp);
    }
  }

  return (PROGRAM_ErrorNo == E_NULL) ? strset1 : NULL;
}
/* $Source: /afs/andrew/system/src/local/depot2/014/src/lib/util/RCS/string.c,v $ */
