/***********************************************************
        Copyright 1994 by Carnegie Mellon University

                      All Rights Reserved

Permission to use, copy, modify, and distribute this software and its
documentation for any purpose and without fee is hereby granted,
provided that the above copyright notice appear in all copies and that
both that copyright notice and this permission notice appear in
supporting documentation, and that the name of CMU not be
used in advertising or publicity pertaining to distribution of the
software without specific, written prior permission.

CMU DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING
ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT SHALL
CMU BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR
ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS,
WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,
ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
SOFTWARE.
******************************************************************/


#ifndef SABER
#ifndef LINT
static char rcs_id[]="$Id: File_Filter.c,v 1.4 1994/12/09 21:30:24 ww0r Exp $";
#endif /* LINT */
#endif /* SABER */

/*
 * Author: Rob Earhart
 */

#include "depotlib.h"

#include "util.h"
#include "DepotErrorCodes.h"
#include "File.h"

static int run_filter(from, to, filter)
    char *from, *to;
    SENTENCE *filter;
{
  /* Okay... let's open up the descriptors */
  close(0);
  if (open(from, O_RDONLY) == -1) {
    FatalError(E_OPENFAILED, "Could not open %s: %s\n", from, strerror(errno));
    _exit(-1);
  }
  close(1);

  if (open(to, O_WRONLY | O_TRUNC | O_CREAT, 0700) == -1) {
    FatalError(E_OPENFAILED, "Could not open %s: %s\n", to, strerror(errno));
    _exit(-1);
  }

  /* I guess it's time to execve... let's do the path search. */
  execvp(SENTENCE_String(filter, 0), SENTENCE_Values(filter));
    
  /* If we EVER get here, we lose.  Abort to plan Q. */
  FatalError(E_EXECFAILED, "Could not exec %s: %s\n",
	     SENTENCE_String(filter, 0), strerror(errno));
  _exit(-1);
  return(0);				/* appease compilers */
}

int File_Filter(from, to, status, flags, filter)
     char *from, *to;
     FILESTAT *status;
     unsigned flags;
     SENTENCE *filter;
{
    int lupe;
    pid_t pid;
    SENTENCE *argv;
    FILESTAT statbuf;
    char newname[MAXPATHLEN];
#ifdef HAVE_UNION_WAIT
    union wait pidstatus;
#else
    pid_t pidstatus;
#endif

    /* First step: copy the sentence, and perform filename substitution. */
    argv = Sentence(filter, SENTENCE_Size(filter));
    
    Filtered_Message(PROGRAM_Verbose, "FILTER \"");
    for(lupe = 0; SENTENCE_String(argv, lupe); lupe++)
	if (! (String_Comparator(SENTENCE_String(argv, lupe), "%to")))
	    {
		String_Free(SENTENCE_String(argv, lupe));
		SENTENCE_String(argv, lupe) = to;
	    } else if (! (String_Comparator(SENTENCE_String(argv, lupe),
					    "%from")))
		{
		    String_Free(SENTENCE_String(argv, lupe));
		    SENTENCE_String(argv, lupe) = from;
		}
    
    /* Print out what we're doing. */
    /* Isn't space insertion fun?  Please excuse the mess... )Rob */
    lupe = 0;
    while(1) {
	Filtered_Message(PROGRAM_Verbose, "%s",
			 SENTENCE_String(argv, lupe));
	lupe++;				/* Gotta love them macros -- )Rob */
	if (! SENTENCE_String(argv, lupe)) break;
	Filtered_Message(PROGRAM_Verbose, " ");
    }


  /* we run the risk that by potentially truncating name2, we 
   * get a non-unique name. It would be too expensive to do this
   * right... *sigh*
   */
    (void)strncpy(newname, to, MAXPATHLEN-4);
    (void)strcat(newname, ".NEW");
    
    Filtered_Message(PROGRAM_Verbose, "\" < %s > %s\n", from, newname);

    /* Okay, now, let's do it. */
    switch(pid = vfork()) {
    case 0:
	run_filter(from, newname, argv);
	break;
    case -1:
	FatalError(E_VFORKFAILED,
		   "Could not vfork %s: %s\n", SENTENCE_String(argv, 0), strerror(errno));
	break;
    default:
	break;
    }

    do { 
      if (waitpid(pid, &pidstatus, 0) == -1)
	FatalError(E_WAITPIDFAILED,
		   "Could not wait for %s (pid %d): %s\n",
		   SENTENCE_String(argv, 0), pid, strerror(errno));
    } while  (WIFSTOPPED(pidstatus));              /* check for stops */
    
    if (WEXITSTATUS(pidstatus) != 0) {
      FatalError(E_EXECFAILED,
		 "Filter %s (pid %d) exited with error code %d\n", 
		 SENTENCE_String(argv, 0), pid, WEXITSTATUS(pidstatus));
    }

    /* And now, let's just frob its status a little */
    if ((PROGRAM_ErrorNo == E_NULL) && (flags != FSTAT_NULL)) {
	if (status == NULL) {
	    status = &statbuf;
	    if (File_GetStatus(from, status, FALSE /* !followlinks */ ) < 0) {
		FatalError(E_GETSTATUSFAILED,
			   "Could not get status of file %s\n", from);
	    }
	}
	if (PROGRAM_ErrorNo == E_NULL) {
	    /* set modes etc. */
	    if (File_SetStatus(newname, status, flags) < 0) {
		FatalError(E_SETSTATUSFAILED,
			   "Could not set status of file %s\n", newname);
	    }
	}
    }

    if (File_Move(newname, to) < 0) {
      return -1;
    }

    /* Hmm.  We can't exactly just call Sentence_Free, because some of our 
       strings are pointers to bad things to free... */
    for (lupe = 0; SENTENCE_String(argv, lupe); lupe++)
	if (SENTENCE_String(argv, lupe) == to ||
	    SENTENCE_String(argv, lupe) == from)
	    SENTENCE_String(argv, lupe) = NULL;

    /* NOW, it's okay.  Sentence_Free deals with internal NULLs. */
    Sentence_Free(argv);
    

    return ((PROGRAM_ErrorNo == E_NULL) ? 0 : -1);
}

/* $Source: /afs/andrew.cmu.edu/system/src/local/depot2/017/src/lib/FileOps/RCS/File_Filter.c,v $ */
