/*
 * cripple - command line CD ripper/encoder wrapper with cddb support
 *
 * Copyright (C) 2003/2004 Neil Phillips
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "config.h"

#define _GNU_SOURCE 1

#if HAVE_W32API_WINDOWS_H
#  ifndef __win32__
#    define __win32__ 1
#  endif
#endif

#if HAVE_SYS_TYPES_H
# include <sys/types.h>
#endif
#if HAVE_SYS_STAT_H
# include <sys/stat.h>
#endif
#if STDC_HEADERS
# include <stdlib.h>
# include <stddef.h>
#else
# if HAVE_STDLIB_H
#  include <stdlib.h>
# endif
#endif
#if HAVE_STRING_H
# if !STDC_HEADERS && HAVE_MEMORY_H
#  include <memory.h>
# endif
# include <string.h>
#endif
#if HAVE_UNISTD_H
# include <unistd.h>
#endif

#if HAVE_FCNTL_H
#  include <fcntl.h>
#elif HAVE_SYS_FCNTL_H
#  include <sys/fcntl.h>
#endif

#include <stdio.h>

/* these are sent in each cddb request */
#define USER		"bob"
#define HOST		"dave"
#define CLIENT		PACKAGE_NAME
#define VERSION		PACKAGE_VERSION

#ifdef __sun
#  define DEF_CD_DEV	"/vol/dev/aliases/cdrom0"
#elif defined(__win32__)
/*
 * on win32 a null path will cause read_cdtoc_from_drive() to search for one.
 * path should be something like "\\\\.\\H:" (with escaped backslashes).
 */
#  define DEF_CD_DEV	""
#else
#  define DEF_CD_DEV	"/dev/cdrom"
#endif

/*
 * default CD drive read speed to set where supported. also passed to ripper.
 * 0 gives maximum on linux at least.
 */
#define DEF_CD_SPEED	16
/*
 * default path prepended to encoded files
 */
#define DEF_PATH_PREFIX	""
/*
 * port and path to use when none is specified by user or DEF_SERVER below
 */
#define DEF_CGI_PATH	"/~cddb/cddb.cgi"
#define DEF_HTTP_PORT	80
/*
 * default cddb server, treated exactly as if it were from --server option,
 */
#define DEF_SERVER	"freedb.freedb.org"
/*
 * default ripper: rip_{cdparanoia|cd_paranoia|cdda2wav}
 */
#ifdef __win32__
#  define DEF_RIPPER	rip_cd_paranoia
#else
#  define DEF_RIPPER	rip_cdparanoia
#endif
/*
 * default encoder: enc_{bladeenc|lame|flac}
 */
#define DEF_ENCODER	enc_lame
/*
 * default capitalisation: NULL or capitalise_{simple|clever|upper|lower}
 */
#define DEF_CAPITAL_FN	NULL
/*
 * default proxy settings. leave undefined for none.
 */
#if 0
#  define DEF_PROXY	"webcache.mydomain:666"
#endif

/*
 * as win32 is only supported with cygwin we always have unix-style paths.
 */
#define PATH_SEPERATOR '/'


struct ripper {
	const char *cmd;
	const char *opts;
};
/*
 * arg1 = options, arg2 = cd device, arg3 = speed, arg4 = arg5 = track number.
 * don't use n$ parameters as they aren't supported everywhere.
 */
#define RIP_CDPARANOIA	     "cdparanoia %s -d \"%s\" -S %d %d-%d -"
#define RIP_CDPARANOIA_OPTS  "-X -w"

#define RIP_CD_PARANOIA	     "cd-paranoia %s -d \"%s\" -S %d %d-%d -"
#define RIP_CD_PARANOIA_OPTS "-X -w -c"

#define RIP_CDDA2WAV         "cdda2wav %s -D \"%s\" -S %d -t %d+%d -"
#define RIP_CDDA2WAV_OPTS    "-paranoia"


struct encoder {
	const char *cmd;	/* see below */
	const char *opts;	/* command line options (bitrate etc.) */
	const char *ext;	/* output filnema extension (e.g. "mp3") */
};
/*
 * arg1 = options, arg2 = filename.
 */
#define ENC_LAME          "lame %s - \"%s\""
#define ENC_LAME_OPTS     "-h -m s -V 1 --quiet"	/* good quality VBR */

#define ENC_BLADEENC      "bladeenc %s stdin \"%s\""
#define ENC_BLADEENC_OPTS "-256 -quiet"			/* 256kb/s CBR */

#define ENC_FLAC          "flac %s -o \"%s\" -"
#define ENC_FLAC_OPTS     ""

/*
 * TODO: configure should test support for static inline functions.
 */
#ifdef __GNUC__
#  ifndef USE_INLINE
#    define USE_INLINE
#  endif
#endif


#define BUFSIZE 4096	/* minimum allocation unit for malloced_buffer */

struct malloced_buffer {
	char *buf;
	int  size;
	int  off;	/* end of valid data */
};
extern struct malloced_buffer mb;



/*
 * CD Table Of Contents - 99 normal tracks plus track 0xAA
 */
struct toc{
	int min, sec, frame;
	unsigned char control;		/* TOC entry control field */
#define CONTROL_PREEMPH	1		/* audio track with preemphasis */
#define CONTROL_COPY	2		/* 0=copy prohibited, 1=copy allowed */
#define CONTROL_DATA	4		/* 0 = audio, 1 = data */
#define CONTROL_4CHAN	8		/* 0 = 2 channel, 1 = 4 channel */
};

#ifndef MAX_TRACKS
#  define MAX_TRACKS 100
#endif
struct disc_info{
	int		tracks;
	struct toc	cdtoc[MAX_TRACKS];
	unsigned long	discid;
	unsigned long	tot_secs;	/* total frames / 75 */
	unsigned long	play_secs;
	char		*artist;
	char		*album;
	char		*year;
	char		*genre;
	char		*comment;
	char		*track_names[MAX_TRACKS];
	char		*artists[MAX_TRACKS];	/* for compilations */
	int		revision;	/* cddb file revision */
	unsigned char	various;	/* set if multiple artists */
};
extern struct disc_info cd;


#ifdef DEBUG
#  define DEBUG_PUTS(str) fputs(str, stderr)
#  ifdef __GNUC__
#    define DEBUG_PRINTF(args...) fprintf(stderr, args)
#  else
#    define DEBUG_PRINTF(...) fprintf(stderr, __VA_ARGS__)
#  endif
#else
#  define DEBUG_PUTS(str)
#  ifdef __GNUC__
#    define DEBUG_PRINTF(args...)
#  else
#    define DEBUG_PRINTF(...)
#  endif
#endif

/* in cripple.c */
extern int check_open(const char *fname, int flags, mode_t mode);
#ifdef USE_INLINE
static __inline__ void extend_mb(struct malloced_buffer *m, int new_size)
{
	m->size = (new_size + BUFSIZE - 1) & ~(BUFSIZE - 1);
	m->buf = realloc(m->buf, m->size);
	if(m->buf == NULL) {
		perror("realloc()");
		exit(1);
	}
}

static __inline__ void check_write(const int fd, const void *data,
		const size_t size)
{
	int i=0;

	do {
		if((i += write(fd, (char*)data+i, size-i)) == -1) {
			perror("write()");
			exit(1);
		}
	} while(size-i);
}
#else
extern __inline__ void extend_mb(struct malloced_buffer *m, int new_size);
extern __inline__ void check_write(int fd, const void *buf, size_t count);
#endif

extern const char *cd_dev;
extern int drive_speed;
	

/* disc.c */
extern int read_cdtoc_from_drive(void);
extern void cddb_discid(void);
extern int eject_cdrom(void);
extern void print_cdtoc();
extern int read_mode2_track(int trk, int fd);

/* net.c */
extern void get_cddb_info(const char *serv_addr, unsigned short serv_port,
			  const char *cgi_path,
			  const char *proxy_addr, unsigned short proxy_port);

/* id3tag.c */
extern void write_id3tag(const char *filename, int track);
