/*
 * cripple - command line CD ripper/encoder wrapper with cddb support
 *
 * Copyright (C) 2003-2005 Neil Phillips
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "cripple.h"

#include <ctype.h>

#if HAVE_STRINGS_H
#  include <strings.h>
#endif

#if HAVE_GETOPT_LONG
#  include <getopt.h>
#else
struct option {
	const char *name;
	int has_arg;
	int *flag;
	int val;
};
#endif

const struct encoder enc_bladeenc = {ENC_BLADEENC, ENC_BLADEENC_OPTS, ".mp3"};
const struct encoder enc_lame     = {ENC_LAME,     ENC_LAME_OPTS,     ".mp3"};
const struct encoder enc_flac     = {ENC_FLAC,     ENC_FLAC_OPTS,     ".flac"};

const struct ripper rip_cdparanoia  = {RIP_CDPARANOIA,  RIP_CDPARANOIA_OPTS};
const struct ripper rip_cd_paranoia = {RIP_CD_PARANOIA, RIP_CD_PARANOIA_OPTS};
const struct ripper rip_cdda2wav    = {RIP_CDDA2WAV,    RIP_CDDA2WAV_OPTS};

/* booleans */
int read_data_trks  = 0;		/* experimental! rip data tracks */
int exit_eject      = 0;		/* eject on exit */
int just_query      = 0;		/* just get and output the cddb info */
int dont_rip        = 0;		/* don't perform actual rip */
int print_toc       = 0;		/* print toc + discid, then exit */

#ifdef __win32__
int fat_compat_names= 1;		/* don't use chars that upset FAT fs */
#else
int fat_compat_names= 0;
#endif

/* values */
int start_track     = 0;		/* optional user-specified start/end */
int end_track       = 0;
int drive_speed     = DEF_CD_SPEED;
const char *blank_file    = NULL;	/* write blank cddb file */
const char *path_prefix   = DEF_PATH_PREFIX;
const char *input_file    = NULL;	/* read from file rather than freedb */
const char *output_file   = NULL;	/* write cddb info to file */
const char *enc_user_opts = NULL;
const char *rip_user_opts = NULL;
const char *cd_dev        = DEF_CD_DEV;
char *cgi_path            = NULL;

char def_server[]         = DEF_SERVER;
char *http_server         = def_server;

#ifdef DEF_PROXY
char def_proxy[]          = DEF_PROXY;
char *http_proxy          = def_proxy;
#else
char *http_proxy          = NULL;
#endif

const struct encoder *enc  = &DEF_ENCODER;
const struct ripper  *rip  = &DEF_RIPPER;
void (*capital_fn)(char *) = NULL;

struct disc_info cd;


#define xstr(s) str(s)
#define str(s) #s
static const char useage[] = "Useage:\n"
"    -a --ripper-opts opts\tpass options to ripper in place of defaults\n"
"    -c --comment text\t\tuse comment in all id3 tags (<=28 chars)\n"
"    -C --capitalise none|simple|clever|upper|lower\n"
"    -d --device dev\t\tcdrom device file. default: '" DEF_CD_DEV "'\n"
"    -D --readdata\t\tread mode2 data tracks (experimental)\n"
"    -e --eject\t\t\teject cdrom when done\n"
"    -E --encoder bladeenc|lame|flac\n"
"    -F --fat\t\t\tmake filenames valid on crappy filesystems\n"
"    -h --help\t\t\tthis\n"
"    -i --infile file\t\tinput xmcd format file (use instead of server)\n"
"    -o --outfile file\t\toutput xmcd format file\n"
"    -O --encoder-opts opts\tpass options to encoder in place of defaults\n"
"    -p --path path\t\toutput path prefix. default: '" DEF_PATH_PREFIX "'\n"
"    -q --justquery\t\tquery only (for use with -o)\n"
"    -r --norip\t\t\tonly pretend to rip (for checking options etc.)\n"
"    -R --ripper cdda2wav|cdparanoia\n"
"    -s --server host:port/path\tspecify http server. port,path optional\n"
"\t\t\t\tdefault: " DEF_SERVER ":" xstr(DEF_HTTP_PORT) DEF_CGI_PATH "\n"
"    -S --speed n\t\tset drive speed to n. default: " xstr(DEF_CD_SPEED) "\n"
"    -t --tracks m[,n]\t\trip from track m to track n\n"
"    -T --toc\t\t\tprint internal representation of TOC, then exit\n"
"    -V --version\t\tprint version\n"
"    -w --blankfile file\t\twrite blank xmcd file for cd, then exit\n"
"    -x --proxy host:port\tuse http proxy." 
#ifdef DEF_PROXY
	" default: " DEF_PROXY
#endif
"\n\nSee manpage for more details (i.e. 'man 1 cripple')\n";

static const char version[] = CLIENT "-" VERSION
	" -- command line CD ripper/encoder wrapper with cddb support\n"
	"Copyright (C) 2003-2005 Neil Phillips\n\n"
	"This program is free software; you can redistribute it and/or"
	" modify it under\nthe terms of the GNU General Public License"
	" version 2 or later.\n";


struct malloced_buffer mb = {NULL, 0, 0};


/*
 * static inline copies of these are in cripple.h
 */
#ifndef USE_INLINE
static __inline__ void extend_mb(struct malloced_buffer *m, int new_size)
{
	m->size = (new_size + BUFSIZE - 1) & ~(BUFSIZE - 1);
	m->buf = realloc(m->buf, m->size);
	if(m->buf == NULL) {
		perror("realloc()");
		exit(1);
	}
}

__inline__ void check_write(const int fd, const void *data, const size_t size)
{
	int i=0;

	do {
		if((i += write(fd, (char*)data+i, size-i)) == -1) {
			perror("write()");
			exit(1);
		}
	} while(size-i);
}
#endif



int check_open(const char *fname, const int flags, const mode_t mode)
{
	int fd;
	
	if(!fname || !*fname) {
		fputs("null filename!\n", stderr);
		exit(1);
	}
	if(*fname == '-' && !fname[1]) {
		switch(flags & (O_RDONLY|O_WRONLY|O_RDWR)) {
			case O_RDONLY:	return 0;	/* stdin */
			case O_WRONLY:	return 1;	/* stdout */
			default:	exit(1);
		}
	}
	if( (fd = open(fname, flags, mode)) == -1) {
		perror("open()");
		fprintf(stderr, "error opening: %s\n", fname);
		exit(1);
	}
	return fd;
}



char *find_info_line(const char *cmp_str)
{
	int len = strlen(cmp_str);
	char *tmp = mb.buf, *tmp2, *tmp3, *ret;
	char *lim = mb.buf + mb.off - len;
	char *lim2 = mb.buf + mb.off;

	while(tmp < lim && memcmp(tmp, cmp_str, len)) {
		for(; tmp < lim && *tmp >= 0x20; tmp++);
		for(; tmp < lim && *tmp < 0x20; tmp++);
	}
	if(tmp >= lim) {
		fprintf(stderr, "no %s found, exiting!\n", cmp_str);
		exit(1);
	}
	tmp += len;
	ret = tmp;

	/*
	 * now we have to deal with concatinating consecutive lines matching
	 * cmp_str. Note may have LF or CR-LF line endings.
	 */
	for(; tmp < lim2 && *(unsigned char *)tmp >= 0x20; tmp++);
	tmp3 = tmp;

	while(tmp3 < lim && (tmp2 = memchr(tmp3, '\n', lim - tmp3)) &&
			!strncmp(++tmp2, cmp_str, len)) {
		tmp2 += len;
		for(tmp3=tmp2; tmp3 < lim2 && *(unsigned char *)tmp3 >= 0x20;
				tmp3++);
		memmove(tmp, tmp2, tmp3 - tmp2);
		tmp += tmp3 - tmp2;
	}
	
	*tmp = '\0';
	return ret;
}



/*
 * filename expansion for xmcd/cddb files
 * 
 * *out->buf should have been malloc()ed or NULL
 */
void parse_filename(const char *in, struct malloced_buffer *out)
{
	out->off = 0;

	do {
		if(out->off + 8 > out->size)
			extend_mb(out, out->off+2);
		if(*in == '%') {
		    switch( *(++in) ) {
			case 'd': out->off += sprintf(out->buf+out->off,
						  "%08lx", cd.discid);
				  break;
			case '%': out->buf[out->off++] = '%';
				  break;
			default : break;
		    }
		} else out->buf[out->off++] = *in;
	} while( *(in++) );
}


/* 
 * backslash escape string for use in shell "" quotes.
 * 
 * *out->buf should have been malloc()ed or NULL
 */
void shell_escape(const char *in, struct malloced_buffer *out)
{
	out->off = 0;

	do {
		if(out->off + 2 > out->size)
			extend_mb(out, out->off+2);
		switch(*in) {
			case '\\':
			case '\"':
			case '`' :
			case '$' : out->buf[out->off++] = '\\';
			default  : break;
		}
		out->buf[out->off++] = *in;
	} while( *(in++) );
}


/*
 * retrieve host, port and path from URL without protocol specifier.
 * in:  str = "host:port/path". if path==NULL, no path is returned
 * out: str = "host", *path = "/path", return port
 */
unsigned short parse_url(char *str, char **path)
{
	char *chrptr;
	unsigned long port = 0;

	if(path) *path = NULL;

	if(( chrptr = strchr(str, ':') )) {
		*chrptr = '\0';		/* replace ':' with '\0' */
		if(!(port = strtoul(chrptr+1, path, 10)) || port > 0xffff) {
			fputs("invalid port number\n", stderr);
			exit(1);
		}
	}
	if(path) {
		if(*path) {	/* there was a :port */
			if(**path && **path != '/') {
				fprintf(stderr, "invalid path: %s\n", *path);
				exit(1);
			}
		} else { /* need to copy path so we can terminate host */
			if((chrptr = strchr(str, '/')) && *chrptr) {
				*path = strdup(chrptr);
				*chrptr = '\0';
			}
		}
				
	}

	return (unsigned short)port;
}


/*
 * Capitalise each word -- upper case first char, lower the rest.
 */
void capitalise_simple(char *str)
{
	if(!*str) return;

	do {
		*str = toupper(*str);
		for(str++; *str && (isalnum((int)*str) || *str == '\''); str++)
			*str = tolower(*str);
		while(*str && !(isalnum((int)*str) || *str=='\''))
			str++;
	} while(*str);
}

/*
 * as above, but some words all lower case. First word is allways Capitalised.
 * Not really very clever as it messes up roman numerals, abbreviations without
 * '.'s etc.
 *
 * suggestions:
 *  - uppering words without vowels.
 *  - uppering roman numerals using roman numeral syntax checker.
 *  - use an up_words list for a few abbreviations not covered above.
 */
void capitalise_clever(char *str)
{
	const char *low_words[] = {"a", "an", "as", "and", "are", "at", "e.g."
		"etc." "feat.", "for", "from", "in", "is", "it", "its", "of",
		"on", "or", "the", "to", "too", "vs.", "with", NULL};
/*	const char *up_words[] = {"AC", "BBC", "CD", "DMX", "DC", "DJ", "EP",
		"II", "III", "IV", "IX", "LP", "NWA", "TV", "UK", "USA", "VI",
		"VII", "VIII", "ZZ"};*/
	int i, len;

	for(; *str && !isalnum((int)*str); str++);
	if(!*str) return;

	do {
		*str = toupper(*str);
		for(str++; *str && (isalnum((int)*str) || *str == '\''); str++)
			*str = tolower(*str);
		while(*str && !(isalnum((int)*str) || *str=='\'')) {
		    if(*(str++) == ' ') {
			for(i=0; low_words[i]; i++) {
				len = strlen(low_words[i]);
				if(!strncasecmp(str, low_words[i], len)
					&& !isalnum((int)str[len])) {
				    memcpy(str, low_words[i], len);
				    str += len;
				    if(!*str) return;
				    i=0;
				}
			}
		    }
		}
	} while(*str);
}


void capitalise_lower(char *str)
{
	for(; *str; str++)
		*str = tolower(*str);
}


void capitalise_upper(char *str)
{
	for(; *str; str++)
		*str = toupper(*str);
}


void read_cddb_info_from_file(const char *fname)
{
	int i, fd;
	
	fd = check_open(fname, O_RDONLY, 0);
	fprintf(stderr, "reading cddb info from file '%s'...\n", fname);

	while( (i = read(fd, mb.buf + mb.off, mb.size - mb.off)) ) {
		mb.off += i;
		if(mb.off >= mb.size)
			extend_mb(&mb, mb.size + BUFSIZE);
	}
	mb.buf[mb.off] = '\0';

	DEBUG_PRINTF("%s\n\n", mb.buf);

	close(fd);
}



static __inline__ void dump_disc_info(void)
{
	int i = 0;
	
	fprintf(stderr, "\nartist:\t\t%s\n"
			"album:\t\t%s\n"
			"year:\t\t%s\n"
			"genre:\t\t%s\n"
			"no. tracks:\t%d\n"
			"play time:\t%lu:%02lu\n"
			"discid:\t\t%08lx\n\n",
			cd.artist, cd.album, cd.year, cd.genre, cd.tracks,
			cd.play_secs/60, cd.play_secs%60, cd.discid);

	if(cd.comment && *cd.comment)
		fprintf(stderr, "comment:\t%s\n\n", cd.comment);

	for(; i<cd.tracks; i++)
		if(cd.various && cd.artists[i] && *cd.artists[i])
			fprintf(stderr, "track %2d:\t%s - %s\n", i+1,
					cd.artists[i], cd.track_names[i]);
		else
			fprintf(stderr, "track %2d:\t%s\n", i+1,
					cd.track_names[i]);
}



static __inline__ void xmcd_parse_names(void)
{
	char tt_str[] = "TTITLE999=", *c1, *c2;
	int i, artist_len;

	cd.artist = find_info_line("DTITLE=");
	cd.year   = find_info_line("DYEAR=");
	cd.genre  = find_info_line("DGENRE=");

	/*
	 * DTITLE=Artist / Album
	 */
	if(( cd.album = strstr(cd.artist, " / ") )) {
		*cd.album = '\0';
		cd.album += 3;
	}

	artist_len = strlen(cd.artist);

	cd.various = ! strcasecmp("Various", cd.artist);

	for(i=0; i < cd.tracks; i++) {
		/*
		 * can have just track name or "artist / track_name". if cd
		 * isn't a compilation, we ignore the artist part
		 */
		sprintf(tt_str+6, "%d=", i);
		c1 = find_info_line(tt_str);
		c2 = strstr(c1, " / ");
		if(cd.various && c2) {
			cd.artists[i] = c1;
			*c2 = '\0';
			cd.track_names[i] = c2 +  3;
		} else if(c2 == c1+artist_len &&
				!strncasecmp(c1, cd.artist, artist_len)) {
			cd.track_names[i] = c2 +  3;
			cd.artists[i] = NULL;
		} else {
			cd.track_names[i] = c1;
			cd.artists[i] = NULL;
		}
	}
}



static __inline__ void xmcd_write_file(char *fname)
{
	FILE *fp;
	int i;

	if(!*fname) {
		fputs("null filename!\n", stderr);
		exit(1);
	}
	if(*fname == '-' && !fname[1]) fp = stdout;
	else if(!(fp = fopen(fname, "w"))) {
		perror("fopen()");
		fprintf(stderr, "unable to open: %s\n", fname);
		exit(1);
	}
	
	fputs("# xmcd\n"
	      "#\n"
	      "# Track frame offsets:\n", fp);

	for(i=0; i<cd.tracks; i++)
		fprintf(fp, "#\t%u\n", cd.cdtoc[i].frame);

	fprintf(fp, "#\n"
		"# Disc length: %lu seconds\n"
		"#\n"
		"# Revision: 0\n"
		"# Submitted via: " CLIENT " " VERSION "\n"
		"#\n"
		"DISCID=%08lx\n"
		"DTITLE=%s / %s\n"
		"DYEAR=%s\n"
		"DGENRE=%s", cd.tot_secs, cd.discid,
		(cd.artist && *cd.artist) ? cd.artist : "<artist>",
		(cd.album && *cd.album) ? cd.album : "<album>",
		cd.year ? cd.year : "", cd.genre ? cd.genre : "");

	for(i=0; i<cd.tracks; i++) {
		fprintf(fp, "\nTTITLE%u=", i);
		if(cd.cdtoc[i].control & CONTROL_DATA)
			fputs("Data Track", fp);
		else if(cd.track_names[i] && *cd.track_names[i]) {
			if(cd.various && cd.artists[i])
				fprintf(fp, "%s / ", cd.artists[i]);
			fputs(cd.track_names[i], fp);
		}
	}

	fputs("\nEXTD=\n", fp);
	
	for(i=0; i<cd.tracks; i++)
		fprintf(fp, "EXTT%u=\n", i);

	fputs("PLAYORDER=\n", fp);

	fprintf(stderr, "Wrote: %s\n", fname);

	fclose(fp);
}
	
		
	
static __inline__ void write_cddb_info_to_file(const char *filename)
{
	int fd;
	
	fprintf(stderr, "writing cddb info to output file '%s' ...\n\n",
			filename);
	fd = check_open(filename, O_WRONLY|O_CREAT|O_TRUNC, 0644);
	check_write(fd, mb.buf, mb.off);
	close(fd);
}



static __inline__ int parse_opts(int argc, char **argv)
{
	const struct option opts[] = 
	{
		{"norip",	0, 0, 'r'},
		{"justquery",	0, 0, 'q'},
		{"eject",	0, 0, 'e'},
		{"help",	0, 0, 'h'},
		{"readdata",	0, 0, 'D'},
		{"fat",         0, 0, 'F'},
		{"toc",		0, 0, 'T'},
		{"version",	0, 0, 'V'},
		{"capitalise",	1, 0, 'C'},
		{"encoder",	1, 0, 'E'},
		{"encoder-opts",1, 0, 'O'},
		{"ripper",	1, 0, 'R'},
		{"ripper-opts", 1, 0, 'a'},
		{"tracks",	1, 0, 't'},
		{"comment",	1, 0, 'c'},
		{"device",	1, 0, 'd'},
		{"path",	1, 0, 'p'},
		{"infile",	1, 0, 'i'},
		{"outfile",	1, 0, 'o'},
		{"blankfile",	1, 0, 'w'},
		{"server",	1, 0, 's'},
		{"speed",	1, 0, 'S'},
		{"proxy",	1, 0, 'x'},
		{0,0,0,0}
	};
	int	c, j, err=0;
	char	*chrptr;

#if HAVE_GETOPT_LONG
	while((c = getopt_long(argc, argv,
		"rqehDFTVC:E:O:R:a:t:c:d:g:p:i:o:w:s:S:x:", opts, &j)) != -1) {
#else
	int i, k;
	char *optarg;

	i=1; j=0;
	while(i<argc && !err) {
		if(!argv[i][j]) {
			i++;
			j=0;
			continue;
		}

		err++;

		if(!j && (argv[i][j] != '-' || !argv[i][++j]))
			break;

		if(j==1 && argv[i][j] == '-') {		/* long option */
			chrptr = argv[i]+j+1;
			for(k=0; opts[k].name &&
				strncmp(chrptr, opts[k].name,
					(int)strlen(opts[k].name)); k++);
			if(!opts[k].name)
				break;
			c = opts[k].val;
			if(opts[k].has_arg) {
				if((chrptr = strchr(argv[i], '=')))
					optarg = chrptr+1;
				else if(++i < argc)
					optarg = argv[i];
				else
					break;
			} else optarg = NULL;
			i++; j=0;
		} else {				/* short option */
			c = argv[i][j++];
			for(k=0; opts[k].name && c != opts[k].val; k++);
			if(!opts[k].name)
				break;
			if(opts[k].has_arg) {
				if(argv[i++][j])
					optarg = argv[i]+j;
				else if(i < argc)
					optarg = argv[i++];
				else
					break;
				j=0;
			} else optarg = NULL;
		}

		err=0;
#endif
		switch(c) {
			case 'r': dont_rip  ^= 1;
				  break;
			case 'q': just_query ^= 1;
				  break;
			case 'e': exit_eject ^= 1;
				  break;
			case 'T': print_toc ^= 1;
				  break;
			case 'D': read_data_trks ^= 1;
				  break;
			case 'F': fat_compat_names ^= 1;
				  break;
			case 'h': fputs(useage, stderr);
				  exit(0);
			case 'V': fputs(version, stderr);
				  exit(0);
			case 't': if(!(start_track =
						strtol(optarg, &chrptr,0))){
				  	err++;
					break;
				  }
				  if(*chrptr && *(++chrptr))
					end_track = atoi(chrptr);
				  break;
			case 'c': cd.comment = optarg;
				  break;
			case 'd': cd_dev = optarg;
				  break;
			case 'p': path_prefix = optarg;
				  break;
			case 'i': input_file = optarg;
				  break;
			case 'o': output_file = optarg;
				  break;
			case 'w': blank_file = optarg;
				  break;
			case 's': http_server = optarg;
				  break;
			case 'x': http_proxy = *optarg ? optarg : NULL;
				  break;
			case 'C': if(!strcmp(optarg, "none"))
					  capital_fn = NULL;
				  else if(!strcmp(optarg, "simple"))
					  capital_fn = capitalise_simple;
				  else if(!strcmp(optarg, "clever"))
					  capital_fn = capitalise_clever;
				  else if(!strcmp(optarg, "lower"))
					  capital_fn = capitalise_lower;
				  else if(!strcmp(optarg, "upper"))
					  capital_fn = capitalise_upper;
				  else err++;
				  break;
			case 'E': if(!strcmp(optarg, "bladeenc")) {
					  enc = &enc_bladeenc;
				  }
				  else if(!strcmp(optarg, "lame")) {
					  enc = &enc_lame;
				  }
				  else if(!strcmp(optarg, "flac")) {
					  enc = &enc_flac;
				  }
				  else err++;
				  break;
			case 'O': enc_user_opts = optarg;
				  break;
			case 'R': if(!strcmp(optarg, "cdda2wav"))
					  rip = &rip_cdda2wav;
				  else if(!strcmp(optarg, "cdparanoia"))
					  rip = &rip_cdparanoia;
				  else if(!strcmp(optarg, "cd-paranoia"))
					  rip = &rip_cd_paranoia;
				  else err++;
				  break;
			case 'a': rip_user_opts = optarg;
				  break;
			case 'S': if(isdigit((int)*optarg)) {
					  drive_speed = atoi(optarg);
					  break;
				  }
			default : err++;
				  break;
		}
	}
	return err;
}



/*
 * not all systems have on_exit(), so we have to improvise.
 */
#define EJECT_RETURN return (exit_eject ? eject_cdrom() : 0);

int main(int argc, char **argv)
{
	int i, j;
	unsigned short server_port = 0, proxy_port = 0;
	struct malloced_buffer filename = {NULL,0,0};
	struct malloced_buffer esc_cd_dev = {NULL,0,0};
	struct malloced_buffer esc_filename = {NULL,0,0};

	memset(&cd, 0, sizeof(struct disc_info));

	if(parse_opts(argc, argv)) {
		fputs("\nunknown / invalid option\n", stderr);
		fputs(useage, stderr);
		return 1;
	}

	if(read_cdtoc_from_drive())
		return 1;

	cddb_discid();

	if(print_toc) {
		print_cdtoc();
		EJECT_RETURN;
	}

	if(blank_file) {
		parse_filename(blank_file, &filename);
		xmcd_write_file(filename.buf);
		EJECT_RETURN;
	}

	if(start_track) start_track--;
	if(!end_track || end_track>cd.tracks) end_track = cd.tracks;

	/*
	 * buffer for server responses / info from files that we can extend
	 */
	extend_mb(&mb, BUFSIZE);
	
	/*
	 * get cddb entry for CD by some means
	 */
	if(input_file) {
		parse_filename(input_file, &filename);
		read_cddb_info_from_file(filename.buf);
	} else {
		if(http_proxy) {
			if(!(proxy_port = parse_url(http_proxy, NULL))){
				fputs("no proxy port specified\n", stderr);
				fputs(useage, stderr);
				return 1;
			}
		}
		server_port = parse_url(http_server, &cgi_path);
		get_cddb_info(http_server, server_port, cgi_path, http_proxy,
				proxy_port);
	}

	/*
	 * done before parsing the contents, so it can be used for debugging
	 */
	if(output_file) {
		parse_filename(output_file, &filename);
		write_cddb_info_to_file(filename.buf);
	}

	xmcd_parse_names();

	if(capital_fn) {
		capital_fn(cd.artist);
		capital_fn(cd.album);
		for(i=0; i<cd.tracks; i++) {
			capital_fn(cd.track_names[i]);
			if(cd.artists[i])
				capital_fn(cd.artists[i]);
		}
	}
	
	/*
	 * dumping parsed data to xmcd file or whatever should go here
	 */

	dump_disc_info();

	shell_escape(cd_dev, &esc_cd_dev);

	if( ! just_query ) {
	    char *fname;
	    int max_len = 0, pp_len = 0;
	    char *request = NULL;

	    /*
	     * determine length needed for filenames...
	     */
	    for(i=0; i<cd.tracks; i++) {
		    j = strlen(cd.track_names[i]);
		    if(cd.artists[i] && *cd.artists[i])
			    j += strlen(cd.artists[i]);
		    if(j > max_len)
			    max_len = j;
	    }

	    if(path_prefix && *path_prefix) {
		    pp_len = strlen(path_prefix);
		    max_len += pp_len + 1;
		    fprintf(stderr, "\npath prefix:\t%s", path_prefix);
	    }
	    fputs("\n\n", stderr);
	    /* (3 * " - ") + "100" + ".xxxx" = 17 chars */
	    max_len += strlen(cd.artist) + strlen(cd.album) + 17 + 0xff;
	    max_len &= ~0xff;
	    if(max_len > filename.size)
		    extend_mb(&filename, max_len);

	    /*
	     * ... and command line.
	     */
	    max_len += strlen(enc->cmd);
	    max_len += strlen(rip->cmd);
	    max_len += strlen(enc_user_opts ? enc_user_opts : enc->opts);
	    max_len += strlen(rip_user_opts ? rip_user_opts : rip->opts);
	    max_len += esc_cd_dev.off + 0xff;
	    max_len &= ~0xff;
	    request = malloc(max_len);
	    if(request == NULL) {
		    perror("malloc()");
		    return 1;
	    }
	    
	    if(pp_len) {
		    memcpy(filename.buf, path_prefix, pp_len);
		    fname = filename.buf + pp_len;
		    if(*(fname-1) != PATH_SEPERATOR)
			    *(fname++) = PATH_SEPERATOR;
	    } else
		    fname = filename.buf;
		    
		    
	    for(i = start_track; i < end_track; i++) {
		char *c;

		if(cd.various)
			if(cd.artists[i] && *cd.artists[i])
				j = sprintf(fname, "%s - %02d - %s - %s",
					cd.album, i+1, cd.artists[i],
					cd.track_names[i]);
			else
				j = sprintf(fname, "%s - %02d - %s",
					cd.album, i+1, cd.track_names[i]);
		else
			j = sprintf(fname, "%s - %s - %02d - %s", cd.artist,
				cd.album, i+1, cd.track_names[i]);
		/*
		 * deal with invalid filename characters.
		 */
		for(c = fname; *c; c++) {
			const char fat_chars[] = "\\:*?\"<>|";
			if(*c == '/') *c = '-';
			else if(fat_compat_names && strchr(fat_chars, *c))
				*c = '_';
		}
	
		if(cd.cdtoc[i].control & CONTROL_DATA) {
		    if(read_data_trks) {
			int fd;
			strcat(fname, ".bin");
			fprintf(stderr, "reading data track %d to \"%s\"\n\n",
					i+1, filename.buf);
			if(!dont_rip) {
				fd = check_open(filename.buf, O_WRONLY|O_CREAT,
						0644);
				read_mode2_track(i, fd);
				close(fd);
			}
		    }
		    continue;
		}
			
		if(cd.cdtoc[i].control & CONTROL_PREEMPH)
			fputs("WARNING: doing bugger all about preemphasis!\n",
					stderr);

		strcat(filename.buf, enc->ext);

		shell_escape(filename.buf, &esc_filename);

		j = sprintf(request, rip->cmd,
				rip_user_opts ? rip_user_opts : rip->opts,
				esc_cd_dev.buf, drive_speed, i+1, i+1);
		strcat(request+j, " | ");
		sprintf(request+j+3, enc->cmd,
				enc_user_opts ? enc_user_opts : enc->opts,
				esc_filename.buf);
		fprintf(stderr, "calling:\n%s\n\n", request);

		/*
		 * rip the track and slap a tag on its ass
		 */
		if( ! dont_rip ) {
		    if( system(request) ) {
			fprintf(stderr, "\nerror ripping track %d, exiting\n\n",
					i+1);
			return 1;
		    }
		    write_id3tag(filename.buf, i);
		}
	    }
	}
	EJECT_RETURN;
}

