#!/bin/sh

# Setup env
set -xe
. /environ

# Clone git repository at branch / tag / commit.
REPO='https://github.com/FFmpeg/FFmpeg'; TAG='n5.1.6'
git clone "$REPO" --depth=1 --branch=${TAG} /ffmpeg
cd /ffmpeg

# Make modifications to flags.
export CFLAGS=$(trim_spaces ${CFLAGS} \
-I/vendor/include \
-D_WASI_EMULATED_MMAN \
-D_WASI_EMULATED_PROCESS_CLOCKS \
-D_WASI_EMULATED_SIGNAL)
export LDFLAGS=$(trim_spaces ${LDFLAGS} \
--no-entry \
--static \
--import-undefined \
--export-memory \
-L/vendor/lib \
-lwasi-emulated-process-clocks \
-lwasi-emulated-mman \
-lwasi-emulated-signal \
${WASI_SYSROOT}/lib/wasm32-wasi/crt1-command.o)
export PKG_CONFIG_PATH='/vendor/lib/pkgconfig'

# Apply our own changes
git apply /ffmpeg.patch

# Configure build parameters
./configure --cc="$CC" \
            --cxx="$CXX" \
            --ld="$LD" \
            --nm="$NM" \
            --ar="$AR" \
            --ranlib="$RANLIB" \
            --strip="$STRIP" \
            --enable-cross-compile \
            --disable-shared \
            --disable-stripping \
            --enable-static \
            --enable-small \
            --arch=x86_32 \
            --target-os=none \
            --disable-doc \
            --disable-debug \
            --disable-network \
            --disable-pthreads \
            --disable-w32threads \
            --disable-os2threads \
            --disable-runtime-cpudetect \
            --disable-asm \
            --disable-ffprobe \
            --disable-ffmpeg \
            --disable-ffplay \
            --enable-ffmpreg \
            --disable-protocols \
            --enable-protocol=file \
            --enable-protocol=pipe \
            --enable-gpl \
            --enable-version3 \
            --enable-libwebp \
            --enable-libx264 \
            --enable-libmp3lame \
            --enable-libopus \
            --enable-libvpx \
            --enable-libaom \
|| { cat ffbuild/config.log; exit 1; }

# Various configuration and other fixes to get build working
sed -i config.h -e 's|#define HAVE_SYSCTL 1|#define HAVE_SYSCT 0L|'
sed -i config.h -e 's|#define HAVE_MKSTEMP 1|#define HAVE_MKSTEMP 0|'
sed -i config.h -e 's|#define HAVE_GETHRTIME 1|#define HAVE_GETHRTIME 0|'
sed -i config.h -e 's|#define HAVE_SETRLIMIT 1|#define HAVE_SETRLIMIT 0|'
sed -i ffbuild/config.mak -e 's|SHFLAGS=.*|SHFLAGS=|'
echo '#define F_DUPFD_CLOEXEC 1' >> config.h
sed -i libavutil/file_open.c -e '1s/^/char *tempnam(const char *dir, const char *pfx);\n/'

# Build all components
make -j $(nproc) all

# Create build
mkdir -p /build

[ -z "$RELEASE" ] && {
    # Skip below opts unless preparing
    # release binary, it takes FOREVER
    for tool in ffmpreg; do
        wasm-opt ${WASMOPTFLAGS} \
                -o /build/${tool} \
                ${tool}
    done
} || {
    # For more information on arguments passed to
    # wasm-opt please see `wasm-opt --help` and:
    # https://github.com/WebAssembly/binaryen/wiki/Optimizer-Cookbook
    #
    # --dce                       : dead code elimination
    # --vacuum                    : remove more obviously un-needed code
    # --precompute-propagate      : compute compile-time evaluatable exprs and propagate through locals
    # -Oz                         : a combined set of optimization passes focused on *size*
    # -O{3,4}                     : a combined set of optimization passes focused on *speed*
    # --flatten                   : flattens out code, removing nesting
    # --rereloop                  : re-optimize control flow using relooper algorithm
    for tool in ffmpreg; do
        wasm-opt ${WASMOPTFLAGS} \
                --dce --vacuum \
                --precompute-propagate \
                --flatten --rereloop -Oz -Oz \
                --flatten -O3 \
                --flatten -O3 \
                -O4 -O4 \
                -o /build/${tool} \
                ${tool}
    done
}

# GZip compress outputs
for tool in ffmpreg; do
    gzip -9 /build/${tool} -c \
        > /build/${tool}.gz
done