/*

        $Id: config.c,v 1.5 1999/07/11 22:03:58 tron Exp $

*/

#include <sys/types.h>
#include <netinet/in.h>
#include <arpa/inet.h>

#include <ctype.h>
#include <netdb.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#include "ipstat.h"

#define MAX_ARGS 32

typedef struct CommandStruct
 {
  char *c_Name;
  int c_MinArgs,c_MaxArgs;
 } Command;

static Command CommandTab[] =
 {
  {"account",1,1},
  {"count",1,11},
  {NULL}
 };

char *Types[] =
 {
  "from",
  "to",
  "proto",
  "fromport",
  "toport",
  NULL,
 };

#define BITS_PER_IP 32

u_int32_t Masks[BITS_PER_IP+1];

static char *ReadLine(FILE *In,char *Buffer,int Max,int *Line)

{
 char *Ptr;
 int Len;

 while ((Ptr=fgets(Buffer,Max,In))!=NULL)
  {
   (*Line)++;

   while (isspace(*Ptr)) Ptr++;
   if (*Ptr=='#') continue;

   Len=strlen(Ptr);
   while ((Len>0)&&isspace(Ptr[Len-1])) Len--;
   if (Len==0) continue;

   Ptr[Len]='\0';
   break;
  }

 return Ptr;
}

static char *GetWord(char **Buffer,int AllowQuotes)

{
 char *Word,*Ptr;

 Ptr=*Buffer;
 while (isspace(*Ptr)) Ptr++;
 if (*Ptr=='\0') return NULL;

 if (AllowQuotes&&(*Ptr=='"'))
  {
   Word=++Ptr;
   while (*Ptr!='"')
    if (*Ptr++=='\0') return NULL;
   *Ptr++='\0';
  }
 else
  {
   Word=Ptr;
   while (*Ptr!='\0')
    if (isspace(*Ptr))
     {
      *Ptr++='\0';
      break;
     }
    else Ptr++;
  }

 *Buffer=Ptr;
 return Word;
}

void DeleteConfig(Config *Cfg)

{
 Account *Acc;
 Match *Rl;

 while ((Acc=Cfg->c_Accounts)!=NULL)
  {
   Cfg->c_Accounts=Acc->a_Next;
   free(Acc);
  }

 while ((Rl=Cfg->c_Matches)!=NULL)
  {
   Cfg->c_Matches=Rl->m_Next;
   free(Rl);
  }

 free(Cfg);
}

static Account *FindAccount(Config *Cfg,char *Name)

{
 Account *Acc;

 Acc=Cfg->c_Accounts;
 while (Acc!=NULL)
  {
   if (strcasecmp(Acc->a_Name,Name)==0) break;
   Acc=Acc->a_Next;
  }
 return Acc;
}

static char *ParseIP(char *Ptr,struct in_addr *TheIP)

{
 u_int32_t MyIP;
 char *EndPtr;
 int Index;

 MyIP=(u_int32_t)strtol(Ptr,&EndPtr,10);
 if (Ptr==EndPtr) return NULL;

 for (Index=0, Ptr=EndPtr; Index<3; Index++, Ptr=EndPtr)
  if (*Ptr++!='.') return NULL;
  else
   {
    MyIP=(MyIP<<8)|(u_int32_t)strtol(Ptr,&EndPtr,10);
    if (Ptr==EndPtr) return NULL;
   }

 TheIP->s_addr=htonl(MyIP);
 return Ptr;
}

static int ParseNet(char *Ptr,struct in_addr *Addr,struct in_addr *Mask)

{
 char *EndPtr;
 long Bits;

 if ((Ptr=ParseIP(Ptr,Addr))==NULL) return FALSE;

 if (*Ptr=='\0')
  {
   Mask->s_addr=0xFFFFFFFF;
   return TRUE;
  }

 if (*Ptr++!='/') return FALSE;

 if (ParseIP(Ptr,Mask)!=NULL) return TRUE;

 Bits=strtol(Ptr,&EndPtr,10);
 if ((Ptr==EndPtr)||(*EndPtr!='\0')||
     (Bits<0)||(Bits>BITS_PER_IP)) return FALSE;
 Mask->s_addr=htonl(Masks[Bits]);

 return ((Addr->s_addr&Mask->s_addr)==Addr->s_addr);
}

static int ParseConfig(char *Filename,FILE *In,Config *Cfg)

{
 int Index,Line,ArgC;
 char Buffer[1024],*Ptr,*Name,*ArgV[MAX_ARGS+1];
 Command *Cmd;
 Account **AccTail;
 Match **MtTail;

 Masks[BITS_PER_IP]=0xFFFFFFFF;
 Index=BITS_PER_IP;
 while (Index-->0) Masks[Index]=Masks[Index+1]<<1;

 Line=0;
 AccTail=&Cfg->c_Accounts;
 MtTail=&Cfg->c_Matches;
 while ((Ptr=ReadLine(In,Buffer,sizeof(Buffer),&Line))!=NULL)
  {
   Name=GetWord(&Ptr,FALSE);
   for (Index=0, Cmd=NULL; CommandTab[Index].c_Name!=NULL; Index++)
    if (strcasecmp(CommandTab[Index].c_Name,Name)==0)
     {
      Cmd=&CommandTab[Index];
      break;
     }
   if (Cmd==NULL)
    {
     (void)fprintf(stderr,
                   "%s: unknown command \"%s\" in line %d.\n",Filename,
                   Name,
                   Line);
     return FALSE;
    }

   for (ArgC=0; ArgC<=MAX_ARGS; ArgC++)
    if ((ArgV[ArgC]=GetWord(&Ptr,TRUE))==NULL) break;
   if (ArgC<Cmd->c_MinArgs)
    {
     (void)fprintf(stderr,
                   "%s: missing argument(s) for command \"%s\" in line %d.\n",
                   Filename,
                   Cmd->c_Name,
                   Line);
     return FALSE;
    }
   if (ArgC>Cmd->c_MaxArgs)
    {
     (void)fprintf(stderr,
                  "%s: too many arguments for command \"%s\" in line %d.\n",
                  Filename,
                  Cmd->c_Name,
                  Line);
     return FALSE;
    }

   switch (Index)
    {
     case 0:
      {
       Account *Acc;

       Acc=MAlloc(sizeof(Account)+strlen(ArgV[0]));
       Acc->a_Next=NULL;
       *AccTail=Acc;
       AccTail=&Acc->a_Next;

       Acc->a_Bytes=0;
       (void)strcpy(Acc->a_Name,ArgV[0]);
       break;
      }
     case 1:
      {
       Match *Mt,*RMt;
       char **Arg;
       int Type;

       Mt=CAlloc(sizeof(Match),1);
       *MtTail=Mt;
       MtTail=&Mt->m_Next;

       if ((Mt->m_Account=FindAccount(Cfg,ArgV[0]))==NULL)
        {
         (void)fprintf(stderr,
                       "%s: undefined account \"%s\" in line %d.\n",
                       Filename,
                       ArgV[0],
                       Line);
         return FALSE;
        }

       Arg=&ArgV[1];
       while (*Arg!=NULL)
        {
         Type=0;
         while (Types[Type]!=NULL)
          if (strcasecmp(Types[Type],*Arg)==0) break;
          else Type++;

         if (Types[Type]==NULL)
          {
           (void)fprintf(stderr,
                         "%s: invalid filter type \"%s\" in line %d.\n",
                         Filename,
                         *Arg,
                         Line);
           return FALSE;
          }
         Arg++;

         if (*Arg==NULL)
          {
           (void)fprintf(stderr,
                         "%s: missing argument for filter \"%s\" "
                         "in line %d.\n",
                         Filename,
                         Types[Type],
                         Line);
           return FALSE;
          }

         switch (Type)
          {
           case 0:
           case 1:
            {
             struct in_addr *Addr,*Mask;

             Addr=&((Type==0)?Mt->m_SrcAddr:Mt->m_DstAddr);
             Mask=&((Type==0)?Mt->m_SrcMask:Mt->m_DstMask);
             if (!ParseNet(*Arg,Addr,Mask))
              {
               (void)fprintf(stderr,
                             "%s: invalid network address \"%s\" "
                             "in line %d.\n",
                             Filename,
                             *Arg,
                             Line);
               return FALSE;
              }
             break;
            }
           case 2:
            {
             struct protoent *Proto;

             if ((Proto=getprotobyname(*Arg))==NULL)
              {
               (void)fprintf(stderr,
                             "%s: invalid protocol \"%s\" "
                             "in line %d.\n",
                             Filename,
                             *Arg,
                             Line);
               return FALSE;
              }
             Mt->m_Proto=Proto->p_proto;
             break;
            }
           case 3:
           case 4:
            {
             long Port;
             char *EndPtr;

             Port=strtol(*Arg,&EndPtr,10);
             if ((*Arg==EndPtr)||(*EndPtr!='\0')||
                 (Port<1)||(Port>65535))
              {
               (void)fprintf(stderr,
                             "%s: invalid port number \"%s\" "
                             "in line %d.\n",
                             Filename,
                             *Arg,
                             Line);
               return FALSE;
              }

             ((Type==3)?Mt->m_SrcPort:Mt->m_DstPort)=Port;
            }
          }

         Arg++;
        }

       if (((Mt->m_SrcPort>0)||(Mt->m_DstPort>0))&&
           ((Mt->m_Proto!=IPPROTO_TCP)&&(Mt->m_Proto!=IPPROTO_UDP)))
        {
         (void)fprintf(stderr,
                       "%s: port numbers can only be used for TCP/UDP "
                       "protocol in line %d.\n",
                       Filename,
                       Line);
         return FALSE;
        }

       RMt=MAlloc(sizeof(Match));
       RMt->m_Next=NULL;
       *MtTail=RMt;
       MtTail=&RMt->m_Next;

       RMt->m_Proto=Mt->m_Proto;
       RMt->m_SrcAddr=Mt->m_DstAddr;
       RMt->m_SrcMask=Mt->m_DstMask;
       RMt->m_DstAddr=Mt->m_SrcAddr;
       RMt->m_DstMask=Mt->m_SrcMask;
       RMt->m_SrcPort=Mt->m_DstPort;
       RMt->m_DstPort=Mt->m_SrcPort;
       RMt->m_Account=Mt->m_Account;
      }
    }
  }

 return TRUE;
}

static char *MaskToString(struct in_addr Mask)

{
 int Bits;
 static char Buffer[3];

 for (Bits=0; Bits<=BITS_PER_IP; Bits++)
  if (ntohl(Mask.s_addr)==Masks[Bits])
   {
    (void)sprintf(Buffer,"%d",Bits);
    return Buffer;
   }

 return inet_ntoa(Mask);
}

static void DumpNetwork(struct in_addr Addr,struct in_addr Mask)

{
 if ((Addr.s_addr!=0)||(Mask.s_addr!=0))
  {
   (void)fputc(' ',stderr);
   (void)fputs(inet_ntoa(Addr),stderr);
   if (Mask.s_addr!=0xFFFFFFFF)
    {
     (void)fputc('/',stderr);
     (void)fputs(MaskToString(Mask),stderr);
    }
  }
 else (void)fputs(" any",stderr);
}

static void DumpAccount(Match *List,Account *Acc)

{
 while (List!=NULL)
  {
   if (List->m_Account==Acc)
    {
     struct protoent *Proto;

     (void)fputs(" from",stderr);
     DumpNetwork(List->m_SrcAddr,List->m_SrcMask);
     if (List->m_SrcPort>0)
      (void)fprintf(stderr," fromport %d",List->m_SrcPort);

     (void)fputs(" to",stderr);
     DumpNetwork(List->m_DstAddr,List->m_DstMask);
     if (List->m_DstPort>0)
      (void)fprintf(stderr," toport %d",List->m_DstPort);

     if ((List->m_Proto>0)&&((Proto=getprotobynumber(List->m_Proto))!=NULL))
      {
       (void)fputs(" proto ",stderr);
       (void)fputs(Proto->p_name,stderr);
      }
     (void)fputc('\n',stderr);
    }
   List=List->m_Next;
  }
}

static void DumpConfig(Config *Cfg)

{
 Account *Acc;

 Acc=Cfg->c_Accounts;
 while (Acc!=NULL)
  {
   (void)fprintf(stderr,"Account \"%s\":\n",Acc->a_Name);
   DumpAccount(Cfg->c_Matches,Acc);
   (void)fputc('\n',stderr);
   Acc=Acc->a_Next;
  }
}

Config *ReadConfig(char *Filename)

{
 FILE *In;
 Config *Cfg;

 Cfg=CAlloc(sizeof(Config),1);

 if ((In=fopen(Filename,"r"))==NULL)
  {
   DeleteConfig(Cfg);
   (void)fprintf(stderr,
                 "%s: unable to open configuration file \"%s\".\n",
                 Progname,Filename);
   return NULL;
  }

 if (!ParseConfig(Filename,In,Cfg))
  {
   DeleteConfig(Cfg);
   Cfg=NULL;
  }
 (void)fclose(In);

 if (Verbose&&(Cfg!=NULL)) DumpConfig(Cfg);

 return Cfg;
}
