/*
 * Copyright (c) 2001 Tommy Bohlin <tommy@gatespace.com>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */
/* raw.c
 */

#include <irda.h>

/**********************************************************************
 * Constants
 **********************************************************************/

static const char id_raw[]="raw";

/**********************************************************************
 * Data structures
 **********************************************************************/

typedef struct Raw {
  SerialDevice sd;
  SerialPort* sp;
} Raw;

/**********************************************************************
 * Methods
 **********************************************************************/

static int rawSetSpeed(SerialDevice* sd, int speed)
{
  Raw* raw=(Raw*)sd;
  SerialPort* sp=raw->sp;

  sp->setSpeed(sp,speed);

  return 0;
}

static int rawGetSpeedMask(SerialDevice* sd)
{
  Raw* raw=(Raw*)sd;

  return raw->sp->getSpeedMask(raw->sp)&
    (SPEED_2400|SPEED_9600|SPEED_19200|SPEED_38400|SPEED_57600|SPEED_115200);
}

static int rawGetMinTurnaroundMask(SerialDevice* sd)
{
  /* NB: Can we decrease this value? */

  return MIN_TA_10ms;
}

static int rawGetChar(SerialDevice* sd)
{
  Raw* raw=(Raw*)sd;

  return raw->sp->getChar(raw->sp);
}

static void rawPutChar(SerialDevice* sd, int c)
{
  Raw* raw=(Raw*)sd;
  raw->sp->putChar(raw->sp,c);
}

static void rawClose(SerialDevice* sd)
{
  Raw* raw=(Raw*)sd;

  raw->sp->handle=0;
  raw->sp->status=0;
  freeMem(raw);
}

static void rawStatus(SerialPort* sp, int event)
{
  Raw* raw=(Raw*)sp->handle;

  if(raw->sd.status) raw->sd.status(&raw->sd,event);
}

/**********************************************************************
 * External interface
 **********************************************************************/

SerialDevice* createRawDevice(SerialPort* sp)
{
  Raw* raw=allocMem(id_raw,sizeof(Raw));

  raw->sd.close=rawClose;
  raw->sd.setSpeed=rawSetSpeed;
  raw->sd.getSpeedMask=rawGetSpeedMask;
  raw->sd.getMinTurnaroundMask=rawGetMinTurnaroundMask;
  raw->sd.getChar=rawGetChar;
  raw->sd.putChar=rawPutChar;
  raw->sd.handle=0;
  raw->sd.status=0;
  raw->sp=sp;

  sp->handle=raw;
  sp->status=rawStatus;

  return &raw->sd;
}
