#!/bin/sh
#
# update.sh
# BannerFilter, http://phroggy.com/bannerfilter/
#
# This script downloads new updated versions of the four data files
# used by redirector.pl, overwrites the data files in the current
# directory, and makes redirector.pl reload the files.  This script
# requires lynx, wget, curl, or another HTTP downloader to be installed
# and working (see below).
#
# Because new versions of these data files may require BannerFilter to
# be upgraded, this script also checks for the current version of
# BannerFilter and notifies you when an update is available.  You will
# only be notified of required updates; if your old version of
# BannerFilter still works fine with the latest data files, you will
# not be notified if an update is available.
#
# If running as a non-root user, be sure you have write access to the
# current directory.  If you're not running as root or the user Squid is 
# running as, or if you don't have killall, change RELOAD to 0.

# Choose your HTTP downloader:
CMD="lynx -source"
# CMD="wget -q -O -"
# CMD="curl -s"

# Reload with "killall -HUP redirector.pl" if the files have changed?
RELOAD=1
# RELOAD=0

##########################################################################

# Earliest version of BannerFilter compatible with your data files:
OLD="1.2"

# Read the data files from the current directory, and get the modification
# date from each.  Redirect errors to /dev/null.  If the files don't exist,
# we get blank values.
OLDBANNERS=`grep "^# last modified " banners.data 2>/dev/null|cut -d " " -f4`
OLDPOPUPS=`grep "^# last modified " popups.data 2>/dev/null|cut -d " " -f4`
OLDFRAMES=`grep "^# last modified " frames.data 2>/dev/null|cut -d " " -f4`
OLDEXCEPTIONS=`grep "^# last modified " exceptions.data 2>/dev/null|cut -d " " -f4`

# Get the earliest version of BannerFilter compatible with the
# new data files, and see whether we need to upgrade
$CMD http://phroggy.com/bannerfilter/version.txt>version.tmp \
|| { echo "Error retrieving version.txt"; exit; }
if ( grep "^version " version.tmp>/dev/null ); then
  NEW=`grep "^version " version.tmp|cut -d " " -f2`
  rm version.tmp
else
  echo "Error: something unexpected in version.txt"
  exit
fi

if [ "$OLD" != "$NEW" ]; then
  echo "The contents of BannerFilter's data files have changed, and you must"
  echo "upgrade to a new version of BannerFilter to be compatible with the"
  echo "changes.  Please upgrade to at least version $NEW at"
  echo "http://phroggy.com/bannerfilter/, and be sure to read the CHANGES."
  exit
fi

# Download all four data files

$CMD http://phroggy.com/bannerfilter/banners.data>banners.data.tmp \
&& grep "^# banners.data$" banners.data.tmp>/dev/null \
|| { echo "Error retrieving banners.data"; exit; }

$CMD http://phroggy.com/bannerfilter/popups.data>popups.data.tmp \
&& grep "^# popups.data$" popups.data.tmp>/dev/null \
|| { echo "Error retrieving popups.data"; exit; }

$CMD http://phroggy.com/bannerfilter/frames.data>frames.data.tmp \
&& grep "^# frames.data$" frames.data.tmp>/dev/null \
|| { echo "Error retrieving frames.data"; exit; }

$CMD http://phroggy.com/bannerfilter/exceptions.data>exceptions.data.tmp \
&& grep "^# exceptions.data$" exceptions.data.tmp>/dev/null \
|| { echo "Error retrieving exceptions.data"; exit; }

# Get the modification dates of the files we just downloaded, compare
# them to what we already have, and only overwrite if necessary.
# By using cat to overwrite, we preserve ownership and permissions.

NEWBANNERS=`grep "^# last modified " banners.data.tmp|cut -d " " -f4`
NEWPOPUPS=`grep "^# last modified " popups.data.tmp|cut -d " " -f4`
NEWFRAMES=`grep "^# last modified " frames.data.tmp|cut -d " " -f4`
NEWEXCEPTIONS=`grep "^# last modified " exceptions.data.tmp|cut -d " " -f4`
CHANGED=0
if [ "$OLDBANNERS" != "$NEWBANNERS" ]; then
  cat banners.data.tmp > banners.data
  CHANGED=1
fi
if [ "$OLDPOPUPS" != "$NEWPOPUPS" ]; then
  cat popups.data.tmp > popups.data
  CHANGED=1
fi
if [ "$OLDFRAMES" != "$NEWFRAMES" ]; then
  cat frames.data.tmp > frames.data
  CHANGED=1
fi
if [ "$OLDEXCEPTIONS" != "$NEWEXCEPTIONS" ]; then
  cat exceptions.data.tmp > exceptions.data
  CHANGED=1
fi

# Clean up our temp files
rm banners.data.tmp
rm popups.data.tmp
rm frames.data.tmp
rm exceptions.data.tmp

# Send all instances of redirector.pl a signal to reload the data files
if [ $RELOAD == 1 ] && [ $CHANGED == 1 ]; then
  killall -HUP redirector.pl
fi
