#include "contours.h"

#include <GL/gl.h>

#define  SIZE_OF_BLOCK  4096

#define  VERSION  3.0
#define  END_RECORD  0.0
/*
#define  END_SUBPLANE  -1.0
*/

#define  MAX_NDIM  10

#define READ_WORDS(N, X, Msg) \
	{   if (FREAD((X), BYTES_PER_WORD, N, fp)) \
	    { \
		sprintf(error_msg, "reading at word %d: %s", c->word, Msg); \
		return  ERROR; \
	    } \
	    c->word += N;   }

#define READ_WORD(X, Msg)  READ_WORDS(1, &(X), Msg)

/*****************************************************************************
 * INTERNAL FUNCTIONS
 *****************************************************************************/

static Status alloc_contours(Contours **p_c, String error_msg)
{
    int pos, neg;
    Contours *c;

    sprintf(error_msg, "allocating contours memory");
    MALLOC(c, Contours, 1);

    pos = glGenLists(1);
    neg = glGenLists(1);

    if ((pos == 0) || (neg ==0))
	RETURN_ERROR_MSG("could not generate display lists");

    c->positive_contours = pos;
    c->negative_contours = neg;
    c->word = 0;
    c->polylines = 0;

    *p_c = c;

    return  OK;
}

static Status read_header(FILE *fp, Contours *c, String error_msg)
{
    int i, n, r, ndim, dim0, dim1;
    float x, lower0, upper0, lower1, upper1;
    Line msg;

    READ_WORD(x, "version");

    if (x != VERSION)
    {
	sprintf(error_msg, "expected version %2.1f, found %2.1f", VERSION, x);
	return  ERROR;
    }

    READ_WORD(x, "ndim");

    ndim = x;
    CHECK_RANGE("ndim", ndim, 2, MAX_NDIM);
    printf("ndim = %d\n", ndim);

    READ_WORD(x, "dim1");
    dim1 = x;
    dim1 = ndim - dim1;
    CHECK_RANGE("dim1", dim1, 0, ndim-1);
 
    READ_WORD(x, "dim0");
    dim0 = x;
    dim0 = ndim - dim0;
    CHECK_RANGE("dim0", dim0, 0, ndim-1);

    printf("dim0 = %d\n", dim0+1);
    printf("dim1 = %d\n", dim1+1);

    for (i = 0; i < ndim - 2; i++)
    {
	sprintf(msg, "orthogonal dim %d", i);
	READ_WORD(x, msg);
    }

    READ_WORD(upper1, "upper1");
    READ_WORD(lower1, "lower1");
    READ_WORD(upper0, "upper0");
    READ_WORD(lower0, "lower0");

    printf("region0 = (%3.2f, %3.2f)\n", lower0, upper0);
    printf("region1 = (%3.2f, %3.2f)\n", lower1, upper1);

    for (i = 0; i < ndim - 2; i++)
    {
	sprintf(msg, "orthogonal lower %d", i);
	READ_WORD(x, msg);

	sprintf(msg, "orthogonal upper %d", i);
	READ_WORD(x, msg);
    }

    READ_WORD(x, "offset1");
    READ_WORD(x, "offset0");

    n = 2; /* positive and negative separate */

    READ_WORD(x, "chunks1");
    n *= (int) x;

    READ_WORD(x, "chunks0");
    n *= (int) x;

    for (i = 0; i < ndim - 2; i++)
    {
	sprintf(msg, "orthogonal npts %d", i);
	READ_WORD(x, msg);

	n *= (int) x;
    }

    sprintf(error_msg, "allocating directory memory");
    MALLOC(c->directory, int, n);

    READ_WORDS(n, c->directory, "directory");

    c->ndim = ndim;
    c->dir_size = n;
    c->lower[0] = lower0;
    c->upper[0] = upper0;
    c->lower[1] = lower1;
    c->upper[1] = upper1;

    r = n + 4*ndim + 4;
    r = SIZE_OF_BLOCK * BLOCK(r, SIZE_OF_BLOCK);

    c->header_size = r;

    return  OK;
}

static Status read_polyline(FILE *fp, Contours *c, int n, String error_msg)
{
    int i;
    float x, y;

    glBegin(GL_LINE_LOOP);

    for (i = 0; i < n; i++)
    {
	READ_WORD(y, "y");
	READ_WORD(x, "x");

	glVertex2f(-x, -y);
    }

    glEnd();

    c->polylines++;

    return  OK;
}

static Status read_polylines(FILE *fp, Contours *c, int d, int e,
							String error_msg)
{
    int n;
    float x;

    if (FSEEK_ABSOLUTE(fp, d))
    {
	sprintf(error_msg, "seeking directory entry %d", e);
	return  ERROR;
    }

    c->word = d;

    READ_WORD(x, "npoints");
    n = x;

    while (n > 0)
    {
	CHECK_STATUS(read_polyline(fp, c, n, error_msg));

	READ_WORD(x, "npoints");

	if (x == END_RECORD)
	{
	    d = SIZE_OF_BLOCK * BLOCK(c->word, SIZE_OF_BLOCK);

	    if (FSEEK_ABSOLUTE(fp, d))
	    {
	        sprintf(error_msg, "seeking next block for entry %d", e);
		return  ERROR;
	    }

	    c->word = d;

	    READ_WORD(x, "npoints");
	}

	n = x;
    }

    return  OK;
}

static Status read_data(FILE *fp, Contours *c, String error_msg)
{
    int i, d;

/*  negative contours  */

    glNewList(c->negative_contours, GL_COMPILE);

    for (i = 0; i < c->dir_size; i += 2)
    {
	d = c->directory[i] - 1;

	if (d < 0)
	    continue; /* no contours */

	CHECK_STATUS(read_polylines(fp, c, d, i, error_msg));
    }

    glEndList();

/*  positive contours  */

    glNewList(c->positive_contours, GL_COMPILE);

    for (i = 1; i < c->dir_size; i += 2)
    {
	d = c->directory[i] - 1;

	if (d < 0)
	    continue; /* no contours */

	CHECK_STATUS(read_polylines(fp, c, d, i, error_msg));
    }

    glEndList();

    return  OK;
}

static Status read_contours(FILE *fp, Contours *c, String error_msg)
{
    CHECK_STATUS(read_header(fp, c, error_msg));
    CHECK_STATUS(read_data(fp, c, error_msg));

    printf("read %d polylines\n", c->polylines);

    return  OK;
}

/*****************************************************************************
 * EXTERNAL FUNCTIONS
 *****************************************************************************/

Contours *load_contours(String name, String error_msg)
{
    FILE *fp;
    Contours *c;

    if (OPEN_FOR_BINARY_READING(fp, name))
    {
	sprintf(error_msg, "opening '%s' for reading", name);
	return  NULL;
    }

    if (alloc_contours(&c, error_msg) == ERROR)
    {
	fclose(fp);

	return  NULL;
    }

    sprintf(error_msg, "reading '%s': ", name);
    error_msg += strlen(error_msg);

    if (read_contours(fp, c, error_msg) == ERROR)
    {
	fclose(fp);

	return  NULL;
    }

    fclose(fp);

    return  c;
}

void draw_contours(Contours *c)
{
    glMatrixMode(GL_PROJECTION);
    glLoadIdentity();
    glOrtho(-c->upper[0], -c->lower[0], -c->upper[1], -c->lower[1], 0.0, 1.0);

    glMatrixMode(GL_MODELVIEW);
    glPushMatrix();

    glColor3f(0.0, 0.0, 1.0); /* blue */
    glCallList(c->positive_contours);

    glColor3f(1.0, 0.0, 0.0); /* red */
    glCallList(c->negative_contours);

    glPopMatrix();
}
