#include "ref.h"

#define  DEFAULT_SPEC_WIDTH     1000.0 /* Hz */
#define  DEFAULT_SPEC_FREQ      500.0 /* MHz */
#define  DEFAULT_REF_PPM        0.0 /* ppm */
#define  DEFAULT_REF_POINT      1.0 /* point */
#define  DEFAULT_NUCLEUS        "1H"

void free_ref_memory(Ref_info **ref)
{
    FREE(*ref, Ref_info);
}

Status initialize_ref(int ndim, Ref_info **ref, String error_msg)
{
    int i;
    Ref_info *ref_info;

    free_ref_memory(ref);

    sprintf(error_msg, "allocating memory for referencing");
    MALLOC(ref_info, Ref_info, ndim);

    for (i = 0; i < ndim; i++)
    {
	ref_info[i].sw = DEFAULT_SPEC_WIDTH;
	ref_info[i].sf = DEFAULT_SPEC_FREQ;
	ref_info[i].refppm = DEFAULT_REF_PPM;
	ref_info[i].refpt = DEFAULT_REF_POINT;
	strcpy(ref_info[i].nuc, DEFAULT_NUCLEUS);
    }

    *ref = ref_info;

    return  OK;
}

void write_ref(FILE *fp, Ref_info *ref)
{
    fprintf(fp, "sw %4.3f\n", ref->sw);
    fprintf(fp, "sf %4.3f\n", ref->sf);
    fprintf(fp, "refppm %5.4f\n", ref->refppm);
    fprintf(fp, "refpt %5.4f\n", ref->refpt);
    fprintf(fp, "nuc %s\n", ref->nuc);
}

void convert_from_point(int ref_type, int npoints, Ref_info *ref, float *point)
{
    float t;

    if (ref_type == REF_PPM)
    {
	t = - npoints * ref->sf / ref->sw;
	*point = (*point - ref->refpt)/t + ref->refppm;
    }
    else if (ref_type == REF_HZ)
    {
	t = - npoints / ref->sw;
	*point = (*point - ref->refpt)/t + ref->sf*ref->refppm;
    }
}

void convert_from_points(int ref_type, int ndim, int *npoints,
						Ref_info *ref, float *points)
{
    int i;

    if (ref_type != REF_POINTS)
    {
	for (i = 0; i < ndim; i++)
	    convert_from_point(ref_type, npoints[i], ref+i, points+i);
    }
}

void convert_to_point(int ref_type, int npoints, Ref_info *ref, float *point)
{
    float t;

    if (ref_type == REF_PPM)
    {
	t = - npoints * ref->sf / ref->sw;
	*point = t*(*point - ref->refppm) + ref->refpt;
    }
    else if (ref_type == REF_HZ)
    {
	t = - npoints / ref->sw;
	*point = t*(*point - ref->sf * ref->refppm) + ref->refpt;
    }
}

void convert_to_points(int ref_type, int ndim, int *npoints,
						Ref_info *ref, float *points)
{
    int i;

    if (ref_type != REF_POINTS)
    {
	for (i = 0; i < ndim; i++)
	    convert_to_point(ref_type, npoints[i], ref+i, points+i);
    }
}

void check_orientation(int ref_type, int ndim, float *lower, float *upper)
{
    int i;

    if (ref_type != REF_POINTS)
    {
	for (i = 0; i < ndim; i++)
	    SWAP(upper[i], lower[i], float);
    }
}

void convert_range_to_points(int ref_type, int ndim, int *npoints,
				Ref_info *ref, float *lower, float *upper)
{
    if (ref_type != REF_POINTS)
    {
	convert_to_points(ref_type, ndim, npoints, ref, lower);
	convert_to_points(ref_type, ndim, npoints, ref, upper);

	check_orientation(ref_type, ndim, lower, upper);
    }
}

void convert_range_from_points(int ref_type, int ndim, int *npoints,
				Ref_info *ref, float *lower, float *upper)
{
    if (ref_type != REF_POINTS)
    {
	convert_from_points(ref_type, ndim, npoints, ref, lower);
	convert_from_points(ref_type, ndim, npoints, ref, upper);

	check_orientation(ref_type, ndim, lower, upper);
    }
}

void find_sub_ref(Ref_info *ref1, Ref_info *ref2,
				int first, int last, int step, int npoints)
{
    ref1->sw = ref2->sw * (last - first) / ((float) npoints);
    ref1->sf = ref2->sf;
    ref1->refppm = ref2->refppm;
    ref1->refpt = 1 + (ref2->refpt - 1 - first) / ((float) step);
    strcpy(ref1->nuc, ref2->nuc);
}

float fractional_ref_offset(int ref_type, Ref_info *ref, float value)
{
    float f;

    if (ref_type == REF_PPM)
	f = value * ref->sf / ref->sw;
    else if (ref_type == REF_HZ)
	f = value / ref->sw;
    else /* (ref_type == REF_POINTS) */
	f = 0; /* arbitrary, should not be called with this type */

    return f;
}

void offset_to_point(int ref_type, int npoints, Ref_info *ref, float *offset)
{
    if (ref_type == REF_PPM)
	*offset *= - npoints * ref->sf / ref->sw;
    else if (ref_type == REF_HZ)
	*offset *= - npoints / ref->sw;
}

void offset_to_points(int ref_type, int ndim, int *npoints, Ref_info *ref,
							float *offsets)
{
    int i;

    if (ref_type != REF_POINTS)
    {
	for (i = 0; i < ndim; i++)
	    offset_to_point(ref_type, npoints[i], ref+i, offsets+i);
    }
}
