#include "project.h"

#include "block.h"
#include "par.h"
#include "parse.h"
#include "ref.h"

static int ndim;
static int *block_size_in;
static int *npoints_in;

static int ndim_proj;
static int dim_proj[MAX_NDIM];
static int block_size_out[MAX_NDIM];
static int npoints_out[MAX_NDIM];

static int nstore_in;
static int nstore_out;
static float *store_in;
static float *store_out;

static Ref_info *ref_in;
static Ref_info ref_out[MAX_NDIM];

static char *input_file;
static Line output_file;
static FILE *file_in;
static FILE *file_out;
static Bool swapped;
static Bool integer;
static Bool blocked;
static int header;
static Bool deflated;
static float level;
static int dir_size;
static int *directory;

static Bool input_found;
static Bool output_found;
static Bool par_found;
static Bool threshold_found;
static Bool project_found;
static Line output_par_file;
static char *project_file;

static float threshold;

static int parse_float[] = { PARSE_FLOAT };
static int parse_string[] = { PARSE_STRING };
static int parse_int_free[] = { PARSE_INT | PARSE_FREE };

#define  FOUND_TWICE(string) \
	 {   sprintf(error_msg, "in \"%s\" '%s' found twice", \
				project_file, string);  return  ERROR;   }

#define  FOUND_BEFORE(string1, string2) \
	 {   sprintf(error_msg, "in \"%s\" '%s' found before '%s'", \
			project_file, string1, string2);  return  ERROR;   }

#define  NOT_FOUND(string) \
	 {   sprintf(error_msg, "in \"%s\" no '%s' found", \
				project_file, string);  return  ERROR;   }

static Status allocate_memory(String error_msg)
{
    sprintf(error_msg, "allocating memory for store_in");
    MALLOC(store_in, float, nstore_in);

    sprintf(error_msg, "allocating memory for store_out");
    MALLOC(store_out, float, nstore_out);

    if (deflated)
    {
	sprintf(error_msg, "allocating memory for directory");
	MALLOC(directory, int, dir_size);
    }

    return  OK;
}

static void determine_params()
{
    int i, j;

    VECTOR_PRODUCT(nstore_in, block_size_in, ndim);

    for (i = 0; i < ndim_proj; i++)
    {
	j = dim_proj[i];
	npoints_out[i] = npoints_in[j];
	block_size_out[i] = block_size_in[j];

	ref_out[i] = ref_in[j];
    }

    nstore_out = nstore_in;

    find_block_sizes(ndim_proj, nstore_out, npoints_out, block_size_out, TRUE);
}

static Status input_parse(Generic_ptr *var, String error_msg)
{
    String input_par_file = (String) (*var);
    Par_info par_info;

    if (input_found)
	FOUND_TWICE("input");

    CHECK_STATUS(read_par_file(input_par_file, &par_info, error_msg));

    input_file = par_info.file;
    ndim = par_info.ndim;
    npoints_in = par_info.npoints;
    block_size_in = par_info.block_size;
    ref_in = par_info.ref;
    swapped = par_info.swapped;
    integer = par_info.integer;
    blocked = par_info.blocked;
    header = par_info.header;
    deflated = par_info.deflated;
    level = par_info.level;
    dir_size = par_info.dir_size;

    if (!blocked)
    {
	sprintf(error_msg, "input file \"%s\" must be blocked", input_file);
	return  ERROR;
    }

    if (deflated)
    {
/*
	sprintf(error_msg, "input file \"%s\" must not be deflated",
								input_file);
	return  ERROR;
*/
	printf("Warning: input file \"%s\" deflated\n", input_file);
    }

    input_found = TRUE;

    return  OK;
}

static Status output_parse(Generic_ptr *var, String error_msg)
{
    String name = (String) (*var);

    if (!input_found)
	FOUND_BEFORE("output", "input");

    if (output_found)
	FOUND_TWICE("output");

    strcpy(output_file, name);

    output_found = TRUE;

    return  OK;
}

static Status par_parse(Generic_ptr *var, String error_msg)
{
    String name = (String) (*var);

    if (!input_found)
	FOUND_BEFORE("par", "input");

    if (par_found)
	FOUND_TWICE("par");

    strcpy(output_par_file, name);

    par_found = TRUE;

    return  OK;
}

static Status threshold_parse(Generic_ptr *var, String error_msg)
{
    if (!input_found)
	FOUND_BEFORE("par", "input");

    if (threshold_found)
	FOUND_TWICE("threshold");

    threshold = *((float *) var[0]);
    threshold_found = TRUE;

    return  OK;
}

static Status project_parse(Generic_ptr *var, String error_msg)
{
    int n;

    if (!output_found)
	FOUND_BEFORE("project", "output");

    if (project_found)
	FOUND_TWICE("project");

    ndim_proj = *((int *) var[0]);

    if (ndim_proj > ndim)
    {
	sprintf(error_msg,
	    "in \"%s\" have 'project' with %d arguments, but 'ndim' = %d",
					project_file, ndim_proj, ndim);
	return  ERROR;
    }

    if (ndim_proj < 1)
    {
	sprintf(error_msg,
		"in \"%s\" have 'project' with no arguments", project_file);
	return  ERROR;
    }

    for (n = 0; n < ndim_proj; n++)
    {
	dim_proj[n] = *((int *) var[n+1]);

	if (dim_proj[n] < 1)
	{
	    sprintf(error_msg, "in \"%s\" projected dim #%d = %d",
                                        project_file, n+1, dim_proj[n]);
	    return  ERROR;
	}

        if (dim_proj[n] > ndim)
	{
            sprintf(error_msg,
                        "in \"%s\" projected dim #%d = %d, but 'ndim' = %d",
                                        project_file, n+1, dim_proj[n], ndim);
	    return  ERROR;
        }

	dim_proj[n]--;
    }

    project_found = TRUE;

    return  OK;
}

static Parse_line project_table[] =
{
    { "input",		1,	parse_string,		input_parse },
    { "output",		1,	parse_string,		output_parse },
    { "par",		1,	parse_string,		par_parse },
    { "threshold",	1,	parse_float,		threshold_parse },
    { "project",	1,	parse_int_free,		project_parse },
    { (String) NULL,	0,	(int *) NULL,		no_parse_func }
};

static Status read_project_file(String error_msg)
{
    int i, j;

    input_found = FALSE;
    output_found = FALSE;
    par_found = FALSE;
    threshold_found = FALSE;
    threshold = 0;
    project_found = FALSE;
    ndim_proj = 0;

    CHECK_STATUS(parse_file(project_file, project_table, TRUE, error_msg));

    if (!input_found)
	NOT_FOUND("input");

    if (!output_found)
	NOT_FOUND("output");

    if (ndim_proj < 1)
	NOT_FOUND("project");

    for (i = 0; i < ndim_proj; i++)
    {
	for (j = i+1; j < ndim_proj; j++)
	{
	    if (dim_proj[j] == dim_proj[i])
	    {
		sprintf(error_msg, "in \"%s\" projected dim #%d = dim#%d = %d",
                                project_file, i+1, j+1, dim_proj[i]+1);
		return  ERROR;
	    }
	}
    }

    return  OK;
}

static void print_project_info()
{
    printf("Number of dimensions of input data = %d\n", ndim);
    printf("Number of dimensions of output data = %d\n", ndim_proj);

    printf("Output dimension%s = ", ndim_proj == 1 ? "" : "s");
    print_integer_array(ndim_proj, dim_proj);
    printf("\n");
}

void main(int argc, char **argv)
{
    Line error_msg;
    Size_info size_in, size_out;
    Store_info store_info;
    File_info file_info;
    Project_info project_info;
    Par_info par_info;
    String par_file;

    printf(product);

    if (help_request(argc, argv, help_table))
	exit (0);

    if (argc != 2)
    {
        sprintf(error_msg, "correct usage: %s <project file>", argv[0]);
        ERROR_AND_EXIT(error_msg);
    }

    project_file = argv[1];

    if (read_project_file(error_msg) == ERROR)
        ERROR_AND_EXIT(error_msg);

    determine_params();

    if (allocate_memory(error_msg) == ERROR)
        ERROR_AND_EXIT(error_msg);

    if (OPEN_FOR_READING(file_in, input_file))
    {
	sprintf(error_msg, "opening \"%s\" for reading", input_file);
	ERROR_AND_EXIT(error_msg);
    }

    if (OPEN_FOR_WRITING(file_out, output_file))
    {
	sprintf(error_msg, "opening \"%s\" for writing", output_file);
	ERROR_AND_EXIT(error_msg);
    }

    size_in.ndim = ndim;
    size_in.block_size = block_size_in;
    size_in.npoints = npoints_in;

    size_out.ndim = ndim_proj;
    size_out.block_size = block_size_out;
    size_out.npoints = npoints_out;

    store_info.store_in = store_in;
    store_info.store_out = store_out;
    store_info.directory = directory;

    file_info.input_file = input_file;
    file_info.output_file = output_file;
    file_info.file_in = file_in;
    file_info.file_out = file_out;
    file_info.swapped = swapped;
    file_info.integer = integer;
    file_info.blocked = blocked;
    file_info.header = header;
    file_info.deflated = deflated;
    file_info.dir_size = dir_size;

    project_info.dim_proj = dim_proj;
    project_info.threshold = threshold;

    print_project_info();
    FLUSH;

    if (block_process(&size_in, &size_out, &store_info,
		&file_info, &project_info, error_msg) == ERROR)
	ERROR_AND_EXIT(error_msg);

    par_info.file = output_file;
    par_info.ndim = ndim_proj;
    par_info.npoints = npoints_out;
    par_info.block_size = block_size_out;
    par_info.ref = ref_out;
    par_info.blocked = TRUE;
    par_info.deflated = FALSE;
    par_info.level = level;
    par_info.param_dim = -1;

    if (par_found)
        par_file = output_par_file;
    else
        par_file = NULL;

    if (write_par_file(par_file, &par_info, error_msg) == ERROR)
	ERROR_AND_EXIT(error_msg);
}
