#include "lp_last.h"

#include "command.h"
#include "lp.h"

static int ncodes = 0;
static int length[MAX_NCODES];
static int number[MAX_NCODES];
static int points[MAX_NCODES];
static float cutoff[MAX_NCODES];
static int step[MAX_NCODES];
static int npoints[MAX_NCODES];
static float **u[MAX_NCODES];
static float **v[MAX_NCODES];
static float *w[MAX_NCODES];
static float *x[MAX_NCODES];
static float *d[MAX_NCODES];
static float *t[MAX_NCODES];

static void do_lp_last(int code, float *data)
{
    lp_last(data, length[code], number[code], points[code],
		step[code], npoints[code], cutoff[code], u[code],
		v[code], w[code], x[code], d[code], t[code]);
}

static void do_lp_last2(int code, float *data)
{
    if (step[code] == 1)
        lp_last(data, length[code], number[code], points[code],
		step[code], npoints[code], cutoff[code], u[code],
		v[code], w[code], x[code], d[code], t[code]);
    else
        lp_last2(data, length[code]/2, number[code]/2, points[code],
		npoints[code], cutoff[code], u[code], v[code],
		w[code], x[code], d[code], t[code]);
}

static Status check_allocation(int code)
{
    int i;

    MALLOC(u[code], float *, number[code]);
    MALLOC(v[code], float *, length[code]);
    MALLOC(w[code], float, length[code]);
    MALLOC(x[code], float, length[code]);
    MALLOC(d[code], float, number[code]);
    MALLOC(t[code], float, length[code]);

    for (i = 0; i < number[code]; i++)
	MALLOC(u[code][i], float, length[code]);

    for (i = 0; i < length[code]; i++)
	MALLOC(v[code][i], float, length[code]);

    return  OK;
}

Status init_lp_last(Generic_ptr *param, String error_msg)
{
    int type, npts, l, m, n, p;
    float c;
    Line msg;

    p = *((int *) param[0]);
	/* number of points to predict in (complex or real) points */

    l = *((int *) param[1]);
	/* length of sequence in (complex or real) points */

    n = *((int *) param[2]);
	/* number of sequences in (complex or real) points */

    c = *((float *) param[3]);
	/* cutoff for throwing out values from svd */

    if (p < 1)
        RETURN_ERROR_MSG("'lp_last': number of predicted points must be >= 1");

    if (l < 1)
        RETURN_ERROR_MSG("'lp_last': length of sequence must be >= 1");

    if (c <= 0)
        RETURN_ERROR_MSG("'lp_last': cutoff must be > 0");

    if (c >= 1)
        RETURN_ERROR_MSG("'lp_last': cutoff must be < 1");

    sprintf(msg, "lp_last %d %d %d %3.2e", p, l, n, c);
    if (setup_command(&type, &npts, ncodes, msg,
					do_lp_last, error_msg) == ERROR)
        return  ERROR;

    if (type == COMPLEX_DATA)
	m = npts/2;
    else
	m = npts;

    if (n < l)
        RETURN_ERROR_MSG("'lp_last': number of sequences must be >= length");

    if (l > (m/2))
        RETURN_ERROR_MSG("'lp_last': length of sequence too large");

    if (n > (m - l))
        RETURN_ERROR_MSG("'lp_last': number of sequences too large");

    length[ncodes] = l;
    number[ncodes] = n;
    points[ncodes] = p;
    cutoff[ncodes] = c;
    npoints[ncodes] = m;

    if (type == COMPLEX_DATA)
	step[ncodes] = 2;
    else
	step[ncodes] = 1;

    if (check_allocation(ncodes) == ERROR)
	RETURN_ERROR_MSG("'lp_last': allocating memory");

    npts += step[ncodes] * p;
    CHECK_STATUS(end_command(type, npts, "lp_last", error_msg));

    ncodes++;

    return  OK;
}

Status init_lp_last2(Generic_ptr *param, String error_msg)
{
    int type, npts, l, m, n, p;
    float c;
    Line msg;

    p = *((int *) param[0]);
	/* number of points to predict in (complex or real) points */

    l = *((int *) param[1]);
	/* length of sequence in (complex or real) points */

    n = *((int *) param[2]);
	/* number of sequences in (complex or real) points */

    c = *((float *) param[3]);
	/* cutoff for throwing out values from svd */

    if (p < 1)
        RETURN_ERROR_MSG("'lp_last2': number of predicted points must be >= 1");

    if (l < 1)
        RETURN_ERROR_MSG("'lp_last2': length of sequence must be >= 1");

    if (c <= 0)
        RETURN_ERROR_MSG("'lp_last2': cutoff must be > 0");

    if (c >= 1)
        RETURN_ERROR_MSG("'lp_last2': cutoff must be < 1");

    sprintf(msg, "lp_last2 %d %d %d %3.2e", p, l, n, c);
    if (setup_command(&type, &npts, ncodes, msg,
					do_lp_last2, error_msg) == ERROR)
        return  ERROR;

    if (type == COMPLEX_DATA)
	m = npts/2;
    else
	m = npts;

    if (n < l)
        RETURN_ERROR_MSG("'lp_last2': number of sequences must be >= length");

    if (l > (m/2))
        RETURN_ERROR_MSG("'lp_last2': length of sequence too large");

    if (n > (m - l))
        RETURN_ERROR_MSG("'lp_last2': number of sequences too large");

    if (type == COMPLEX_DATA)
	step[ncodes] = 2;
    else
	step[ncodes] = 1;

    length[ncodes] = step[ncodes] * l;
    number[ncodes] = step[ncodes] * n;
    points[ncodes] = p;
    cutoff[ncodes] = c;
    npoints[ncodes] = m;

    if (check_allocation(ncodes) == ERROR)
	RETURN_ERROR_MSG("'lp_last2': allocating memory");

    npts += step[ncodes] * p;
    CHECK_STATUS(end_command(type, npts, "lp_last2", error_msg));

    ncodes++;

    return  OK;
}
