#include "lp_fit.h"

#include "command.h"
#include "lp.h"

#define  MAX_POLES  20

static int ncodes = 0;
static int npoints[MAX_NCODES];
static int npred[MAX_NCODES];
static int step[MAX_NCODES];
static int npoles[MAX_NCODES];
static float *w1[MAX_NCODES];
static float *w2[MAX_NCODES];
static float *w3[MAX_NCODES];
static float *w4[MAX_NCODES];

static void do_lp_fit(int code, float *data)
{
    lp_fit(data, npoints[code], npred[code], step[code], npoles[code],
				w1[code], w2[code], w3[code], w4[code]);
}

static Status check_allocation(int code)
{
    int n;

    n = npoles[code];
    MALLOC(w1[code], float, n);

    n = MAX(npoints[code]-1, 2*(npoles[code]+1));
    MALLOC(w2[code], float, n);

    n = MAX(npoints[code]-1, 2*npoles[code]);
    MALLOC(w3[code], float, n);

    n = 2*(npoles[code]+1);
    MALLOC(w4[code], float, n);

    return  OK;
}

static Status init_lp_fit(Generic_ptr *param, int direction,
					String cmd, String error_msg)
{
    int type, npts, pred, poles;
    Line msg;

    pred = *((int *) param[0]);
	/* number of points to predict in (complex or real) points */

    poles = *((int *) param[1]);
	/* number of poles */

    if (pred < 1)
    {
        sprintf(error_msg,
		"'%s': number of predicted points (%d) must be >= 1",
		cmd, pred);
	return  ERROR;
    }

    if ((poles < 1) || (poles > MAX_POLES))
    {
        sprintf(error_msg,
		"'%s': number of poles (%d) must be >= 1 and <= %d",
		cmd, pred, MAX_POLES);
	return  ERROR;
    }

    sprintf(msg, "%s %d %d", cmd, pred, poles);
    if (setup_command(&type, &npts, ncodes, msg,
					do_lp_fit, error_msg) == ERROR)
        return  ERROR;

/* for forward prediction end up with extra points */
/* for backward prediction overwrite existing points at beginning */
    if (type == COMPLEX_DATA)
    {
        if (direction == 1)
	{
	    step[ncodes] = 2;
	    npoints[ncodes] = npts/2;
            npts += 2*pred;
	}
        else /* (direction == -1) */
	{
	    step[ncodes] = -2;
	    npoints[ncodes] = npts/2 - pred;

	}
    }
    else /* (type == REAL_DATA) */
    {
        if (direction == 1)
	{
	    step[ncodes] = 1;
	    npoints[ncodes] = npts;
            npts += pred;
	}
        else /* (direction == -1) */
	{
	    step[ncodes] = -1;
	    npoints[ncodes] = npts - pred;

	}
    }

    if (npoints[ncodes] < poles) /* get serious! */
    {
        sprintf(error_msg,
		"'%s': number of usable points (%d) < number poles (%d)",
		cmd, npoints[ncodes], poles);
	return  ERROR;
    }

    npred[ncodes] = pred;
    npoles[ncodes] = poles;

    if (check_allocation(ncodes) == ERROR)
    {
        sprintf(error_msg, "'%s': allocating memory", cmd);
	return  ERROR;
    }

    CHECK_STATUS(end_command(type, npts, cmd, error_msg));

    ncodes++;

    return  OK;
}

Status init_lp_forward(Generic_ptr *param, String error_msg)
{
    return  init_lp_fit(param, 1, "lp_forward", error_msg);
}

Status init_lp_backward(Generic_ptr *param, String error_msg)
{
    return  init_lp_fit(param, -1, "lp_backward", error_msg);
}
