#include "complex.h"

#include "command.h"

static int ncodes = 0;
static int data_type[MAX_NCODES];
static int npoints[MAX_NCODES];

static void do_complexify(int code, float *data)
{
    int i, n = npoints[code];

    for (i = n-2; i >= 0; i -= 2)
    {
	data[i] = data[i/2];
	data[i+1] = 0;
    }
}

static void do_conjugate(int code, float *data)
{
    int i, n = npoints[code];

    for (i = 1; i < n; i += 2)
	data[i] = - data[i];
}

static void do_exchange(int code, float *data)
{
    int i, n = npoints[code];

    for (i = 0; i < n; i += 2)
	SWAP(data[i], data[i+1], float);
}

static void do_reduce(int code, float *data)
{
    int i, n = npoints[code];

    for (i = 0; i < n; i++)
	data[i] = data[2*i];
}

static void do_magnitude(int code, float *data)
{
    int i, j, k, n = npoints[code], type = data_type[code];
    double d;

    if (type == REAL_DATA)
    {
	for (i = 0; i < n; i++)
	    data[i] = ABS(data[i]);
    }
    else /* COMPLEX_DATA */
    {
	for (i = 0; i < n; i++)
	{
	    j = 2*i;  k = j + 1;
	    d = data[j]*data[j] + data[k]*data[k];
	    data[i] = sqrt(d);
	}
    }
}

static void do_magnitude2(int code, float *data)
{
    int i, j, k, n = npoints[code], type = data_type[code];

    if (type == REAL_DATA)
    {
	for (i = 0; i < n; i++)
	    data[i] = data[i] * data[i];
    }
    else /* COMPLEX_DATA */
    {
	for (i = 0; i < n; i++)
	{
	    j = 2*i;  k = j + 1;
	    data[i] = data[j]*data[j] + data[k]*data[k];
	}
    }
}

Status init_complex(Generic_ptr *param, String error_msg)
{
    int type, npts, n = 0;

    if (setup_command(&type, &npts, n, "complex", do_nothing, error_msg)
                                                                == ERROR)
        return  ERROR;

    type = COMPLEX_DATA;

    CHECK_STATUS(end_command(type, npts, "complex", error_msg));

    return  OK;
}

Status init_complexify(Generic_ptr *param, String error_msg)
{
    int type, npts;

    if (setup_command(&type, &npts, ncodes, "complexify",
					do_complexify, error_msg) == ERROR)
        return  ERROR;

    if (type != REAL_DATA)
	RETURN_ERROR_MSG("'complexify': must have real data");

    type = COMPLEX_DATA;
    npts *= 2;

    npoints[ncodes] = npts;

    CHECK_STATUS(end_command(type, npts, "complexify", error_msg));

    ncodes++;

    return  OK;
}

Status init_conjugate(Generic_ptr *param, String error_msg)
{
    int type, npts;

    if (setup_command(&type, &npts, ncodes, "conjugate",
					do_conjugate, error_msg) == ERROR)
        return  ERROR;

    if (type != COMPLEX_DATA)
	RETURN_ERROR_MSG("'conjugate': must have complex data");

    npoints[ncodes] = npts;

    CHECK_STATUS(end_command(type, npts, "conjugate", error_msg));

    ncodes++;

    return  OK;
}

Status init_exchange(Generic_ptr *param, String error_msg)
{
    int type, npts;

    if (setup_command(&type, &npts, ncodes, "exchange",
					do_exchange, error_msg) == ERROR)
        return  ERROR;

    if (type != COMPLEX_DATA)
	RETURN_ERROR_MSG("'exchange': must have complex data");

    npoints[ncodes] = npts;

    CHECK_STATUS(end_command(type, npts, "exchange", error_msg));

    ncodes++;

    return  OK;
}

Status init_real(Generic_ptr *param, String error_msg)
{
    int type, npts, n = 0;

    if (setup_command(&type, &npts, n, "real", do_nothing, error_msg)
                                                                == ERROR)
        return  ERROR;

    type = REAL_DATA;

    CHECK_STATUS(end_command(type, npts, "real", error_msg));

    return  OK;
}

Status init_reduce(Generic_ptr *param, String error_msg)
{
    int type, npts;

    if (setup_command(&type, &npts, ncodes, "reduce",
					do_reduce, error_msg) == ERROR)
        return  ERROR;

    if (type != COMPLEX_DATA)
	RETURN_ERROR_MSG("'reduce': must have complex data");

    type = REAL_DATA;
    npts /= 2;

    npoints[ncodes] = npts;

    CHECK_STATUS(end_command(type, npts, "reduce", error_msg));

    ncodes++;

    return  OK;
}

Status init_magnitude(Generic_ptr *param, String error_msg)
{
    int type, npts;

    if (setup_command(&type, &npts, ncodes, "magnitude",
					do_magnitude, error_msg) == ERROR)
        return  ERROR;

    if (type == COMPLEX_DATA)
	npts /= 2;

    npoints[ncodes] = npts;
    data_type[ncodes] = type;

    type = REAL_DATA;
    CHECK_STATUS(end_command(type, npts, "magnitude", error_msg));

    ncodes++;

    return  OK;
}

Status init_magnitude2(Generic_ptr *param, String error_msg)
{
    int type, npts;

    if (setup_command(&type, &npts, ncodes, "magnitude2",
					do_magnitude2, error_msg) == ERROR)
        return  ERROR;

    if (type == COMPLEX_DATA)
	npts /= 2;

    npoints[ncodes] = npts;
    data_type[ncodes] = type;

    type = REAL_DATA;
    CHECK_STATUS(end_command(type, npts, "magnitude2", error_msg));

    ncodes++;

    return  OK;
}
