#include "slice.h"

#include "color.h"
#include "ref.h"
#include "ticks.h"

#define  SMALL_NUMBER  1.0e-4
#define  LARGE_NUMBER  1.0e20

static float lower[DISPLAY_DIM];
static float upper[DISPLAY_DIM];

static int npoints;
static float *data = NULL;
static int ref_type;
static Ref_info *ref = NULL;

static void initialize_range(float *l, float *u, int ndim)
{
    int i;

    for (i = 0; i < ndim; i++)
    {
	lower[i] = l[i];
	upper[i] = u[i];
    }
}

float calculate_slice_point(int x, int width, Slice_info *info)
{
    float p;

    initialize_range(info->lower, info->upper, 1);
    convert_range_to_points(info->ref_type, 1, &(info->npoints), info->ref,
								lower, upper);

    if (width > 0)
	p = lower[0] + (x * (upper[0] - lower[0])) / (width - 1);
    else
	p = 0;

    return  p;
}

void do_slice(Draw_funcs *funcs, Slice_info *info)
{
    int i, begin, end;
    float a0, b0, a1, b1;

    npoints = info->npoints;
    data = info->data;
    ref_type = info->ref_type;
    ref = info->ref;

    (*(funcs->start_draw))(funcs->data);
    (*(funcs->set_draw_color))(BLACK);

    initialize_range(info->lower, info->upper, DISPLAY_DIM);
    check_orientation(ref_type, 1, lower, upper);

    (*(funcs->new_draw_range))(lower[0], lower[1], upper[0], upper[1], FALSE);

    draw_minor_ticks(info->minor_ticks, info->ruler, lower, upper, funcs);
    draw_major_ticks(info->major_ticks, info->ruler, lower, upper, funcs);

    (*(funcs->new_draw_range))(lower[0], lower[1], upper[0], upper[1], TRUE);

    if (info->axis[0])
	(*(funcs->draw_line))(lower[0], 0.0, upper[0], 0.0);

    if (info->axis[1])
	(*(funcs->draw_line))(0.0, lower[1], 0.0, upper[1]);

    initialize_range(info->lower, info->upper, DISPLAY_DIM);

    convert_range_to_points(ref_type, 1, &npoints, ref, lower, upper);

    (*(funcs->new_draw_range))(lower[0], lower[1], upper[0], upper[1], TRUE);

    begin = lower[0] - 1;
		/* floor, taking into account that points start at 1 */
    begin = MAX(0, begin);

    end = upper[0] - SMALL_NUMBER;
		/* ceiling, taking into account that points start at 1 */
    end = MIN(npoints-1, end);

    for (i = begin; i < end; i++)
    {
	a0 = i+1;  b0 = data[i];
	a1 = i+2;  b1 = data[i+1];
	(*(funcs->draw_line))(a0, b0, a1, b1);
    }

    (*(funcs->end_draw))();
}

void print_slice_point_stats(float *point, Print_funcs *print_funcs)
{
    int p;
    float pnt;
    Line error_msg, message;

    if (!data)
	return;

    if ((*(print_funcs->start_print))(error_msg) == ERROR)
	ERROR_AND_RETURN(error_msg);

    sprintf(message, "Chosen point is (%5.2f %s)\n", point[0],
		ref_type == REF_POINTS  ?  "\b"  :  ref_names[ref_type]);

    (*(print_funcs->print_message))(message);

    if (ref_type != REF_POINTS)
    {
	convert_to_points(ref_type, 1, &npoints, ref, point);

	sprintf(message, "Chosen point is (%5.2f)\n", point[0]);
    	(*(print_funcs->print_message))(message);
    }

    sprintf(message, "\nClosest data point(s) and value(s):\n");
    (*(print_funcs->print_message))(message);

    pnt = point[0] + SMALL_NUMBER - 1;
    if (pnt < 0)
    {
	p = 0;
	sprintf(message, "\tPoint %d:\t%5.2f\n", p+1, data[p]);
	(*(print_funcs->print_message))(message);
    }
    else if (pnt >= npoints)
    {
	p = npoints - 1;
	sprintf(message, "\tPoint %d:\t%5.2f\n", p+1, data[p]);
	(*(print_funcs->print_message))(message);
    }
    else
    {
	p = FLOOR(pnt);
	sprintf(message, "\tPoint %d:\t%5.2f\n", p+1, data[p]);
	(*(print_funcs->print_message))(message);

	p++;
	sprintf(message, "\tPoint %d:\t%5.2f\n", p+1, data[p]);
	(*(print_funcs->print_message))(message);
    }

    sprintf(message, "\nCrosshair value = %5.2f\n", point[1]);
    (*(print_funcs->print_message))(message);

    (*(print_funcs->end_print))();
}

void print_slice_limits_stats(float *limits, Print_funcs *print_funcs)
{
    int i, n, begin, end;
    float d, data_min, data_max, data_sum, data_sqr;
    Line error_msg, message;

    if (!data)
	return;

    if ((*(print_funcs->start_print))(error_msg) == ERROR)
	ERROR_AND_RETURN(error_msg);

    sprintf(message, "Chosen limits are (%5.2f %5.2f %s)\n",
		limits[0], limits[1],
		ref_type == REF_POINTS  ?  "\b"  :  ref_names[ref_type]);

    (*(print_funcs->print_message))(message);

    if (ref_type != REF_POINTS)
    {
	convert_to_points(ref_type, 1, &npoints, ref, limits);
	convert_to_points(ref_type, 1, &npoints, ref, limits+1);

	sprintf(message, "Chosen limits are (%5.2f %5.2f)\n",
						limits[0], limits[1]);
	(*(print_funcs->print_message))(message);
    }

    begin = limits[0] - SMALL_NUMBER;
		/* points start at 1, and want ceiling of lower */
    begin = MAX(0, begin);

    end = limits[1];
		/* points start at 1, and want floor of upper */
    end = MIN(npoints, end);

    if (end > begin)
    {
	data_min = LARGE_NUMBER;
	data_max = - LARGE_NUMBER;
	data_sum = 0;
	data_sqr = 0;

	for (i = begin; i < end; i++)
	{
	    d = data[i];
	    data_min = MIN(d, data_min);
	    data_max = MAX(d, data_max);
	    data_sum += d;
	    data_sqr += d*d;
	}

	n = end - begin;

	sprintf(message, "\tNumber points used:\t%d\n", n);
    	(*(print_funcs->print_message))(message);

	sprintf(message, "\tMinimum data value:\t%5.2f\n", data_min);
    	(*(print_funcs->print_message))(message);

	sprintf(message, "\tMaximum data value:\t%5.2f\n", data_max);
    	(*(print_funcs->print_message))(message);

	sprintf(message, "\tSum of data values:\t%5.2f\n", data_sum);
    	(*(print_funcs->print_message))(message);

	data_sum /= (float) n;
	sprintf(message, "\tAverage of data values:\t%5.2f\n", data_sum);
    	(*(print_funcs->print_message))(message);

	data_sqr /= (float) n;
	data_sqr -= data_sum*data_sum;
	data_sqr = sqrt(data_sqr);

	if (n > 1)
	    data_sqr *= ((float) n) / ((float) (n-1));

	sprintf(message, "\tStandard deviation of data values:\t%5.2f\n",
								data_sqr);
    	(*(print_funcs->print_message))(message);
    }

    (*(print_funcs->end_print))();
}
