#include "peak.h"

#include "utility.h"

static int npeaks = 0;
static int npeaks_alloc = 0;
static int npeaks_incr = 20;
static Peak **peaks;

static int npeak_sets = 0;
static int npeak_sets_alloc = 0;
static int npeak_sets_incr = 20;
static Peak_set **peak_sets;

static Status alloc_peaks_memory(String error_msg)
{
    int npeaks_new;

    if (npeaks < npeaks_alloc)
	return  OK;

    sprintf(error_msg, "allocating peaks memory");

    npeaks_new = npeaks_alloc + npeaks_incr;

    if (npeaks_alloc == 0)
    {
	MALLOC(peaks, Peak *, npeaks_new);
    }
    else
    {
	REALLOC(peaks, Peak *, npeaks_new);
    }

    npeaks_alloc = npeaks_new;

    return  OK;
}

static Status alloc_peak_sets_memory(String error_msg)
{
    int npeak_sets_new;

    if (npeak_sets < npeak_sets_alloc)
	return  OK;

    sprintf(error_msg, "allocating peak sets memory");

    npeak_sets_new = npeak_sets_alloc + npeak_sets_incr;

    if (npeak_sets_alloc == 0)
    {
	MALLOC(peak_sets, Peak_set *, npeak_sets_new);
    }
    else
    {
	REALLOC(peak_sets, Peak_set *, npeak_sets_new);
    }

    npeak_sets_alloc = npeak_sets_new;

    return  OK;
}

static Status alloc_peak_memory(Peak **p_peak, String name, String error_msg)
{
    Peak *peak;

    sprintf(error_msg, "allocating peak memory");
    MALLOC(peak, Peak, 1);

    STRING_MALLOC_COPY(peak->name, name);

    *p_peak = peak;

    return  OK;
}

static Status alloc_peak_set_memory(Peak_set **p_peak_set, String set,
							String error_msg)
{
    Peak_set *peak_set;

    sprintf(error_msg, "allocating peak set memory");
    MALLOC(peak_set, Peak_set, 1);
    peak_set->nalloc = 0;
    peak_set->npeaks = 0;

    STRING_MALLOC_COPY(peak_set->set, set);

    *p_peak_set = peak_set;

    return  OK;
}

static Status alloc_peak_set_peaks_memory(Peak_set *peak_set, String error_msg)
{
    int n, nalloc = peak_set->nalloc, npeaks = peak_set->npeaks;

    if (npeaks < nalloc)
	return  OK;

    sprintf(error_msg, "allocating peak set peaks memory");

    n = nalloc + npeaks_incr;

    if (nalloc == 0)
    {
	MALLOC(peak_set->peaks, Peak *, n);
    }
    else
    {
	REALLOC(peak_set->peaks, Peak *, n);
    }

    peak_set->nalloc = n;

    return  OK;
}

static void free_peak_memory(Peak *peak)
{
    FREE(peak->name, char);

    FREE(peak, Peak);
}

static void free_peak_set_memory(Peak_set *peak_set)
{
    FREE(peak_set->set, char);

    if (peak_set->nalloc > 0)
	FREE(peak_set->peaks, Peak *);

    FREE(peak_set, Peak_set);
}

Status add_peak(String set, String name, int x, float x_ppm,
			int y, float y_ppm, float extremum,
			Add_peak_func add_func, String error_msg)
{
/*
    int data_set;
*/
    int n;
    Peak *peak;
    Peak_set *peak_set = NULL;

/*  do this at fitting stage only
    if (!data_name_exists(name, &data_set))
    {
	sprintf(error_msg, "unknown data set %s", name);
	return  ERROR;
    }

    data_info = get_data_set(data_set);

    if (!data_info->have_param)
    {
	sprintf(error_msg, "data set %s has no param", name);
	return  ERROR;
    }
*/

    CHECK_STATUS(alloc_peaks_memory(error_msg));

    n = find_peak_set(set);

    if (n < 0)
    {
	CHECK_STATUS(alloc_peak_sets_memory(error_msg))

	CHECK_STATUS(alloc_peak_set_memory(&peak_set, set, error_msg));

	peak_sets[npeak_sets++] = peak_set;
    }
    else
    {
	peak_set = peak_sets[n];
    }

    CHECK_STATUS(alloc_peak_memory(&peak, name, error_msg));

    CHECK_STATUS(alloc_peak_set_peaks_memory(peak_set, error_msg));

    peak->x = x;
    peak->x_ppm = x_ppm;
    peak->y = y;
    peak->y_ppm = y_ppm;
    peak->extremum = extremum;
    peak->have_fit = FALSE;
    peak->peak_set = peak_set;

    n = peak_set->npeaks;
    peak_set->method = -1;
    peak_set->peaks[n++] = peak;
    peak_set->npeaks = n;

    if (add_func)
    {
	if ((*add_func)(peak, error_msg) == ERROR)
	{
	    free_peak_memory(peak);
	    peak_set->npeaks--;

	    return  ERROR;
	}
    }

    peaks[npeaks++] = peak;

    return  OK;
}

static void remove_peak_set_peak(Peak_set *peak_set, Peak *peak)
{
    int i, j, n = peak_set->npeaks;

    if (peak->peak_set != peak_set)
	return;

    if (n == 1)
    {
	for (i = 0; i < npeak_sets; i++)
	{
	    if (peak_sets[i] == peak_set)
		break;
	}

	for (j = i; j < npeak_sets-1; j++)
	    peak_sets[j] = peak_sets[j+1];

	npeak_sets--;

	free_peak_set_memory(peak_set);

	return;
    }

    for (i = 0; i < n; i++)
    {
	if (peak_set->peaks[i] == peak)
	    break;
    }

    for (; i < n-1; i++)
	peak_set->peaks[i] = peak_set->peaks[i+1];

    peak_set->npeaks = n - 1;
}

void delete_peak(int peak)
{
    int i;
    Peak *p = peaks[peak];

    remove_peak_set_peak(p->peak_set, p);

    free_peak_memory(p);

    for (i = peak; i < npeaks-1; i++)
	peaks[i] = peaks[i+1];

    npeaks--;
}

Bool peak_name_exists(String name, int *peak)
{
    int i;
    Line value;

    STRIP_LEADING_SPACE(name);
    STRIP_TRAILING_SPACE(name);

    for (i = 0; i < npeaks; i++)
    {
	sprintf(value, "%s: %s", peaks[i]->peak_set->set, peaks[i]->name);

	if (equal_strings(name, value))
	{
	    *peak = i;
	    return  TRUE;
	}
    }

    return  FALSE;
}

int find_peak_set(String set)
{
    int i;

    for (i = 0; i < npeak_sets; i++)
    {
	if (equal_strings(peak_sets[i]->set, set))
	    return i;
    }

    return -1;
}

Peak_set **get_peak_sets(int *p_npeak_sets)
{
    *p_npeak_sets = npeak_sets;

    return  peak_sets;
}

Peak **get_peaks(int *p_npeaks)
{
    *p_npeaks = npeaks;

    return  peaks;
}

