#include "ft.h"

/* This file contains various Fourier transform routines */
/* and the corresponding inverse Fourier transforms. */
/* Note that the inverse transform is not normalised. */
/* If the routine is *ft then the inverse is i*ft. */
/* This file also contains a Hilbert Fourier transform routine. */

/* Current list of routines (inverses not listed): */

/* fft:		complex Fourier transform */
/* rft:		real Fourier transform */
/* sft:		sine Fourier transform */
/* cft:		cosine Fourier transform */
/* hft:		Hilbert Fourier transform */

/* Normalised routines (and inverses) added 24 October 1995 */

/* fftn:	normalised complex Fourier transform */
/* rftn:	normalised real Fourier transform */
/* sftn:	normalised sine Fourier transform */
/* cftn:	normalised cosine Fourier transform */

static void normalise_data(float *data, int n, int m)
{
    float t = 1.0 / sqrt((double) (n / m));

    SCALE_VECTOR(data, data, t, n);
}

/* Note: fft and ifft are not normalised, fftn and ifftn are. */
/* fft followed by ifft leads to scaling the data by n/2 */

/* fft takes the complex Fourier transform of the array data, */
/* which is of size n real (n/2 complex) points.  The data is */
/* output so that the negative parts are first in the array */
/* (this is contrary to the usual convention of engineers). */
/* The real and imaginary numbers of data are interlaced. */
/* Note that n must be a power of 2 (this is not checked). */
/* Idea for code comes from Numerical Recipes, pp 390 - 395. */

void fft(float *data, int n)
{
    int i, j, k, irev, mask, step, two_step;
    double c, s, angle, dr, di;
    double cos_angle, sin_angle, t;

    if (n < 4)  return;

    irev = n / 2;

    for (i = 2; i < n - 4;  i += 2)
    {
	if (irev > i)
	{
	    SWAP(data[i], data[irev], float);
	    SWAP(data[i+1], data[irev+1], float);
	}

	mask = n / 2;
	while (mask & irev)
	{
	    irev &= ~mask;  mask >>= 1;
	}

	irev |= mask;
    }

    for (step = 2; step < n/2; step = two_step)
    {
	two_step = (step << 1);

	angle = TWOPI / step;

	cos_angle = cos(angle);  sin_angle = sin(angle);
	c = 1.0;  s = 0.0;

	for (i = 0; i < step; i += 2)
	{
	    for (j = i; j < n; j += two_step)
	    {
		k = j + step;
		dr = c*data[k] - s*data[k+1];
		di = s*data[k] + c*data[k+1];

		data[k] = data[j] - dr;
		data[k+1] = data[j+1] - di;
		data[j] += dr;
		data[j+1] += di;
	    }

	    t = cos_angle*c - sin_angle*s;
	    s = sin_angle*c + cos_angle*s;
	    c = t;
	}
    }

    angle = TWOPI / step;  /* step = n/2 now */

    cos_angle = cos(angle);  sin_angle = sin(angle);
    c = 1.0;  s = 0.0;

    for (i = 0; i < step; i += 2)
    {
	k = i + step;
	dr = c*data[k] - s*data[k+1];
	di = s*data[k] + c*data[k+1];

	data[k] = data[i] + dr;
	data[k+1] = data[i+1] + di;
	data[i] -= dr;
	data[i+1] -= di;

	t = cos_angle*c - sin_angle*s;
	s = sin_angle*c + cos_angle*s;
	c = t;
    }
}

/* ifft takes the inverse complex Fourier transform of the array */
/* data, which is of size n real (n/2 complex) points.  It is */
/* assumed that the negative parts are first in the array */
/* (this is contrary to the usual convention of engineers). */
/* The real and imaginary numbers of data are interlaced. */
/* Note that n must be a power of 2 (this is not checked). */
/* Idea for code comes from Numerical Recipes, pp 390 - 395. */

void ifft(float *data, int n)
{
    int i, j, k, irev, mask, step, two_step;
    double c, s, angle, dr, di;
    double cos_angle, sin_angle, t;

    if (n < 4)  return;

    irev = n / 2;

    for (i = 0; i < irev; i++)
	SWAP(data[i], data[i+irev], float);

    for (i = 2; i < n - 4;  i += 2)
    {
	if (irev > i)
	{
	    SWAP(data[i], data[irev], float);
	    SWAP(data[i+1], data[irev+1], float);
	}

	mask = n / 2;
	while (mask & irev)
	{
	    irev &= ~mask;  mask >>= 1;
	}

	irev |= mask;
    }

    for (step = 2; step < n; step = two_step)
    {
	two_step = (step << 1);

	angle = - TWOPI / step;

	cos_angle = cos(angle);  sin_angle = sin(angle);
	c = 1.0;  s = 0.0;

	for (i = 0; i < step; i += 2)
	{
	    for (j = i; j < n; j += two_step)
	    {
		k = j + step;
		dr = c*data[k] - s*data[k+1];
		di = s*data[k] + c*data[k+1];

		data[k] = data[j] - dr;
		data[k+1] = data[j+1] - di;
		data[j] += dr;
		data[j+1] += di;
	    }

	    t = cos_angle*c - sin_angle*s;
	    s = sin_angle*c + cos_angle*s;
	    c = t;
	}
    }
}

void fftn(float *data, int n)
{
    fft(data, n);

    normalise_data(data, n, 2);
}

void ifftn(float *data, int n)
{
    ifft(data, n);

    normalise_data(data, n, 2);
}

/* Note: rft and irft are not normalised, rftn and irftn are approximately. */
/* rft followed by irft leads to scaling the data by n/2 */

/* rft takes the Fourier transform of the real array data, which */
/* is of size n real points, returning the positive frequency half. */
/* Note that n must be a power of 2 (this is not checked). */
/* Idea for code comes from Numerical Recipes, pp 398 - 400. */

void rft(float *data, int n)
{
    int i;
    double c, s, angle, d1r, d1i, d2r, d2i;
    double cos_angle, sin_angle, t, d;

    if (n < 2)  return;

    fft(data, n);

    for (i = 0; i < n/2; i++)
	SWAP(data[i], data[i+n/2], float);

    angle = TWOPI / n;
    c = cos_angle = cos(angle);  s = sin_angle = sin(angle);

    for (i = 2; i < n/2; i += 2)
    {
	d1r = HALF * (data[i] + data[n-i]);
	d1i = HALF * (data[i+1] - data[n-i+1]);
	d2r = HALF * (data[i+1] + data[n-i+1]);
	d2i = - HALF * (data[i] - data[n-i]);

	d = c*d2r - s*d2i;
	data[i] = d1r + d;
	data[n-i] = d1r - d;

	d = c*d2i + s*d2r;
	data[i+1] = d1i + d;
	data[n-i+1] = - d1i + d;

	t = cos_angle*c - sin_angle*s;
	s = sin_angle*c + cos_angle*s;
	c = t;
    }

    d = data[0];
    data[0] = d + data[1];
    data[1] = d - data[1];
}

/* irft takes the inverse Fourier transform of the array data, */
/* which is assumed to be the Fourier transform of a real array. */
/* Note that n must be a power of 2 (this is not checked). */
/* Idea for code comes from Numerical Recipes, pp 398 - 400. */

void irft(float *data, int n)
{
    int i;
    double c, s, angle, d1r, d1i, d2r, d2i;
    double cos_angle, sin_angle, t, d;

    if (n < 2)  return;

    angle = - TWOPI / n;
    c = cos_angle = cos(angle);  s = sin_angle = sin(angle);

    for (i = 2; i < n/2; i += 2)
    {
	d1r = HALF * (data[i] + data[n-i]);
	d1i = HALF * (data[i+1] - data[n-i+1]);
	d2r = - HALF * (data[i+1] + data[n-i+1]);
	d2i = HALF * (data[i] - data[n-i]);

	d = c*d2r - s*d2i;
	data[i] = d1r + d;
	data[n-i] = d1r - d;

	d = c*d2i + s*d2r;
	data[i+1] = d1i + d;
	data[n-i+1] = - d1i + d;

	t = cos_angle*c - sin_angle*s;
	s = sin_angle*c + cos_angle*s;
	c = t;
    }

    d = data[0];
    data[0] = HALF * (d + data[1]);
    data[1] = HALF * (d - data[1]);

    for (i = 0; i < n/2; i++)
	SWAP(data[i], data[i+n/2], float);

    ifft(data, n);
}

/* rftt is the transpose of rft, which is needed for real MEM. */
/* Note that n must be a power of 2 (this is not checked). */
/* And in addition must have n >= 2. */

void rftt(float *data, int n)
{
   data[0] *= 2;
   data[n-1] *= 2;

   irft(data, n);
}

void rftn(float *data, int n)
{
    rft(data, n);

    normalise_data(data, n, 1);
}

void irftn(float *data, int n)
{
    irft(data, n);

    normalise_data(data, n, 4);
}

/* Note: sft and isft (= sft) are not normalised, sftn and isftn are. */
/* sft followed by isft leads to scaling the data by n/2 */

/* sft takes the sine Fourier transform of the array data, */
/* which is of size n real points. */
/* Note that n must be a power of 2 (this is not checked). */
/* Idea for code comes from Numerical Recipes, pp 401 - 407. */

void sft(float *data, int n)
{
    int i;
    double c, s, angle, d1, d2;
    double cos_angle, sin_angle, t, d;

    data[0] = 0;

    if (n < 2)  return;

    angle = PI / n;
    c = cos_angle = cos(angle);  s = sin_angle = sin(angle);

    for (i = 1; i < n/2+1; i++)
    {
	d1 = s * (data[i] + data[n-i]);
	d2 = HALF * (data[i] - data[n-i]);

	data[i] = d1 + d2;
	data[n-i] = d1 - d2;

	t = cos_angle*c - sin_angle*s;
	s = sin_angle*c + cos_angle*s;
	c = t;
    }

    rft(data, n);

    d = 0;
    data[0] *= HALF;
    data[1] = 0;

    for (i = 0; i < n; i += 2)
    {
	d += data[i];
	data[i] = data[i+1];
	data[i+1] = d;
    }
}

void isft(float *data, int n)
{
    sft(data, n);
}

void sftn(float *data, int n)
{
    sft(data, n);

    normalise_data(data, n, 2);
}

void isftn(float *data, int n)
{
    sftn(data, n);
}

/* Note: cft and icft are not normalised, cftn and icftn are approximately. */
/* cft followed by icft leads to scaling the data by n/2 */

/* cft takes the sine Fourier transform of the array data, */
/* which is of size n real points. */
/* Note that n must be a power of 2 (this is not checked). */
/* Idea for code comes from Numerical Recipes, pp 401 - 407. */

void cft(float *data, int n)
{
    int i;
    double c, s, angle, d1, d2;
    double cos_angle, sin_angle, t, d;

    if (n < 2)  return;

    angle = PI / n;
    c = cos_angle = cos(angle);  s = sin_angle = sin(angle);

    d = data[0];

    for (i = 1; i < n/2; i++)
    {
	d1 = HALF * (data[i] + data[n-i]);
	d2 = data[i] - data[n-i];

	data[i] = d1 - s*d2;
	data[n-i] = d1 + s*d2;
	d += c*d2;

	t = cos_angle*c - sin_angle*s;
	s = sin_angle*c + cos_angle*s;
	c = t;
    }

    rft(data, n);

    data[1] = d;

    for (i = 3; i < n; i += 2)
    {
	d += data[i];
	data[i] = d;
    }
}

void icft(float *data, int n)
{
    int i;
    double d, d1, d2;

    if (n < 2)  return;

    cft(data, n);

    d1 = d2 = 0;

    for (i = 0; i < n; i += 2)
    {
	d1 += data[i];
	d2 += data[i+1];
    }

    d = d1 - d2;
    d1 = data[0] - 2.0*d;
    d2 = 2.0*d2/n - d1;

    data[0] = d;
    data[1] -= d2;

    for (i = 2; i < n; i += 2)
    {
	data[i] -= d1;
	data[i+1] -= d2;
    }
}

void cftn(float *data, int n)
{
    cft(data, n);

    normalise_data(data, n, 1);
}

void icftn(float *data, int n)
{
    icft(data, n);

    normalise_data(data, n, 4);
}

/* hft takes the Hilbert Fourier transform of the array data, */
/* which is real and of size n (real) points. */
/* On output data is complex of size 2n (real) points. */
/* Note that n must be a power of 2 (this is not checked). */

void hft(float *data, int n)
{
    int i;

    SCALE_VECTOR(data, data, 2.0/n, n);

    for (i = n-1; i >= 0; i--)
    {
	data[2*i] = data[i];
	data[2*i+1] = 0;
    }

    ifft(data, 2*n);
    data[0] /= 2;

    ZERO_VECTOR(data+n, n);

    fft(data, 2*n);
}
