#include "display_popup.h"

#include "color.h"
#include "display.h"
#include "draw.h"
#include "event_handle.h"
#include "info_popup.h"
#include "levels_func.h"
#include "levels_popup.h"
#include "object_popup.h"
#include "param.h"
#include "region_func.h"
#include "region_popup.h"
#include "timer.h"
#include "track_popup.h"
#include "utility.h"
#include "window.h"

#define  MIN_WIDTH	200
#define  MIN_HEIGHT	200

static Widget display_popup = (Widget) NULL;
static Widget display_area;

static Display *display_display = (Display *) NULL;
static Window display_window;
static GC display_gc = (GC) NULL;
static Pixmap display_pixmap = (Pixmap) NULL;
static Drawable display_drawable;

static int width = MIN_WIDTH;
static int height = MIN_HEIGHT;
static int depth = 1;

static float ax, bx, ay, by;

static Draw_info draw_info;
static Draw_rowcol_info rowcol_info;

static Bool have_valid_display = FALSE;

static Bool have_rectangle = FALSE;

static int begin_rect_x, end_rect_x;
static int begin_rect_y, end_rect_y;

static int cursor_x, cursor_y;

static int translate_x, translate_y;

static Bool rowcol_update = TRUE;

static float rowcol_scale = 1;
static float rowcol_mult = 1.5;

static int background = WHITE;

void update_display_params()
{
    if (display_popup)
    {
	get_widget_size(display_area, &width, &height);
	sprintf(display_width, "%d", width);
	sprintf(display_height, "%d", height);
    }
    else
    {
	if (*display_width)
	    width = atoi(display_width);

	width = MAX(width, MIN_WIDTH);

	if (*display_height)
	    height = atoi(display_height);

	height = MAX(height, MIN_HEIGHT);
    }
}

static void clear_display()
{
    if (*background_color)
    {
	background = get_color_index(background_color);
	set_gc_color(display_display, display_gc, background);
    }
    else
    {
	set_gc_color(display_display, display_gc, WHITE);
    }

    clear_region(display_display, display_pixmap, display_gc,
						0, 0, width, height);
}

static void show_display()
{
    copy_region(display_display, display_pixmap, display_window, display_gc,
						0, 0, width, height, 0, 0);
}

static void start_display(Generic_ptr data)
{
/*
    void create_display_popup(Widget parent);

    if (!display_popup)
	create_display_popup(get_topshell());
*/

    display_drawable = display_pixmap;
    clear_display();
}

static void end_display()
{
    show_display();
    have_rectangle = FALSE;
}

static void start_rowcol(Generic_ptr data)
{
    display_drawable = display_window;
}

static void end_rowcol()
{
}

static void new_display_range(float x0, float y0, float x1, float y1, Bool clip)
{
    ax = width / (x1 - x0);  bx = - ax * x0;
    ay = - height / (y1 - y0);  by = - ay * y1;
	/* y formulas different from x ones because y = 0 on top of screen */
}

#define  CONVERT_X(x)  (ax*(x) + bx)
#define  CONVERT_Y(y)  (ay*(y) + by)

static void do_display_line(float x0, float y0, float x1, float y1)
{
    int w0, w1, h0, h1;

    w0 = CONVERT_X(x0);
    w1 = CONVERT_X(x1);
    h0 = CONVERT_Y(y0);
    h1 = CONVERT_Y(y1);

    display_line(display_display, display_drawable, display_gc, w0, h0, w1, h1);
}

static void do_display_text(String text, float x, float y, float a, float b)
{
    int w, h;

    w = CONVERT_X(x);
    h = CONVERT_Y(y);

    display_text(display_display, display_drawable, display_gc,
							w, h, a, b, text);
}

static void set_display_color(int color)
{
    set_gc_color(display_display, display_gc, color);
}

static void set_display_font(String name, int size)
{
}

static void set_display_line_style(int line_style)
{
}

static void draw_bounding_box()
{
    set_display_color(BLACK);

    display_line(display_display, display_drawable, display_gc,
					0, 0, width-1, 0);
    display_line(display_display, display_drawable, display_gc,
					width-1, 0, width-1, height-1);
    display_line(display_display, display_drawable, display_gc,
					width-1, height-1, 0, height-1);
    display_line(display_display, display_drawable, display_gc,
					0, height-1, 0, 0);
}

static void do_display()
{
    Line error_msg;
    Status status;
    Timer_funcs *t_funcs;
    static Draw_funcs draw_funcs = { SCREEN_DISPLAY, (Generic_ptr) NULL,
	start_display, end_display, new_display_range,
	do_display_line, do_display_text, set_display_color,
	set_display_font, set_display_line_style };
    static Timer_funcs timer_funcs = { start_timer, update_timer, stop_timer };
    static Timer_funcs no_timer_funcs =
			{ start_no_timer, update_no_timer, stop_no_timer };

    have_valid_display = FALSE;

    if (use_timer())
	t_funcs = &timer_funcs;
    else
	t_funcs = &no_timer_funcs;

    if ((status = do_drawing(&draw_info, &draw_funcs, t_funcs,
							error_msg)) != OK)
    {
	if (status == ERROR)
	{
	    ERROR_AND_RETURN(error_msg);
	}
	else
	{
	    return;
	}
    }

    draw_bounding_box(); /* needed for raster dump */

    have_valid_display = TRUE;

    flush_display(display_display);

    do_tracking_drawing();
}

static void display_expose_callback(Widget widget, Callback_ptr data,
							Callback_ptr cbs)
{
    show_display();
}

static void display_resize_callback(Widget widget, Callback_ptr data,
							Callback_ptr cbs)
{
    int old_width = width, old_height = height;

    update_display_params();

    if (!display_pixmap || 
	(have_valid_display &&
			((old_width != width) || (old_height != height))))
    {
	free_pixmap(display_display, display_pixmap);
	display_pixmap = create_pixmap(display_display, display_window,
						width, height, depth);
    }

    if (have_valid_display)
    {
	do_display();
    }
    else
    {
    	clear_display();
	show_display();
    }
}

static void do_rows_and_cols()
{
    Line error_msg;
    static Draw_funcs draw_funcs = { SCREEN_DISPLAY, (Generic_ptr) NULL,
	start_rowcol, end_rowcol, new_display_range,
	do_display_line, do_display_text, set_display_color,
	set_display_font, set_display_line_style };
    int fold_type[DISPLAY_DIM], flip_type[DISPLAY_DIM];

    if (rowcol_update)
    {
    	if (region_apply(&(rowcol_info.ref_type), rowcol_info.lower,
		rowcol_info.upper, fold_type, flip_type, error_msg) == ERROR)
	    return;

    	if (objects_apply(&(rowcol_info.ndata_sets), &(rowcol_info.data_info),
				&(rowcol_info.global_info), error_msg) == ERROR)
	    return;

	rowcol_info.w = width;
	rowcol_info.h = height;

	rowcol_update = FALSE;
    }

    rowcol_info.x = cursor_x;
    rowcol_info.y = height - 1 - cursor_y;
    rowcol_info.scale = rowcol_scale;

    draw_rows_and_cols(&rowcol_info, &draw_funcs);
}

static void do_rectangle(int color)
{
    if ((begin_rect_x == end_rect_x) && (begin_rect_y == end_rect_y))
	return;

    set_gc_color(display_display, display_gc, color);
/*
    set_gc_function(display, display_gc, GXinvert);
*/
    display_rectangle(display_display, display_drawable, display_gc, 
			begin_rect_x, begin_rect_y, end_rect_x, end_rect_y);
/*
    set_gc_function(display, display_gc, GXcopy);
*/

    have_rectangle = TRUE;
}

static void do_xor_rectangle(int color)
{
    set_gc_function(display_display, display_gc, GXxor);
    display_drawable = display_pixmap;

    do_rectangle(color);

    set_gc_function(display_display, display_gc, GXcopy);
    display_drawable = display_window;
}

static void rectangle_limits(int *begin, int *end)
{
    begin[0] = MIN(begin_rect_x, end_rect_x);
    end[0] = MAX(begin_rect_x, end_rect_x);
    begin[1] = height - 1 - MAX(begin_rect_y, end_rect_y);
    end[1] = height - 1 - MIN(begin_rect_y, end_rect_y);
}

static Bool inside_rectangle(int x, int y, int *begin, int *end)
{
    rectangle_limits(begin, end);
    y = height - 1 - y;

    if (x < begin[0])  return  FALSE;
    if (x > end[0])  return  FALSE;
    if (y < begin[1])  return  FALSE;
    if (y > end[1])  return  FALSE;

    return  TRUE;
}

Bool rectangle_displayed(int *size, int *begin, int *end)
{
    if (!have_rectangle)
	return  FALSE;

    size[0] = width;  size[1] = height;
    rectangle_limits(begin, end);

    return  TRUE;
}

static void button_press(Button button, int x, int y, Time time)
{
    int original[DISPLAY_DIM], begin[DISPLAY_DIM], end[DISPLAY_DIM];
    int point[DISPLAY_DIM], size[DISPLAY_DIM];
    Line error_msg;
    static Print_funcs print_funcs = { start_info, print_info, end_info };

    if (button == Button1)
    {
	if (have_rectangle)
	{
	    original[0] = width;  original[1] = height;

	    if (inside_rectangle(x, y, begin, end))
		(void) change_region(original, begin, end, EXPAND_REGION,
								error_msg);
	    else
		(void) change_region(original, begin, end, CONTRACT_REGION,
								error_msg);

    	    rowcol_update = TRUE;
	    have_rectangle = FALSE;

	    cursor_x = x;  cursor_y = y;
	    do_rows_and_cols();
	}

	translate_x = x;  translate_y = y;
    }
    else if (button == Button2)
    {
	if (have_rectangle)
	{
	    do_xor_rectangle(WHITE);
	}

	show_display();
	have_rectangle = FALSE;
	begin_rect_x = end_rect_x = x;
	begin_rect_y = end_rect_y = y;
    }
    else if (button == Button3)
    {
	size[0] = width;  size[1] = height;
	point[0] = x;  point[1] = height - 1 - y;

	(void) find_point_stats(&print_funcs, size, point, error_msg);
    }
}

static void button_release(Button button, int x, int y, Time time)
{
    int original[DISPLAY_DIM], translation[DISPLAY_DIM];
    Line error_msg;

    if (button == Button1)
    {
	if ((x != translate_x) || (y != translate_y))
	{
	    original[0] = width;  original[1] = height;
	    translation[0] = translate_x - x;
	    translation[1] = y - translate_y;

	    (void) move_region(original, translation, error_msg);

    	    rowcol_update = TRUE;

	    cursor_x = x;  cursor_y = y;
	    do_rows_and_cols();
	}
    }
    else if (button == Button2)
    {
	if (have_rectangle)
	{
	    end_rect_x = x;  end_rect_y = y;

	    do_xor_rectangle(WHITE);
	    show_display();
	}
    }
}

static void button_motion(Button button, int x, int y, Time time)
{
    show_display();

    if (button == Button1)
    {
    }
    else if (button == Button2)
    {
	end_rect_x = x;  end_rect_y = y;

	if (background == BLACK)
	    do_rectangle(WHITE);
	else
	    do_rectangle(BLACK);
    }

    if ((x < 0) || (x > width))  return;
    if ((y < 0) || (y > height))  return;

    cursor_x = x;  cursor_y = y;
    do_rows_and_cols();
}

static void button_enter(Button button, int x, int y, Time time)
{
    rowcol_update = TRUE;
}

static void button_leave(Button button, int x, int y, Time time)
{
    show_display();
}

static void key_press(Key_symbol key, int x, int y, Time time)
{
    if (key == UP_KEY)
	rowcol_scale *= rowcol_mult;
    else if (key == DOWN_KEY)
	rowcol_scale /= rowcol_mult;
    else
	return;

    show_display();
    do_rows_and_cols();
}

static void quit_callback(Widget widget, Callback_ptr data, Callback_ptr cbs)
{
    free_pixmap(display_display, display_pixmap);
    free_gc(display_display, display_gc);
    destroy_widget(display_popup);
    display_popup = (Widget) NULL;
    display_gc = (GC) NULL;
}

static Event_handlers handlers =
{   button_press, button_release, button_motion,
    button_enter, button_leave, key_press, no_key_handler   };

static void create_display_popup(Widget parent)
{
    Drawing_area_info drawing_info;
    static Event_handle handle =
    { ButtonPressMask | ButtonReleaseMask | PointerMotionMask |
	EnterWindowMask | LeaveWindowMask | KeyPressMask, &handlers };

    update_display_params();

    display_popup = create_iconifiable_popup(parent, "2D Display");
    CHECK_WIDGET_WARNING(display_popup);
    change_delete_protocol(display_popup, quit_callback);

    drawing_info.width = width;
    drawing_info.height = height;
    drawing_info.expose = display_expose_callback;
    drawing_info.resize = display_resize_callback;

    display_area = create_drawing(display_popup, &drawing_info);
    CHECK_WIDGET_DESTROY_WARNING(display_area, display_popup);

    realize_widget(display_popup);

    add_event_handler(display_area, &handle);

    display_display = WIDGET_DISPLAY(display_area);
    initialize_display(display_display);
    initialize_font(display_display);

    depth = DISPLAY_DEPTH(display_display);

    display_window = WIDGET_WINDOW(display_area);
    display_pixmap = create_pixmap(display_display, display_window,
						width, height, depth);
    display_gc = create_gc(display_display, display_window);

    clear_display();
}

void display_popup_callback(Widget parent, Callback_ptr data, Callback_ptr cbs)
{
    if (!display_popup)
        create_display_popup(parent);

    if (display_popup)
    {
	popup(display_area);

	if (have_valid_display)
	    do_display_drawing();
    }
}

void do_display_drawing()
{
    Line error_msg;

    have_valid_display = FALSE;

    if (levels_apply(&(draw_info.nlevels), &(draw_info.levels),
						error_msg) == ERROR)
	ERROR_AND_RETURN(error_msg);

    if (region_apply(&(draw_info.ref_type), draw_info.lower,
		draw_info.upper, draw_info.fold_type,
		draw_info.flip_type, error_msg) == ERROR)
	ERROR_AND_RETURN(error_msg);

    if (objects_apply(&(draw_info.ndata_sets), &(draw_info.data_info),
				&(draw_info.global_info), error_msg) == ERROR)
	ERROR_AND_RETURN(error_msg);

    if (!display_popup)
	create_display_popup(get_topshell());

    popup(display_area);

    do_display();
}

#ifdef USE_XPM
Status do_display_dump(String file, String error_msg)
{
    XpmAttributes attributes;

    attributes.valuemask = 0;

    do_display_drawing();

    if (have_valid_display)
	(void) XpmWriteFileFromPixmap(display_display, file,
			display_pixmap, (Pixmap) NULL, &attributes);

    return  OK;
}
#endif /* USE_XPM */

Status display_popup_command(String value, Generic_ptr data, String error_msg)
{
    display_popup_callback(get_topshell(), (Callback_ptr) NULL,
							(Callback_ptr) NULL);

    return  OK;
}
