#include "baseline_popup.h"

#include "baseline.hlp"

#include "callback.h"
#include "param.h"
#include "baseline.h"
#include "baseline_func.h"
#include "slice.h"
#include "slice_popup.h"
#include "window.h"

static Widget baseline_popup = (Widget) NULL;
static Widget baseline_form;

static Widget baseline_boxes[NBASELINE_TYPES];
static Radiobox_item baseline_items[] =
{
    { "no baseline", NO_CALLBACK, NO_DATA, baseline_boxes + NO_BASELINE },
    { "automatic", NO_CALLBACK, NO_DATA, baseline_boxes + AUTO_BASELINE },
    { "hand-picked", NO_CALLBACK, NO_DATA, baseline_boxes + PICK_BASELINE }
};

static int baseline_type = NO_BASELINE;
static int nbaseline_types = ARRAY_SIZE(baseline_boxes);

static Widget base_fit_boxes[NBASE_FIT_TYPES];
static Radiobox_item base_fit_items[] =
{
    { "constant", NO_CALLBACK, NO_DATA, base_fit_boxes + CONST_BASELINE },
    { "polynomial", NO_CALLBACK, NO_DATA, base_fit_boxes + POLY_BASELINE },
    { "trigonometric", NO_CALLBACK, NO_DATA, base_fit_boxes + TRIG_BASELINE }
};

static int base_fit_type = CONST_BASELINE;
static int nbase_fit_types = ARRAY_SIZE(base_fit_boxes);

static Widget baseline_field;

static Textfield_item textfield_items[] =
{
    { "baseline file: ", baseline_file, NO_CALLBACK, NO_DATA, &baseline_field }
};

static int ntexts = ARRAY_SIZE(textfield_items);

#define  HALF_WIDTH		0
#define  DEGREE_ORDER		1
#define  NBUTTON_TEXTS		2

static int nbutton_texts = NBUTTON_TEXTS;
static Textfield_item button_text[NBUTTON_TEXTS];
static Widget button_field[NBUTTON_TEXTS];
static char *button_param[] = { half_width, degree_order };
static char *button_label[] = { "half width", "degree/order" };

static char next_label[] = "next";
static char previous_label[] = "previous";

static Bool have_baseline_apply = FALSE;

static void set_baseline_fields()
{
    set_textfields(textfield_items, ntexts);
    set_textfields(button_text, nbutton_texts);
}

void update_baseline_params()
{
    if (baseline_popup)
    {
	baseline_type = get_radio_state(baseline_boxes, nbaseline_types);
	base_fit_type = get_radio_state(base_fit_boxes, nbase_fit_types);
	get_textfields(textfield_items, ntexts);
	get_textfields(button_text, nbutton_texts);

	sprintf(baseline_used, "%d", baseline_type);
    }
    else
    {
	if (*baseline_used)
	    baseline_type = atoi(baseline_used);

	if (*base_fit_used)
	    base_fit_type = atoi(base_fit_used);

	set_baseline_params();
    }
}

Status baseline_apply(Slice_info *info, String error_msg)
{
    update_baseline_params();

    have_baseline_apply = FALSE;

    CHECK_STATUS(baseline_apply_func(info, error_msg));

    have_baseline_apply = TRUE;

    return  OK;
}

static int find_button_text(Widget *data)
{
    int n;

    n = (int) (data - button_field);

    if ((n < 0) || (n >= nbutton_texts))
	n = -1;

    return  n;
}

static void next_command(int n)
{
    if (n == HALF_WIDTH)
	baseline_half_width_next_func();
    else /* (n == DEGREE_ORDER) */
	baseline_degree_order_next_func();

    if (baseline_popup)
	set_textfields(button_text+n, 1);

    if (have_baseline_apply)
	do_slice_drawing(TRUE);
}

static void next_callback(Widget widget, Callback_ptr data, Callback_ptr cbs)
{
    int n;

    if ((n = find_button_text((Widget *) data)) == -1)
	ERROR_AND_RETURN("should not be here in next_callback");

    get_textfields(button_text+n, 1);

    next_command(n);
}

Status baseline_half_width_next_command(String value, Generic_ptr data,
							String error_msg)
{
    next_command(HALF_WIDTH);

    return  OK;
}

Status baseline_degree_order_next_command(String value, Generic_ptr data,
							String error_msg)
{
    next_command(DEGREE_ORDER);

    return  OK;
}

static void previous_command(int n)
{
    if (n == HALF_WIDTH)
	baseline_half_width_previous_func();
    else /* (n == DEGREE_ORDER) */
	baseline_degree_order_previous_func();

    if (baseline_popup)
	set_textfields(button_text+n, 1);

    if (have_baseline_apply)
	do_slice_drawing(TRUE);
}

static void previous_callback(Widget widget, Callback_ptr data,
							Callback_ptr cbs)
{
    int n;

    if ((n = find_button_text((Widget *) data)) == -1)
	ERROR_AND_RETURN("should not be here in previous_callback");

    get_textfields(button_text+n, 1);

    previous_command(n);
}

Status baseline_half_width_previous_command(String value, Generic_ptr data,
							String error_msg)
{
    previous_command(HALF_WIDTH);

    return  OK;
}

Status baseline_degree_order_previous_command(String value, Generic_ptr data,
							String error_msg)
{
    previous_command(DEGREE_ORDER);

    return  OK;
}

static void base_callback(Widget widget, Callback_ptr data, Callback_ptr cbs)
{
    if (!toggle_being_set(cbs))
	return;

    if (have_baseline_apply)
	do_slice_drawing(TRUE);
}

static Status apply_command(String error_msg)
{
    do_slice_drawing(TRUE);

    return  OK;
}

static void apply_callback(Widget widget, Callback_ptr data, Callback_ptr cbs)
{
    Line error_msg;

    if (apply_command(error_msg) == ERROR)
	ERROR_AND_RETURN(error_msg);
}

Status baseline_apply_command(String value, Generic_ptr data, String error_msg)
{
    CHECK_STATUS(apply_command(error_msg));

    return  OK;
}

static Status load_command(String error_msg)
{
    CHECK_STATUS(baseline_load_func(error_msg));

    if (have_baseline_apply)
	do_slice_drawing(TRUE);

    return  OK;
}

static void load_callback(Widget widget, Callback_ptr data, Callback_ptr cbs)
{
    Line error_msg;

    if (load_command(error_msg) == ERROR)
	ERROR_AND_RETURN(error_msg);
}

Status baseline_load_command(String value, Generic_ptr data, String error_msg)
{
    CHECK_STATUS(load_command(error_msg));

    return  OK;
}

static Status save_command(String error_msg)
{
    CHECK_STATUS(baseline_save_func(error_msg));

    return  OK;
}

static void save_callback(Widget widget, Callback_ptr data, Callback_ptr cbs)
{
    Line error_msg;

    if (save_command(error_msg) == ERROR)
	ERROR_AND_RETURN(error_msg);
}

Status baseline_save_command(String value, Generic_ptr data, String error_msg)
{
    CHECK_STATUS(save_command(error_msg));

    return  OK;
}

static void clear_command()
{
    baseline_clear_func();

    if (have_baseline_apply)
	do_slice_drawing(TRUE);
}

static void clear_callback(Widget widget, Callback_ptr data, Callback_ptr cbs)
{
    clear_command();
}

Status baseline_clear_command(String value, Generic_ptr data, String error_msg)
{
    clear_command();

    return  OK;
}

static void create_baseline_popup(Widget parent)
{
    int i;
    Widget buttons, buttons2, radio, texts, separator, label;
    Widget text_button, previous_widget;
    Text_buttons_info text_buttons_info;
    Button_item text_button_items[2];
    static Button_item button_items[] =
    {
	{ "load",	load_callback,		NO_DATA },
	{ "save",	save_callback,		NO_DATA },
	{ "clear",	clear_callback,		NO_DATA }
    };
    static int nbuttons = ARRAY_SIZE(button_items);
    Apply_dismiss_help_info adh_info;
    char apply_label[] = "apply";

    update_baseline_params();

    baseline_popup = create_popup(parent, "Baseline Correction");
    CHECK_WIDGET_WARNING(baseline_popup);

    baseline_form = create_form(baseline_popup);
    CHECK_WIDGET_DESTROY_WARNING(baseline_form, baseline_popup);

    label = create_label(baseline_form, "Baseline: ");
    CHECK_WIDGET_DESTROY_WARNING(label, baseline_popup);
    attachments(label, FORM_ATTACH, NO_ATTACH, FORM_ATTACH, NO_ATTACH);

    for (i = 0; i < nbaseline_types; i++)
	baseline_items[i].callback = base_callback;

    radio = create_radiobox(baseline_form, baseline_items,
				nbaseline_types, baseline_type, HORIZONTAL);
    CHECK_WIDGET_DESTROY_WARNING(radio, baseline_popup);
    attachments(radio, FORM_ATTACH, NO_ATTACH, label, FORM_ATTACH);

    texts = create_textfields(baseline_form, textfield_items, ntexts);
    CHECK_WIDGET_DESTROY_WARNING(texts, baseline_popup);
    attachments(texts, radio, NO_ATTACH, FORM_ATTACH, FORM_ATTACH);

    separator = create_separator(baseline_form);
    CHECK_WIDGET_DESTROY_WARNING(separator, baseline_popup);
    attachments(separator, texts, NO_ATTACH, FORM_ATTACH, FORM_ATTACH);

    label = create_label(baseline_form, "Fitting: ");
    CHECK_WIDGET_DESTROY_WARNING(label, baseline_popup);
    attachments(label, separator, NO_ATTACH, FORM_ATTACH, NO_ATTACH);

    for (i = 0; i < nbase_fit_types; i++)
	base_fit_items[i].callback = base_callback;

    radio = create_radiobox(baseline_form, base_fit_items,
				nbase_fit_types, base_fit_type, HORIZONTAL);
    CHECK_WIDGET_DESTROY_WARNING(radio, baseline_popup);
    attachments(radio, separator, NO_ATTACH, label, FORM_ATTACH);

    text_buttons_info.nbuttons = 2;
    text_buttons_info.buttons = text_button_items;

    text_button_items[0].callback = next_callback;
    text_button_items[1].callback = previous_callback;

    previous_widget = radio;

    for (i = 0; i < nbutton_texts; i++)
    {
	button_text[i].value = button_param[i];
	button_text[i].callback = (Callback_proc) NULL;
	button_text[i].data = (Callback_ptr) NULL;
	button_text[i].field = &button_field[i];
	button_text[i].label = button_label[i];
	text_buttons_info.text = button_text + i;

	text_button_items[0].label = next_label;
	text_button_items[0].data = (Generic_ptr) (button_field + i);

	text_button_items[1].label = previous_label;
	text_button_items[1].data = (Generic_ptr) (button_field + i);

	text_button = create_text_buttons(baseline_form, &text_buttons_info);
	CHECK_WIDGET_DESTROY_WARNING(text_button, baseline_popup);
	attachments(text_button, previous_widget, NO_ATTACH, NO_ATTACH,
								FORM_ATTACH);

	previous_widget = text_button;
    }

    separator = create_separator(baseline_form);
    CHECK_WIDGET_DESTROY_WARNING(separator, baseline_popup);
    attachments(separator, previous_widget, NO_ATTACH, FORM_ATTACH,
								FORM_ATTACH);

    buttons = create_horizontal_buttons(baseline_form, button_items, nbuttons);
    CHECK_WIDGET_DESTROY_WARNING(buttons, baseline_popup);
    attachments(buttons, separator, NO_ATTACH, FORM_ATTACH, FORM_ATTACH);
    offsets(buttons, UNIT_OFFSET, UNIT_OFFSET, NO_OFFSET, NO_OFFSET);

    adh_info.apply_label = apply_label;
    adh_info.apply_callback = apply_callback;
    adh_info.dismiss_form = baseline_form;
    adh_info.help_message = baseline_help;

    buttons2 = create_apply_dismiss_help(baseline_form, &adh_info);
    CHECK_WIDGET_DESTROY_WARNING(buttons2, baseline_popup);
    attachments(buttons2, buttons, FORM_ATTACH, FORM_ATTACH, FORM_ATTACH);
    offsets(buttons2, UNIT_OFFSET, UNIT_OFFSET, NO_OFFSET, NO_OFFSET);

    set_baseline_fields();

    manage_widget(baseline_form);
}

void baseline_popup_callback(Widget parent, Callback_ptr data, Callback_ptr cbs)
{
    if (!baseline_popup)
        create_baseline_popup(parent);

    if (baseline_popup)
        popup(baseline_form);
}

Status baseline_popup_command(String value, Generic_ptr data, String error_msg)
{
    baseline_popup_callback(get_topshell(), (Callback_ptr) NULL,
							(Callback_ptr) NULL);

    return  OK;
}
